/* GIMP PSD Plug-in
 * Copyright 2007 by John Marshall
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


/* ----- Known Image Resource Block Types -----
  All image resources not otherwise handled, including unknown types
  are added as image parasites. 
  The data is attached as-is from the file (i.e. in big endian order).

  PSD_PS2_IMAGE_INFO    = 1000,    Dropped      * Obsolete - ps 2.0 image info *
  PSD_MAC_PRINT_INFO    = 1001,    PS Only      * Optional - Mac print manager print info record *
  PSD_PS2_COLOR_TAB     = 1002,    Dropped      * Obsolete - ps 2.0 indexed colour table *
  PSD_RESN_INFO         = 1005,    Loaded       * ResolutionInfo structure *
  PSD_ALPHA_NAMES       = 1006,    Loaded       * Alpha channel names *
  PSD_DISPLAY_INFO      = 1007,    Loaded       * DisplayInfo structure *
  PSD_CAPTION           = 1008,    Loaded       * Optional - Caption string *
  PSD_BORDER_INFO       = 1009,                 * Border info *
  PSD_BACKGROUND_COL    = 1010,                 * Background colour *
  PSD_PRINT_FLAGS       = 1011,                 * Print flags *
  PSD_GREY_HALFTONE     = 1012,                 * Greyscale and multichannel halftoning info *
  PSD_COLOR_HALFTONE    = 1013,                 * Colour halftoning info *
  PSD_DUO_HALFTONE      = 1014,                 * Duotone halftoning info *
  PSD_GREY_XFER         = 1015,                 * Greyscale and multichannel transfer functions *
  PSD_COLOR_XFER        = 1016,                 * Colour transfer functions *
  PSD_DUO_XFER          = 1017,                 * Duotone transfer functions *
  PSD_DUO_IMAGE_INFO    = 1018,                 * Duotone image information *
  PSD_EFFECTIVE_BW      = 1019,                 * Effective black & white values for dot range *
  PSD_OBSOLETE_01       = 1020,    Dropped      * Obsolete *
  PSD_EPS_OPT           = 1021,                 * EPS options *
  PSD_QUICK_MASK        = 1022,                 * Quick mask info *
  PSD_OBSOLETE_02       = 1023,    Dropped      * Obsolete *
  PSD_LAYER_STATE       = 1024,    Loaded       * Layer state info *
  PSD_WORKING_PATH      = 1025,                 * Working path (not saved) *
  PSD_LAYER_GROUP       = 1026,                 * Layers group info *
  PSD_OBSOLETE_03       = 1027,    Dropped      * Obsolete *
  PSD_IPTC_NAA_DATA     = 1028,                 * IPTC-NAA record (IMV4.pdf) *
  PSD_IMAGE_MODE_RAW    = 1029,                 * Image mode for raw format files *
  PSD_JPEG_QUAL         = 1030,    PS Only      * JPEG quality *
  PSD_GRID_GUIDE        = 1032,    Loaded       * Grid & guide info *
  PSD_THUMB_RES         = 1033,    Special      * Thumbnail resource *
  PSD_COPYRIGHT_FLG     = 1034,                 * Copyright flag *
  PSD_URL               = 1035,                 * URL string *
  PSD_THUMB_RES2        = 1036,    Special      * Thumbnail resource *
  PSD_GLOBAL_ANGLE      = 1037,                 * Global angle *
  PSD_COLOR_SAMPLER     = 1038,                 * Colour samplers resource *
  PSD_ICC_PROFILE       = 1039,    Loaded       * ICC Profile *
  PSD_WATERMARK         = 1040,                 * Watermark *
  PSD_ICC_UNTAGGED      = 1041,                 * Do not use ICC profile flag *
  PSD_EFFECTS_VISIBLE   = 1042,                 * Show  hide all effects layers *
  PSD_SPOT_HALFTONE     = 1043,                 * Spot halftone *
  PSD_DOC_IDS           = 1044,                 * Document specific IDs *
  PSD_ALPHA_NAMES_UNI   = 1045,    Loaded       * Unicode alpha names *
  PSD_IDX_COL_TAB_CNT   = 1046,                 * Indexed colour table count *
  PSD_IDX_TRANSPARENT   = 1047,                 * Index of transparent colour (if any) *
  PSD_GLOBAL_ALT        = 1049,                 * Global altitude *
  PSD_SLICES            = 1050,                 * Slices *
  PSD_WORKFLOW_URL_UNI  = 1051,                 * Workflow URL - Unicode string *
  PSD_JUMP_TO_XPEP      = 1052,                 * Jump to XPEP (?) *
  PSD_ALPHA_ID          = 1053,    Loaded       * Alpha IDs *
  PSD_URL_LIST_UNI      = 1054,                 * URL list - unicode *
  PSD_VERSION_INFO      = 1057,                 * Version info *
  PSD_EXIF_DATA         = 1058,    Loaded       * Exif data block *
  PSD_XMP_DATA          = 1060,    Loaded       * XMP data block *
  PSD_PATH_INFO_FIRST   = 2000,    Loaded       * First path info block *
  PSD_PATH_INFO_LAST    = 2998,    Loaded       * Last path info block *
  PSD_CLIPPING_PATH     = 2999,                 * Name of clipping path *
  PSD_PRINT_FLAGS_2     = 10000                 * Print flags *
  
*/

#include "config.h"

#include <string.h>
#include <errno.h>

#include <glib/gstdio.h>
#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>

#include "psd.h"
#include "psd_util.h"
#include "psd_image_res.h"

#ifdef HAVE_EXIF
#include <libexif/exif-content.h>
#include <libexif/exif-data.h>
#include <libexif/exif-utils.h>

#include "gimpexif.h"
#endif /* HAVE_EXIF */

#include "plug-ins/metadata/xmp-model.h"
#include "libgimp/stdplugins-intl.h"


/*  Local function prototypes  */
static gint     load_resource_unknown           (const PSDimageres      *res_a,
                                                 const gint32           image_id,
                                                 FILE                   *f);

static gint     load_resource_ps_only           (const PSDimageres      *res_a,
                                                 const gint32           image_id,
                                                 FILE                   *f);

static gint     load_resource_1005              (const PSDimageres      *res_a,
                                                 const gint32           image_id,
                                                 FILE                   *f);

static gint     load_resource_1006              (const PSDimageres      *res_a,
                                                 const gint32           image_id,
                                                 FILE                   *f);

static gint     load_resource_1007              (const PSDimageres      *res_a,
                                                 const gint32           image_id,
                                                 FILE                   *f);

static gint     load_resource_1008              (const PSDimageres      *res_a,
                                                 const gint32           image_id,
                                                 FILE                   *f);

static gint     load_resource_1024              (const PSDimageres      *res_a,
                                                 const gint32           image_id,
                                                 FILE                   *f);

static gint     load_resource_1032              (const PSDimageres      *res_a,
                                                 const gint32           image_id,
                                                 FILE                   *f);

static gint     load_resource_1039              (const PSDimageres      *res_a,
                                                 const gint32           image_id,
                                                 FILE                   *f);

static gint     load_resource_1045              (const PSDimageres      *res_a,
                                                 const gint32           image_id,
                                                 FILE                   *f);

static gint     load_resource_1053              (const PSDimageres      *res_a,
                                                 const gint32           image_id,
                                                 FILE                   *f);

static gint     load_resource_1058              (const PSDimageres      *res_a,
                                                 const gint32           image_id,
                                                 FILE                   *f);

static gint     load_resource_1060              (const PSDimageres      *res_a,
                                                 const gint32           image_id,
                                                 FILE                   *f);

static gint     load_resource_2000              (const PSDimageres      *res_a,
                                                 const gint32           image_id,
                                                 FILE                   *f);

/* Public Functions */
gint
get_image_resource_header (PSDimageres *res_a, FILE *f)
{
  gssize               read_len,
                       write_len;
  gchar                *name;

  if (fread (&res_a->type, 4, 1, f) < 1
      || fread (&res_a->id, 2, 1, f) < 1)
    {
      g_warning (_("Error reading image resource block header"));
      return -1;
    }
  res_a->id = GUINT16_FROM_BE (res_a->id);
  name = fread_pascal_string (&read_len, &write_len, 2, f);
  if (name != NULL)
    g_strlcpy (res_a->name, name, write_len + 1); 
  else
    res_a->name[0] = 0;
  g_free (name); 
  if (fread (&res_a->data_len, 4, 1, f) < 1)
    {
      g_warning (_("Error image resource block length"));
      return -1;
    }
  res_a->data_len = GUINT32_FROM_BE (res_a->data_len);
  res_a->data_start = ftell (f);

  IFDBG(2) g_debug ("Type: %.4s, id: %d, start: %d, len: %d",
                        res_a->type, res_a->id, res_a->data_start, res_a->data_len);

  return 0;
}

gint
load_image_resource (PSDimageres *res_a, const gint32 image_id, FILE *f)
{
  gint  pad;

  /* Set file position to start of image resource data block */
  if (fseek (f, res_a->data_start, SEEK_SET) < 0)
    {
      g_warning (_("Error setting file position"));
      return -1;
    }

   /* Process image resource blocks */
  if (memcmp (res_a->type, "8BIM", 4) != 0 &&
      memcmp (res_a->type, "MeSa", 4) !=0)
      g_message (_("Unknown image resource type signature %.4s"), res_a->type);

  switch (res_a->id)
    {
      case PSD_PS2_IMAGE_INFO:
      case PSD_PS2_COLOR_TAB:
      case PSD_OBSOLETE_01:      
      case PSD_OBSOLETE_02:      
      case PSD_OBSOLETE_03:      
        /* Drop obsolete image resource blocks */
        IFDBG(2) g_debug ("Obsolete image resource block: %d",
                           res_a->id);
        break;

      case PSD_THUMB_RES:      
      case PSD_THUMB_RES2:
        /* Drop thumbnails from standard file load */
        IFDBG(2) g_debug ("Thumbnail resource block: %d",
                           res_a->id);
        break;

      case PSD_MAC_PRINT_INFO:
      case PSD_JPEG_QUAL:
        /* Save photoshop resources with no meaning for GIMP
          as image parasites */
        load_resource_ps_only (res_a, image_id, f);
        break;

      case PSD_RESN_INFO:
        load_resource_1005 (res_a, image_id, f);
        break;
        
      case PSD_ALPHA_NAMES:
        load_resource_1006 (res_a, image_id, f);
        break;

      case PSD_DISPLAY_INFO:
        load_resource_1007 (res_a, image_id, f);
        break;

      case PSD_CAPTION:
        load_resource_1008 (res_a, image_id, f);
        break;

      case PSD_LAYER_STATE:
        load_resource_1024 (res_a, image_id, f);
        break;

      case PSD_GRID_GUIDE:
        load_resource_1032 (res_a, image_id, f);
        break;

      case PSD_ICC_PROFILE:
        load_resource_1039 (res_a, image_id, f);
        break;

      case PSD_ALPHA_NAMES_UNI:
        load_resource_1045 (res_a, image_id, f);
        break;

      case PSD_ALPHA_ID:
        load_resource_1053 (res_a, image_id, f);
        break;

      case PSD_EXIF_DATA:
        load_resource_1058 (res_a, image_id, f);
        break;

      case PSD_XMP_DATA:
        load_resource_1060 (res_a, image_id, f);
        break;

      default:
        if (res_a->id >= 2000 &&
            res_a->id <  2999)
          load_resource_2000 (res_a, image_id, f);
        else
          load_resource_unknown (res_a, image_id, f);
    }

  /* Image blocks are null padded to even length */
  if (res_a->data_len % 2 == 0)
    pad = 0;
  else
    pad = 1;

  /* Set file position to end of image resource block */
  if (fseek (f, res_a->data_start + res_a->data_len + pad, SEEK_SET) < 0)
    {
      g_warning (_("Error setting file position"));
      return -1;
    }
  
  return 0;
}

gint
load_thumbnail_resource (PSDimageres *res_a, const gint32 image_id, FILE *f)
{
  gint  pad;

  /* Set file position to start of image resource data block */
  if (fseek (f, res_a->data_start, SEEK_SET) < 0)
    {
      g_warning (_("Error setting file position"));
      return -1;
    }

   /* Process image resource blocks */
 if (res_a->id == PSD_THUMB_RES
     || res_a->id == PSD_THUMB_RES2)      
   {
        /* Load thumbnails from standard file load */
        IFDBG(2) g_debug ("Process thumbnail resource block: %d",
                           res_a->id);
  }

  /* Image blocks are null padded to even length */
  if (res_a->data_len % 2 == 0)
    pad = 0;
  else
    pad = 1;

  /* Set file position to end of image resource block */
  if (fseek (f, res_a->data_start + res_a->data_len + pad, SEEK_SET) < 0)
    {
      g_warning (_("Error setting file position"));
      return -1;
    }
  
  return 0;
}

/* Private Functions */

static gint
load_resource_unknown (const PSDimageres *res_a, const gint32 image_id, FILE *f)
{
  /* Unknown image resources attached as parasites to re-save later */
  GimpParasite  *parasite;
  gchar         *data;
  gchar         *name;
  
  IFDBG(2) g_debug ("Process unknown image resource block: %d", res_a->id);

  data = g_malloc (res_a->data_len);
  if (fread (data, 4, 1, f) < 1)
    {
      g_warning (_("Error reading image resource block %d"), res_a->id);
      return -1;
    }

  name = g_strdup_printf ("psd-image-resource-%.4s-%.4x", 
                            res_a->type, res_a->id);
  IFDBG(2) g_debug ("Parasite name: %s", name);

  parasite = gimp_parasite_new (name, 0, res_a->data_len, data);
  g_free (name);
  g_free (data);
  gimp_image_parasite_attach (image_id, parasite);
  gimp_parasite_free (parasite);

  return 0;
}

static gint
load_resource_ps_only (const PSDimageres *res_a, const gint32 image_id, FILE *f)
{
  /* Save photoshop resources with no meaning for GIMP as image parasites 
     to re-save later */
  GimpParasite  *parasite;
  gchar         *data;
  gchar         *name;
  
  IFDBG(3) g_debug ("Process image resource block: %d", res_a->id);

  data = g_malloc (res_a->data_len);
  if (fread (data, 4, 1, f) < 1)
    {
      g_warning (_("Error reading image resource block %d"), res_a->id);
      return -1;
    }

  name = g_strdup_printf ("psd-image-resource-%.4s-%.4x", 
                            res_a->type, res_a->id);
  IFDBG(2) g_debug ("Parasite name: %s", name);

  parasite = gimp_parasite_new (name, 0, res_a->data_len, data);
  g_free (name);
  g_free (data);
  gimp_image_parasite_attach (image_id, parasite);
  gimp_parasite_free (parasite);

  return 0;
}

static gint
load_resource_1005 (const PSDimageres *res_a, const gint32 image_id, FILE *f)
{
  /* Load image resolution and unit of measure */

  /* FIXME  width unit and height unit unused at present */
  
  ResolutionInfo        res_info;
  GimpUnit              image_unit;
  
  IFDBG(2) g_debug ("Process image resource block 1005: Resolution Info");
  
  if (fread (&res_info.hRes, 4, 1, f) < 1
      || fread (&res_info.hResUnit, 2, 1, f) < 1
      || fread (&res_info.widthUnit, 2, 1, f) < 1
      || fread (&res_info.vRes, 4, 1, f) < 1
      || fread (&res_info.vResUnit, 2, 1, f) < 1
      || fread (&res_info.heightUnit, 2, 1, f) < 1)
    {
      g_warning (_("Error reading image resource block %d"), res_a->id);
      return -1;
    }
  res_info.hRes = GINT32_FROM_BE (res_info.hRes);
  res_info.hResUnit = GINT16_FROM_BE (res_info.hResUnit);
  res_info.widthUnit = GINT16_FROM_BE (res_info.widthUnit);
  res_info.vRes = GINT32_FROM_BE (res_info.vRes);
  res_info.vResUnit = GINT16_FROM_BE (res_info.vResUnit);
  res_info.heightUnit = GINT16_FROM_BE (res_info.heightUnit);

  IFDBG(3) g_debug ("Resolution: %d, %d, %d, %d, %d, %d",
                      res_info.hRes,
                      res_info.hResUnit,
                      res_info.widthUnit,
                      res_info.vRes,
                      res_info.vResUnit,
                      res_info.heightUnit);

  /* Resolution always record as pixels / inch in a fixed point implied
     decimal int32 with 16 bits before point and 16 after (i.e. cast as
     double and divide resolution by 2^16 */
  gimp_image_set_resolution (image_id, res_info.hRes / 65536.0, res_info.vRes / 65536.0);
  
  /* GIMP only has one display unit so use ps horizontal resolution unit */
  switch (res_info.hResUnit)
    {
      case PSD_RES_INCH:
        image_unit = GIMP_UNIT_INCH;
        break;
      case PSD_RES_CM:
        image_unit = GIMP_UNIT_MM;
        break;
      default:
        image_unit = GIMP_UNIT_INCH;
    }

  gimp_image_set_unit (image_id, image_unit);

  return 0;
}

static gint
load_resource_1006 (const PSDimageres *res_a, const gint32 image_id, FILE *f)
{
  /* Load alpha channel names stored as a series of pascal strings 
     unpadded between strings */

  GimpParasite  *parasite;
  gchar         *name[MAX_CHANNELS],
                *str;
  gint32        block_rem;
  gint          cidx = 0;
  gssize        read_len,
                write_len;

  IFDBG(2) g_debug ("Process image resource block 1006: Alpha Channel Names");

  parasite = gimp_image_parasite_find (image_id, PSD_PARASITE_ALPHA_NAMES);
  if (parasite != NULL)
    {
      IFDBG(3) g_debug ("Alpha names loaded from unicode resource block");
      gimp_parasite_free (parasite);
      return 0;
    }

  block_rem = res_a->data_len;
  while (block_rem > 1)
    {
      str = fread_pascal_string (&read_len, &write_len, 1, f);
      IFDBG(3) g_debug ("String: %s, %d, %d", str, read_len, write_len);
      if (write_len >= 0)
        {
          name[cidx] = g_strndup (str, write_len);
          cidx++;
          g_free (str);
        }
      block_rem -= read_len;
    }

  if (cidx > 0)
    {
      name[cidx] = NULL;
      str = g_strjoinv (STRING_SEP, name);
      IFDBG(3) g_debug ("String: %s", str);
      parasite = gimp_parasite_new (PSD_PARASITE_ALPHA_NAMES, 0, 
                                    strlen (str), str);
      g_strfreev (name);
      g_free (str);
      gimp_image_parasite_attach (image_id, parasite);
      gimp_parasite_free (parasite);
    }
      
  return 0;
}

static gint
load_resource_1007 (const PSDimageres *res_a, const gint32 image_id, FILE *f)
{
  /* Load alpha channel display info */

  GimpParasite   *parasite;
  DisplayInfo     dsp_info;
  PSDchanneldata *ch_data;
  CMColor         ps_color; 
  GimpRGB         gimp_rgb;
  GimpHSV         gimp_hsv;
  GimpCMYK        gimp_cmyk;
  gint16          tot_rec;
  gint            cidx;

  IFDBG(2) g_debug ("Process image resource block 1007: Display Info");
  tot_rec = res_a->data_len / 14;
  if (tot_rec ==0)
    return 0;

  ch_data = g_new (PSDchanneldata, tot_rec);
  for (cidx = 0; cidx < tot_rec; ++cidx)
    {
      if (fread (&dsp_info.colorSpace, 2, 1, f) < 1
          || fread (&dsp_info.color, 8, 1, f) < 1
          || fread (&dsp_info.opacity, 2, 1, f) < 1
          || fread (&dsp_info.kind, 1, 1, f) < 1
          || fread (&dsp_info.padding, 1, 1, f) < 1)
        {
          g_warning (_("Error reading image resource block %d"), res_a->id);
          return -1;
        }
      dsp_info.colorSpace = GINT16_FROM_BE (dsp_info.colorSpace);
      ps_color.cmyk.cyan = GUINT16_FROM_BE (dsp_info.color[0]);
      ps_color.cmyk.magenta = GUINT16_FROM_BE (dsp_info.color[1]);
      ps_color.cmyk.yellow = GUINT16_FROM_BE (dsp_info.color[2]);
      ps_color.cmyk.black = GUINT16_FROM_BE (dsp_info.color[3]);
      dsp_info.opacity = GINT16_FROM_BE (dsp_info.opacity);

      switch (dsp_info.colorSpace)
        {
          case PSD_CS_RGB:
            gimp_rgb_set (&gimp_rgb, ps_color.rgb.red / 65535.0,
                          ps_color.rgb.green / 65535.0,
                          ps_color.rgb.blue / 65535.0);
            break;
          
          case PSD_CS_HSB:
            gimp_hsv_set (&gimp_hsv, ps_color.hsv.hue / 65535.0,
                          ps_color.hsv.saturation / 65535.0,
                          ps_color.hsv.value / 65535.0);
            gimp_hsv_to_rgb (&gimp_hsv, &gimp_rgb);
            break;
          
          case PSD_CS_CMYK:
            gimp_cmyk_set (&gimp_cmyk, 1.0 - ps_color.cmyk.cyan / 65535.0,
                           1.0 - ps_color.cmyk.magenta / 65535.0,
                           1.0 - ps_color.cmyk.yellow / 65535.0,
                           1.0 - ps_color.cmyk.black / 65535.0);
            gimp_cmyk_to_rgb (&gimp_cmyk, &gimp_rgb);
            break;
          
          case PSD_CS_GRAYSCALE:
            gimp_rgb_set (&gimp_rgb, ps_color.gray.gray / 10000.0,
                          ps_color.gray.gray / 10000.0,
                          ps_color.gray.gray / 10000.0);
            break;
          
          case PSD_CS_FOCOLTONE:
          case PSD_CS_TRUMATCH:
          case PSD_CS_HKS:
          case PSD_CS_LAB:
          case PSD_CS_PANTONE:
          case PSD_CS_TOYO:
          case PSD_CS_DIC:
          case PSD_CS_ANPA:
          default:
            IFDBG(2) g_debug ("Color space %d not supported by GIMP", dsp_info.colorSpace);
            gimp_rgb_set (&gimp_rgb, 1.0, 0.0, 0.0);
        }
      IFDBG(2) g_debug ("PS cSpace: %d, col: %d %d %d %d, opacity: %d, kind: %d",
             dsp_info.colorSpace, ps_color.cmyk.cyan, ps_color.cmyk.magenta,
             ps_color.cmyk.yellow, ps_color.cmyk.black, dsp_info.opacity,
             dsp_info.kind);

      IFDBG(2) g_debug ("cSpace: %d, col: %g %g %g, opacity: %d, kind: %d",
             dsp_info.colorSpace, gimp_rgb.r * 255 , gimp_rgb.g * 255,
             gimp_rgb.b * 255, dsp_info.opacity, dsp_info.kind);

      ch_data[cidx].gimp_color = gimp_rgb;
      ch_data[cidx].opacity = dsp_info.opacity;
      ch_data[cidx].ps_kind = dsp_info.kind;
      ch_data[cidx].ps_color = ps_color;
    }

  if (cidx > 0)
    {
      parasite = gimp_parasite_new (PSD_PARASITE_DISPLAY_INFO, 0, 
                                    sizeof (PSDchanneldata) * tot_rec,
                                    ch_data);
      g_free (ch_data);
      gimp_image_parasite_attach (image_id, parasite);
      gimp_parasite_free (parasite);
    }
   
  return 0;
}

static gint
load_resource_1008 (const PSDimageres *res_a, const gint32 image_id, FILE *f)
{
  /* Load image caption */
  GimpParasite  *parasite;
  gchar         *caption;
  gssize        read_len,
                write_len;
  
  IFDBG(2) g_debug ("Process image resource block: 1008: Caption");
  caption = fread_pascal_string (&read_len, &write_len, 1, f);

  IFDBG(2) g_debug ("Caption: %s", caption);
  parasite = gimp_parasite_new (GIMP_PARASITE_COMMENT, GIMP_PARASITE_PERSISTENT,
                                 write_len, caption);
  g_free (caption);
  gimp_image_parasite_attach (image_id, parasite);
  gimp_parasite_free (parasite);

  return 0;
}

static gint
load_resource_1024 (const PSDimageres *res_a, const gint32 image_id, FILE *f)
{
  /* Load image layer state - current active layer */
  GimpParasite  *parasite;
  gint16        layer_state;
  
  IFDBG(2) g_debug ("Process image resource block: 1024: Layer State");

  if (fread (&layer_state, 2, 1, f) < 1)
    {
      g_warning (_("Error reading image resource block %d"), res_a->id);
      return -1;
    }
  layer_state = GINT16_FROM_BE (layer_state);

  IFDBG(2) g_debug ("Layer state: %d", layer_state);
  parasite = gimp_parasite_new (PSD_PARASITE_LAYER_STATE, 0,
                                sizeof (layer_state), &layer_state);
  gimp_image_parasite_attach (image_id, parasite);
  gimp_parasite_free (parasite);

  return 0;
}

static gint
load_resource_1032 (const PSDimageres *res_a, const gint32 image_id, FILE *f)
{
  /* Load grid and guides */

  /* Grid info is not used (CS2 or earlier) */

  GuideHeader           hdr;
  GuideResource         guide;
  gint                  i;
  
  IFDBG(2) g_debug ("Process image resource block 1032: Grid and Guide Info");
  
  if (fread (&hdr.fVersion, 4, 1, f) < 1
      || fread (&hdr.fGridCycleV, 4, 1, f) < 1
      || fread (&hdr.fGridCycleH, 4, 1, f) < 1
      || fread (&hdr.fGuideCount, 4, 1, f) < 1)
    {
      g_warning (_("Error reading image resource block %d"), res_a->id);
      return -1;
    }
  hdr.fVersion = GUINT32_FROM_BE (hdr.fVersion);
  hdr.fGridCycleV = GUINT32_FROM_BE (hdr.fGridCycleV);
  hdr.fGridCycleH = GUINT32_FROM_BE (hdr.fGridCycleH);
  hdr.fGuideCount = GUINT32_FROM_BE (hdr.fGuideCount);

  IFDBG(2) g_debug ("Grids & Guides: %d, %d, %d, %d",
                     hdr.fVersion,
                     hdr.fGridCycleV,
                     hdr.fGridCycleH,
                     hdr.fGuideCount);

  for (i = 0; i < hdr.fGuideCount; ++i)
    {
      if (fread (&guide.fLocation, 4, 1, f) < 1
          || fread (&guide.fDirection, 1, 1, f) < 1)
        {
          g_warning (_("Error reading image resource block %d"), res_a->id);
          return -1;
        }
      guide.fLocation = GUINT32_FROM_BE (guide.fLocation);
      guide.fLocation /= 32;

      IFDBG(2) g_debug ("Guide: %d px, %d",
                         guide.fLocation,
                         guide.fDirection);

      if (guide.fDirection == PSD_VERTICAL)
        gimp_image_add_vguide (image_id, guide.fLocation);
      else
        gimp_image_add_hguide (image_id, guide.fLocation);
    }

  return 0;
}

static gint
load_resource_1039 (const PSDimageres *res_a, const gint32 image_id, FILE *f)
{
  /* Load ICC profile */
  GimpParasite  *parasite;
  gchar         *icc_profile;
  
  IFDBG(2) g_debug ("Process image resource block: 1039: ICC Profile");

  icc_profile = g_malloc (res_a->data_len);
  if (fread (icc_profile, res_a->data_len, 1, f) < 1)
    {
      g_warning (_("Error reading image resource block %d"), res_a->id);
      return -1;
    }

  parasite = gimp_parasite_new (GIMP_PARASITE_ICC_PROFILE, GIMP_PARASITE_PERSISTENT,
                                 res_a->data_len, icc_profile);
  gimp_image_parasite_attach (image_id, parasite);
  g_free (icc_profile);
  gimp_parasite_free (parasite);

  return 0;
}

static gint
load_resource_1045 (const PSDimageres *res_a, const gint32 image_id, FILE *f)
{
  /* Load alpha channel names stored as a series of unicode strings 
     unpadded between strings */

  GimpParasite *parasite;
  gchar        *name[MAX_CHANNELS + 1],
               *str;
  gint32        block_rem;
  gint          cidx = 0;
  gssize        read_len,
                write_len;

  IFDBG(2) g_debug ("Process image resource block 1045: Unicode alpha Channel Names");

  parasite = gimp_image_parasite_find (image_id, PSD_PARASITE_ALPHA_NAMES);
  if (parasite != NULL)
    {
      IFDBG(3) g_debug ("Deleting localised alpha channel names");
      gimp_image_parasite_detach (image_id, PSD_PARASITE_ALPHA_NAMES);
      gimp_parasite_free (parasite);
    }

  block_rem = res_a->data_len;
  while (block_rem > 1)
    {
      str = fread_unicode_string (&read_len, &write_len, 1, f);
      IFDBG(3) g_debug ("String: %s, %d, %d", str, read_len, write_len);
      if (write_len >= 0)
        {
          name[cidx] = g_strndup (str, write_len);
          cidx++;
          g_free (str);
        }
      block_rem -= read_len;
    }

  if (cidx > 0)
    {
      name[cidx] = NULL;
      str = g_strjoinv (STRING_SEP, name);
      IFDBG(3) g_debug ("String: %s", str);
      parasite = gimp_parasite_new (PSD_PARASITE_ALPHA_NAMES, 0, 
                                    strlen (str), str);
      g_strfreev (name);
      g_free (str);
      gimp_image_parasite_attach (image_id, parasite);
      gimp_parasite_free (parasite);
    }
      
  return 0;
}

static gint
load_resource_1053 (const PSDimageres *res_a, const gint32 image_id, FILE *f)
{
  /* Load image alpha channel ids (tattoos) */
  GimpParasite *parasite;
  guint32      *id;
  gint16        tot_rec,
                cidx;

  IFDBG(2) g_debug ("Process image resource block: 1053: Channel ID");

  tot_rec = res_a->data_len / 4; 
  if (tot_rec ==0)
    return 0;

  id = g_malloc (sizeof (id) * tot_rec);
  for (cidx = 0; cidx < tot_rec; ++cidx)
    {
      if (fread (&id[cidx], 4, 1, f) < 1)
        {
          g_warning (_("Error reading image resource block %d"), res_a->id);
          return -1;
        }
      id[cidx] = GUINT32_FROM_BE (id[cidx]);

      IFDBG(3) g_debug ("Channel id: %d", id[cidx]);
    }

  parasite = gimp_parasite_new (PSD_PARASITE_ALPHA_TATTOOS, 0,
                                (sizeof (id) * tot_rec), id);
  g_free (id);
  gimp_image_parasite_attach (image_id, parasite);
  gimp_parasite_free (parasite);

  return 0;
}

static gint
load_resource_1058 (const PSDimageres *res_a, const gint32 image_id, FILE *f)
{
  /* Load EXIF data */
#ifdef HAVE_EXIF
  ExifData      *exif_data = NULL;
#else
  GimpParasite  *parasite;
  gchar         *name;
#endif
  gchar         *data;

  IFDBG(2) g_debug ("Process image resource block: 1058: EXIF data");

  data = g_malloc (res_a->data_len);
  if (fread (data, 4, 1, f) < 1)
    {
      g_warning (_("Error reading image resource block %d"), res_a->id);
      return -1;
    }

#ifdef HAVE_EXIF
  exif_data = exif_data_new_from_data (data, res_a->data_len);
  if (!exif_data)
    g_warning (_("Unable to process exif data"));
    return -1;

  /* return if there is no data */
  if (!exif_data->data || exif_data->size == 0)
    return -1;

  /*
   * Unfortunately libexif may return a non-null exif_data even if the file
   * contains no exif data.  We check for validity by making sure it
   * has an ExifVersion tag.
  */
  if (! exif_content_get_entry (exif_data->ifd[EXIF_IFD_EXIF],
                                EXIF_TAG_EXIF_VERSION))
    return -1;

  gimp_metadata_store_exif (image_id, exif_data);
  exif_data_unref (exif_data);
#else
  /* If we don't have the EXIF library - store EXIF data as image resource */
  name = g_strdup_printf ("psd-image-resource-%.4s-%.4x", 
                            res_a->type, res_a->id);
  IFDBG(2) g_debug ("Parasite name: %s", name);

  parasite = gimp_parasite_new (name, 0, res_a->data_len, data);
  g_free (name);
  gimp_image_parasite_attach (image_id, parasite);
  gimp_parasite_free (parasite);
#endif /* HAVE_EXIF */

  g_free (data);
  return 0;
}

static gint
load_resource_1060 (const PSDimageres *res_a, const gint32 image_id, FILE *f)
{
  /* Load XMP Metadata block */
  GimpParasite  *parasite;
  gchar         *res_data;
  GString       *xmp_data;
    
  IFDBG(2) g_debug ("Process image resource block: 1060: XMP Data");

  res_data = g_malloc (res_a->data_len);
  if (fread (res_data, res_a->data_len, 1, f) < 1)
    {
      g_warning (_("Error reading image resource block %d"), res_a->id);
      return -1;
    }

  xmp_data = g_string_new (METADATA_MARKER);
  g_string_append (xmp_data, res_data);
  g_free (res_data);

  parasite = gimp_parasite_new (METADATA_PARASITE,
                                GIMP_PARASITE_PERSISTENT,
                                xmp_data->len,
                                (gpointer) xmp_data->str);
  gimp_image_parasite_attach (image_id, parasite);
  g_string_free (xmp_data, TRUE);
  gimp_parasite_free (parasite);

  return 0;
}

static gint
load_resource_2000 (const PSDimageres *res_a, const gint32 image_id, FILE *f)
{
  gchar        *name;
  gdouble      *controlpoints;
  gint32        x[3],
                y[3],
                vector_id = -1;         
  gint16        type,
                init_fill,
                num_rec,
                path_rec,
                cntr;
  gint          image_width,
                image_height,
                i;
  gboolean      closed,
                fill;

  /* Load path data from image resources 2000-2998 */
  
  IFDBG(2) g_debug ("Process image resource block: %d :Path data", res_a->id);
  path_rec = res_a->data_len / 26; 
  if (path_rec ==0)
    return 0;

  if (fread (&type, 2, 1, f) < 1)
    {
      g_warning (_("Error reading image resource block %d"), res_a->id);
      return -1;
    }
  type = GINT16_FROM_BE (type); 
  if (type != PSD_PATH_FILL_RULE)
    {
      g_warning (_("Unexpected path record type: %d"), type);
      return -1;
    }
  else
    fill = FALSE;

  if (fseek (f, 24, SEEK_CUR) < 0)
    {
      g_warning (_("Error setting file position"));
      return -1;
    }

  path_rec--;
  if (path_rec ==0)
    return 0;
  
  image_width = gimp_image_width (image_id);
  image_height = gimp_image_height (image_id);

  /* Create path */  
  name = g_locale_to_utf8 (res_a->name, -1, NULL, NULL, NULL);
  vector_id = gimp_vectors_new (image_id, name);
  g_free (name);
  gimp_image_add_vectors (image_id, vector_id, -1);

  while (path_rec > 0)
    {
      if (fread (&type, 2, 1, f) < 1)
        {
          g_warning (_("Error reading image resource block %d"), res_a->id);
          return -1;
        }
      type = GINT16_FROM_BE (type); 
      IFDBG(2) g_debug ("Path record type %d", type);    

      if (type == PSD_PATH_FILL_RULE)
        {
          fill = FALSE;
          if (fseek (f, 24, SEEK_CUR) < 0)
            {
              g_warning (_("Error setting file position"));
              return -1;
            }
        }

      else if (type == PSD_PATH_FILL_INIT)
        {
          if (fread (&init_fill, 2, 1, f) < 1)
            {
              g_warning (_("Error reading image resource block %d"), res_a->id);
              return -1;
            }
          if (init_fill != 0)
            fill = TRUE;

          if (fseek (f, 22, SEEK_CUR) < 0)
            {
              g_warning (_("Error setting file position"));
              return -1;
            }
        }
      
      else if (type == PSD_PATH_CL_LEN
               || type == PSD_PATH_OP_LEN)
        {
          if (fread (&num_rec, 2, 1, f) < 1)
            {
              g_warning (_("Error reading image resource block %d"), res_a->id);
              return -1;
            }
          num_rec = GINT16_FROM_BE (num_rec);
          if (num_rec > path_rec)
            {
              g_warning (_("Too many path point records"));
              return - 1;
            }
          IFDBG(2) g_debug ("Num path records %d", num_rec);    

          if (type == PSD_PATH_CL_LEN)
            closed = TRUE;
          else
            closed = FALSE;
          cntr = 0;
          controlpoints = g_malloc (sizeof (gdouble) * num_rec * 6);
          if (fseek (f, 22, SEEK_CUR) < 0)
            {
              g_warning (_("Error setting file position"));
              return -1;
            }

          while (num_rec > 0)
            {
              if (fread (&type, 2, 1, f) < 1)
                {
                  g_warning (_("Error reading image resource block %d"), res_a->id);
                  return -1;
                }
              type = GINT16_FROM_BE (type); 
              IFDBG(2) g_debug ("Path record type %d", type);    

              if (type == PSD_PATH_CL_LNK
                  || type == PSD_PATH_CL_UNLNK
                  || type == PSD_PATH_OP_LNK
                  || type == PSD_PATH_OP_UNLNK)
                {
                  if (fread (&y[0], 4, 1, f) < 1 
                    || fread (&x[0], 4, 1, f) < 1
                    || fread (&y[1], 4, 1, f) < 1
                    || fread (&x[1], 4, 1, f) < 1
                    || fread (&y[2], 4, 1, f) < 1
                    || fread (&x[2], 4, 1, f) < 1)
                    {
                      g_warning (_("Error reading image resource block %d"), res_a->id);
                      return -1;
                    }
                  for (i = 0; i < 3; ++i)
                    {
                      x[i] = GINT32_FROM_BE (x[i]);
                      controlpoints[cntr] = x[i] / 16777216.0 * image_width;
                      cntr++;
                      y[i] = GINT32_FROM_BE (y[i]);
                      controlpoints[cntr] = y[i] / 16777216.0 * image_height;
                      cntr++;
                    }
                  IFDBG(2) g_debug ("Path points (%d,%d), (%d,%d), (%d,%d)",
                                    x[1], y[1], x[2], y[2], x[3], y[3]);
                }
              else
                {
                  g_warning (_("Unexpected path type record %d"), type);
                  if (fseek (f, 24, SEEK_CUR) < 0)
                    {
                      g_warning (_("Error setting file position"));
                      return -1;
                    }
                }
              path_rec--;
              num_rec--;
            }
          /* Add sub-path */
          gimp_vectors_stroke_new_from_points (vector_id, GIMP_VECTORS_STROKE_TYPE_BEZIER,
                                               cntr, controlpoints, closed);
          g_free (controlpoints);
        }      

      else  
        {
          if (fseek (f, 24, SEEK_CUR) < 0)
            {
              g_warning (_("Error setting file position"));
              return -1;
            }
        }

      path_rec--;
    }

 return 0;
}
