
#include "config.h"

#include <string.h>

#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>

#include "webx_main.h"
#include "webx_source.h"
#include "webx_dialog.h"
#include "webx_widgets.h"
#include "webx_indexed.h"

#include "libgimp/stdplugins-intl.h"

void
webx_indexed_init (WebxIndexed *settings)
{
  settings->dither_type = GIMP_NO_DITHER;
  if (webx_source_get_indexed (NULL) != -1)
    settings->palette_type = GIMP_REUSE_PALETTE;
  else
    settings->palette_type = GIMP_MAKE_PALETTE;
  settings->custom_palette = NULL;
  settings->alpha_dither = FALSE;
  settings->remove_unused = FALSE;
  settings->num_colors = 256;
  settings->image = -1;
  settings->layer = -1;
}

void
webx_indexed_cleanup (WebxIndexed *settings)
{
  webx_indexed_invalidate (NULL, settings);
}

void
webx_indexed_invalidate (void        *widget,
                         WebxIndexed *settings)
{
  if (settings->image != -1)
    {
      gimp_image_delete (settings->image);
      settings->image = -1;
    }
  settings->layer = -1;
}

static void
webx_indexed_palette_mode_update (GtkWidget         *widget,
                                  gpointer           data)
{
  GtkWidget   *adj;
  GtkWidget   *num_colors;
  GtkWidget   *remove_unused;
  WebxIndexed *settings;
  gint         old_type;
    
  old_type = settings->palette_type;
    
  settings = g_object_get_data (G_OBJECT (widget), "settings");
  num_colors = g_object_get_data (G_OBJECT (widget), "num_colors");
  remove_unused = g_object_get_data (G_OBJECT (widget), "remove_unused");
  gtk_widget_set_sensitive (GTK_WIDGET (remove_unused), TRUE);

  adj = g_object_get_data (G_OBJECT (widget), "reuse_palette");
  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (adj)))
    {
      g_assert (webx_source_get_indexed (NULL) != -1);
      settings->palette_type = GIMP_REUSE_PALETTE;
      gtk_widget_set_sensitive (GTK_WIDGET (remove_unused), FALSE);
    }

  adj = g_object_get_data (G_OBJECT (widget), "make_palette");
  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (adj)))
    {
      settings->palette_type = GIMP_MAKE_PALETTE;
      gtk_widget_set_sensitive (GTK_WIDGET (num_colors), TRUE);
      gtk_widget_set_sensitive (GTK_WIDGET (remove_unused), FALSE);
    }
  else
    {
      gtk_widget_set_sensitive (GTK_WIDGET (num_colors), FALSE);
    }

  adj = g_object_get_data (G_OBJECT (widget), "web_palette");
  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (adj)))
    {
      settings->palette_type = GIMP_WEB_PALETTE;
    }

  adj = g_object_get_data (G_OBJECT (widget), "mono_palette");
  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (adj)))
    {
      settings->palette_type = GIMP_MONO_PALETTE;
    }
   
  if (settings->palette_type != old_type)
    {
      webx_indexed_invalidate (widget, settings);
      webx_dialog_queue_update ();
    }
}

static void
webx_dither_type_changed (GimpIntComboBox   *combo,
                          WebxIndexed       *settings)
{
  gint    dither_type;

  if (gimp_int_combo_box_get_active (combo, &dither_type)
      && settings->dither_type != dither_type)
    {
      settings->dither_type = dither_type;
      webx_indexed_invalidate (combo, settings);
      webx_dialog_queue_update ();
    }
}

static void
webx_indexed_num_colors_changed (GtkSpinButton *spinbtn,
				 WebxIndexed   *settings)
{
  settings->num_colors =
    gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spinbtn));
  webx_indexed_invalidate (spinbtn, settings);
  webx_dialog_queue_update ();
}

GtkWidget*
webx_indexed_create_widgets (WebxIndexed *settings)
{
  GtkWidget   *adj;
  GtkWidget   *label;
  GtkWidget   *table;
  GtkWidget   *w;
  GSList      *group;
  gint         row = 0;

  table = gtk_table_new (12, 3, FALSE);
    
  g_object_set_data (G_OBJECT (table), "settings", settings);
    
  w = gtk_radio_button_new_with_label (NULL, _("Reuse existing palette"));
  gtk_table_attach (GTK_TABLE (table), w,
		    0, 3, row, row+1,
		    GTK_FILL, GTK_FILL, 0, 0);
  if (webx_source_get_indexed (NULL) == -1)
    gtk_widget_set_sensitive (w, FALSE);
  group = gtk_radio_button_get_group (GTK_RADIO_BUTTON (w));
  g_signal_connect_swapped (GTK_WIDGET (w), "toggled",
			    G_CALLBACK (webx_indexed_palette_mode_update),
			    table);
  g_object_set_data (G_OBJECT (table), "reuse_palette", w);

  row++;
  w = gtk_radio_button_new_with_label (group, _("Generate optimum palette"));
  gtk_table_attach (GTK_TABLE (table), w,
		    0, 3, row, row+1,
		    GTK_FILL, GTK_FILL, 0, 0);
  group = gtk_radio_button_get_group (GTK_RADIO_BUTTON (w));
  g_signal_connect_swapped (GTK_WIDGET (w), "toggled",
			    G_CALLBACK (webx_indexed_palette_mode_update),
			    table);
  g_object_set_data (G_OBJECT (table), "make_palette", w);
  row++;
  label = gtk_label_new (_("Number of colors:"));
  gtk_table_attach (GTK_TABLE (table), label,
		    1, 2, row, row+1,
		    GTK_FILL, GTK_FILL, 0, 0);
  adj = gtk_spin_button_new_with_range (2, 256, 1);
  g_signal_connect (adj, "value-changed",
		    G_CALLBACK (webx_indexed_num_colors_changed),
		    settings);
  gtk_table_attach (GTK_TABLE (table), adj,
		    2, 3, row, row+1,
		    GTK_FILL, GTK_FILL, 0, 0);
  g_object_set_data (G_OBJECT (table), "num_colors", adj);

  row++;
  w = gtk_radio_button_new_with_label (group, _("Use web-optimized palette"));
  gtk_table_attach (GTK_TABLE (table), w,
		    0, 3, row, row+1,
		    GTK_FILL, GTK_FILL, 0, 0);
  group = gtk_radio_button_get_group (GTK_RADIO_BUTTON (w));
  g_signal_connect_swapped (GTK_WIDGET (w), "toggled",
			    G_CALLBACK (webx_indexed_palette_mode_update),
			    table);
  g_object_set_data (G_OBJECT (table), "web_palette", w);

  row++;
  w = gtk_radio_button_new_with_label (group, _("Use black and white (1-bit) palette"));
  gtk_table_attach (GTK_TABLE (table), w,
		    0, 3, row, row+1,
		    GTK_FILL, GTK_FILL, 0, 0);
  group = gtk_radio_button_get_group (GTK_RADIO_BUTTON (w));
  g_signal_connect_swapped (GTK_WIDGET (w), "toggled",
			    G_CALLBACK (webx_indexed_palette_mode_update),
			    table);
  g_object_set_data (G_OBJECT (table), "mono_palette", w);

  row++;
  w = gtk_check_button_new_with_label (_("Remove unused colors"));
  gtk_table_attach (GTK_TABLE (table), w,
		    0, 3, row, row+1,
		    GTK_FILL, GTK_FILL, 0, 0);
  g_signal_connect (w, "toggled",
		    G_CALLBACK (webx_indexed_invalidate),
		    settings);
  g_signal_connect (w, "toggled",
		    G_CALLBACK (webx_widgets_boolean_update),
		    &settings->remove_unused);
  g_object_set_data (G_OBJECT (table), "remove_unused", w);

  row++;
  w = gtk_hseparator_new ();
  gtk_table_attach (GTK_TABLE (table), w,
		    0, 3, row, row+1,
		    GTK_FILL, GTK_FILL, 0, 8);
    
  row++;
  label = gtk_label_new (_("Dither:"));
  gtk_table_attach (GTK_TABLE (table), label,
		    0, 1, row, row+1,
		    GTK_FILL, GTK_FILL, 0, 0);
  w = gimp_int_combo_box_new (_("None"), GIMP_NO_DITHER,
			      _("F-S (normal)"), GIMP_FS_DITHER,
			      _("F-S (reduced bleeding)"),
			      GIMP_FSLOWBLEED_DITHER,
			      _("Positioned"), GIMP_FIXED_DITHER,
			      NULL);
  gtk_table_attach (GTK_TABLE (table), w,
		    1, 3, row, row+1,
		    GTK_FILL | GTK_SHRINK, GTK_FILL, 0, 8);
  gimp_int_combo_box_connect (GIMP_INT_COMBO_BOX (w),
			      settings->dither_type,
			      G_CALLBACK (webx_dither_type_changed),
			      settings);
  g_object_set_data (G_OBJECT (table), "dither_type", w);
    
  row++;
  w = gtk_check_button_new_with_label (_("Enable dithering of transparency"));
  gtk_table_attach (GTK_TABLE (table), w,
		    0, 3, row, row+1,
		    GTK_FILL, GTK_FILL, 0, 0);
  g_signal_connect (w, "toggled",
		    G_CALLBACK (webx_indexed_invalidate),
		    settings);
  g_signal_connect (w, "toggled",
		    G_CALLBACK (webx_widgets_boolean_update),
		    &settings->alpha_dither);
  g_object_set_data (G_OBJECT (table), "alpha_dither", w);

  row++;
  w = gtk_hseparator_new ();
  gtk_table_attach (GTK_TABLE (table), w,
		    0, 3, row, row+1,
		    GTK_FILL, GTK_FILL, 0, 8);

  return table;
}

void
webx_indexed_update_widgets (GtkWidget *widget, WebxIndexed *settings)
{
  GtkWidget   *adj;
  GtkWidget   *num_colors;
  GtkWidget   *remove_unused;

  num_colors = g_object_get_data (G_OBJECT (widget), "num_colors");
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (num_colors),
			      settings->num_colors);

  remove_unused = g_object_get_data (G_OBJECT (widget), "remove_unused");
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (remove_unused),
				settings->remove_unused);
  gtk_widget_set_sensitive (remove_unused, TRUE);
    
  adj = g_object_get_data (G_OBJECT (widget), "dither_type");
  gimp_int_combo_box_set_active (GIMP_INT_COMBO_BOX (adj),
				 settings->dither_type);

  adj = g_object_get_data (G_OBJECT (widget), "alpha_dither");
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (adj),
				settings->alpha_dither);

  adj = g_object_get_data (G_OBJECT (widget), "reuse_palette");
  if (settings->palette_type == GIMP_REUSE_PALETTE)
    {
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (adj), TRUE);
      gtk_widget_set_sensitive (remove_unused, FALSE);
    }

  adj = g_object_get_data (G_OBJECT (widget), "make_palette");
  if (settings->palette_type == GIMP_MAKE_PALETTE)
    {
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (adj), TRUE);
      gtk_widget_set_sensitive (num_colors, TRUE);
      gtk_widget_set_sensitive (remove_unused, FALSE);
    }
  else
    {
      gtk_widget_set_sensitive (num_colors, FALSE);
    }

  adj = g_object_get_data (G_OBJECT (widget), "web_palette");
  if (settings->palette_type == GIMP_WEB_PALETTE)
    {
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (adj), TRUE);
    }
    
  adj = g_object_get_data (G_OBJECT (widget), "mono_palette");
  if (settings->palette_type == GIMP_MONO_PALETTE)
    {
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (adj), TRUE);
    }
}

gint
webx_indexed_get_image (WebxIndexed *settings, gint *drawable)
{
  gint        image;
  gint        layer;
  gint       *layers;
  gint        num_layers;
  gchar      *custom_palette;
  gint        num_colors;
  gboolean    converted;
    
  if (settings->palette_type == GIMP_REUSE_PALETTE)
    return webx_source_get_indexed (drawable);

  /* precached */
  if (settings->image != -1)
    {
      *drawable = settings->layer;
      return settings->image;
    }
    
  if (! settings->custom_palette)
    custom_palette = "";
  else
    custom_palette = settings->custom_palette;

  image = webx_source_get_rgb (&layer);
  num_colors = settings->num_colors;
  if (num_colors == 256 && gimp_drawable_has_alpha (layer))
    num_colors = 255;
  image = gimp_image_duplicate (image);
  converted = gimp_image_convert_indexed (image, settings->dither_type,
					  settings->palette_type,
					  num_colors,
					  settings->alpha_dither,
					  settings->remove_unused,
					  custom_palette);
  if (! converted)
    {
      gimp_image_delete (image);
      *drawable = -1;
      return -1;
    }
  layers = gimp_image_get_layers (image, &num_layers);
  g_assert (num_layers == 1);
  layer = layers[0];
  settings->image = image;
  settings->layer = layer;
  *drawable = layer;
  return image;
}

