#include "config.h"

#include <string.h>

#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

#include "webx_main.h"
#include "webx_source.h"
#include "webx_dialog.h"

#include "libgimp/stdplugins-intl.h"

static WebxSource      webx_src;

void
webx_source_init (gint32    image_ID,
                  gint32    drawable_ID)
{
  memset (&webx_src, 0, sizeof(webx_src));
  webx_src.user_image = image_ID;
  webx_src.user_drawable = drawable_ID;
  webx_src.original_width = gimp_image_width (webx_src.user_image);
  webx_src.original_height = gimp_image_height (webx_src.user_image);
  webx_src.resize_width = webx_src.original_width;
  webx_src.resize_height = webx_src.original_height;
  webx_src.crop_width = webx_src.original_width;
  webx_src.crop_height = webx_src.original_height;
  webx_src.crop_offsx = 0;
  webx_src.crop_offsy = 0;
  webx_src.rgb_image = -1;
  webx_src.indexed_image = -1;
  webx_src.background_pixbuf = NULL;
  webx_src.crop_scale_x = 1.0;
  webx_src.crop_scale_y = 1.0;
  webx_source_update ();
}

void
webx_source_cleanup ()
{
  if (webx_src.rgb_image != -1)
    gimp_image_delete (webx_src.rgb_image);
  if (webx_src.indexed_image != -1)
    gimp_image_delete (webx_src.indexed_image);
  if (! webx_src.background_pixbuf)
    g_object_unref (webx_src.background_pixbuf);
}

void
webx_source_get_rect (GdkRectangle *rect)
{
  rect->x = webx_src.crop_offsx;
  rect->y = webx_src.crop_offsy;
  rect->width = webx_src.crop_width;
  rect->height = webx_src.crop_height;
}

GdkPixbuf*
webx_background_get_pixbuf ()
{
  return webx_src.background_pixbuf;
}

gint
webx_background_get_width ()
{
  return webx_src.resize_width;
}

gint
webx_background_get_height ()
{
  return webx_src.resize_height;
}

gboolean
webx_source_resize (gint    newwidth,
                    gint    newheight)
{
  /* clip dimensions */
  if (newwidth < 1)
    newwidth = 1;
  if (newheight < 1)
    newheight = 1;
  if (webx_src.resize_width == newwidth
      && webx_src.resize_height == newheight)
    return FALSE;
  /* scale crop */
  if (webx_src.resize_width == webx_src.crop_width
      && webx_src.resize_height == webx_src.crop_height
      && webx_src.crop_offsx == 0
      && webx_src.crop_offsy == 0)
    {
      /* avoid roundoff errors when user doesn't want to crop */
      webx_src.crop_width = newwidth;
      webx_src.crop_height = newheight;
        
      webx_src.crop_scale_x = 1.0;
      webx_src.crop_scale_y = 1.0;
    }
  else
    {
      webx_src.crop_scale_x *= (gdouble)newwidth/webx_src.resize_width;
      webx_src.crop_scale_y *= (gdouble)newheight/webx_src.resize_height;
    }
  webx_src.resize_width = newwidth;
  webx_src.resize_height = newheight;
  webx_source_invalidate (TRUE);
  return TRUE;
}

static void
webx_source_crop_clip ()
{
  /* clip to image bounds */
  if (webx_src.crop_offsx < 0)
    webx_src.crop_offsx = 0;
  else if (webx_src.crop_offsx >= webx_src.resize_width)
    webx_src.crop_offsx = webx_src.resize_width - 1;
  if (webx_src.crop_offsy < 0)
    webx_src.crop_offsy = 0;
  else if (webx_src.crop_offsy >= webx_src.resize_height)
    webx_src.crop_offsy = webx_src.resize_height - 1;
  if (webx_src.crop_width < 1)
    webx_src.crop_width = 1;
  else if (webx_src.crop_width > webx_src.resize_width)
    webx_src.crop_width = webx_src.resize_width;
  if (webx_src.crop_height < 1)
    webx_src.crop_height = 1;
  else if (webx_src.crop_height > webx_src.resize_height)
    webx_src.crop_height = webx_src.resize_height;
  if (webx_src.crop_offsx + webx_src.crop_width > webx_src.resize_width)
    webx_src.crop_width = webx_src.resize_width - webx_src.crop_offsx;
  if (webx_src.crop_offsy + webx_src.crop_height > webx_src.resize_height)
    webx_src.crop_height = webx_src.resize_height - webx_src.crop_offsy;
}

gboolean
webx_source_crop (gint     width,
                  gint     height,
                  gint     offsx,
                  gint     offsy,
                  gboolean clip_offsets)
{
  /* clip to image bounds */
  if (offsx < 0)
    {
      width += offsx;
      offsx = 0;
    }
  else if (offsx >= webx_src.resize_width)
    {
      offsx = webx_src.resize_width - 1;
    }
  if (offsy < 0)
    {
      height += offsy;
      offsy = 0;
    }
  else if (offsy >= webx_src.resize_height)
    {
      offsy = webx_src.resize_height - 1;
    }
  if (width < 1)
    width = 1;
  else if (width > webx_src.resize_width)
    width = webx_src.resize_width;
  if (height < 1)
    height = 1;
  else if (height > webx_src.resize_height)
    height = webx_src.resize_height;
  if (clip_offsets)
    {
      if (offsx + width > webx_src.resize_width)
	offsx = webx_src.resize_width - width;
      if (offsy + height > webx_src.resize_height)
	offsy = webx_src.resize_height - height;
    }
  else
    {
      if (offsx + width > webx_src.resize_width)
	width = webx_src.resize_width - offsx;
      if (offsy + height > webx_src.resize_height)
	height = webx_src.resize_height - offsy;
    }
  if (width == webx_src.crop_width
      && height == webx_src.crop_height
      && offsx == webx_src.crop_offsx
      && offsy == webx_src.crop_offsy)
    return FALSE; /* crop options not changed */
  webx_src.crop_width = width;
  webx_src.crop_height = height;
  webx_src.crop_offsx = offsx;
  webx_src.crop_offsy = offsy;
  webx_src.crop_scale_x = 1.0;
  webx_src.crop_scale_y = 1.0;
  webx_source_invalidate (FALSE);
  return TRUE;
}

gint32
webx_source_get_rgb (gint32    *layer)
{
  if (webx_src.rgb_image == -1)
    webx_source_update ();

  if (layer)
    *layer = webx_src.rgb_layer;
  return webx_src.rgb_image;
}

gint32
webx_source_get_indexed (gint32    *layer)
{
  if (webx_src.rgb_image == -1)
    webx_source_update ();

  if (layer)
    *layer = webx_src.indexed_layer;
  return webx_src.indexed_image;
}

void
webx_source_invalidate (gboolean delete_bg)
{
  if (webx_src.rgb_image != -1)
    {
      gimp_image_delete (webx_src.rgb_image);
      webx_src.rgb_image = -1;
    }
  if (webx_src.indexed_image != -1)
    {
      gimp_image_delete (webx_src.indexed_image);
      webx_src.indexed_image = -1;
    }
  if (delete_bg && webx_src.background_pixbuf)
    {
      g_object_unref (webx_src.background_pixbuf);
      webx_src.background_pixbuf = NULL;
    }
}

static void
webx_background_create ()
{
  gint32           bg_image;
  guchar          *pixels;
  guchar          *row;
  gint             rowstride;
  gint             width;
  gint             height;
  gint             i;
  gint             j;
  gboolean         has_alpha;

  bg_image = gimp_image_duplicate (webx_src.rgb_image);
  gimp_image_undo_disable (bg_image);
  webx_src.background_pixbuf = webx_image_to_pixbuf (bg_image);
  gimp_image_delete (bg_image);
  pixels = gdk_pixbuf_get_pixels (webx_src.background_pixbuf);
  width = gdk_pixbuf_get_width (webx_src.background_pixbuf);
  height = gdk_pixbuf_get_height (webx_src.background_pixbuf);
  rowstride = gdk_pixbuf_get_rowstride (webx_src.background_pixbuf);
  has_alpha = gdk_pixbuf_get_has_alpha (webx_src.background_pixbuf);
  for (i = 0; i < height; i++)
    {
      row = pixels;
      for (j = 0; j < width; j++)
        {
	  row[0] /= 4;
	  row[1] /= 4;
	  row[2] /= 4;
	  row += 3;
	  if (has_alpha)
	    row++;
        }
      pixels += rowstride;
    }
}

gboolean
webx_source_update (void)
{
  if (webx_src.rgb_image != -1)
    return FALSE;
        
  if (webx_src.background_pixbuf)
    {
      g_object_unref (webx_src.background_pixbuf);
      webx_src.background_pixbuf = NULL;
    }
        
  webx_src.rgb_image = gimp_image_duplicate (webx_src.user_image);
  gimp_image_undo_disable (webx_src.rgb_image);
  webx_src.rgb_layer =
    gimp_image_merge_visible_layers (webx_src.rgb_image,
				     GIMP_CLIP_TO_IMAGE);
  /* we don't want layer to be smaller than image */
  gimp_layer_resize_to_image_size (webx_src.rgb_layer);
  if (webx_src.resize_width != webx_src.original_width
      || webx_src.resize_height != webx_src.original_width)
    {
      gimp_image_scale (webx_src.rgb_image,
                        webx_src.resize_width, webx_src.resize_height);
    }
 
  webx_background_create ();
    
  webx_src.crop_offsx *= webx_src.crop_scale_x;
  webx_src.crop_offsy *= webx_src.crop_scale_y;
  webx_src.crop_width *= webx_src.crop_scale_x;
  webx_src.crop_height *= webx_src.crop_scale_y;
  webx_src.crop_scale_x = 1.0;
  webx_src.crop_scale_y = 1.0;
  webx_source_crop_clip ();

  if (webx_src.crop_width != webx_src.resize_width
      || webx_src.crop_height != webx_src.resize_height )
    {
      gimp_image_crop (webx_src.rgb_image,
		       webx_src.crop_width, webx_src.crop_height,
		       webx_src.crop_offsx, webx_src.crop_offsy);
    }
    
  if (gimp_drawable_is_indexed (webx_src.rgb_layer))
    {
      webx_src.indexed_image = gimp_image_duplicate (webx_src.rgb_image);
      gimp_image_undo_disable (webx_src.indexed_image);
      webx_src.indexed_layer =
	gimp_image_merge_visible_layers (webx_src.indexed_image,
                                         GIMP_CLIP_TO_IMAGE);
    }
  else
    {
      webx_src.indexed_image = -1;
    }

  if ( ! gimp_drawable_is_rgb (webx_src.rgb_layer))
    gimp_image_convert_rgb (webx_src.rgb_image);
        
  return TRUE;
}
