
#include "config.h"

#include <string.h>

#include <glib.h>
#include <glib/gstdio.h>

#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>

#include "webx_main.h"
#include "webx_source.h"
#include "webx_dialog.h"
#include "webx_jpeg.h"
#include "webx_widgets.h"

#include "libgimp/stdplugins-intl.h"

static WebxJPEGSettings    webx_jpeg;

void
webx_jpeg_init ()
{
  webx_jpeg.quality       = 0.85;
  webx_jpeg.smoothing     = 0.0;
  webx_jpeg.subsmp        = 0;
  webx_jpeg.restart       = 0;
  webx_jpeg.dct           = 0;
  webx_jpeg.optimize      = TRUE;
  webx_jpeg.progressive   = FALSE;
  webx_jpeg.baseline      = TRUE;
}

void
webx_jpeg_cleanup ()
{
}

GtkWidget*
webx_jpeg_toolbox_new ()
{
  GtkWidget   *table;
  GtkWidget   *check;
  GtkObject   *adj;
  gint         row = 0;

  table = gtk_table_new (5, 3, FALSE);

  adj = gimp_scale_entry_new (GTK_TABLE (table), 0, row++,
			      _("_Quality:"), 90, 6,
			      webx_jpeg.quality, 6.0, 100.0, 1.0, 1.0, 0,
			      TRUE, 0, 0,
			      NULL, NULL);
  g_signal_connect (adj, "value-changed",
		    G_CALLBACK (webx_widgets_percent_update),
		    &webx_jpeg.quality);
  g_object_set_data (G_OBJECT (table), "quality", adj);

  adj = gimp_scale_entry_new (GTK_TABLE (table), 0, row++,
			      _("_Smoothing:"), 90, 6,
			      webx_jpeg.smoothing, 0.0, 100.0, 1.0, 1.0, 0,
			      TRUE, 0, 0,
			      NULL, NULL);
  g_signal_connect (adj, "value-changed",
		    G_CALLBACK (webx_widgets_percent_update),
		    &webx_jpeg.smoothing);
  g_object_set_data (G_OBJECT (table), "smoothing", adj);

  row++;
  check = gtk_check_button_new_with_label (_("Optimize"));
  gtk_table_attach (GTK_TABLE (table), check,
		    0, 3, row, row+1,
		    GTK_FILL, GTK_FILL, 0, 0);
  g_signal_connect (check, "toggled",
		    G_CALLBACK (webx_widgets_boolean_update),
		    &webx_jpeg.optimize);
  g_object_set_data (G_OBJECT (table), "optimize", check);

  row++;
  check = gtk_check_button_new_with_label (_("Progressive"));
  gtk_table_attach (GTK_TABLE (table), check,
		    0, 3, row, row+1,
		    GTK_FILL, GTK_FILL, 0, 0);
  g_signal_connect (check, "toggled",
		    G_CALLBACK (webx_widgets_boolean_update),
		    &webx_jpeg.progressive);
  g_object_set_data (G_OBJECT (table), "progressive", check);

  row++;
  check = gtk_check_button_new_with_label (_("Baseline"));
  gtk_table_attach (GTK_TABLE (table), check,
		    0, 3, row, row+1,
		    GTK_FILL, GTK_FILL, 0, 0);
  g_signal_connect (check, "toggled",
		    G_CALLBACK (webx_widgets_boolean_update),
		    &webx_jpeg.baseline);
  g_object_set_data (G_OBJECT (table), "baseline", check);

  return table;
}

void
webx_jpeg_toolbox_update (GtkWidget *toolbox)
{
  GtkObject   *adj;
  GtkWidget   *check;
    
  adj = GTK_OBJECT (g_object_get_data (G_OBJECT (toolbox), "quality"));
  gtk_range_set_value (GTK_RANGE (GIMP_SCALE_ENTRY_SCALE (adj)),
		       webx_jpeg.quality * 100.0);

  adj = GTK_OBJECT (g_object_get_data (G_OBJECT (toolbox), "smoothing"));
  gtk_range_set_value (GTK_RANGE (GIMP_SCALE_ENTRY_SCALE (adj)),
		       webx_jpeg.smoothing * 100.0);

  check = g_object_get_data (G_OBJECT (toolbox), "optimize");
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check), webx_jpeg.optimize);

  check = g_object_get_data (G_OBJECT (toolbox), "progressive");
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check),
				webx_jpeg.progressive);

  check = g_object_get_data (G_OBJECT (toolbox), "baseline");
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check), webx_jpeg.baseline);
}

gboolean
webx_jpeg_save (const gchar *file_name)
{
  GimpParam    *return_vals;
  gint          n_return_vals;
  gint32        image;
  gint32        layer;
  gint32        save_image;
  gint32        save_layer;
  gboolean      save_res;
  
  image = webx_source_get_rgb (&layer);
  if (gimp_drawable_has_alpha (layer))
    {
      save_image = gimp_image_duplicate (image);
      gimp_image_undo_disable (image);
      save_layer = gimp_image_flatten (save_image);
    }
  else
    {
      save_image = image;
      save_layer = layer;
    }

  return_vals = gimp_run_procedure ("file-jpeg-save", &n_return_vals,
				    GIMP_PDB_INT32, GIMP_RUN_NONINTERACTIVE,
				    GIMP_PDB_IMAGE, save_image,
				    GIMP_PDB_DRAWABLE, save_layer,
				    GIMP_PDB_STRING, file_name,
				    GIMP_PDB_STRING, file_name,
				    GIMP_PDB_FLOAT, webx_jpeg.quality,
				    GIMP_PDB_FLOAT, webx_jpeg.smoothing,
				    GIMP_PDB_INT32, (gint)webx_jpeg.optimize,
				    GIMP_PDB_INT32, (gint)webx_jpeg.progressive,
				    GIMP_PDB_STRING, "",
				    GIMP_PDB_INT32, webx_jpeg.subsmp,
				    GIMP_PDB_INT32, (gint)webx_jpeg.baseline,
				    GIMP_PDB_INT32, webx_jpeg.restart,
				    GIMP_PDB_INT32, webx_jpeg.dct,
				    GIMP_PDB_END);
  if (return_vals[0].data.d_int32 == GIMP_PDB_SUCCESS)
    save_res = TRUE;
  else
    save_res = FALSE;
  gimp_destroy_params (return_vals, n_return_vals);

  if (gimp_drawable_has_alpha (layer))
    {
      gimp_image_delete (save_image);
    }
  return save_res;
}

static gint32
webx_jpeg_load (const gchar *file_name)
{
  GimpParam  *return_vals;
  gint        n_return_vals;
  gint32      image;

  return_vals = gimp_run_procedure ("file-jpeg-load", &n_return_vals,
				    GIMP_PDB_INT32, GIMP_RUN_NONINTERACTIVE,
				    GIMP_PDB_STRING, file_name,
				    GIMP_PDB_STRING, file_name,
				    GIMP_PDB_END);
  image = return_vals[1].data.d_image;
  gimp_destroy_params (return_vals, n_return_vals);

  return image;
}

GdkPixbuf*
webx_jpeg_generate_preview ()
{
  gchar       *file_name;
  GdkPixbuf   *pixbuf = NULL;
  gint32       image;

  file_name = gimp_temp_name ("JPEG");
  if (webx_jpeg_save (file_name))
    {
      webx_dialog_update_file_size (file_name);
      image = webx_jpeg_load (file_name);
      if (image != -1)
	{
	  pixbuf = webx_image_to_pixbuf (image);
	  gimp_image_delete (image);
	}
    }
  g_unlink (file_name);
  g_free (file_name);

  return pixbuf;
}

void
webx_jpeg_invalidate (void)
{
  /* We are not caching images for JPEG. */
}
