/*
 * ExtensionDialog.cs
 *
 * Authors:
 *   Stephane Delcroix <stephane@delcroix.org>
 *
 * Copyright (C) 2006 Stephane Delcroix
 */


using System;
using System.Collections;
using Mono.Posix;

using FSpot;
using FSpot.Extensions;

namespace FSpot.Extensions {

	public class ExtensionDialogExtension: FSpot.Extensions.Extension
	{
		protected override string ConfigurationName { get { return "ExtensionDialogExtension"; } }
		public override string DisplayName { get { return "Extension Dialog Extension"; } }
		public override string Description { get { return "This extension creates a dialog to enable/disable extensions"; } }
		public override ExtensionCategory Category { get { return ExtensionCategory.Misc; } }
		public override ExtensionRevision Revision { get { return new ExtensionRevision(0, 2, 2); } }
		public override string [] Authors {
			get { 
				return new string [] {
					"Stephane Delcroix"
				};
			} 
		}

		public ExtensionDialogExtension () : base()
		{
			Console.WriteLine("new PDP instanciated !");	
			//Preferences.Get("Enabled", true);

			//Enable this extension everytime
			Preferences.Set("Enabled", true);
		}


		protected override void ExtensionInitialize()
		{
			Console.WriteLine("Initializing PDP");	
			MainWindow.Toplevel.ExtensionMenuItem.Sensitive = true;
			MainWindow.Toplevel.ExtensionMenuItem.Activated += HandleExtensionMenuItemActivated;
		}

		protected void HandleExtensionMenuItemActivated (object o, EventArgs e)
		{
			try {
				new ExtensionDialog();
			} catch (Exception ex) {
				Console.WriteLine(e);
			}
		}

		protected override void ExtensionDispose()
		{
			Console.WriteLine("Disposing PDP");
			MainWindow.Toplevel.ExtensionMenuItem.Sensitive = false;
			MainWindow.Toplevel.ExtensionMenuItem.Activated -= HandleExtensionMenuItemActivated;
		}
	}

	public class ExtensionDialog
	{
		private string dialog_name = "extension_dialog";
		private Glade.XML xml;
		private Gtk.Dialog dialog;
		private CategoryTree ctree;

		[Glade.Widget] Gtk.Label extensions_label;
		[Glade.Widget] Gtk.Label loaded_label;
		[Glade.Widget] Gtk.Viewport extensions_vp;

		public ExtensionDialog()
		{
			CreateDialog();
			Dialog.Modal = false;
			Dialog.TransientFor = null;
			
			ctree = new CategoryTree();
			extensions_vp.Add(ctree);

			Populate();

			Dialog.ShowAll();
			Dialog.Response += HandleResponse;
		}

		private void Populate()
		{
			int extensions = 0;
			int loaded = 0;

			foreach (Extension p in ExtensionCore.Factory) {
				ExtensionCheck cb = new ExtensionCheck(p);
				extensions ++;
				if ((bool) p.Preferences.Get("Enabled", true))
					loaded++;
				cb.Toggled += HandleToggled;
				ctree.AddInCategory(cb, p.Category);
			}
			extensions_label.Text = extensions.ToString();
			loaded_label.Text = loaded.ToString();
		}

		private void ApplyChanges ()
		{
			ExtensionCore.Reload();
			ctree = new CategoryTree();
			extensions_vp.Add(ctree);
			//Populate();
		}

		private void CreateDialog()
		{
			xml = new Glade.XML (null, "extensiondialog.glade", dialog_name, "f-spot");
			xml.Autoconnect (this);
		}

		private Gtk.Dialog Dialog {
			get {
				if (dialog == null)
					dialog = (Gtk.Dialog) xml.GetWidget (dialog_name);
				return dialog;
			}
		}

		private void HandleToggled (object sender, EventArgs args)
		{
			if (((ExtensionCheck)sender).Active)
				((ExtensionCheck)sender).Extension.Preferences.Set("Enabled", true);
			else
				((ExtensionCheck)sender).Extension.Preferences.Set("Enabled", false);

		}

		private void HandleResponse (object sender, Gtk.ResponseArgs args)
		{
			if (args.ResponseId != Gtk.ResponseType.Apply){
				Dialog.Destroy();
				return;
			}

			ApplyChanges();

			return;
		}

		private class CategoryExpander: Gtk.Expander
		{
			private string label;
			private int loaded = 0;
			private int available = 0;

			public CategoryExpander(string label): base(label)
			{
				UseMarkup = true;
				UpdateLabel();
				this.label = label;
				Gtk.Alignment al = new Gtk.Alignment(0,0,1,0);
				al.LeftPadding = 12;
				al.Add(new Gtk.VBox());
				Add(al);
			}

			private void UpdateLabel()
			{
				Label = String.Format("<b>{0} </b> (<b>{1}</b>/<i>{2}</i>)", label, loaded, available);
			}

			private Gtk.VBox VBox {
				get { return (Child as Gtk.Alignment).Child as Gtk.VBox; }	
			}

			public void AddChild(ExtensionCheck w)
			{
				VBox.PackStart(w, false,false, 2);	
				available ++;
				if ((bool)w.Extension.Preferences.Get("Enabled",true))
					loaded++;
				UpdateLabel();
			}
		}

		private class CategoryTree: Gtk.VBox
		{
			private Hashtable categories = new Hashtable();

			public void AddInCategory(ExtensionCheck w, ExtensionCategory category)
			{
				if (!categories.ContainsKey(category)) {
					Gtk.VBox vb = new Gtk.VBox();
					CategoryExpander exp = new CategoryExpander(category.ToString());
					PackStart(exp, false, false, 4);
					categories.Add(category,exp);
				}
				(categories[category] as CategoryExpander).AddChild(w);
			}

		}

		private class ExtensionCheck: Gtk.HBox
		{
			Gtk.Label label;
			Gtk.CheckButton check;
			Extension extension;

			public event EventHandler Toggled;

			public ExtensionCheck(Extension extension)
			{
				this.extension = extension;
				label = new Gtk.Label(extension.Name);
				label.Xalign = 0;
				
				check = new Gtk.CheckButton();
				check.Active = (bool) extension.Preferences.Get("Enabled", true);
				check.Toggled += HandleToggled;

				PackStart(label, true, true, 0);
				PackStart(check, false, false, 0);
				Homogeneous = false;
				Spacing = 4;
				ShowAll();
			}

			private void HandleToggled(object o, EventArgs e)
			{
				Toggled(this, e);	
			}

			public bool Active {
				get { return check.Active; }
			}

			public Extension Extension
			{
				get { return extension; }
			}
		}
	}
}
