#!/usr/bin/env python3

from gi.repository import GObject, Gio
from gi.repository import Ide

import shutil

def detect_ninja():
    if shutil.which('ninja-build'):
        return 'ninja-build'
    return 'ninja'

def detect_meson():
    if shutil.which('meson.py'):
        return 'meson.py'
    return 'meson'

class MesonBuildSystem(Ide.Object, Ide.BuildSystem, Gio.AsyncInitable):
    project_file = GObject.Property(type=Gio.File)

    def do_init_async(self, io_priorty, cancellable, callback, data):
        task = Gio.Task.new(self, cancellable, callback)
        meson_command = detect_meson()

        # This is all done synchronously, doing it in a thread would probably
        # be somewhat ideal although unnecessary at this time.

        try:
            # Maybe this is a build definition file
            if self.props.project_file.get_basename() == 'meson.build':
                task.return_boolean(True)
                return

            # Maybe this is a directory with a build definition file
            if self.props.project_file.query_file_type() == Gio.FileType.DIRECTORY:
                child = self.props.project_file.get_child('meson.build')
                if child.query_exists(None):
                    self.props.project_file = child
                    task.return_boolean(True)
                    return
        except Exception as ex:
            task.return_error(ex)

        # At this point do the following:
        # rc = subprocess.call([meson_command, build_dir, source_dir])
        # error out if rc is not zero

        raise NotImplemented

    def do_init_finish(self, result):
        return result.propagate_boolean()

    def do_get_priority(self):
        # Priority is used to determine the order of discovery
        return 1000

    def do_get_build_flags_async(self, ifile, cancellable, callback, data):
        # GTask sort of is painful from Python.
        # We can use it to attach some data to return from the finish
        # function though.
        task = Gio.Task.new(self, cancellable, callback)
        task.build_flags = []
        task.return_boolean(True)
        # Getting build flags in Meson (and also CMake) is done with
        # the compile_commands database.
        #
        # Steps are as follows:
        #
        # Open $build_dir/compile_commands.json
        #
        # Go through the array of objects and match the "file" attribute
        # against the filename to be checked. Note that the file may be either
        # an absolute path or relative in the format
        # path_from_build_dir_to_src_dir/subdir/filename.c
        #
        # Extract build arguments from the "command" attribute.

    def do_get_build_flags_finish(self, result):
        if task.propagate_boolean():
            return result.build_flags
        raise RuntimeError

    def do_get_builder(self, config):
        return MesonBuilder(config)

class MesonBuilder(Ide.Builder):
    def __init__(self, config, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.config = config
        self.ninja_command = detect_ninja()

    def do_build_async(self, flags, cancellable, callback, data):
        task = Gio.Task.new(self, cancellable, callback)
        task.build_result = MesonBuildResult(self.config, flags, self.ninja_command)

        def wrap_execute():
            try:
                task.build_result.execute()
                task.return_boolean(True)
            except Exception as ex:
                task.return_error(ex)

        thread = threading.Thread(target=wrap_execute)
        thread.start()

    def do_build_finish(self, result):
        return result.build_result

class MesonBuildResult(Ide.BuildResult):
    def __init__(self, config, flags, ninja_command, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.flags = flags
        self.ninja_command = ninja_command

        # execute() runs in a thread, so probably want to extract what
        # you need from the configuration here in the main loop.
        self.config = config

    def execute(self):
        rc = subprocess.call(self.ninja_command)

        if rc == 0:
            self.set_mode('Successful')
            self.set_failed(False)
            self.set_running(False)
        else:
            self.set_mode('Failure')
            self.set_failed(True)
            self.set_running(False)

