#!/bin/powershell
#
# $Maintainer: michael.waeber $
#    $Created: michael.waeber 2014-08-26 $
#  $Copyright: Michael Waeber. All rights reserved. $
#    $License: Modified BSD license $
#
<#
.SYNOPSIS
    Transform an input diagram to an image file used by doxygen

    diagram_converter.ps1 -o OUTFILE -f INFILE

.DESCRIPTION
    Automated build: Export UMLet diagram files to image files for
    including in the generated doxygen documentation. This script is
    called from the doxygen.exe program (see doxygen configuration
    EXTIMAGE_CONVERTER).

.EXAMPLE
    diagram_converter.ps1 -o OUTFILE -f INFILE

.PARAMETER f
    Input file
.PARAMETER o
     Output file, format is recognized from file extension
#>


#-------------------------------------------------------------------------------
# Script input parameters
#-------------------------------------------------------------------------------
PARAM
(
    # output file
    [Parameter(Mandatory=$true)]
    [string]
    $o,

    # input file
    [Parameter(Mandatory=$true)]
    [string]
    $f
)


#-------------------------------------------------------------------------------
# Configuration
#-------------------------------------------------------------------------------
$umletJar   = "d:/M/thirdparty/umlet/current/umlet.jar";
$diaExe     = "d:/PDev/dia/DiaPortable/App/Dia/bin/diaw.exe";


#-------------------------------------------------------------------------------
# Variable definitions
#-------------------------------------------------------------------------------
$scriptName     = $MyInvocation.MyCommand.Name
$inputFile      = $f
$outputFile     = $o


#-------------------------------------------------------------------------------
# Functions
#-------------------------------------------------------------------------------
function Get-Format([string]$filename)
{
    $format  = [System.IO.Path]::GetExtension($filename).ToLower()
    if($format -eq $null -or $format.Length -lt 2)
    {
        throw [system.FormatException]"Invalid file extension"
    }

    $format.SubString(1)
}

function Call-Exe([string]$exeFile, [string]$arguments)
{
    $info = New-Object System.Diagnostics.ProcessStartInfo
    $info.FileName                  = $exeFile;
    $info.RedirectStandardError     = $true
    $info.RedirectStandardOutput    = $true
    $info.UseShellExecute           = $false
    $info.Arguments                 = $arguments
    $proc = New-Object System.Diagnostics.Process
    $proc.StartInfo = $info
    $proc.Start() | Out-Null
    $proc.WaitForExit()
    if($proc.ExitCode -ne 0)
    {
        throw "$exeFile exit with code " + $proc.ExitCode
    }
}

function Call-Jar([string]$jarFile, [string]$arguments)
{
    Call-Exe "java.exe" "-jar $jarFile $arguments"
}


#-------------------------------------------------------------------------------
# Main
#-------------------------------------------------------------------------------
try
{
    $inputFormat        = Get-Format $inputFile
    $outputFormat       = Get-Format $outputFile
    $outputFileNoExt    = [System.IO.Path]::Combine(
        [System.IO.Path]::GetDirectoryName($outputFile),
        [System.IO.Path]::GetFileNameWithoutExtension($outputFile))

    switch($inputFormat)
    {
        "uxf"
        {
            $supportedFormats = "png", "svg", "eps", "pdf", "bmp", "gif", "jpg"
            if($supportedFormats -contains $outputFormat)
            {
                $umletArgs =
                    "-action=convert",
                    "-format=$outputFormat",
                    "-filename=`"$inputFile`"",
                    "-output=`"$outputFileNoExt`"";
                Call-Jar $umletJar $umletArgs;
            }
            else
            {
                throw [system.NotImplementedException]"Output format $outputFormat for UMLet is not supported"
            }
        }

        "dia"
        {
            $supportedFormats = "png", "eps", "pdf"
            if($supportedFormats -contains $outputFormat)
            {
                $diaArgs = "--filter=$outputFormat", "-n", "--export=`"$outputFile`"", "`"$inputFile`"";
                Write-Host $diaExe $diaArgs;
                &$diaExe @($diaArgs);
            }
            else
            {
                throw [system.NotImplementedException]"Output format $outputFormat for Dia is not supported"
            }
        }

        default
        {
            throw [system.NotImplementedException]"Input format $inputFormat is not supported"
        }
    }

    [System.Environment]::Exit(0)
}
catch
{
    echo "Error in $scriptName : $error"
    [System.Environment]::Exit(1)
}

# eof

