#!/bin/bash
#.SYNOPSIS
#    Transform an input diagram to an image file used by doxygen
#
#    diagram_converter.sh -o OUTFILE -f INFILE
#
#.DESCRIPTION
#    Automated build: Export UMLet diagram files to image files for
#    including in the generated doxygen documentation. This script is
#    called from the doxygen.exe program (see doxygen configuration
#    EXTIMAGE_CONVERTER).
#
#.EXAMPLE
#    diagram_converter.sh -o OUTFILE -f INFILE
#
#.PARAMETER f
#    Input file
#.PARAMETER o
#     Output file, format is recognized from file extension
#

#-------------------------------------------------------------------------------
# Configuration
#-------------------------------------------------------------------------------
UMLET_EXE="d:/M/thirdparty/umlet/current/Umlet.exe"
DIA_EXE="d:/PDev/dia/DiaPortable/App/Dia/bin/diaw.exe"


#-------------------------------------------------------------------------------
# Functions
#-------------------------------------------------------------------------------
# Print an error message and exit script with error code 2
#
# @param $1 message
exit_error()
{
  echo Error: $1 >&2
  exit 2;
}


#-------------------------------------------------------------------------------
# Parse script arguments
#-------------------------------------------------------------------------------
inputFile=
ouputFile=

while getopts o:f: flag; do
  case $flag in
    o)
      outputFile=$OPTARG;
      ;;
    f)
      inputFile=$OPTARG;
      ;;
    ?)
      echo Usage: diagram_converter.sh -o OUTFILE -f INFILE
      exit 2;
      ;;
  esac
done
shift $(( OPTIND - 1 ));

if [[ -z $inputFile ]]; then
  exit_error "no input file specified";
fi
if [[ -z $outputFile ]]; then
  exit_error "no output file specified";
fi

# file formats (from file extension)
inputFormat=${inputFile##*.}
outputFormat=${outputFile##*.}


#-------------------------------------------------------------------------------
# Main
#-------------------------------------------------------------------------------
case $inputFormat in
  "uxf")
    case $outputFormat in
      "png"|"svg"|"eps"|"pdf"|"bmp"|"gif"|"jpg")
        $UMLET_EXE -action=convert -format=$outputFormat -filename="$inputFile" -output="${outputFile%.*}"  || exit 1;
        ;;

      *)
        exit_error "Unsupported output format for UMLet files";
        exit 1;
        ;;
    esac
    ;;

  "dia")
    case $outputFormat in
      "png"|"eps"|"pdf")
        $DIA_EXE --filter=$outputFormat -n --export="$outputFile" "$inputFile" || exit 1;
        ;;

      *)
        exit_error "Unsupported output format for Dia files";
        ;;
    esac
    ;;

  *)
    exit_error "Unsupported input format $inputFormat";
esac

exit 0;
