#include "stdafx.h"
#include "GstreamerDemo.h"
#include <stdio.h>
#include <string.h>
#include <gst\video\videooverlay.h>




/* This method is called by the idle GSource in the mainloop, to feed CHUNK_SIZE bytes into appsrc.
* The ide handler is added to the mainloop when appsrc requests us to start sending data (need-data signal)
* and is removed when appsrc has enough data (enough-data signal).
*/
gboolean GstreamerDemo::push_data( CustomData *data )
{
	GstBuffer *buffer;
	GstFlowReturn ret;
	GstMemory* mem;
	GstMapInfo info;
	int i;
	gint16 *raw;
	gint num_samples = CHUNK_SIZE / 2; /* Because each sample is 16 bits */
	gfloat freq;

	/* Create a new empty buffer */
	buffer = gst_buffer_new();

	mem = gst_allocator_alloc(NULL, CHUNK_SIZE, NULL);
	gst_buffer_append_memory(buffer, mem);
	gst_buffer_map(buffer, &info, GST_MAP_WRITE);

	/* Set its timestamp and duration */
	GST_BUFFER_TIMESTAMP (buffer) = gst_util_uint64_scale (data->num_samples, GST_SECOND*5, SAMPLE_RATE);
	GST_BUFFER_DURATION (buffer) = gst_util_uint64_scale (CHUNK_SIZE, GST_SECOND*5, SAMPLE_RATE);

	/* Generate some psychodelic waveforms */
	raw = (gint16*) malloc(sizeof(guint16) * num_samples);
	data->c += data->d;
	data->d -= data->c / 1000;
	freq = 1100 + 1000 * data->d;
	for (i = 0; i < num_samples; i++) {
		data->a += data->b;
		data->b -= data->a / freq;
		raw[i] = (gint16)(500 * data->a);
	}
	data->num_samples += num_samples;

	memcpy(info.data, raw, num_samples);

	free(raw);

	/* Push the buffer into the appsrc */
	g_signal_emit_by_name (data->app_source, "push-buffer", buffer, &ret);

	/* Free the buffer now that we are done with it */
	gst_buffer_unmap(buffer, &info);
	//gst_memory_unmap(mem, &info);
	gst_buffer_unref(buffer);

	g_object_set(data->videobox,
		"left", data->i,
		"right", data->i,
		"top", data->i,
		"bottom", data->i,
		NULL);

	if (data->i == 10)
		data->add = FALSE;
	else if (data->i == -10)
		data->add = TRUE;

	if (data->add)
		data->i += 1;
	else
		data->i -= 1;

	if (ret != GST_FLOW_OK) {
		/* We got some error, stop sending data */
		return FALSE;
	}

	return TRUE;
}

/* This signal callback triggers when appsrc needs data. Here, we add an idle handler
* to the mainloop to start pushing data into the appsrc */
void GstreamerDemo::start_feed( GstElement *source, guint size, CustomData *data )
{
	if (data->sourceid == 0) {
		g_print ("Start feeding\n");
		data->sourceid = g_idle_add ((GSourceFunc) push_data, data);
	}
}

/* This callback triggers when appsrc has enough data and we can stop sending.
* We remove the idle handler from the mainloop */
void GstreamerDemo::stop_feed( GstElement *source, CustomData *data )
{
	if (data->sourceid != 0) {
		g_print ("Stop feeding\n");
		g_source_remove (data->sourceid);
		data->sourceid = 0;
	}
}

/* This function is called when an error message is posted on the bus */
gboolean GstreamerDemo::error_cb( GstBus *bus, GstMessage *msg, CustomData *data )
{
	GST_DEBUG("got message %s", gst_message_type_get_name(GST_MESSAGE_TYPE(msg)));

	switch (GST_MESSAGE_TYPE(msg))
	{
	case GST_MESSAGE_ERROR: 
		{
			GError* err = NULL;
			gchar* dbg_info = NULL;

			gst_message_parse_error(msg, &err, &dbg_info);
			printf("ERROR from element %s: %s\n", GST_OBJECT_NAME(msg->src), err->message);
			printf("Debugging info: %s\n", (dbg_info) ? dbg_info : "none");
			g_error_free(err);
			g_free(dbg_info);
			g_main_loop_quit(data->main_loop);
			break;
		}
	case GST_MESSAGE_EOS:
		g_main_loop_quit(data->main_loop);
		break;
	default:
		break;
	}
	return TRUE;
}

GstreamerDemo::GstreamerDemo()
{
	/* Initialize cumstom data structure */
	memset (&data, 0, sizeof (data));
	data.b = 1; /* For waveform generation */
	data.d = 1;
	data.i = 5;
	data.add = FALSE;

	thread_gstreamer = NULL;
}

GstreamerDemo::~GstreamerDemo()
{
	/* Free resources */
	g_main_loop_quit(data.main_loop);
	gst_element_set_state (data.pipeline, GST_STATE_NULL);
	gst_object_unref (data.pipeline);
	g_main_loop_unref(data.main_loop);

	if (thread_gstreamer)
	{
		thread_gstreamer->join();
		delete thread_gstreamer;
	}
}

void GstreamerDemo::init(HWND hwnd)
{
	GstBus *bus;
	gchar *audio_caps_text;
	GstCaps *audio_caps;

	/* Initialize GStreamer */
	gst_init (NULL, NULL);

	/* Create the elements */
	data.app_source = gst_element_factory_make ("appsrc", "audio_source");
	data.visual = gst_element_factory_make ("wavescope", "visual");
	data.video_convert = gst_element_factory_make ("videoconvert", "csp");
	data.videobox = gst_element_factory_make("videobox", "filter");
	data.video_sink = gst_element_factory_make ("d3dvideosink", "video_sink");

	/* Create the empty pipeline */
	data.pipeline = gst_pipeline_new ("test-pipeline");

	if (!data.pipeline || !data.app_source || !data.visual || !data.video_convert || !data.videobox || !data.video_sink) {
		g_printerr ("Not all elements could be created.\n");
		return;
	}

	/* Configure wavescope */
	g_object_set (data.visual, "shader", 0, "style", 1, NULL);

	/* Configure appsrc */
	audio_caps_text = g_strdup_printf (AUDIO_CAPS, SAMPLE_RATE);
	audio_caps = gst_caps_from_string (audio_caps_text);
	g_object_set (data.app_source, "caps", audio_caps, "format", GST_FORMAT_TIME, NULL);
	g_signal_connect (data.app_source, "need-data", G_CALLBACK (start_feed), &data);
	g_signal_connect (data.app_source, "enough-data", G_CALLBACK (stop_feed), &data);

	/* Link all elements that can be automatically linked because they have "Always" pads */
	gst_bin_add_many (GST_BIN (data.pipeline), data.app_source, data.visual, data.video_convert, data.videobox, data.video_sink, NULL);
	if (gst_element_link_many (data.app_source, data.visual, data.video_convert, data.videobox, data.video_sink, NULL) != TRUE) 
	{
		g_printerr ("Elements could not be linked.\n");
		gst_object_unref (data.pipeline);
		return;
	}

	GstVideoOverlay* ov = (GstVideoOverlay*)data.video_sink;	// nastavit d3dvideosink na handler existujuceho okna
	gst_video_overlay_set_window_handle(ov, (gulong)hwnd);

	/* Start playing the pipeline */
	gst_element_set_state (data.pipeline, GST_STATE_PLAYING);
	/* Create a GLib Main Loop and set it to run */
	data.main_loop = g_main_loop_new (NULL, FALSE);
	
	thread_gstreamer = new boost::thread(boost::BOOST_BIND(&GstreamerDemo::StartTask, this));

	return;
}

void GstreamerDemo::StartTask()
{
	g_main_loop_run (data.main_loop);	
}