/*
 * MpegTimestampParser.c
 * Air Video Server
 *
 * Copyright 2012 InMethod s.r.o.
 * Author: matej.knopp@inmethod.com
 */

#include "MpegTimestampParser.h"
#include <stdio.h>

GST_DEBUG_CATEGORY_STATIC (mpeg_timestamp_parser_debug);
#define GST_CAT_DEFAULT mpeg_timestamp_parser_debug

#define CAPS \
    GST_STATIC_CAPS ("video/mpeg, " \
        "mpegversion = (int) [1, 2], parsed=(boolean) true, systemstream = (boolean) false") 
    

static GstStaticPadTemplate sink_factory = GST_STATIC_PAD_TEMPLATE("sink",
                                                                   GST_PAD_SINK,
                                                                   GST_PAD_ALWAYS,
                                                                   CAPS);

static GstStaticPadTemplate src_factory = GST_STATIC_PAD_TEMPLATE ("src",
                                                                   GST_PAD_SRC,
                                                                   GST_PAD_ALWAYS,
                                                                   CAPS);

G_DEFINE_TYPE(MpegTimestampParser, mpeg_timestamp_parser, GST_TYPE_ELEMENT);

static void mpeg_timestamp_parser_set_property(GObject *object, guint prop_id,
                                       const GValue *value, GParamSpec *pspec);

static void mpeg_timestamp_parser_get_property(GObject *object, guint prop_id,
                                       GValue *value, GParamSpec *pspec);


static void mpeg_timestamp_parser_reset(MpegTimestampParser *parser);

static GstFlowReturn mpeg_timestamp_parser_chain(GstPad *pad, GstObject *object, GstBuffer *buf);

static gboolean mpeg_timestamp_parser_sink_event(GstPad *pad, GstObject *object, GstEvent *event);

static GstStateChangeReturn mpeg_timestamp_parser_change_state(GstElement *element, GstStateChange transition);

enum {
    PROP_0,
    PROP_SILENT
};

static void
mpeg_timestamp_parser_class_init(MpegTimestampParserClass *klass)
{
    GObjectClass *gobject_class;
    GstElementClass *element_class;
    
    gobject_class = (GObjectClass *) klass;
    element_class = (GstElementClass *) klass;
    
    gobject_class->set_property = mpeg_timestamp_parser_set_property;
    gobject_class->get_property = mpeg_timestamp_parser_get_property;
    
    element_class->change_state = mpeg_timestamp_parser_change_state;
    
    g_object_class_install_property(gobject_class, PROP_SILENT,
                                    g_param_spec_boolean ("silent", "Silent",
                                                          "Whether the element shouldn't log anything", FALSE,
                                                          G_PARAM_CONSTRUCT | G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));
    
    gst_element_class_set_details_simple(element_class,
                                         "MpegTimestampParser",
                                         "Codec/Parser/Video",
                                         "Corrects timestamps for MPEG stream",
                                         "matej.knopp@inmethod.com");
    
    gst_element_class_add_pad_template(element_class,
                                       gst_static_pad_template_get(&src_factory));
    gst_element_class_add_pad_template(element_class,
                                       gst_static_pad_template_get(&sink_factory));
    
}

static void
mpeg_timestamp_parser_init(MpegTimestampParser *filter)
{
    filter->sinkpad = gst_pad_new_from_static_template(&sink_factory, "sink");
    
    gst_pad_set_chain_function(filter->sinkpad,
                               GST_DEBUG_FUNCPTR(mpeg_timestamp_parser_chain));

    gst_pad_set_event_function(filter->sinkpad,
                               GST_DEBUG_FUNCPTR(mpeg_timestamp_parser_sink_event));
    
    GST_PAD_SET_PROXY_CAPS(filter->sinkpad);
    
    filter->srcpad = gst_pad_new_from_static_template(&src_factory, "src");    
    
    gst_element_add_pad(GST_ELEMENT(filter), filter->sinkpad);
    gst_element_add_pad(GST_ELEMENT(filter), filter->srcpad);
    
    mpeg_timestamp_parser_reset(filter);
}

static void
mpeg_timestamp_parser_set_property(GObject *object, guint prop_id,
                           const GValue *value, GParamSpec *pspec)
{
    MpegTimestampParser *filter = MPEG_TIMESTAMP_PARSER(object);
    
    switch (prop_id) 
    {
        case PROP_SILENT:
            filter->silent = g_value_get_boolean(value);
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID(filter, prop_id, pspec);
            break;
    }
}

static void
mpeg_timestamp_parser_get_property(GObject *object, guint prop_id,
                           GValue *value, GParamSpec *pspec)
{
    MpegTimestampParser *filter = MPEG_TIMESTAMP_PARSER(object);
    
    switch (prop_id) 
    {
        case PROP_SILENT:
            g_value_set_boolean(value, filter->silent);
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (filter, prop_id, pspec);
            break;
    }
}

static GstFlowReturn
mpeg_timestamp_parser_push_buffer(MpegTimestampParser *filter, GstBuffer *buffer)
{
//    fprintf(stderr, "OUT %" GST_TIME_FORMAT " %" GST_TIME_FORMAT " %i\n", GST_TIME_ARGS(GST_BUFFER_PTS(buffer)), GST_TIME_ARGS(GST_BUFFER_DTS(buffer)), (int)gst_buffer_get_size(buffer));
    return gst_pad_push(filter->srcpad, buffer);
}

// Set buffer PTS to next PTS and update next_pts
static void
mpeg_timestamp_parser_update_ts(MpegTimestampParser *filter, GstBuffer *buffer, gboolean update_pts, gboolean update_dts)
{
    if (update_pts)
    {
        if (GST_BUFFER_IS_DISCONT(buffer))
        {
            filter->next_pts = GST_CLOCK_TIME_NONE;
        }

        if (!GST_BUFFER_PTS_IS_VALID(buffer))
        {
            GST_BUFFER_PTS(buffer) = filter->next_pts;
        }

        if (GST_BUFFER_PTS_IS_VALID(buffer) && GST_BUFFER_DURATION_IS_VALID(buffer))
        {
            filter->next_pts = GST_BUFFER_PTS(buffer) + GST_BUFFER_DURATION(buffer);
        }
        else
        {
            filter->next_pts = GST_CLOCK_TIME_NONE;
        }
    }

    if (update_dts)
    {
        if (GST_BUFFER_IS_DISCONT(buffer))
        {
            filter->next_dts = GST_CLOCK_TIME_NONE;
        }
        
        if (!GST_BUFFER_DTS_IS_VALID(buffer))
        {
            GST_BUFFER_DTS(buffer) = filter->next_dts;
        }
        
        if (GST_BUFFER_DTS_IS_VALID(buffer) && GST_BUFFER_DURATION_IS_VALID(buffer))
        {
            filter->next_dts = GST_BUFFER_DTS(buffer) + GST_BUFFER_DURATION(buffer);
        }
        else
        {
            filter->next_dts = GST_CLOCK_TIME_NONE;
        }
    }
}

static GstFlowReturn
mpeg_timestamp_parser_flush(MpegTimestampParser *filter)
{
    GList *l;
    GstFlowReturn ret = GST_FLOW_OK;
    
    // The b-frames will be displayed before the stored I or P frame
    for (l = filter->b_frames; l; l = l->next)
    {
        mpeg_timestamp_parser_update_ts(filter, l->data, TRUE, FALSE);
    }
    
    if (filter->i_or_p_frame)
    {
        // then the I or P frame is displayed
        mpeg_timestamp_parser_update_ts(filter, filter->i_or_p_frame, TRUE, TRUE);
        
        // I or P frame came first
        ret = mpeg_timestamp_parser_push_buffer(filter, filter->i_or_p_frame);
    }
    
    // after I or P send the B frames
    for (l = filter->b_frames; l; l = l->next)
    {
        mpeg_timestamp_parser_update_ts(filter, l->data, FALSE, TRUE);
        
        ret = mpeg_timestamp_parser_push_buffer(filter, GST_BUFFER(l->data));
    }
        
    // cleanup
    g_list_free(filter->b_frames);
    filter->b_frames = NULL;
    filter->i_or_p_frame = NULL;
    
    // return last result
    return ret;
}

// Determine the timestamps according to 6.1.1.1 of ISO/IEC 13818-2
static GstFlowReturn
mpeg_timestamp_parser_process_buffer(MpegTimestampParser *filter, GstBuffer *buffer, int pic_type)
{
    if (pic_type == GST_MPEG_VIDEO_PICTURE_TYPE_D)
    {
        gst_buffer_unref(buffer); // we don't need this
        return GST_FLOW_OK;
    }
    
    if (pic_type == GST_MPEG_VIDEO_PICTURE_TYPE_B)
    {
        if (filter->i_or_p_frame != NULL) // there is a frame waiting for timestamp; it will be displayed after all b-frames, so we need to queue the b-frame
        {
            filter->b_frames = g_list_append(filter->b_frames, buffer);
            return GST_FLOW_OK;
        }
        else
        {
            // no frame waiting for timestamp, we can output the b-frame
            mpeg_timestamp_parser_update_ts(filter, buffer, TRUE, TRUE);
            return mpeg_timestamp_parser_push_buffer(filter, buffer);
        }
    }
    else
    {
        GstFlowReturn ret = GST_FLOW_OK;
        if (filter->i_or_p_frame)
        {
            // got I or P waiting for timestamp, all b-frames (if any) are queued already, we can flush it
            ret = mpeg_timestamp_parser_flush(filter);
        }

        if (!GST_BUFFER_PTS_IS_VALID(buffer))
        {
            // to calculate the timestamp, we might need to wait for all b frames that follow (if any), so queue the buffer
            filter->i_or_p_frame = buffer;
        }
        else
        {
            mpeg_timestamp_parser_update_ts(filter, buffer, TRUE, TRUE); // update DTS (if necessary), calculate next_pts and next_dts
            ret = mpeg_timestamp_parser_push_buffer(filter, buffer);
        }
        return ret;
    }
}

static GstFlowReturn
mpeg_timestamp_parser_chain(GstPad *pad, GstObject *object, GstBuffer *buf)
{
	MpegTimestampParser *filter = MPEG_TIMESTAMP_PARSER(object);

    int repeat_count = 1;
    guint offset = 0;
    GstMapInfo map;
    GstMpegVideoPacket packet;
    int pic_type = -1;
    
    //fprintf(stderr, "IN: %" GST_TIME_FORMAT " %" GST_TIME_FORMAT " %i\n", GST_TIME_ARGS(GST_BUFFER_PTS(buf)), GST_TIME_ARGS(GST_BUFFER_DURATION(buf)), (int) gst_buffer_get_size(buf));
    
    gst_buffer_map(buf, &map, GST_MAP_READ);
    
    while (offset < map.size && gst_mpeg_video_parse(&packet, map.data, map.size, offset))
    {
        if (packet.type == GST_MPEG_VIDEO_PACKET_PICTURE)
        {
            GstMpegVideoPictureHdr hdr;
            if (gst_mpeg_video_parse_picture_header(&hdr, map.data, map.size, packet.offset))
            {
                if (pic_type != -1)
                {
                    GST_WARNING_OBJECT(filter, "Received multiple frames in one buffer. "
                                       "Parser doesn't seem to be splitting frames properly.");
                }
                pic_type = hdr.pic_type;
            }
        }
        
        if (packet.type == GST_MPEG_VIDEO_PACKET_SEQUENCE)
        {
            GstMpegVideoSequenceHdr hdr;
            if (gst_mpeg_video_parse_sequence_header(&hdr, map.data, map.size, packet.offset))
            {
                filter->sequence_hdr = hdr;
                if (filter->fps_num == 0 || filter->fps_den == 0)
                {
                    filter->fps_num = hdr.fps_n;
                    filter->fps_den = hdr.fps_d;
                }
            }
        }
        
        if (packet.type == GST_MPEG_VIDEO_PACKET_EXTENSION)
        {
            GstMpegVideoSequenceExt sequence_ext;
            GstMpegVideoPictureExt picture_ext;
            
            if (gst_mpeg_video_parse_sequence_extension(&sequence_ext, map.data, map.size, packet.offset))
            {
                filter->sequence_ext = sequence_ext;     
                filter->fps_num = filter->sequence_hdr.fps_n * (filter->sequence_ext.fps_n_ext + 1);
                filter->fps_den = filter->sequence_hdr.fps_d * (filter->sequence_ext.fps_d_ext + 1);
            }
            
            if (gst_mpeg_video_parse_picture_extension(&picture_ext, map.data, map.size, packet.offset))
            {
                if (picture_ext.repeat_first_field) 
                {
                    if (filter->sequence_ext.progressive)
                    {
                        if (picture_ext.top_field_first)
                            repeat_count = 5;
                        else
                            repeat_count = 3;
                    } 
                    else if (picture_ext.progressive_frame) 
                    {
                        repeat_count = 2;
                    }
                }
            }
        }
        if (packet.size > 0)
        {
            offset = packet.offset + packet.size;
        }
        else
        {
            break;
        }
    }

    gst_buffer_unmap(buf, &map);
    
    if (filter->fps_num != 0 && filter->fps_den != 0) 
    {
        GST_BUFFER_DURATION(buf) = GST_SECOND * filter->fps_den / filter->fps_num;
    }
    
    if (GST_CLOCK_TIME_IS_VALID (GST_BUFFER_DURATION(buf))) 
    {
        GST_BUFFER_DURATION(buf) = (1 + repeat_count) * GST_BUFFER_DURATION (buf) / 2;
    }
    
//    fprintf(stderr, "IN %i: %" GST_TIME_FORMAT " %" GST_TIME_FORMAT " %i\n",  pic_type, GST_TIME_ARGS(GST_BUFFER_PTS(buf)), GST_TIME_ARGS(GST_BUFFER_DURATION(buf)), (int) gst_buffer_get_size(buf));
    
    return mpeg_timestamp_parser_process_buffer(filter, buf, pic_type);
}

static void
mpeg_timestamp_parser_reset(MpegTimestampParser *parser)
{
    if (parser->i_or_p_frame)
    {
        gst_buffer_unref(parser->i_or_p_frame);
    }
    g_list_free_full(parser->b_frames, (GDestroyNotify)gst_buffer_unref);
    parser->i_or_p_frame = NULL;
    parser->b_frames = NULL;
    parser->next_pts = GST_CLOCK_TIME_NONE;
    parser->next_dts = GST_CLOCK_TIME_NONE;
}

static gboolean
mpeg_timestamp_parser_sink_event(GstPad *pad, GstObject *object, GstEvent *event)
{
    MpegTimestampParser *filter = MPEG_TIMESTAMP_PARSER(object);
    
    if (GST_EVENT_TYPE(event) == GST_EVENT_EOS)
    {
        mpeg_timestamp_parser_flush(filter);
    }
    else if (GST_EVENT_TYPE(event) == GST_EVENT_FLUSH_STOP)
    {
        mpeg_timestamp_parser_reset(filter);
    }
    return gst_pad_event_default(pad, object, event);
}

static GstStateChangeReturn
mpeg_timestamp_parser_change_state(GstElement *element, GstStateChange transition)
{
    if (transition == GST_STATE_CHANGE_READY_TO_NULL ||
        transition == GST_STATE_CHANGE_NULL_TO_READY)
    {
        mpeg_timestamp_parser_reset(MPEG_TIMESTAMP_PARSER(element));
    }
    return GST_ELEMENT_CLASS(mpeg_timestamp_parser_parent_class)->change_state(element, transition);
}

static gboolean
plugin_init(GstPlugin *mpeg_timestamp_parser)
{
    GST_DEBUG_CATEGORY_INIT (mpeg_timestamp_parser_debug, 
                             "mpegtimestampparser",
                             0, 
                             "Corrects timestamps for MPEG stream");
    
    return gst_element_register(mpeg_timestamp_parser, 
                                "mpegtimestampparser",
                                GST_RANK_PRIMARY,
                                TYPE_MPEG_TIMESTAMP_PARSER);
}

void mpeg_timestamp_parser_register(void)
{
    gst_plugin_register_static(GST_VERSION_MAJOR,
                               GST_VERSION_MINOR,
                               "mpegtimestampparser",
                               "Corrects timestamps for MPEG stream",
                               plugin_init,
                               "1",
                               "Proprietary",
                               "InMethod",
                               "Air Video",
                               "http://inmethod.com"
                               );
}

