--------------------------------------------------------------------------------
-- addressable_shift_register_rtl.vhdl
--------------------------------------------------------------------------------
--! \file addressable_shift_register_rtl.vhdl
--! \brief Implementation of the addressable_shift_register component.
--!
--! \class addressable_shift_register
--! \ingroup my_design_lib
--! \brief Simple, dynamic shift register using distributed memory.
--!
--! This module implements a small, 2**LOG2_DEPTH-deep by NBITS_DATA-bit
--! wide shift register with an addressable output.
--!
--! Data are written to the shift register at the input clock rate whilst
--! DATA_IN_CE is asserted.
--!
--! The output data are registered.  This means that there is a one
--! clock cycle delay on the addressable data port.
--!
--------------------------------------------------------------------------------
-- $Revision: 6$
--------------------------------------------------------------------------------

-- hds interface_start
library ieee;
use     ieee.std_logic_1164.all;
use     ieee.numeric_std.all;

entity addressable_shift_register is

    generic
    (

        --! Width of the data ports
        NBITS_DATA              : positive := 16;

        --! Depth of the shift register, 5 -> 32, 6 -> 64
        LOG2_DEPTH              : positive := 5

    );

    port
    (

        --! Clock
        CLK                     : in    std_logic;

        --! Input data
        DATA_IN                 : in    std_logic_vector (NBITS_DATA - 1 downto 0);

        --! Input data clock enable
        DATA_IN_CE              : in    std_logic;

        --! Output data
        DATA_OUT                : out   std_logic_vector (NBITS_DATA - 1 downto 0);

        --! Output data address
        DATA_OUT_ADDR           : in    std_logic_vector (LOG2_DEPTH - 1 downto 0);

        --! Output data clock enable
        DATA_OUT_CE             : in    std_logic;

        --! Output data synchronous reset
        DATA_OUT_RST            : in    std_logic

    );

end addressable_shift_register;
-- hds interface_end

--------------------------------------------------------------------------------
-- Start of architecture rtl for module my_design_lib.addressable_shift_register
--------------------------------------------------------------------------------


--! Implementation of the addressable_shift_register component.
architecture rtl of addressable_shift_register is
--! \cond

    ----------------------------------------------------------------------------
    -- Data types
    ----------------------------------------------------------------------------

    subtype DATA_WORD is std_logic_vector (NBITS_DATA - 1 downto 0);
    type DATA_WORD_ARRAY is array (natural range <>) of DATA_WORD;


    ----------------------------------------------------------------------------
    -- Constants
    ----------------------------------------------------------------------------

    constant    DEPTH                       : natural := 2**LOG2_DEPTH;


    ----------------------------------------------------------------------------
    -- Internal signals
    ----------------------------------------------------------------------------

    -- Shift register
    signal      SREG_DATA                   : DATA_WORD_ARRAY (DEPTH - 1 downto 0) := (others => (others => '0'));

    -- Output data
    signal      SREG_OUT                    : DATA_WORD := (others => '0');


--! \endcond
begin


    ----------------------------------------------------------------------------
    --! Process to implement the shift register
    ----------------------------------------------------------------------------
    SHIFT_REGISTER : process (CLK)
    begin

        -- Synchronous operation
        if rising_edge (CLK) then

            -- Add new data on DATA_IN_CE - SREG_DATA(0) is the latest value inserted
            if DATA_IN_CE = '1' then

                SREG_DATA <= SREG_DATA ( DEPTH - 2 downto 0 ) & DATA_IN;

            end if;

        end if;

    end process SHIFT_REGISTER;


    ----------------------------------------------------------------------------
    --! Register the output of the addressable shift register
    ----------------------------------------------------------------------------
    OUTPUT_REGISTER : process (CLK)
    begin

        -- Synchronous operation
        if rising_edge (CLK) then

            -- Synchronous reset
            if DATA_OUT_RST = '1' then
                SREG_OUT <= ( others => '0' );

            -- Update
            elsif DATA_OUT_CE = '1' then
                SREG_OUT <= SREG_DATA (TO_INTEGER (unsigned(DATA_OUT_ADDR)));
            end if;

        end if;

    end process OUTPUT_REGISTER;


    ----------------------------------------------------------------------------
    -- Map internal signals to outputs
    ----------------------------------------------------------------------------
    DATA_OUT <= SREG_OUT;


end architecture rtl;


--------------------------------------------------------------------------------
-- End of file addressable_shift_register_rtl.vhdl
--------------------------------------------------------------------------------

