using Cairo;
using Gdk;

int main(string[] args)
{
	regExp = /^(.*)\.([^.]*)$/;
	loop = new MainLoop();
	
	OptionContext optCtx = new OptionContext("OptionContext");
	optCtx.add_main_entries(cliOptions, null);
	optCtx.parse(ref args);
	
	main_async();
	
	loop.run();
	return global::exitCode;
}

const string fntFamily = "Liberation Serif";
const int fntSize = 10;

const OptionEntry[] cliOptions =
{
	{ "dir-in",  'i', 0, OptionArg.FILENAME, ref pathIn,  "directory containig input images",  null },
	{ "dir-out", 'o', 0, OptionArg.FILENAME, ref pathOut, "directory for processed images", null },
	{ null }
};

int exitCode;
MainLoop loop;

string pathIn;
string pathOut;
Regex regExp;

void lQuit(int exitCode = 0)
{
	global::exitCode = exitCode;
	global::loop.quit();
}

void reportError(Error err)
{
	stdout.printf("Error\n\tdomain: %s (%d)\n\tcode: %d\n\tmessage: %s\n", err.domain.to_string(), (int)err.domain, err.code, err.message);
}

async void main_async()
{
	File pathInFile;
	File pathOutFile;
	
	Idle.add(main_async.callback);
	yield;
	
	try
	{
		pathInFile = yield check_directory(pathIn, "--dir-in");
		pathOutFile = yield check_directory(pathOut, "--dir-out");
	}
	catch (Error err)
	{
		reportError(err);
		lQuit(1);
	}
	
	do_work_async(pathInFile, pathOutFile);
}

async File check_directory(string pathSrt, string optionName)
{
	FileType fileType;
	File pathFile = null;
	if (pathSrt!=null)
	{
		pathFile = File.new_for_path(pathSrt);
		if (yield file_get_type_async(pathFile, out fileType))
		{
			if (fileType!=FileType.DIRECTORY)
			{
				stdout.printf("Error: invalid value for the option %s: \"%s\" is not a directory.\n", optionName, pathFile.get_path());
				lQuit(1);
			}
		}
		else
		{
			stdout.printf("Error: invalid value for the option %s: \"%s\" does not exist.\n", optionName, pathFile.get_path());
			lQuit(1);
		}
	}
	else
	{
		stdout.printf("Error: %s value is not specified.\n", optionName);
		lQuit(1);
	}
	
	return pathFile;
}

async bool file_get_type_async(File file, out FileType fileType)
{
	FileInfo fileInfo;
	try
	{
		fileInfo = yield file.query_info_async (FileAttribute.STANDARD_TYPE, FileQueryInfoFlags.NONE);
	}
	catch(IOError err)
	{
		if (err is IOError.NOT_FOUND) return false;
		else throw err;
	}
	
	fileType = fileInfo.get_file_type();
	return true;
}

async void do_work_async(File dirIn, File dirOut)
{
	MatchInfo matchInfo;
	
	try
	{
		FileEnumerator dirEnumer = yield dirIn.enumerate_children_async(FileAttribute.STANDARD_NAME, FileQueryInfoFlags.NONE);
		
		while (true)
		{
			List<FileInfo> inFiles = yield dirEnumer.next_files_async(1);
			
			if (inFiles == null) break;
			
			var inName = inFiles.data.get_name();
			
			regExp.match(inName, 0, out matchInfo);
			if (!matchInfo.matches())
			{
				stdout.printf("%s - skipped\n", inName);
				continue;
			}
			stdout.printf("%s\n", inName);
			var name = matchInfo.fetch(1);
			
			try
			{
				yield write_title_to_file_async(name, dirIn.get_child(inName), dirOut.get_child(name + ".jpg"));
			}
			catch (Error err) { reportError(err); }
		}
	}
	catch (Error err) { reportError(err); }
	finally
	{
		lQuit();
	}
}

async void write_title_to_file_async(string title, File inFile, File outFile)
{
	FileInputStream inStream = yield inFile.read_async();
	FileOutputStream outStream = yield outFile.replace_async(null, false, FileCreateFlags.NONE);
	Pixbuf outPixbuf = write_title(title, yield Pixbuf.new_from_stream_async(inStream));
	//gboolean gdk_pixbuf_save_to_stream_finish (GAsyncResult  *async_result, GError **error);
	//         gdk_pixbuf_save_to_stream_finish (_data_->_tmp15_, _data_->_res_, &_data_->_inner_error_);
	yield outPixbuf.save_to_stream_async(outStream, "jpeg");
	yield inStream.close_async();
	yield outStream.close_async();
}

Pixbuf write_title(string title, Pixbuf inPixbuf)
{
	var width = inPixbuf.get_width();
	var height = inPixbuf.get_height();
	stdout.printf("\twidth: %d\n\theight: %d\n", width, height);
	ImageSurface srf = new Cairo.ImageSurface(Format.ARGB32, width, height);
	Context cairoCtx = new Context(srf);
	Gdk.cairo_set_source_pixbuf(cairoCtx, inPixbuf, 0, 0);
	cairoCtx.paint();
	
	var pangoLout = Pango.cairo_create_layout(cairoCtx);
	pangoLout.set_width (-1);
	pangoLout.set_height (-1);
	pangoLout.set_text (title , -1);
	
	Pango.FontDescription fntDescr = new Pango.FontDescription();
	fntDescr.set_family(fntFamily);
	fntDescr.set_size((int)(fntSize * Pango.SCALE));
	fntDescr.set_weight(Pango.Weight.ULTRAHEAVY);
	var inkRect = Pango.Rectangle();
	var logicalRect = Pango.Rectangle();
	pangoLout.set_font_description(fntDescr);
	pangoLout.get_line_readonly(0).get_extents(out inkRect, out logicalRect);
	
	var x0 = -(double)logicalRect.x / Pango.SCALE;
	var y0 = -(double)logicalRect.y / Pango.SCALE;
	//var k = width / (logicalRect.width / Pango.SCALE);
	var indentLeft = width / 40;
	var indentRight = 5;
	var k1 = ((double)height / 14) / (inkRect.height / Pango.SCALE);
	var k2 = ((double)width - indentLeft - indentRight) / (logicalRect.width / Pango.SCALE);
	var k = k1<k2 ? k1 : k2;
	var indentBottom = 5;
	var indentTop = height - indentBottom - (logicalRect.height / Pango.SCALE) * k;
	stdout.printf("\tindentLeft: %.2f\n\tindentTop:%.2f\n", indentLeft, indentTop);
	stdout.printf("\tk1: %.2f\n\tk2:%.2f\n", k1, k2);
	
	cairoCtx.save();
		cairoCtx.translate(indentLeft, indentTop);
		cairoCtx.scale(k, k);
		cairoCtx.move_to(x0, y0);
		Pango.cairo_layout_line_path (cairoCtx, pangoLout.get_line_readonly(0));
	cairoCtx.restore();
	cairoCtx.set_source_rgba(1, 1, 1, 1);
	cairoCtx.fill_preserve();
	cairoCtx.set_line_width(1);
	cairoCtx.set_source_rgba(0, 0, 0, 1);
	cairoCtx.stroke();
	
	return Gdk.pixbuf_get_from_surface(srf, 0, 0, width, height);
}