--------------------------------------------------------------------------------
-- test_component_rtl.vhdl
--------------------------------------------------------------------------------
-- Project: Test project
-- Library: my_design_lib
-- Unit:    test_component
-- View:    rtl
-- Type:    Combined VHDL entity and architecture
--------------------------------------------------------------------------------
--! \file test_component_rtl.vhdl
--! \brief Implementation of the test_component component.
--! \class test_component
--! \ingroup my_design_lib
--!
--! \brief Simple test component
--!
--! Detailed description of test component.
--!
--------------------------------------------------------------------------------
-- $Revision: 244 $
--------------------------------------------------------------------------------

-- hds interface_start
library ieee; --! IEEE library
use     ieee.std_logic_1164.all; --! For std_logic definition

entity test_component is

    port
    (

        --! System clock
        CLK                 : in    std_logic;

        --! Bypass enable (active high)
        BYPASS              : in    std_logic;

        -- Input
        I_IN                : in    std_logic_vector (17 downto 0); --! In-phase input data
        Q_IN                : in    std_logic_vector (17 downto 0); --! Quadrature input data
        ND                  : in    std_logic; --! New data strobe

        -- Output
        I_OUT               : out   std_logic_vector (17 downto 0); --! In-phase output data
        Q_OUT               : out   std_logic_vector (17 downto 0); --! Quadrature output data
        RDY                 : out   std_logic --! Data ready strobe

    );

end test_component;
-- hds interface_end

--------------------------------------------------------------------------------
-- Start of architecture rtl for module my_design_lib.test_component
--------------------------------------------------------------------------------

library my_design_lib; --! Design library
use     my_design_lib.design_components_pkg.all; --! Component declarations


--! Implementation of the test_component component.
architecture rtl of test_component is
--! \cond


    ----------------------------------------------------------------------------
    -- Internal signals
    ----------------------------------------------------------------------------

    -- Filter output
    signal      I_OUT_FILTER            : std_logic_vector (17 downto 0); -- lowpass_filter output
    signal      Q_OUT_FILTER            : std_logic_vector (17 downto 0); -- lowpass_filter output
    signal      RDY_FILTER              : std_logic; -- lowpass_filter output

    -- Module output
    signal      I_OUT_INT               : std_logic_vector (17 downto 0) := (others => '0');
    signal      Q_OUT_INT               : std_logic_vector (17 downto 0) := (others => '0');
    signal      RDY_INT                 : std_logic := '0';


    ----------------------------------------------------------------------------
    -- lowpass_filter component declaration
    ----------------------------------------------------------------------------
    component lowpass_filter is

        port
        (

            -- Clock
            CLK                 : in    std_logic;

            -- Input
            I_IN                : in    std_logic_vector (17 downto 0);
            Q_IN                : in    std_logic_vector (17 downto 0);
            ND                  : in    std_logic;

            -- Output
            I_OUT               : out   std_logic_vector (17 downto 0);
            Q_OUT               : out   std_logic_vector (17 downto 0);
            RDY                 : out   std_logic

        );

    end component;


    ----------------------------------------------------------------------------
    -- Embedded configurations
    ----------------------------------------------------------------------------

    -- synthesis translate_off
    for all : lowpass_filter use entity my_design_lib.lowpass_filter;
    -- synthesis translate_on


--! \endcond
begin


    ----------------------------------------------------------------------------
    --! Low pass filter
    ----------------------------------------------------------------------------
    U_FILTER : lowpass_filter

        port map
        (

            -- Clock
            CLK                 => CLK,

            -- Input
            I_IN                => I_IN,
            Q_IN                => Q_IN,
            ND                  => ND,

            -- Output
            I_OUT               => I_OUT_FILTER,
            Q_OUT               => Q_OUT_FILTER,
            RDY                 => RDY_FILTER

        );


    ----------------------------------------------------------------------------
    --! Multiplexer to allow the filter to be bypassed
    ----------------------------------------------------------------------------
    FILTER_BYPASS : process (CLK)
    begin

        -- Synchronous operation
        if rising_edge (CLK) then

            -- Default assignments
            RDY_INT <= '0';

            -- Filter is bypassed
            if BYPASS = '1' then
                if ND = '1' then
                    I_OUT_INT <= I_IN;
                    Q_OUT_INT <= Q_IN;
                    RDY_INT <= '1';
                end if;

            -- Filter is enabled
            else
                if RDY_FILTER = '1' then
                    I_OUT_INT <= I_OUT_FILTER;
                    Q_OUT_INT <= Q_OUT_FILTER;
                    RDY_INT <= '1';
                end if;
            end if;

        end if;

    end process FILTER_BYPASS;


    ----------------------------------------------------------------------------
    -- Map the registered output data to the outputs
    ----------------------------------------------------------------------------
    I_OUT <= I_OUT_INT;
    Q_OUT <= Q_OUT_INT;
    RDY <= RDY_INT;


end architecture rtl;


--------------------------------------------------------------------------------
-- End of file test_component_rtl.vhdl
--------------------------------------------------------------------------------

