//!	extended data definitions for RTL
/**
		\file		support.h
*/

//----------------------------------------------------------------------------
// R T L   K E R N E L   S U P P O R T
//----------------------------------------------------------------------------
// Name:    SYKAM/RTL/SUPPORT.H
// Purpose: RTL KERNEL SUPPORT
// Author:  hschoettner
//----------------------------------------------------------------------------
// Copyright (c) 2008-2011 Sykam GmbH. All rights reserved.
//----------------------------------------------------------------------------

#ifndef SYKAM_RTL_SUPPORT_H
#define SYKAM_RTL_SUPPORT_H 1


// Includes
#include <c55x.h>
#include <string.h>

#include <RTL.h>

// Suppress warning for "#111-D: statement is unreachable" to allow ALWAYS the macro DETACH at tasks end
#pragma diag_remark 111

// Suppress warning for "550-D: variable was set but never used" for debugging
#ifdef DEBUG
#pragma diag_remark 550
#endif


// Externals

extern U32 const os_clockrate;
extern U32 os_timernum;
extern BIT irq_lock;
extern BIT force_irq_enable;


// Definitions


// Task priorities

//!   <b>Task priority:</b>: maximum (maximum available)
#define OS_P_MAX           255

//!   <b>Task priority:</b>: realtime
#define OS_P_REALTIME      240

//!   <b>Task priority:</b>: above high
#define OS_P_ABOVEHIGH     210

//!   <b>Task priority:</b>: high
#define OS_P_HIGH          180

//!   <b>Task priority:</b>: above normal
#define OS_P_ABOVENORMAL   150

//!   <b>Task priority:</b>: normal
#define OS_P_NORMAL        120

//!   <b>Task priority:</b>: below normal
#define OS_P_BELOWNORMAL   90

//!   <b>Task priority:</b>: low
#define OS_P_LOW           60

//!   <b>Task priority:</b>: below low
#define OS_P_BELOWLOW      30

//!   <b>Task priority:</b>: idle (minimum available)
#define OS_P_IDLE          0


// Booleans

//!   <b>Boolean:</b> true
#ifndef TRUE
   #define TRUE 1
#endif

//!   <b>Boolean: </b> false
#ifndef FALSE
   #define FALSE 0
#endif


// SSemaphore

//!   <b>Macro:</b> os_sem_take (os_sem_wait) for #_SSemaphore (time-out of #_SSemaphore) (result == result of os_sem_take)
#define SSEM_TAKE(s) os_sem_take(s.oiID, s.u16To)

//!   <b>Macro:</b> os_sem_take (os_sem_wait) for #_SSemaphore (user specified time-out) (result == result of os_sem_take)
#define SSEM_TAKEUSR(s, to) os_sem_take(s.oiID, to)

//!   <b>Macro:</b> os_sem_give (os_sem_semd) for #_SSemaphore
#define SSEM_GIVE(s) os_sem_give(s.oiID)


//!   <b>Macro:</b> take semaphore at loc, copy loc.var to var and give semaphore (result == #TRUE for success)
#define SSEM_READ(sem, loc, var) SSEM_READVAR(sem, loc, var, var)

//!   <b>Macro:</b> take semaphore at loc, copy loc.var to dst and give semaphore (result == #TRUE for success)
#define SSEM_READVAR(sem, loc, var, dst) OS_SEM_TAKE_CHK(SSEM_TAKE(loc.sem)) ? (dst)= loc.var, SSEM_GIVE(loc.sem), TRUE : FALSE


//!   <b>Macro:</b> take semaphore at loc, copy var to loc.var and give semaphore (result == #TRUE for success)
#define SSEM_WRITE(sem, loc, var) SSEM_WRITEVAR(sem, loc, var, var)

//!   <b>Macro:</b> take semaphore at loc, copy src to loc.var and give semaphore (result == #TRUE for success)
#define SSEM_WRITEVAR(sem, loc, var, src) OS_SEM_TAKE_CHK(SSEM_TAKE(loc.sem)) ? loc.var= (src), SSEM_GIVE(loc.sem), TRUE : FALSE


//!   <b>Macro:</b> os_sem_take (os_sem_wait) for #_SSemaphore (time-out of #_SSemaphore) (result == #TRUE for success)
#define SSEM_LOCK(sem, loc) OS_SEM_TAKE_CHK(SSEM_TAKE(loc.sem))

//!   <b>Macro:</b> os_sem_take (os_sem_wait) for #_SSemaphore (user specified time-out) (result == #TRUE for success)
#define SSEM_LOCKUSR(sem, loc, to) OS_SEM_TAKE_CHK(SSEM_TAKEUSR(loc.sem, to))

//!   <b>Macro:</b> os_sem_give (os_sem_semd) for #_SSemaphore
#define SSEM_UNLOCK(sem, loc) SSEM_GIVE(loc.sem)

#endif
