#include <glibmm.h>
#include <giomm.h>
#include <iostream>
#include <stdlib.h>

// Description of a property: (name, value)
typedef std::pair<Glib::ustring, Glib::ustring> PropertyDescription;

static Glib::RefPtr<Gio::DBus::NodeInfo> refIntrospectionData;

// Introspection data for the service we are exporting
static const Glib::ustring introspection_xml =
  "<node>"
  "  <interface name='org.gtkmm.GDBus.TestInterface'>"
  "    <annotation name='org.gtkmm.GDBus.Annotation' value='OnInterface'/>"
  "    <annotation name='org.gtkmm.GDBus.Annotation' value='AlsoOnInterface'/>"
  "    <method name='HelloWorld'>"
  "      <annotation name='org.gtkmm.GDBus.Annotation' value='OnMethod'/>"
  "      <arg type='s' name='greeting' direction='in'/>"
  "      <arg type='s' name='response' direction='out'/>"
  "    </method>"
  "    <method name='EmitSignal'>"
  "      <arg type='d' name='speed_in_mph' direction='in'>"
  "        <annotation name='org.gtkmm.GDBus.Annotation' value='OnArg'/>"
  "      </arg>"
  "    </method>"
  "    <signal name='VelocityChanged'>"
  "      <annotation name='org.gtkmm.GDBus.Annotation' value='Onsignal'/>"
  "      <arg type='d' name='speed_in_mph'/>"
  "      <arg type='s' name='speed_as_string'>"
  "        <annotation name='org.gtkmm.GDBus.Annotation' value='OnArg_NonFirst'/>"
  "      </arg>"
  "    </signal>"
  "    <property type='s' name='FluxCapicitorName' access='read'>"
  "      <annotation name='org.gtkmm.GDBus.Annotation' value='OnProperty'>"
  "        <annotation name='org.gtkmm.GDBus.Annotation' value='OnAnnotation_YesThisIsCrazy'/>"
  "      </annotation>"
  "    </property>"
  "    <property type='s' name='Title' access='readwrite'/>"
  "    <property type='s' name='ReadingAlwaysThrowsError' access='read'/>"
  "    <property type='s' name='WritingAlwaysThrowsError' access='readwrite'/>"
  "    <property type='s' name='OnlyWritable' access='write'/>"
  "    <property type='s' name='Foo' access='read'/>"
  "    <property type='s' name='Bar' access='read'/>"
  "  </interface>"
  "</node>";

static Glib::ustring globalTitle;
static bool swap_a_and_b = false;

static void emit_signal_velocity_changed(const Glib::RefPtr<Gio::DBus::Connection>& refConnection,
  double speed_in_mph)
{
  Glib::ustring speed_as_string = Glib::ustring::compose("%1 mph!", speed_in_mph);

  // We construct a "(ds)" variant for the signal parameters
  Glib::VariantContainerBase varSignalParameters;
  Glib::Variant<double> varParam1 = Glib::Variant<double>::create(speed_in_mph);
  Glib::Variant<Glib::ustring> varParam2 =
    Glib::Variant<Glib::ustring>::create(speed_as_string);
  std::vector<Glib::VariantBase> paramVector;
  paramVector.push_back(varParam1);
  paramVector.push_back(varParam2);
  varSignalParameters = Glib::VariantContainerBase::create_tuple(paramVector);

  // At this time, we cannot emit the signal to "everybody" (i.e. with a null busname)
  //connection->emit_signal(object_path, interface_name, "VelocityChanged", "", vc);
  //  connection->emit_signal("/org/gtkmm/GDBus/TestObject", "org.freedesktop.DBus.Properties",
  //    "VelocityChanged", sender, signal_parameters);

  GError *error = NULL;
  g_dbus_connection_emit_signal(refConnection->gobj(), NULL,
    "/org/gtkmm/GDBus/TestObject", "org.freedesktop.DBus.Properties",
    "PropertiesChanged",
    varSignalParameters.gobj(), &error);
  g_assert_no_error (error);
}

static void emit_signal_properties_changed(const Glib::RefPtr<Gio::DBus::Connection>& refConnection,
  const std::vector<PropertyDescription>& properties)
{
  // We create a t-uple of parameters: (sa{sv}as)

  // s: interface name
  Glib::Variant< Glib::ustring > varInterfaceName =
    Glib::Variant< Glib::ustring >::create("org.gtkmm.GDBus.TestInterface");

  // a{sv}: changed properties
  std::map<Glib::ustring, Glib::VariantBase > changedProperties;

  Glib::Variant<Glib::ustring> varValue;

  std::vector<PropertyDescription>::const_iterator it;
  for (it = properties.begin(); it != properties.end(); it++)
  {
    varValue =
        Glib::Variant<Glib::ustring>::create(it->second);
      changedProperties[it->first] = varValue;
  }

  Glib::Variant<std::map<Glib::ustring, Glib::VariantBase > > varChangedProperties =
    Glib::Variant<std::map<Glib::ustring, Glib::VariantBase > >::create(changedProperties);

  // as: invalidated properties
  std::vector<Glib::ustring> invalidatedProperties;
  Glib::Variant< std::vector<Glib::ustring> > varInvalidatedProperties =
      Glib::Variant< std::vector<Glib::ustring> >::create(invalidatedProperties);

  // (sa{sv}as): the parameters of the signal PropertiesChanged
  std::vector<Glib::VariantBase> parameters;
  parameters.push_back(varInterfaceName);
  parameters.push_back(varChangedProperties);
  parameters.push_back(varInvalidatedProperties);
  Glib::VariantContainerBase varParameters =
    Glib::VariantContainerBase::create_tuple(parameters);

//  connection->emit_signal("/org/gtkmm/GDBus/TestObject", "org.freedesktop.DBus.Properties",
//   "PropertiesChanged", "", varParameters);

  GError *error = NULL;
  g_dbus_connection_emit_signal(refConnection->gobj(), NULL,
    "/org/gtkmm/GDBus/TestObject", "org.freedesktop.DBus.Properties",
    "PropertiesChanged",
    varParameters.gobj(), &error);
  g_assert_no_error (error);

}

static void on_method_call(const Glib::RefPtr<Gio::DBus::Connection>& refConnection,
  const Glib::ustring& /* sender */, const Glib::ustring& /* objectPath */,
  const Glib::ustring& /* interfaceName */, const Glib::ustring& methodName,
  const Glib::VariantBase& varParameters,
  const Glib::RefPtr<Gio::DBus::MethodInvocation>& invocation)
{
  std::cout << "debug: method call" << std::endl;
  const Glib::VariantContainerBase& varTuple = (Glib::VariantContainerBase&)varParameters;

  if(methodName == "HelloWorld")
  {
    Glib::Variant<Glib::ustring> varString;
    varTuple.get(varString);
    Glib::ustring greeting;
    greeting = varString.get();

    if(greeting == "Return Unregistered")
    {
      Gio::Error error(Gio::Error::FAILED_HANDLED,
        "As requested, here's an error not registered (Gio::Error::FAILED_HANDLED)");
      invocation->return_gerror(error);
    }
    else if(greeting == "Return Registered")
    {
      Gio::DBus::Error error(Gio::DBus::Error::MATCH_RULE_NOT_FOUND,
        "As requested, here's an error that is registered (Gio::DBus::Error::MATCH_RULE_NOT_FOUND)");
      invocation->return_gerror(error);
    }
    else if(greeting == "Return Raw")
    {
      invocation->return_dbus_error("org.gtkmm.GDBus.SomeErrorName",
        "As requested, here's a raw D-Bus error");
    }
    else
    {
      Glib::ustring response =
        Glib::ustring::compose("You greeted me with '%1'. Thanks!", greeting);

      Glib::Variant<Glib::ustring> varString = Glib::Variant<Glib::ustring>::create(response);
      std::vector<Glib::VariantBase> results;
      results.push_back(varString);
      Glib::VariantContainerBase varContainer = Glib::VariantContainerBase::create_tuple(results);

      invocation->return_value(varContainer);
    }
  }
  else if(methodName == "EmitSignal")
  {
    double speed_in_mph;

    // We retrieve the parameter (a double: speed_in_mph)
    //const Glib::VariantContainerBase& varTuple = (Glib::VariantContainerBase&) varParameters;
    Glib::Variant<double> varDouble;
    varTuple.get(varDouble);
    speed_in_mph = varDouble.get();

    emit_signal_velocity_changed(refConnection, speed_in_mph);

    Glib::VariantBase varNil;
    invocation->return_value(varNil);
  }
}

static void on_get_property(Glib::VariantBase& varRet,
  const Glib::RefPtr<Gio::DBus::Connection>& /* refConnection */, const Glib::ustring& sender,
  const Glib::ustring& /* objectPath */, const Glib::ustring& /* interfaceName */,
  const Glib::ustring& propertyName)
{
  std::cout << "debug: get property " << propertyName << std::endl;

  //varRet = Glib::VariantBase();
  if(propertyName == "FluxCapicitorName")
  {
      varRet = Glib::Variant<Glib::ustring>::create("DeLorean");
  }
  else if(propertyName == "Title")
  {
    if(globalTitle == "")
      globalTitle = "Back To C!";
    varRet = Glib::Variant<Glib::ustring>::create(globalTitle);
  }
  else if(propertyName == "ReadingAlwaysThrowsError")
  {
    Glib::ustring msg = "Hello ";
    msg += sender;
    msg += ". I thought I said reading this property always results in an error. kthxbye";

    throw Gio::Error(Gio::Error::FAILED, msg);
  }
  else if(propertyName == "WritingAlwaysThrowsError")
  {
    varRet = Glib::Variant<Glib::ustring>::create("There's no home like home");
  }
  else if(propertyName == "Foo")
  {
    varRet = Glib::Variant<Glib::ustring>::create(swap_a_and_b ? "Tock" : "Tick");
  }
  else if(propertyName == "Bar")
  {
    varRet = Glib::Variant<Glib::ustring>::create(swap_a_and_b ? "Tick" : "Tock");
  }
}

static bool on_set_property(const Glib::RefPtr<Gio::DBus::Connection>& refConnection,
  const Glib::ustring& sender, const Glib::ustring& /* objectPath */,
  const Glib::ustring& /* interfaceName */, const Glib::ustring& propertyName,
  const Glib::Variant<Glib::VariantBase>& varValue)
{
  std::cout << "debug: set property" << std::endl;
  if(propertyName == "Title")
  {
    Glib::Variant<Glib::ustring> varString;
    varValue.get(varString, 0);
    if(varString.get() != globalTitle)
    {
      globalTitle = varString.get();
      std::cout << "title: "<< globalTitle << std::endl;

      std::vector<PropertyDescription> properties(1);
      properties[0].first = "Title";
      properties[0].second = globalTitle;
      emit_signal_properties_changed(refConnection, properties);

    }
  }
  else if(propertyName == "ReadingAlwaysThrowsError")
  {
    /* do nothing - they can't read it after all! */
  }
  else if(propertyName == "WritingAlwaysThrowsError")
  {
    Glib::ustring msg = "Hello AGAIN ";
    msg += sender;
    msg += ". I thought I said writing this property always results in an error. kthxbye";
    throw Gio::Error(Gio::Error::FAILED, msg);
  }

  return true;
}

static bool on_timeout_properties(const Glib::RefPtr<Gio::DBus::Connection> refConnection)
{
  std::cout << "debug: swap Tick Tock" << std::endl;

  swap_a_and_b = !swap_a_and_b;
  std::vector<PropertyDescription> properties(2);
  properties[0] = PropertyDescription("Foo", swap_a_and_b ? "Tock" : "Tick");
  properties[1] = PropertyDescription("Bar", swap_a_and_b ? "Tick" : "Tock");
  emit_signal_properties_changed(refConnection, properties);

  return true;
}

static void on_bus_acquired(const Glib::RefPtr<Gio::DBus::Connection>& refConnection,
  const Glib::ustring& /* name */)
{
  std::cout << "debug: bus acquired" << std::endl;
  guint registration_id;

  Gio::DBus::InterfaceVTable* pVTable = new Gio::DBus::InterfaceVTable(
    sigc::ptr_fun(on_method_call),
    sigc::ptr_fun(on_get_property),
    sigc::ptr_fun(on_set_property));

  registration_id = refConnection->register_object("/org/gtkmm/GDBus/TestObject",
    refIntrospectionData->lookup_interface("org.gtkmm.GDBus.TestInterface"), pVTable);
  g_assert(registration_id > 0);

  /* swap value of properties Foo and Bar every two seconds */
  Glib::signal_timeout().connect_seconds(
    sigc::bind<const Glib::RefPtr<Gio::DBus::Connection> >(
        sigc::ptr_fun(on_timeout_properties), refConnection), 2);
}

static void on_name_acquired(const Glib::RefPtr<Gio::DBus::Connection>& /* refConnection */,
  const Glib::ustring& /* name */)
{
  std::cout << "debug: name acquired" << std::endl;
}

static void on_name_lost(const Glib::RefPtr<Gio::DBus::Connection>& /* refConnection */,
  const Glib::ustring& /* name */)
{
  std::cerr << "error: name lost" << std::endl;
  exit(EXIT_FAILURE);
}

int main(int argc, char *argv[])
{
  guint owner_id;
  Glib::RefPtr<Glib::MainLoop> loop;

  Gio::init();
  Glib::init();

  /* We are lazy here - we don't want to manually provide
   * the introspection data structures - so we just build
   * them from XML.
   */
  try
  {
    refIntrospectionData = Gio::DBus::NodeInfo::create_for_xml(introspection_xml);
  } catch (Glib::Error & e)
  {
    std::cerr << e.what() << std::endl;
    exit(EXIT_FAILURE);
  }

  owner_id = Gio::DBus::own_name(Gio::DBus::BUS_TYPE_SESSION,
    "org.gtkmm.GDBus.TestServer",
    sigc::ptr_fun(on_bus_acquired),
    sigc::ptr_fun(on_name_acquired),
    sigc::ptr_fun(on_name_lost),
    Gio::DBus::BUS_NAME_OWNER_FLAGS_NONE);

  loop = Glib::MainLoop::create();
  loop->run();

  return EXIT_SUCCESS;
}
