#ifndef SAMPLEHEADERFILE1_MULTIGROUPS_C_H_
#define SAMPLEHEADERFILE1_MULTIGROUPS_C_H_

/**
@file SampleHeaderFile1_multigroups_C.h
@brief 
First sample header file with Doxygen markup examples.

This file includes sample code needed to generate a sample PDF
document developed with standard C files, which include multiple
groups. This file is also used for training and provides the
examples shown in the QCT Doxygen Markup Standards (80-VP989-1).
*/
/*-------------------------------------------------------------------------------
     Copyright (c) 2010 QUALCOMM Incorporated.
	 All rights reserved.
	 Qualcomm Confidential and Proprietary
-----------------------------------------------------------------------------*/
/*=============================================================================
                             Edit History
$Header: //depot/Living_Documents/Training/Current_Doxygen_Standards/
           DoxygenMarkupStandards/SampleHeaderFile1_multigroups_C.h $

when       who     what, where, why
--------   ---     ------------------------------------------------------------
09/22/10   LB     Updated to reflect Rev C standards.
06/07/10   LB     Corrected param name in CreateInstance and FancyInterfere
                  functions from 'IEnv' to 'pEnvironment'. Added \n following
                  'SUCCESS' return for CreateInstance and removed hyphens
                  for each return.
05/26/10   LB     Updated from LD CCB review to create final for Rev B.
05/18/10   LB     Updated to include Module grouping samples and other 
                  advanced Doxygen markup examples for the Rev B Doxygen
                  Markup Standards
01/14/10   LB     Fixed the corrupted file and regenerated PDF.
01/13/10   LB     Initial revision for Rev A Doxygen Markup Standards.
===============================================================================*/

#include <stdio.h>
#include <cstdio>

#pragma diag_suppress 494

#pragma message( "FancyMessage (" #_FANCY_DATA_PROFILE_VERSION \
                 "): Depends_on_target" )

/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  BEGIN EXAMPLE: #ifdef macros used for conditional cases
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/

/** @addtogroup fancy_device_common_data
@{ */
/** @name
@{ */
#if defined(FANCY_DATA_TARGET1)
#define  FANCY_DATA_PROFILE_VERSION   7  
    /**< Profile version for Fancy Data Target 1. 
    The value of the Fancy Device target's profile version depends on the following 
    conditional settings:
    - If FANCY_DATA_TARGET1 is TRUE, FANCY_DATA_PROFILE_VERSION = 7.
    - If FANCY_DATA_TARGET2 is TRUE, FANCY_DATA_PROFILE_VERSION = 9.
    - If FANCY_DATA_TARGET3 is TRUE, FANCY_DATA_PROFILE_VERSION = 12.
    - If none of the above conditions are TRUE, FANCY_DEFAULT_DATA_PROFILE_VERSION = 3.*/
#elif defined(FANCY_DATA_TARGET2)
#define  FANCY_DATA_PROFILE_VERSION   9
    /**< Profile version for Fancy Data Target 2. 
    The value of the Fancy Device target's profile version depends on the following 
    conditional settings:
    - If FANCY_DATA_TARGET1 is TRUE, FANCY_DATA_PROFILE_VERSION = 7.
    - If FANCY_DATA_TARGET2 is TRUE, FANCY_DATA_PROFILE_VERSION = 9.
    - If FANCY_DATA_TARGET3 is TRUE, FANCY_DATA_PROFILE_VERSION = 12.
    - If none of the above conditions are TRUE, FANCY_DEFAULT_DATA_PROFILE_VERSION = 3.*/
#elif defined(FANCY_DATA_TARGET3)
#define  FANCY_DATA_PROFILE_VERSION  12
    /**< Profile version for Fancy Data Target 3. 
    The value of the Fancy Device target's profile version depends on the following 
    conditional settings:
    - If FANCY_DATA_TARGET1 is TRUE, FANCY_DATA_PROFILE_VERSION = 7.
    - If FANCY_DATA_TARGET2 is TRUE, FANCY_DATA_PROFILE_VERSION = 9.
    - If FANCY_DATA_TARGET3 is TRUE, FANCY_DATA_PROFILE_VERSION = 12.
    - If none of the above conditions are TRUE, FANCY_DEFAULT_DATA_PROFILE_VERSION = 3.*/
#else
#define  FANCY_DEFAULT_DATA_PROFILE_VERSION   3 
    /**< Default profile version. 
    The value of the Fancy Device target's profile version depends on the following 
    conditional settings:
    - If FANCY_DATA_TARGET1 is TRUE, FANCY_DATA_PROFILE_VERSION = 7.
    - If FANCY_DATA_TARGET2 is TRUE, FANCY_DATA_PROFILE_VERSION = 9.
    - If FANCY_DATA_TARGET3 is TRUE, FANCY_DATA_PROFILE_VERSION = 12.
    - If none of the above conditions are TRUE, FANCY_DEFAULT_DATA_PROFILE_VERSION = 3.*/
#endif

typedef struct fancy_handle  fancy_handle;
typedef fancy_handle         *fancy_handle_t;

/** @} */ /* end_name */
/** @} */ /* end_addtogroup fancy_device_common_data */

/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  END EXAMPLE: #ifdef macros used for conditional cases
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/

/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  START fancy_device_apis group level 1
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/

/** @addtogroup first_fancy_device_api
@{ */

/** 
@brief Keeps track of out-of-bounds block area information and the size of each block.
*/
struct oob_area_info
{
  uint8 block_count;         /**< Number of blocks in the out-of-bounds area.*/
  uint8 block_size_in_bytes; /**< Size of each block in the out-of-bounds area. */
};

/** @} */ /* end_first_fancy_device_api */
/** @addtogroup first_fancy_device_api
@{ */

/*
@brief Keeps track of local area block information. 

This is an optional detailed description. This structure is used to keep track 
of the number and size of blocks in the local area.
*/
struct local_area
{
  uint32 area_count;         /**< Total number of blocks in the local area. */
  uint32 area_size_in_bytes; /**< Size of each block in the local area. */
};

/** @} */ /* end_addtogroup_first_fancy_device_api */

/** @addtogroup second_fancy_device_api
@{ */

/** 
@brief Access rights namespace.
*/
namespace AR {

   const ::FANCYID FANCYID_IRightsChange = 0x1074feb; /**< Changes the access 
                                                      rights. */

   enum RightsChangeReason
   {
      RightsChangeReason_Added,
      RightsChangeReason_Deleted,
      RightsChangeReason_Modified,
      RightsChangeReason_Unknown,
      _AR_PLACEHOLDER_RightsChangeReason = 0x7fffffff
   };
}; /* end_AR_namespace */

  
/*=============================================================================
  Class: CRectangle
=============================================================================*/
/** 
@brief Takes the height and length of a rectangle and returns the area value.

This class (CRectangle) contains four members: two input parameter members of type 
integer (x, y, where x=height and y=length) and two member functions (set_values, area) 
with public access. 

This class calls the set_values member function to set the input values (x, y) 
to integers. This function is void, which indicates that is does not return a result; it 
just holds the results temporarily. It then calls the area member function, which takes 
the values in the set_values member function (x, y), multiplies them (x*y), and returns
the result (area of the rectangle).

@param[in] x  Data member x of type integer with private access. This member is private 
              by default.
@param[in] y  Data member y of type integer with private access. This member is private 
              by default.

@return 
Area of the rectangle as an integer value.

@dependencies
None.

@msc
  hscale="0.75";

  CS,WS,Init;

  CS->CS [ label = "cs()" ] ;
  CS->Init [ label = "Init(TRUE)", ID="1"];
  Init=>CS [ label = "process(1)" ];
  Init=>CS [ label = "process(2)" ];
  CS box Init [label="Sample box"];
  ...;
  Init=>Init [ label = "process(n)" ];
  Init=>Init [ label = "process(END)" ];
  CS rbox WS [ label = "Round box with blue border", linecolour="#0000ff"];
  CS<<=Init [ label = "callback()"];
  ---  [ label = "If more to run", ID="*" ];
  WS->WS [ label = "next()"];
  WS->Init [ label = "ac1()\nac2()"];
  Init<-WS [ label = "Init(TRUE)"];
  WS abox Init [ label = "Angular box with red text", textcolour="red" ];
  Init->Init [ label = "stalled(...)"];
  WS<-Init [ label = "ws() = FALSE"];
@endmsc
*/
class CRectangle {
  public:
    int x, y;
  public:
    void set_values (int,int);
    int area () {return (x*y);}
}; /* end_class */

void CRectangle::set_values (int a, int b) {
  x = a;
  y = b;
};

/** @} */ /* end_addtogroup second_fancy_device_api */

/** @addtogroup third_fancy_device_api
@{ */

/**
@brief Holds the new and previous cold boot block numbers. 

This is an optional detailed description. This structure is used in Fancy 
Device tools during software initialization.
*/
struct fancy_cold_boot_block
{
  int cold_boot_new_block;              /**< New cold boot block number. */
  int cold_boot_old_block;              /**< Old cold boot block number. */
};

/** @} */ /* end_addtogroup third_fancy_device_api */

/** @addtogroup first_fancy_device_api
@{ */
/** @name WFD Cold Boot Values
@{ */
#define FANCY_COLD_BOOT_START_BLOCK_VALUE        0
  /**< Initial value for the start block. */

#define FANCY_MAX_COLD_BOOT_END_BLOCK_VALUE      1
  /**< Maximum value for the end block. */

#define FANCY_COLD_BOOT_BLOCK_VALUE_INVALID      ((cold_blk_invalid_type) 0xFF)
  /**< Value for the cold boot block is invalid. */

#define FANCY_COLD_BOOT_BLOCK_VALUE_UNASSIGNED   ((cold_blk_unassigned_type) 0xFE)
  /**< Value for the cold boot block is unassigned. */
/** @} */ /* end_name WFD Cold Boot Values */
/** @} */ /* end_addtogroup first_fancy_device_api */

/** @addtogroup second_fancy_device_api
@{ */
/**
@brief Holds the new and previous warm boot block numbers. 

This is an optional detailed description. This structure is used in Fancy 
Device tools during software download.
*/
struct fancy_warm_boot_block
{
  int warm_boot_new_block;              /**< New warm boot block number. */
  int warm_boot_old_block;              /**< Old warm boot block number. */
};

/** @} */ /* end_addtogroup second_fancy_device_api */
/** @addtogroup second_fancy_device_api
@{ */

#define FANCY_WARM_BOOT_START_BLOCK_VALUE        0
  /**< Initial value for the start block. */

#define FANCY_MAX_WARM_BOOT_END_BLOCK_VALUE      1
  /**< Maximum value for the end block. */

#define FANCY_WARM_BOOT_BLOCK_VALUE_INVALID      ((warm_blk_invalid_type) 0xFF)
  /**< Warm boot block value is invalid. */

#define FANCY_WARM_BOOT_BLOCK_VALUE_UNASSIGNED   ((warm_blk_unassigned_type) 0xFE)
  /**< Warm boot block value is unassigned. */

/** @} */ /* end_addtogroup second_fancy_device_api */
/** @addtogroup second_fancy_device_api
@{ */

/** Identifies the device types for downloading software to one or more devices. */
typedef enum download_type
{
  download_OOB,      /**< Out-Of-Bounds device. */
  download_NOOB,     /**< More than one Not-Out-Of-Bounds device.\n
					      @note1 This is an example of a note for an enum
						  member (note1 command). This can also be used
						  in param descriptions. */
  download_ONENOOB,  /**< OneNOOB device. */
  download_MULTI     /**< Multiple devices. */
};

/** Identifies the bits per cell for the device. */
typedef enum download_tech
{
  download_SLC,        /**< Single-Level Cell device. */
  download_MLC,        /**< Multi-Level Cell device. */
};

/** Identifies the download ID. */
typedef enum download_ID
{
  download_x8  = 8,    /**< 8-bit interface download ID. */
  download_x16 = 16    /**< 16-bit interface download ID. */
};

/* Identifies the type of data to be read with the Fancy Read API. */
typedef enum
{
  FANCY_READ_MAIN = 0,   /* Read main data only. main_data is not NULL. */
  FANCY_READ_SPARE,      /* Read spare data only. spare_data is not NULL and
                            spare_size_bytes is set. */ 
  FANCY_READ_MAIN_SPARE, /* Read both main and spare data. main_data and
                            spare_data are not NULL, and spare_size_bytes is set. */ 
  FANCY_READ_RAW,        /* Read all RAW bytes in the page. main_data is not 
                            NULL and main_size_bytes includes main+spare space. */ 
  FANCY_READ_BYTES       /* Read the specified number of bytes within the page. 
                            main_data is not NULL, main_size_bytes is less than
                            or equal to the page size bytes. */
};

/** @} */ /* end_addtogroup second_fancy_device_api */

/** @addtogroup third_fancy_device_api
@{ */

/*=============================================================================
  Function: CreateInstance
=============================================================================*/
/**
Provides a public entry point for the module. A new instance of FancyInterfere is 
created for FancyModule, and the default interface is returned.

@note Note text 1.
@note Note text 2.
@note Note text 3.

@param[in] clsid         Class ID of the module.
@param[in] pEnvironment  Interface to the Fancy Services environment.
@param[out] pPrivSet      Privilege set of the caller.
@param[in,out] ppNewObj     Set to the interface pointer of the new instance.

@return 
SUCCESS -- Instance was created successfully. \n
ENOMEMORY -- Indicates a memory allocation failure.

@dependencies
None.

@sideeffects 
An invalid class ID may produce erroneous results.

@sa
NDQueryInterface
*/
static int CreateInstance(FANCYCLID clsid,
                             IEnv *pEnvironment,
                             IPrivSet *pPrivSet,
                             void **ppNewObj
                            );

/** @} */ /* end_addtogroup third_fancy_device_api */
/** @addtogroup third_fancy_device_api
@{ */

#define FANCY_DEVICE_DONE                 0  /**< Operation passed. */
#define FANCY_DEVICE_FAIL               (-1) /**< Operation failed. */
#define FANCY_DEVICE_NOT_SUPPORTED      (-2) /**< Device is not supported. */
#define FANCY_DEVICE_CP_READ_FAIL       (-3) /**< Copy page read failure. */

/** @} */ /* end_addtogroup third_fancy_device_api */
/** @addtogroup third_fancy_device_api
@{ */

/** Defines the interface version.
The interface version can be 0 (the default) to indicate that this is the first 
instance, or it can be 1 to indicate that this is the latest version. */
#define FANCY_INTERFACE_VERSION DALINTERFACE_VERSION(1,0) 

/** @} */ /* end_addtogroup third_fancy_device_api */
/** @addtogroup third_fancy_device_api
@{ */
/*=============================================================================
  Function: NDQueryInterface
=============================================================================*/
/**
Handles interface pointers for non-based classes.

This method must be implemented if the FancyInterfere module supports any 
interfaces not handled by its base classes.

@param[in] idReq     Unique ID of the requested interface.
@param[out] ppIface  Interface pointer of the requested interface.

@return
SUCCESS -- Interface is returned successfully.\n
ECLASSNOTSUPPORT -- Interface is not supported.
*/
virtual int NDQueryInterface(FANCYID idReq,
                                    IQI **ppIface
                                    );

/*=============================================================================
  Function: NameThread
=============================================================================*/
/**
Handles thread name intialization.

The deriving class can override this method to provide a descriptive name for the 
optional output thread. This helps distinguish it in debugging applications.
The default implementation provides a generic name.

@note1hang 
This is an example of a single hanging indent note (note1hang command). It is primarily 
intended for use in fancy tools during software download. 

@return
SUCCESS -- Initialization was successful.\n
FAILURE -- Initialization failed.
*/
FancyResult NameThread() {
      return DoNameThread("FancyInterfere");
   };

/*=============================================================================
  Function: FancyInterfere
=============================================================================*/
/**
Provides a class constructor.

This is the component constructor, meant to be called by the class factory 
of the component.

@param[in] clsid         Class ID of the module.
@param[in] pEnvironment  Interface to the Fancy Services environment.
@param[out] result       Result of the function.

@return 
The following results may be returned:
- Valid class object
- Invalid class object
*/
FancyInterfere(FANCYCLSID clsid, pEnv
					  *pEnvironment,
					  int &result
                     );

/** @} */ /* end_addtogroup third_fancy_device_api */
/** @addtogroup third_fancy_device_api
@{ */

/** Standard print function.
The extern keyword is used to inform the compiler about variables declared outside 
of the current file. Variables described by extern statements do not have any 
space allocated for them because they have been previously defined elsewhere.
*/
extern "C" int printf( const char *fmt, ... );

/** @} */ /* end_addtogroup third_fancy_device_api */
/** @addtogroup third_fancy_device_api
@{ */

/** 
@brief Accesses the header values. This union includes two data structures. 
*/
union _AfancyAudioEvent {
   struct AfancyAudioAnyEvent any;
   struct AfancyAudioAnyEvent header; 
      /* For convenience to access header values. */
   };

/** @} */ /* end_addtogroup third_fancy_device_api */

/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  BEGIN EXAMPLE: PACKED STRUCT AND PACKED ENUM EXAMPLES. Code using Pack  
                 macros (such as PACKED and PACKED_POST) must be defined in the 
				 Doxy_Config.ldd under PREDEFINED and EXPAND AS DEFINED, so that 
				 Doxygen can see the code comments for parsing.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/

/** @addtogroup third_fancy_device_api
@{ */

/** 
@brief Stores Fancy Quality of Service parameters.

This is an optional detailed description. This structure includes three members.
*/
typedef PACKED struct PACKED_POST
{
  boolean   valid_flg;            /**< Indicates whether the parameters are set 
                                       and valid. This is a test detailed sentence. */
  uint32    precedence;           /**< Precedence class. This is a test detailed 
                                       sentence. */
  uint32    mean;                 /**< Mean throughput class. */
} fancy_qos_params_type;

/** @} */ /* end_addtogroup third_fancy_device_api */
/** @addtogroup third_fancy_device_api
@{ */

/** Fancy header compression types.
*/
typedef PACKED enum PACKED_POST
{
  FANCY_HEADER_COMP_OFF =    0,   /**< Compression is off (default). */
  FANCY_HEADER_COMP_ON  =    1,   /**< Compression is on when selected. */
  FANCY_HEADER_COMP_MAX = 0xFF    /**< Forces the maximum compression to 0xff
                                       so the enumeration is defined as a byte. */
} fancy_header_comp_e_type;

/** @} */ /* end_addtogroup third_fancy_device_api */

/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  END EXAMPLE: PACKED STRUCT AND PACKED ENUM EXAMPLES.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/

/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  BEGIN EXAMPLE: ANONYMOUS ENUM EXAMPLE. An anonymous enum is an enum with no 
                 name following the enum keyword. In the Doxy_Config.ldd file, 
				 set MACRO_EXPANSION = YES to force Doxygen to parse these and 
				 place them into the output documentation.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/

/** @addtogroup third_fancy_device_api
@{ */
/** Fancy device validation modes.
*/
typedef enum
{
  FANCY_DEVICE_VAL_OFF =    0,   /**< Device validation mode is off (default). */
  FANCY_DEVICE_VAL_ON  =    1,   /**< Device validation mode is on when selected. */

}fancy_dev_val_e_type;

/** @} */ /* end_addtogroup third_fancy_device_api */

/** @addtogroup third_fancy_device_api
@{ */

/* System time types are required. */
typedef enum
{
   IAO_SYSTIME_UMTS  = 0x00,      /* UMTS system time. */
   IAO_SYSTIME_GSM   = 0x01,      /**< GSM system time. */
   IAO_SYSTIME_TOT                /**< Total number of system time types. */
}FancySysTimeType1;

/** @} */ /* end_addtogroup third_fancy_device_api */

/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  END EXAMPLE: ANONYMOUS ENUM EXAMPLE. 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/

/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  BEGIN EXAMPLE: #ifndef macro (DOXYGEN_SHOULD_SKIP_THIS). When this macro is 
    placed in the Doxy Config file after PREDEFINED, the code between this
    macro should be skipped by Doxygen and not included in the PDF.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/

#ifndef DOXYGEN_SHOULD_SKIP_THIS

/** 
@brief Fancy input and output device handles.

These handles allow the device to be configured for opening, closing, and synchronizing. 
*/
typedef struct 
{
   FancyDevice FancyDevice; /** Main fancy device. */
   FancyResult (*FancyOpen)(FancyDeviceHandle * _h, uint32 fancyFifoNum); /** Open fancy device. *.
   FancyResult (*FancyClose)(FancyDeviceHandle  * _h, uint32 fancyFifoNum); /** Close fancy device. */
   FancyResult (*FancyAsyncRead)(FancyDeviceHandle * _h, uint32 fancyFifoNum, FANCYSYSMemHandle hXferBufMem,uint32 fancyLen, FANCYSYSEventHandle hEvent); /** Asynchronous read of fancy device. */
   FancyResult (*FancyAsyncWrite)(FancyDeviceHandle *  _h, uint32 fancyFifoNum, FANCYSYSMemHandle hXferBufMem, uint32 fancyLen, FANCYSYSEventHandle hEvent); /** Asynchronous write of fancy device. */
}FancyIAO;

#endif #ifndef DOXYGEN_SHOULD_SKIP_THIS

/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  END EXAMPLE: #ifndef macro (DOXYGEN_SHOULD_SKIP_THIS). When this macro is 
    placed in the Doxy Config file after PREDEFINED, the code between this
    macro should be skipped by Doxygen and not included in the PDF.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/

/** @addtogroup third_fancy_device_api
@{ */ 

/** Supported WFD warm start events. */
typedef enum
{
  WFD_WS_EVENT_TOT        /**< Total number of warm start events */
}WfdWSEvtType;

/** @} */ /* end_addtogroup third_fancy_device_api */

#endif /* SAMPLEHEADERFILE1_MULTIGROUPS_C_H_ */
