/*
 * GStreamer gltexture video sink
 * Example: Texture Streaming Application
 * Copyright (C) 2010 Nokia Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>

#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/keysym.h>
#include <X11/extensions/Xrender.h>

#include <gst/gst.h>
#include <gst/gstvalue.h>
#include <gst/video/video.h>
#include <gst/interfaces/videotexture.h>

#include <EGL/egl.h>
#include <EGL/eglext.h>
#include <GLES2/gl2.h>
#include <GLES2/gl2ext.h>

#include "shaders.h"
#include "extdefs.h"

#define XRES 854
#define YRES 480

Display *dpy;
EGLDisplay egldpy;
EGLSurface win_surface = EGL_NO_SURFACE;

GstElement *playbin, *vsink, *asink;

gchar choice;


// Lock surface functions
_PFNEGLLOCKSURFACEKHRPROC eglLockSurfaceKHR;
_PFNEGLUNLOCKSURFACEKHRPROC eglUnlockSurfaceKHR;

// EGLSync functions
_PFNEGLCREATESYNCKHRPROC eglCreateSyncKHR;
PFNGLEGLIMAGETARGETTEXTURE2DOESPROC glEGLImageTargetTexture2DOES;

void checkCompileErrors(GLint shaderHandle)
{
  GLint shaderCompiled;
  glGetShaderiv(shaderHandle, GL_COMPILE_STATUS, &shaderCompiled);
  if(!shaderCompiled) {
    // An error happened, first retrieve the length of the log message
    int i32InfoLogLength, i32CharsWritten;
    glGetShaderiv(shaderHandle, GL_INFO_LOG_LENGTH, &i32InfoLogLength);

    // Allocate enough space for the message and retrieve it
    char* pszInfoLog = (char *) malloc (i32InfoLogLength);
    glGetShaderInfoLog(shaderHandle, i32InfoLogLength, &i32CharsWritten, pszInfoLog);

    printf("error log length %d, chars written %d\n", i32InfoLogLength, i32CharsWritten);

    // Displays the error
    printf("Failed to compile shader: %s\n", pszInfoLog);
    free (pszInfoLog);
  }
}

GLint createProgram(const char* vertexShaderString, const char* fragmentShaderString)
{
  GLint programHandle, fragmentShaderHandle, vertexShaderHandle;

  fragmentShaderHandle = glCreateShader (GL_FRAGMENT_SHADER);
  glShaderSource (fragmentShaderHandle, 1, &fragmentShaderString, NULL);

  glCompileShader (fragmentShaderHandle);
  checkCompileErrors(fragmentShaderHandle);

  vertexShaderHandle   = glCreateShader (GL_VERTEX_SHADER);
  glShaderSource (vertexShaderHandle,   1, &vertexShaderString,   NULL);

  glCompileShader (vertexShaderHandle);
  checkCompileErrors(vertexShaderHandle);

  // create program from compiled shaders
  programHandle = glCreateProgram();

  glAttachShader (programHandle, vertexShaderHandle);
  glAttachShader (programHandle, fragmentShaderHandle);

  return programHandle;
}

void 
frame_ready (GstVideoTexture *sink, gint frame, gpointer data)
{

  static float k = 0;
  GLfloat identityMatrix_normal[16] =
     {
       1.0, 0.0, 0.0, 0.0,
       0.0, 1.0, 0.0, 0.0,
       0.0, 0.0, 1.0, 0.0,
       0.0, 0.0, 0.0, 1.0
     };
  
  GLfloat identityMatrix_rotation[16] =
    {
      cos(k), sin(k), 0.0, 0.0,
      -sin(k), cos(k), 0.0, 0.0,
      0.0, 0.0, 1.0, 0.0,
      0.0, 0.0, 0.0, 1.0
    };
  
  // trianglestrip
  static float quad[] = {
    -1.0,  1.0,
    1.0,  1.0,
    -1.0, -1.0,
    1.0, -1.0
  };

  static const GLfloat texCoords[] = {
    0., 0.,
    1., 0.,
    0., 1.,
    1., 1.,
  };
 
  EGLContext context;
  static GLint programHandle;
  static unsigned int init = 0;
  static struct timeval now, prev;

  if (!init) {

    if (eglMakeCurrent(egldpy, win_surface, win_surface, (EGLContext)data) == EGL_TRUE) {
       g_object_set(G_OBJECT(vsink),"egl-context", (EGLContext)data, NULL);	    
       printf("make egl context as  current in rendering thread: \n");
    }   
    else
    {	    
       printf ("failed to make  egl context as current in rendering thread: \n");
       exit (0);
    }   
    
    gettimeofday(&prev, NULL);
    gettimeofday(&now,  NULL);

    programHandle = createProgram(vertexShader, streamShader);

    glBindAttribLocation(programHandle, 1, "myUV");

    glLinkProgram(programHandle);
    glUseProgram(programHandle);

    // Set the sampler2D uniforms to corresponding texture units
    glUniform1i(glGetUniformLocation(programHandle, "texture0"), 0);

    glViewport (0, 0, XRES, YRES);
    glClearColor (0, 0, 0.0, 0);

 
    if(eglGetCurrentContext() == EGL_NO_CONTEXT)
    {	    
       printf("failed to get the context in rendering thread\n");
       exit (0);
    }   
    else
	printf("got the rendering context successfully in rendering thread, starting the render process......\n");
    init++;
  }

  if (!gst_video_texture_acquire_frame(sink,frame))
  {
     printf ("application failed to acquire the frame..exiting.. \n");
  }

  glClear(GL_COLOR_BUFFER_BIT);


  int loc = glGetUniformLocation(programHandle, "projectionModelView");
  if (choice == 'y') {
    glUniformMatrix4fv(loc, 1, GL_FALSE, identityMatrix_rotation);
    k += .02;
  }  
  else
    glUniformMatrix4fv(loc, 1, GL_FALSE, identityMatrix_normal);

  glEnableVertexAttribArray(0);
  glEnableVertexAttribArray(1);

  glVertexAttribPointer(0, 2, GL_FLOAT, GL_FALSE, 0, quad);
  glVertexAttribPointer(1, 2, GL_FLOAT, GL_FALSE, 0, texCoords);
  
  if (!gst_video_texture_bind_frame(sink, GL_TEXTURE_EXTERNAL_OES, frame))
  {
     printf("application failed to  bind with the frame \n");
  }  
  
  glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);

  if (!gst_video_texture_bind_frame(sink, GL_TEXTURE_EXTERNAL_OES, -1))
  {
     printf("application failed to  unbind  the frame \n");
  }  

  EGLSyncKHR sync = eglCreateSyncKHR(egldpy, EGL_SYNC_FENCE_KHR, NULL);

  eglSwapBuffers(egldpy, win_surface);

  gst_video_texture_release_frame(sink, frame, (gpointer) sync);

}


static gboolean
bus_callback (GstBus *bus, GstMessage *message, gpointer data)
{
  if (GST_MESSAGE_TYPE(message) == GST_MESSAGE_EOS) {
    fprintf(stderr, "end of stream, exit.\n");
    if (data) {
      g_main_loop_quit((GMainLoop*)data);
    }
    return FALSE;
  }

  else if (GST_MESSAGE_TYPE(message) == GST_MESSAGE_ERROR) {
    GError *error =NULL;
    gst_message_parse_error (message, &error, NULL);
    fprintf(stderr, "gstreamer error : %s\n", error->message);
    g_error_free (error);
    return FALSE;
  }
  return TRUE;
}


static Bool WaitForNotify(Display *d, XEvent *e, char *arg)
{
  return (e->type == MapNotify) && (e->xmap.window == (Window) arg);
}

static int is_extension_supported(const char* extensions, const char* name)
{
    const char *start;
    const char *where, *terminator;

    /* Extension names should not have spaces. */
    where = strchr(name, ' ');
    if (where || !name)
    {
        return 0;
    }

    /* It takes a bit of care to be fool-proof about parsing the
     OpenGL extensions string. Don't be fooled by sub-strings,
     etc. */
    start = extensions;
    for (;;)
    {
	where = strstr(start, name);
	if (!where)
	    break;
	terminator = where + strlen(name);
	if (where == start || *(where - 1) == ' ')
	    if (*terminator == ' ' || *terminator == '\0')
	        return 1;
	start = terminator;
    }
    return 0;
}

int
main (int argc, char** argv)
{
  EGLint major, minor;
  EGLint attribList[] = { EGL_BUFFER_SIZE, 16, EGL_RENDERABLE_TYPE, EGL_OPENGL_ES2_BIT, EGL_NONE };
  EGLConfig configs[10];
  EGLint matchingConfigs;
  EGLint contextAttribs[] = { EGL_CONTEXT_CLIENT_VERSION, 2, EGL_NONE };
  EGLContext context;

  Window win;
  XEvent event;
  int k;
  gint flags=99;

  XSetWindowAttributes wa;
  Colormap cmap;
  Visual *vis;

  GstBus *messagebus;
  GMainLoop* loop;
  char uri[256];
  char vsinkname[256];
  EGLint depth;

  struct stat stat_info;
  char *current_dir;
  char path[256];

  if (argc < 2) {
    fprintf(stderr, "oops, please give a file to play\n");
    return 1;
  }

  snprintf(path, 256, "%s", argv[1]);

  if(stat(path, &stat_info) != 0) {
    printf("file does not exist\n");
    return 1;
  }

  printf("thread %p : %s\n", g_thread_self(), __func__);

  // use current directory if path not clearly specified
  if (path[0] != '/') {
    snprintf(path, 256, "%s/%s", getenv("PWD"), argv[1]);
  }
  
  snprintf(uri, 256, "file://%s", path);

  /* user choice for rotation */
  printf ("Enable Rotation? (y/n): ");
  scanf ("%c", &choice);
  
  g_thread_init(NULL);
  gst_init(&argc, &argv);

  dpy = XOpenDisplay (NULL);

  if (!dpy) {
    fprintf (stderr, "error, no display\n");
    return 1;
  }


  egldpy = eglGetDisplay ((EGLNativeDisplayType) dpy);

  if (!eglInitialize(egldpy, &major, &minor)) {
    fprintf (stderr, "error, could not init display\n");
    return 1;
  }

  if (!eglChooseConfig(egldpy, attribList, configs, 10, &matchingConfigs)) {
    fprintf (stderr, "error, could not choose config\n");
    return 1;
  }

  if (matchingConfigs < 1) {
    fprintf (stderr, "error, did not find suitable config\n");
    return 1;
  }

  context = eglCreateContext(egldpy, configs[0], 0, contextAttribs);

  vis = XDefaultVisual(dpy, DefaultScreen(dpy));
  cmap = XCreateColormap(dpy, RootWindow(dpy, DefaultScreen(dpy)), vis, AllocNone);

  wa.background_pixel = 0x00000000;
  wa.colormap = cmap;
  wa.border_pixel = 0;

  win = XCreateWindow(dpy,
		      RootWindow(dpy, DefaultScreen(dpy)),
		      0,
		      0,
		      XRES,
		      YRES,
		      0,
		      16,
		      InputOutput,
		      vis,
		      CWBackPixel | CWBorderPixel | CWColormap,
		      &wa);

  Atom a = XInternAtom(dpy,"_KDE_NET_WM_WINDOW_TYPE_OVERRIDE", False);
  XChangeProperty(dpy, win, XInternAtom(dpy, "_NET_WM_WINDOW_TYPE", False), XA_ATOM, 32,           
		  PropModeReplace,(unsigned char *) &a, 1);

  XSelectInput(dpy, win, StructureNotifyMask | VisibilityNotify | ExposureMask | KeyPressMask);
  XMapWindow(dpy, win);
  XIfEvent(dpy, &event, WaitForNotify, (char*)win);

  win_surface = eglCreateWindowSurface(egldpy, configs[0], (EGLNativeWindowType) win, NULL);

  if (!win_surface) {
    fprintf (stderr, "panic, cannot create window surface\n");
    return 1;
  }

  if (eglMakeCurrent(egldpy, win_surface, win_surface, context) == EGL_TRUE)
    printf("make  the context as current: success in main thread\n");
  else
    printf ("make th context as current:  failed in main thread\n");


  // Make sure the required extensions are supported
  const char* egl_exts = eglQueryString(egldpy, EGL_EXTENSIONS);
  const char* gl_exts = glGetString(GL_EXTENSIONS);

  if (!is_extension_supported(egl_exts, "EGL_KHR_lock_surface2")) {
    fprintf(stderr, "EGL_KHR_lock_surface2 is not supported, aborting\n");
    exit(1);
  }

  if (!is_extension_supported(egl_exts, "EGL_KHR_fence_sync")) {
    fprintf(stderr, "EGL_KHR_fence_sync is not supported, aborting\n");
    exit(1);
  }

  if (!is_extension_supported(gl_exts, "GL_OES_EGL_image_external")) {
    fprintf(stderr, "GL_OES_EGL_image_external is not supported, aborting\n");
    exit(1);
  }

  // Lock surface functions
  eglLockSurfaceKHR =
    (_PFNEGLLOCKSURFACEKHRPROC)eglGetProcAddress("eglLockSurfaceKHR");
  eglUnlockSurfaceKHR =
    (_PFNEGLUNLOCKSURFACEKHRPROC)eglGetProcAddress("eglUnlockSurfaceKHR");


  // EGLSync functions
  eglCreateSyncKHR =
    (_PFNEGLCREATESYNCKHRPROC)eglGetProcAddress("eglCreateSyncKHR");

  // ..........................................................
  loop = g_main_loop_new (NULL, TRUE);

  playbin = gst_element_factory_make("playbin2", "playbin");

  vsink = gst_element_factory_make("gltexturesink", "egl-texture-sink");
  if (!vsink) 
  {
        fprintf (stderr, "error,failed to create gltexturesink...exiting\n");
	exit(1);
  }

  asink = gst_element_factory_make("pulsesink", "asink");

  messagebus = gst_pipeline_get_bus (GST_PIPELINE (playbin));
  gst_bus_add_watch (messagebus, bus_callback, loop);
  gst_object_unref (messagebus);

  g_object_set (G_OBJECT (playbin), "uri", uri, 
		                    "video-sink", vsink,
			 	    "audio-sink", asink,
				    "flags", flags,  
				     NULL);
  
  g_object_set (G_OBJECT (vsink), "x-display", dpy, 
		  		  "egl-display", egldpy,
				   NULL);
  
  g_signal_connect (G_OBJECT (vsink),"frame-ready",G_CALLBACK (frame_ready),context);

  eglMakeCurrent (egldpy, EGL_NO_SURFACE, EGL_NO_SURFACE, EGL_NO_CONTEXT);


  gst_element_set_state(GST_ELEMENT (playbin), GST_STATE_PLAYING);

  g_main_loop_run(loop);

  gst_element_set_state(playbin, GST_STATE_NULL);
  gst_object_unref(GST_OBJECT (playbin));

  gst_deinit();

  eglMakeCurrent(egldpy, EGL_NO_SURFACE, EGL_NO_SURFACE, EGL_NO_CONTEXT);
  eglTerminate(egldpy);

  XDestroyWindow(dpy, win);

  XCloseDisplay(dpy);

  return 0;
}
