/***************************************************************************//**
 *	\file	adc.c
 *
 *	\brief	This file provides all the functionality provided by ADC driver  
 *			module of Freescale MPC563 power-pc MCU
 *
 *	\version	1.0A - Initial Draft
 *	\date		25 June 2010
 *
 *	\version	1.1A - Design changed to support run-time CCW addition and 
 *						configuration
 *	\date		16 July 2010
 ******************************************************************************/

/******************************************************************************
								Include Files
******************************************************************************/
#include "mpc563.h"
#include "m_common.h"
#include "m_qadc64.h"
#include "hal_types.h"
#include "adc_Cfg.h"
#include "adc.h"


/******************************************************************************
								Type defines
******************************************************************************/
/**	\brief	This structure defines the data for each CCW and the 
			Application call-back function for that conversion */
typedef struct
{
	/** Call-back function to be called after this CCW conversion 
		is complete */
	ConvComplCB ApplCB;
	/** Upper two bits will be used for mask configuration and lower 
		10 bits for CCW Info */
	U16 u16CCW_Data;
}QADC_CCW_Data;

/**	\brief	This structure holds all global data of teh ADC module */
 typedef struct
 {
	/** Provides ADC converter active status */
	QADC_Drv_Status DrvStatus;
	/** Application call-back function to be called after conversion 
		complete. Assign 'NULL', if no call-back needed */
	ConvComplCB fptrCB;
	/** Provides ccurrent mode of operation for ADC converter module */
	QADC_OpMode ConvModOpMode;
	QADC_CCW_Data CCW_Data[ADC_MOD_CCW_TOT];
	/** Provides the Queue-2 CCW begin position in the Queue */
	U8 u8Q2BeginPos;
 }QADC_Drv;

/******************************************************************************
							Global Variables
******************************************************************************/
/**	\var	self
 *	\brief	This global static variable holds ADC driver related data for each
			ADC converter module */
static QADC_Drv self[ADC_TOT_MOD];

/**	\brief	This variable holds the address for register array of all the 
			ADC converters */
struct QADC64_tag* const ModRegAddr[ADC_TOT_MOD] = 
									{QADC_MOD_A, QADC_MOD_B};
/******************************************************************************
							Function Prototypes
******************************************************************************/
 INTERRUPT void ADC_MOD_A_Q1_CI_ISR (void);
 INTERRUPT void ADC_MOD_A_Q2_CI_ISR (void);
 INTERRUPT void ADC_MOD_A_Q1_PI_ISR (void);
 INTERRUPT void ADC_MOD_A_Q2_PI_ISR (void);
 INTERRUPT void ADC_MOD_B_Q1_CI_ISR (void);
 INTERRUPT void ADC_MOD_B_Q2_CI_ISR (void);
 INTERRUPT void ADC_MOD_B_Q1_PI_ISR (void);
 INTERRUPT void ADC_MOD_B_Q2_PI_ISR (void);
void QADC_CompletionIntCB(QADC_Module_ID ModID);
void QADC_PauseIntCB(QADC_Module_ID ModID);
/******************************************************************************
							Function Definitions
******************************************************************************/

/**************************************************************************//**
 *	\fn		QADC_DrvOpStatus ADC_Init(QADC_ConvModConf* ModConf)
 *
 *	\brief	This API will initialize a ADC converter module and start its
 * 			operation
 *
 *	\param	ModConf - QADC converter module configuration data
 *	\return	Driver operation status
 *			- ADC_OP_FAIL			- Module Initialization failed
 *			- ADC_OP_INVALID_MOD_ID	- Invalid Module ID
 *			- ADC_OP_SUCCESS		- Module Initialized successfully
 *
 *  - The followwing actions will be taken on each ADC converetr module
 *	- If the Module ID is valid,
 *      - It resets the ADC converter module status to uninitialized mode
 *      - Get the register array address for the ADC converter module
 *      - Update the Operation mode, Register access level from application
 *      - Update Queue-1 and Queue-2 interrupt priority
 *      - Update external MUX selection, presclar clock high and low time,
 *        and External Triggering types for Queue-1 and Queue-2
 *      - Update Queue-1 operational mode and single scan option
 *      - Update Queue-2 operational mode, single scan option, operation
 *        resume type and Queue-2 begin location in the queue
 *      - Configure the ADC converter module to Initialized mode
 *****************************************************************************/
 QADC_DrvOpStatus ADC_Init(QADC_ConvModConf* ModConf)
 {
	QADC_DrvOpStatus RetVal = ADC_OP_FAIL;
	struct QADC64_tag* QADC_Reg;
	if(ModConf->ModID < ADC_TOT_MOD)
	{
		/* Set the default status of the driver */
		self[ModConf->ModID].DrvStatus = ADC_DRV_UNINIT;
		QADC_Reg = ModRegAddr[ModConf->ModID];
		/* If QADCMCR.STOP is set to enable stop mode/ low power mode,
			- The DDRQA, PORTQA/PORTQB, QACR0 are not reset and are read-only accessible
			- QACR1, QACR2, QASR0 and QASR1 are reset and are read-only accessible
			- QADCMCR and QADCINT are fully accessible and are not reset
			So before configuring the operation modes, the converter
			needs to be active
		*/
		/* QADCMCR - Module Configuration Register settings */
		QADC_Reg->QADC64MCR.B.FRZ	= ADC_ENABLE_FREEZE;
		QADC_Reg->QADC64MCR.B.STOP	= ADC_DISABLE_STOP_MODE;
		QADC_Reg->QADC64MCR.B.SUPV	= ModConf->RegAcc;
		QADC_Reg->QADC64MCR.B.LOCK	= ADC_MODE_LOCK_DISABLE;
		QADC_Reg->QADC64MCR.B.FLIP	= ModConf->OpMode;
		self[ModConf->ModID].ConvModOpMode = ModConf->OpMode;
		QADC_Reg->QADC64MCR.B.LOCK	= ADC_MODE_LOCK_ENABLE;

		/* Configure ADC interrupts */
		QADC_Reg->QADC64INT.B.IRL1	= ModConf->u8Q1IntPrio;
		QADC_Reg->QADC64INT.B.IRL2	= ModConf->u8Q2IntPrio;

		/* Control register-0 settings */
		QADC_Reg->QACR0.B.EMUX = ModConf->ExtMux;
		QADC_Reg->QACR0.B.TRG = ModConf->ExtTrigType;
		QADC_Reg->QACR0.B.PSH = ModConf->u8PresclClkHighTime;
		QADC_Reg->QACR0.B.PSA = 0;
		QADC_Reg->QACR0.B.PSL = ModConf->u8PresclClkLowTime;

		/* Control register-1 settings */
		QADC_Reg->QACR1.B.CIE1	= ADC_Q1_COMP_INT_ENABLE;
		QADC_Reg->QACR1.B.PIE1	= ADC_Q1_PAUSE_INT_ENABLE;
		QADC_Reg->QACR1.B.MQ1	= ModConf->Q1_OpMode;

		/* Control register-2 settings */
		QADC_Reg->QACR2.B.CIE2	= ADC_Q2_COMP_INT_ENABLE;
		QADC_Reg->QACR2.B.PIE2	= ADC_Q2_PAUSE_INT_ENABLE;
		QADC_Reg->QACR2.B.MQ2	= ModConf->Q2_OpMode;
		QADC_Reg->QACR2.B.RESUME	= ModConf->Q2_ResumeType;
		QADC_Reg->QACR2.B.BQ2	= ModConf->u8Q2Begin;
		/* Keep the Queue-2 begin position for record */
		self[ModConf->ModID].u8Q2BeginPos = ModConf->u8Q2Begin;

		/* Update ADc converter module status */
		self[ModConf->ModID].DrvStatus	= ADC_DRV_INIT;

		RetVal = ADC_OP_SUCCESS;
	}/* if(ModID < ADC_TOT_MOD) */
	else
	{
		return ADC_OP_INVALID_MOD_ID;
	}
	return RetVal;
 }


/**************************************************************************//**
 *	\fn		QADC_DrvOpStatus ADC_Read (QADC_CCW_ApplResultData* ResultData)
 *
 *	\brief	This API will read ADC conversion result value for the request
 *			client ID/ (CCW) as per the described result format
 *
 *	\param	ResultData - Structure to hold user request data to
 *							read the ADC conversion result
 *
 *	\return	Operation status
 *			- ADC_OP_FAIL			- Operation failed due to unknown reason
 *			- ADC_OP_SUCCESS		- Operation success
 *			- ADC_OP_UNSUPP_RD_FMT	- Unsupported result read format
 *			- ADC_OP_UNSUPP_STATE	- Current state of driver does not
 *										allow this action. The analog
 *										conversion for the drive should
 *										be active before calling this API
 *			- ADC_OP_INVALID_CCW_ID	- Invalid CCW ID provided by the user
 *			- ADC_OP_INVALID_MOD_ID	- Invalid ADC converter module ID
 *										provided by the user
 *			- ADC_OP_INVALID_DATA	- Invalid user Data
 *	\note	This API will only process the request, if the ADC
 *			conversion is started
 *****************************************************************************/
 QADC_DrvOpStatus ADC_Read (QADC_CCW_ApplResultData* ResultData)
 {
	QADC_DrvOpStatus RetVal = ADC_OP_FAIL;

	if(ResultData != NULL)
	{
		/* Check if the user provided ADC converter module ID is correct */
		if(ResultData->ModID < ADC_TOT_MOD)
		{
			if(ResultData->QueueID < ADC_MOD_CCW_TOT)
			{
				struct QADC64_tag* QADC_Reg = ModRegAddr[ResultData->ModID];
				/* If QADCMCR.STOP is set to Enable stop mode/ driver is in not active,
					- The CCW and result is not reset and is not accessible */
				if(ADC_DRV_INIT == self[ResultData->ModID].DrvStatus)
				{
					RetVal = ADC_OP_SUCCESS;
					/* Get the conversion result */
					switch(ResultData->ReadFormat)
					{
						case ADC_RIGHT_JUST_UNSIGN:
							ResultData->u16Data = QADC_Reg->RJURR[ResultData->QueueID].R;
							break;
						case ADC_LEFT_JUST_SIGN:
							ResultData->u16Data = QADC_Reg->LJSRR[ResultData->QueueID].R;
							break;
						case ADC_LEFT_JUST_UNSIGN:
							ResultData->u16Data = QADC_Reg->LJURR[ResultData->QueueID].R;
							break;
						default:
							ResultData->u16Data = 0;
							RetVal = ADC_OP_UNSUPP_RD_FMT;
							break;
					}
				}
				else
				{
					RetVal = ADC_OP_UNSUPP_STATE;
				}
			}
			else
			{
				RetVal = ADC_OP_INVALID_CCW_ID;
			}
		}
		else
		{
			RetVal = ADC_OP_INVALID_MOD_ID;
		}
	}
	else
	{
		RetVal = ADC_OP_INVALID_DATA;
	}
	return RetVal;
 }

 /**************************************************************************//**
 *	\fn		QADC_IO_OpStatus QADC_PortConfig(QADC_Module_ID ModID, QADC_Ports PortID, QADC_PortConfigType Type)
 *
 *	\brief	This API will help in configuraing any port A pins of
 *			any ADC converetr module in Input/ Output mode. Port B
 *			is always configured in input mode
 *
 *	\param	ModID	- ADC converter module ID
 *	\param	PortID	- Port ID to be configured
 *	\param	Type	- Configuration type (Output/Input mode)
 *
 *	\return	Operation status
 *			- QADC_IO_INVALID_CFG_TYPE	- Invalid configuration type selected
 *			- QADC_IO_INVALID_PORT_ID	- Invalid port ID selected
 *			- QADC_IO_INVALID_MOD_ID	- Invalid Module ID selected
 *****************************************************************************/
 QADC_IO_OpStatus QADC_PortConfig(QADC_Module_ID ModID, QADC_Ports PortID, QADC_PortConfigType Type)
 {
	 if((QADC_CONFIG_OUT == Type) || (QADC_CONFIG_IN == Type))
	 {
		/* Check if the user provided ADC converter module ID is correct */
		if(ModID < ADC_TOT_MOD)
		{
			switch(PortID)
			{
				case QADC_PORT_A0:
					ModRegAddr[ModID]->DDRQA.B.DDQA0 = Type;
					break;
				case QADC_PORT_A1:
					ModRegAddr[ModID]->DDRQA.B.DDQA1 = Type;
					break;
				case QADC_PORT_A2:
					ModRegAddr[ModID]->DDRQA.B.DDQA2 = Type;
					break;
				case QADC_PORT_A3:
					ModRegAddr[ModID]->DDRQA.B.DDQA3 = Type;
					break;
				case QADC_PORT_A4:
					ModRegAddr[ModID]->DDRQA.B.DDQA4 = Type;
					break;
				case QADC_PORT_A5:
					ModRegAddr[ModID]->DDRQA.B.DDQA5 = Type;
					break;
				case QADC_PORT_A6:
					ModRegAddr[ModID]->DDRQA.B.DDQA6 = Type;
					break;
				case QADC_PORT_A7:
					ModRegAddr[ModID]->DDRQA.B.DDQA7 = Type;
					break;
				default:
					return QADC_IO_INVALID_PORT_ID;
					break;
			}
			return QADC_IO_SUCCESS;
		}
		else
		{
			return QADC_IO_INVALID_MOD_ID;
		}
	 }
	 else
	 {
		 return QADC_IO_INVALID_CFG_TYPE;
	 }
 }
 /**************************************************************************//**
 *	\fn		QADC_IO_OpStatus QADC_PortSet(QADC_Module_ID ModID, QADC_Ports PortID, QADC_PortVal Value)
 *
 *	\brief	This API will help to set the value of any pin of port
 *			A to the defined value
 *
 *	\param	ModID	- ADC converter module ID
 *	\param	PortID	- Port ID to be configured
 *	\param	Value	- Value of the port to be set
 *
 *	\return	Operation status
 *			- QADC_IO_INVALID_CFG_TYPE	- Invalid configuration type selected
 *			- QADC_IO_INVALID_PORT_ID	- Invalid port ID selected
 *			- QADC_IO_INVALID_MOD_ID	- Invalid Module ID selected
 *****************************************************************************/
 QADC_IO_OpStatus QADC_PortSet(QADC_Module_ID ModID, QADC_Ports PortID, QADC_PortVal Value)
 {
	/* Check if the user provided ADC converter module ID is correct */
	if(ModID < ADC_TOT_MOD)
	{
		switch(PortID)
		{
			case QADC_PORT_A0:
				ModRegAddr[ModID]->PORTQA.B.PQA0 = Value;
				break;
			case QADC_PORT_A1:
				ModRegAddr[ModID]->PORTQA.B.PQA1 = Value;
				break;
			case QADC_PORT_A2:
				ModRegAddr[ModID]->PORTQA.B.PQA2 = Value;
				break;
			case QADC_PORT_A3:
				ModRegAddr[ModID]->PORTQA.B.PQA3 = Value;
				break;
			case QADC_PORT_A4:
				ModRegAddr[ModID]->PORTQA.B.PQA4 = Value;
				break;
			case QADC_PORT_A5:
				ModRegAddr[ModID]->PORTQA.B.PQA5 = Value;
				break;
			case QADC_PORT_A6:
				ModRegAddr[ModID]->PORTQA.B.PQA6 = Value;
				break;
			case QADC_PORT_A7:
				ModRegAddr[ModID]->PORTQA.B.PQA7 = Value;
				break;
			default:
				return QADC_IO_INVALID_PORT_ID;
				break;
		}
		return QADC_IO_SUCCESS;
	}
	else
	{
		return QADC_IO_INVALID_MOD_ID;
	}
 }

/**************************************************************************//**
 *	\fn		QADC_IO_OpStatus QADC_PortGet(QADC_Module_ID ModID, QADC_Ports PortID, QADC_PortVal* Value)
 *
 *	\brief	This API will help to get the value of any pin of port
 *			A and B
 *
 *	\param	ModID	- ADC converter module ID
 *	\param	PortID	- Port ID to be configured
 *	\param	Value	- Value of the port to be set
 *
 *	\return	Operation status
 *			- QADC_IO_INVALID_CFG_TYPE	- Invalid configuration type selected
 *			- QADC_IO_INVALID_PORT_ID	- Invalid port ID selected
 *			- QADC_IO_INVALID_MOD_ID	- Invalid Module ID selected
 *****************************************************************************/
 QADC_IO_OpStatus QADC_PortGet(QADC_Module_ID ModID, QADC_Ports PortID, QADC_PortVal* Value)
 {
	/* Check if the user provided ADC converter module ID is correct */
	if(ModID < ADC_TOT_MOD)
	{
		switch(PortID)
		{
			case QADC_PORT_A0:
				*Value = ModRegAddr[ModID]->PORTQA.B.PQA0;
				break;
			case QADC_PORT_A1:
				*Value = ModRegAddr[ModID]->PORTQA.B.PQA1;
				break;
			case QADC_PORT_A2:
				*Value = ModRegAddr[ModID]->PORTQA.B.PQA2;
				break;
			case QADC_PORT_A3:
				*Value = ModRegAddr[ModID]->PORTQA.B.PQA3;
				break;
			case QADC_PORT_A4:
				*Value = ModRegAddr[ModID]->PORTQA.B.PQA4;
				break;
			case QADC_PORT_A5:
				*Value = ModRegAddr[ModID]->PORTQA.B.PQA5;
				break;
			case QADC_PORT_A6:
				*Value = ModRegAddr[ModID]->PORTQA.B.PQA6;
				break;
			case QADC_PORT_A7:
				*Value = ModRegAddr[ModID]->PORTQA.B.PQA7;
				break;
			case QADC_PORT_B0:
				*Value = ModRegAddr[ModID]->PORTQB.B.PQB0;
				break;
			case QADC_PORT_B1:
				*Value = ModRegAddr[ModID]->PORTQB.B.PQB1;
				break;
			case QADC_PORT_B2:
				*Value = ModRegAddr[ModID]->PORTQB.B.PQB2;
				break;
			case QADC_PORT_B3:
				*Value = ModRegAddr[ModID]->PORTQB.B.PQB3;
				break;
			case QADC_PORT_B4:
				*Value = ModRegAddr[ModID]->PORTQB.B.PQB4;
				break;
			case QADC_PORT_B5:
				*Value = ModRegAddr[ModID]->PORTQB.B.PQB5;
				break;
			case QADC_PORT_B6:
				*Value = ModRegAddr[ModID]->PORTQB.B.PQB6;
				break;
			case QADC_PORT_B7:
				*Value = ModRegAddr[ModID]->PORTQB.B.PQB7;
				break;
			default:
				return QADC_IO_INVALID_PORT_ID;
				break;
		}
		return QADC_IO_SUCCESS;
	}
	else
	{
		return QADC_IO_INVALID_MOD_ID;
	}
 }

