/******************************************************************************/
/**	\file	adc.h
 *
 *	\brief	This file provides all the APIs provided by QADC64E module 
 *			of Freescale MPC563 power-pc MCU
 *
 *	\version	1.0A - Initial Draft
 *	\date		25 June 2010
 *
 *	\version	1.1A - Design changed to support run-time CCW addition and 
 *						configuration
 *	\date		16 July 2010
 ******************************************************************************/
#ifndef ADC_H
#define ADC_H
/******************************************************************************
Include Files
******************************************************************************/
#include "mpc563.h"

/******************************************************************************
Type defines
******************************************************************************/
/**	\def	QADC_PORT_A
 *	\brief	This defines the PORT-A mask */
#define QADC_PORT_A	0x0000
/**	\def	QADC_PORT_B
 *	\brief	This defines the PORT-B mask */
#define QADC_PORT_B	0x0080

typedef enum
{
	QADC_PORT_A0 = QADC_PORT_A,
	QADC_PORT_A1,
	QADC_PORT_A2,
	QADC_PORT_A3,
	QADC_PORT_A4,
	QADC_PORT_A5,
	QADC_PORT_A6,
	QADC_PORT_A7,
	QADC_PORT_B0 = QADC_PORT_B,
	QADC_PORT_B1,
	QADC_PORT_B2,
	QADC_PORT_B3,
	QADC_PORT_B4,
	QADC_PORT_B5,
	QADC_PORT_B6,
	QADC_PORT_B7,
}QADC_Ports;

typedef enum
{
	QADC_CONFIG_IN,
	QADC_CONFIG_OUT,
}QADC_PortConfigType;

typedef enum
{
	PORT_VAL_LOW,
	PORT_VAL_HIGH,
}QADC_PortVal;

typedef enum
{
	QADC_IO_FAIL,
	QADC_IO_SUCCESS,
	QADC_IO_INVALID_CFG_TYPE,
	QADC_IO_INVALID_MOD_ID,
	QADC_IO_INVALID_PORT_ID,
	
}QADC_IO_OpStatus;

/***************************************************************************
************* QADCMCR - Module Configuration Register settings *************
****************************************************************************/
/** \addtogroup QADCMCR_REG_SETTINGS	QADCMCR register settings */ /*@{*/

/* QADC mode of operation */
typedef enum
{
	/**	\brief	ADC "Legacy Mode" of operation */
	ADC_LEGACY_OP_MODE,
	/**	\brief	ADC "Enhanced Mode"	of operation */
	ADC_ENHANCED_OP_MODE
}QADC_OpMode;

/* QADC register access permission */
typedef enum
{
	/**	Defines the value to configure the module configuration register, 
		test register, and interrupt register as supervisor-only 
		data space. Access to all other locations is unrestricted */
	ADC_LIMITED_SUPV_ACC,
	/**	Defines the value to configure all QADC64E registers and CCW/result 
		tables as supervisor-only data space */
	ADC_FULL_SUPV_ACC,
}QADC_RegAcc;
/*@}*/

/***************************************************************************
************************** QACR0 - Control Reg-0 ***************************
****************************************************************************/
/**	\defgroup	QACR0_REG_SETTINGS	QACR0 register settings */ /*@{*/

/** \brief	This enum provides the External Multiplexed mode selection */
typedef enum
{
	/**	Select internal MUX configuration */
	ADC_MUX_MODE_INT,
	/**	Sselect external MUX configuration */
	ADC_MUX_MODE_EXT,
}QADC_ExtMuxSel;

/** \brief	This enum provides the ADC External trigger assignment */
typedef enum
{
	/**	Selects external trigger signal ETRIG1 for queue-1 and external 
		trigger signal ETRIG2 for queue-2 */
	ADC_ETRIG1_Q1_ETRIG2_Q2,
	/**	Select external trigger signal ETRIG1 for queue-2 and external 
		trigger signal ETRIG2 for queue-1 */
	ADC_ETRIG1_Q2_ETRIG2_Q1,
}QADC_ExtTrig_Sel;

/*@}*/
/***************************************************************************
****************************** QACR1 - Control Reg-1 ***********************
****************************************************************************/
/**	\addtogroup	QACR1_REG_SETTINGS	QACR1 register settings */ /*@{*/

/** \brief	Provides Queue-1 Single scan operation status */
typedef enum
{
	/**	Disables single scan operation for Queue-1 */
	ADC_Q1_SINGLE_SCAN_DISABLE,
	/**	Enables single scan operation for Queue-1 */
	ADC_Q1_SINGLE_SCAN_ENABLE,
}QADC_Q1_SingleScanOpt;

/* Queue-1 Operating mode */
/**	\brief	This enum provides all Queue-1 operation modes */
typedef enum 
{
	/**	\brief	Disabled mode, conversions do not occur */
	ADC_OP_Q1_DISABLE,
	/**	\brief	Software triggered single-scan mode (started with SSE1) */
	ADC_OP_Q1_SW_TRIG_SINGLE_SCAN,
	/**	\brief	External trigger rising edge single-scan mode */
	ADC_OP_Q1_EXT_TRIG_RISING_EDGE_SINGLE_SCAN,
	/**	\brief	External trigger falling edge single-scan mode */
	ADC_OP_Q1_EXT_TRIG_FALLING_EDGE_SINGLE_SCAN,
	/**	\brief	Interval timer single-scan mode: time = QCLK period x (2^7) */
	ADC_OP_Q1_INTVAL_TIMER_SINGLE_SCAN_128_QCLK,
	/**	\brief	Interval timer single-scan mode: time = QCLK period x (2^8) */
	ADC_OP_Q1_INTVAL_TIMER_SINGLE_SCAN_256_QCLK,
	/**	\brief	Interval timer single-scan mode: time = QCLK period x (2^9) */
	ADC_OP_Q1_INTVAL_TIMER_SINGLE_SCAN_512_QCLK,
	/**	\brief	Interval timer single-scan mode: time = QCLK period x (2^10) */
	ADC_OP_Q1_INTVAL_TIMER_SINGLE_SCAN_1024_QCLK,
	/**	\brief	Interval timer single-scan mode: time = QCLK period x (2^11) */
	ADC_OP_Q1_INTVAL_TIMER_SINGLE_SCAN_2048_QCLK,
	/**	\brief	Interval timer single-scan mode: time = QCLK period x (2^12) */
	ADC_OP_Q1_INTVAL_TIMER_SINGLE_SCAN_4096_QCLK,
	/**	\brief	Interval timer single-scan mode: time = QCLK period x (2^13) */
	ADC_OP_Q1_INTVAL_TIMER_SINGLE_SCAN_8192_QCLK,
	/**	\brief	Interval timer single-scan mode: time = QCLK period x (2^14) */
	ADC_OP_Q1_INTVAL_TIMER_SINGLE_SCAN_16384_QCLK,
	/**	\brief	Interval timer single-scan mode: time = QCLK period x (2^15) */
	ADC_OP_Q1_INTVAL_TIMER_SINGLE_SCAN_32768_QCLK,
	/**	\brief	Interval timer single-scan mode: time = QCLK period x (2^16) */
	ADC_OP_Q1_INTVAL_TIMER_SINGLE_SCAN_65536_QCLK,
	/**	\brief	Interval timer single-scan mode: time = QCLK period x (2^17) */
	ADC_OP_Q1_INTVAL_TIMER_SINGLE_SCAN_131072_QCLK,
	/**	\brief	External gated single-scan mode (started with SSE1) */
	ADC_OP_Q1_EXT_GATED_SINGLE_SCAN,
	/**	\brief	Reserved mode */
	ADC_OP_Q1_RESERVED_1,
	/**	\brief	Software triggered continuous-scan mode */
	ADC_OP_Q1_SW_TRIG_CONTINOUS_SCAN,
	/**	\brief	External trigger rising edge continuous-scan mode */
	ADC_OP_Q1_EXT_TRIG_RISING_EDGE_CONTINOUS_SCAN,
	/**	\brief	External trigger falling edge continuous-scan mode */
	ADC_OP_Q1_EXT_TRIG_FALLING_EDGE_CONTINOUS_SCAN,
	/**	\brief	Periodic timer continuous-scan mode: time = QCLK period x (2^7) */
	ADC_OP_Q1_PERIO_TIMER_CONT_SCAN_128_QCLK,
	/**	\brief	Periodic timer continuous-scan mode: time = QCLK period x (2^8) */
	ADC_OP_Q1_PERIO_TIMER_CONT_SCAN_256_QCLK,
	/**	\brief	Periodic timer continuous-scan mode: time = QCLK period x (2^9) */
	ADC_OP_Q1_PERIO_TIMER_CONT_SCAN_512_QCLK,
	/**	\brief	Periodic timer continuous-scan mode: time = QCLK period x (2^10) */
	ADC_OP_Q1_PERIO_TIMER_CONT_SCAN_1024_QCLK,
	/**	\brief	Periodic timer continuous-scan mode: time = QCLK period x (2^11) */
	ADC_OP_Q1_PERIO_TIMER_CONT_SCAN_2048_QCLK,
	/**	\brief	Periodic timer continuous-scan mode: time = QCLK period x (2^12) */
	ADC_OP_Q1_PERIO_TIMER_CONT_SCAN_4096_QCLK,
	/**	\brief	Periodic timer continuous-scan mode: time = QCLK period x (2^13) */
	ADC_OP_Q1_PERIO_TIMER_CONT_SCAN_8192_QCLK,
	/**	\brief	Periodic timer continuous-scan mode: time = QCLK period x (2^14) */
	ADC_OP_Q1_PERIO_TIMER_CONT_SCAN_16384_QCLK,
	/**	\brief	Periodic timer continuous-scan mode: time = QCLK period x (2^15) */
	ADC_OP_Q1_PERIO_TIMER_CONT_SCAN_32768_QCLK,
	/**	\brief	Periodic timer continuous-scan mode: time = QCLK period x (2^16) */
	ADC_OP_Q1_PERIO_TIMER_CONT_SCAN_65536_QCLK,
	/**	\brief	Periodic timer continuous-scan mode: time = QCLK period x (2^17) */
	ADC_OP_Q1_PERIO_TIMER_CONT_SCAN_131072_QCLK,
	/**	\brief	External gated continuous-scan mode */
	ADC_OP_Q1_EXT_GATED_CONTINOUS_SCAN,
}ADC_Q1_OpModes;

/*@}*/

/***************************************************************************
******************************** QACR2 - Control Reg-2 *********************
****************************************************************************/
/**	\addtogroup	QACR2_REG_SETTINGS	QACR2 register settings */ /*@{*/
/** \brief	Provides Queue-2 Single scan operation status */
typedef enum
{
	/**	Disables single scan operation for Queue-2 */
	ADC_Q2_SINGLE_SCAN_DISABLE,
	/**	Enables single scan operation for Queue-2 */
	ADC_Q2_SINGLE_SCAN_ENABLE,
}QADC_Q2_SingleScanOpt;

/* Queue-2 Operating mode */
/**	\brief	This enum provides all Queue-2 operation modes */
typedef enum 
{
	/**	\brief	Disabled mode, conversions do not occur */
	ADC_OP_Q2_DISABLE,
	/**	\brief	Software triggered single-scan mode (started with SSE2) */
	ADC_OP_Q2_SW_TRIG_SINGLE_SCAN,
	/**	\brief	External trigger rising edge single-scan mode */
	ADC_OP_Q2_EXT_TRIG_RISING_EDGE_SINGLE_SCAN,
	/**	\brief	External trigger falling edge single-scan mode */
	ADC_OP_Q2_EXT_TRIG_FALLING_EDGE_SINGLE_SCAN,
	/**	\brief	Interval timer single-scan mode: time = QCLK period x (2^7) */
	ADC_OP_Q2_INTVAL_TIMER_SINGLE_SCAN_128_QCLK,
	/**	\brief	Interval timer single-scan mode: time = QCLK period x (2^8) */
	ADC_OP_Q2_INTVAL_TIMER_SINGLE_SCAN_256_QCLK,
	/**	\brief	Interval timer single-scan mode: time = QCLK period x (2^9) */
	ADC_OP_Q2_INTVAL_TIMER_SINGLE_SCAN_512_QCLK,
	/**	\brief	Interval timer single-scan mode: time = QCLK period x (2^10) */
	ADC_OP_Q2_INTVAL_TIMER_SINGLE_SCAN_1024_QCLK,
	/**	\brief	Interval timer single-scan mode: time = QCLK period x (2^11) */
	ADC_OP_Q2_INTVAL_TIMER_SINGLE_SCAN_2048_QCLK,
	/**	\brief	Interval timer single-scan mode: time = QCLK period x (2^12) */
	ADC_OP_Q2_INTVAL_TIMER_SINGLE_SCAN_4096_QCLK,
	/**	\brief	Interval timer single-scan mode: time = QCLK period x (2^13) */
	ADC_OP_Q2_INTVAL_TIMER_SINGLE_SCAN_8192_QCLK,
	/**	\brief	Interval timer single-scan mode: time = QCLK period x (2^14) */
	ADC_OP_Q2_INTVAL_TIMER_SINGLE_SCAN_16384_QCLK,
	/**	\brief	Interval timer single-scan mode: time = QCLK period x (2^15) */
	ADC_OP_Q2_INTVAL_TIMER_SINGLE_SCAN_32768_QCLK,
	/**	\brief	Interval timer single-scan mode: time = QCLK period x (2^16) */
	ADC_OP_Q2_INTVAL_TIMER_SINGLE_SCAN_65536_QCLK,
	/**	\brief	Interval timer single-scan mode: time = QCLK period x (2^17) */
	ADC_OP_Q2_INTVAL_TIMER_SINGLE_SCAN_131072_QCLK,
	/**	\brief	Reserved mode */
	ADC_OP_Q2_RESERVED_1,
	/**	\brief	Reserved mode */
	ADC_OP_Q2_RESERVED_2,
	/**	\brief	Software triggered continuous-scan mode */
	ADC_OP_Q2_SW_TRIG_CONTINOUS_SCAN,
	/**	\brief	External trigger rising edge continuous-scan mode */
	ADC_OP_Q2_EXT_TRIG_RISING_EDGE_CONTINOUS_SCAN,
	/**	\brief	External trigger falling edge continuous-scan mode */
	ADC_OP_Q2_EXT_TRIG_FALLING_EDGE_CONTINOUS_SCAN,
	/**	\brief	Periodic timer continuous-scan mode: time = QCLK period x (2^7) */
	ADC_OP_Q2_PERIO_TIMER_CONT_SCAN_128_QCLK,
	/**	\brief	Periodic timer continuous-scan mode: time = QCLK period x (2^8) */
	ADC_OP_Q2_PERIO_TIMER_CONT_SCAN_256_QCLK,
	/**	\brief	Periodic timer continuous-scan mode: time = QCLK period x (2^9) */
	ADC_OP_Q2_PERIO_TIMER_CONT_SCAN_512_QCLK,
	/**	\brief	Periodic timer continuous-scan mode: time = QCLK period x (2^10) */
	ADC_OP_Q2_PERIO_TIMER_CONT_SCAN_1024_QCLK,
	/**	\brief	Periodic timer continuous-scan mode: time = QCLK period x (2^11) */
	ADC_OP_Q2_PERIO_TIMER_CONT_SCAN_2048_QCLK,
	/**	\brief	Periodic timer continuous-scan mode: time = QCLK period x (2^12) */
	ADC_OP_Q2_PERIO_TIMER_CONT_SCAN_4096_QCLK,
	/**	\brief	Periodic timer continuous-scan mode: time = QCLK period x (2^13) */
	ADC_OP_Q2_PERIO_TIMER_CONT_SCAN_8192_QCLK,
	/**	\brief	Periodic timer continuous-scan mode: time = QCLK period x (2^14) */
	ADC_OP_Q2_PERIO_TIMER_CONT_SCAN_16384_QCLK,
	/**	\brief	Periodic timer continuous-scan mode: time = QCLK period x (2^15) */
	ADC_OP_Q2_PERIO_TIMER_CONT_SCAN_32768_QCLK,
	/**	\brief	Periodic timer continuous-scan mode: time = QCLK period x (2^16) */
	ADC_OP_Q2_PERIO_TIMER_CONT_SCAN_65536_QCLK,
	/**	\brief	Periodic timer continuous-scan mode: time = QCLK period x (2^17) */
	ADC_OP_Q2_PERIO_TIMER_CONT_SCAN_131072_QCLK,
	/**	\brief	External gated continuous-scan mode */
	ADC_OP_Q2_EXT_GATED_CONTINOUS_SCAN,
	/**	\brief	Reserved mode */
	ADC_OP_Q2_RESERVED_3,
}ADC_Q2_OpModes;

/**	\brief	This enum defines the Queue-2 CCW Resume position */
typedef enum
{
	/**	Begin executing with the first CCW in queue 2 or the current 
		sub-queue, After suspension */
	ADC_Q2_RESUME_FROM_START_POS,
	/**Begin executing with the aborted CCW in queue 2,	After suspension */
	ADC_Q2_RESUME_FROM_SUSPENDED_POS,
}QADC_Q2_ResumePos;

/*@}*/
/***************************************************************************
*************************** QASR0 - Status Register ************************
****************************************************************************/
/**	\defgroup	QASR0_REG_SETTINGS	QASR0 register settings */ /*@{*/
/* Queue (Q1 and Q2) status */
/**	\brief	This enum provides Queue-1 and Queue-2 operation status */
typedef enum
{
	/**	\brief	queue 1 idle, queue 2 idle */
	ADC_Q_IDLE,
	/**	\brief	queue 1 idle, queue 2 paused */
	ADC_Q1_IDLE_Q2_PAUSED,
	/**	\brief	queue 1 idle, queue 2 active */
	ADC_Q1_IDLE_Q2_ACTIVE,
	/**	\brief	queue 1 idle, queue 2 trigger pending */
	ADC_Q1_IDLE_Q2_PENDING,
	/**	\brief	queue 1 paused, queue 2 idle */
	ADC_Q1_PAUSED_Q2_IDLE,
	/**	\brief	queue 1 paused, queue 2 paused */
	ADC_Q1_PAUSED_Q2_PAUSED,
	/**	\brief	queue 1 paused, queue 2 active */
	ADC_Q1_PAUSED_Q2_ACTIVE,
	/**	\brief	queue 1 paused, queue 2 trigger pending */
	ADC_Q1_PAUSED_Q2_PENDING,
	/**	\brief	queue 1 active, queue 2 idle */
	ADC_Q1_ACTIVE_Q2_IDLE,
	/**	\brief	queue 1 active, queue 2 paused */
	ADC_Q1_ACTIVE_Q2_PAUSED,
	/**	\brief	queue 1 active, queue 2 suspended */
	ADC_Q1_ACTIVE_Q2_SUSPENDED,
	/**	\brief	queue 1 active, queue 2 trigger pending */
	ADC_Q1_ACTIVE_Q2_PENDING,
	/**	\brief	Reserved_1 */
	ADC_Q_STATUS_1,
	/**	\brief	Reserved_2 */
	ADC_Q_STATUS_2,
	/**	\brief	Reserved_3 */
	ADC_Q_STATUS_3,
	/**	\brief	Reserved_4 */
	ADC_Q_STATUS_4	
}ADC_Q_Status;
/*@}*/

/***************************************************************************
******************** CCW - Conversion Command Word Format ******************
****************************************************************************/
/**	\defgroup	CCW_SETTINGS	CCW settings */ /*@{*/

/**	\brief Enum provides pause status after each CCW execution */
typedef enum
{
	/**	Defines the value to disable pause after the current CCW execution */
	ADC_PAUSE_DISABLE,
	/**	Defines the value to enable pause after the current CCW execution */
	ADC_PAUSE_ENABLE,
}QADC_CCW_PauseVal;

/* Amplifier bypass selection in legacy mode of operation */
typedef enum
{
	/**	Disables the Sample amplifier bypass for the signal conversion */
	ADC_AMPL_BYPASS_DISABLE,
	/**	Enables the Sample amplifier bypass for a conversion, and 
		subsequently changes the timing */
	ADC_AMPL_BYPASS_ENABLE,
}QADC_CCW_AmplBypassVal;

/* High reference selection for enhanced mode of operation */
typedef enum
{
	/**	VRH is used as high reference */
	ADC_REF_VRH,
	/**	AltRef signal is used as the high reference */
	ADC_REF_ALTREF,
}QADC_CCW_RefVal;


/* Input sample time */
/**	\brief	This enum provides possible sampling times for a conversion

for Legacy mode of operation: select clocks of 2x, 4x, 8x, 16x
for Enhanced mode of operation: select clocks of 2x, 8x
*/
typedef enum
{
	/**	\brief	Sample time is 2x times QADC clock */
	ADC_SAMPL_TIME_2_QCLK,
	/**	\brief	Sample time is 4x times QADC clock */
	ADC_SAMPL_TIME_4_QCLK,
	/**	\brief	Sample time is 8x times QADC clock */
	ADC_SAMPL_TIME_8_QCLK,
	/**	\brief	Sample time is 16x times QADC clock */
	ADC_SAMPL_TIME_16_QCLK
}ADC_InputSampleTime;

/*@}*/


/**	\typedef	QADC_ResultFormat
 *	\brief		This typedefine enum provides the ADC conversion result read 
				format */
 typedef enum 
 {
	/** \brief	Read the result in right justified unsigned format */
	ADC_RIGHT_JUST_UNSIGN,
	/** \brief	Read the result in left justified signed format */
	ADC_LEFT_JUST_SIGN,
	/** \brief	Read the result in left justified unsigned format */
	ADC_LEFT_JUST_UNSIGN
 }QADC_ResultFormat;
 
/**	\typedef	QADC_Module_ID
 *	\brief		This typedefine enum provides the ADC conversion modules */
 typedef enum
 {
	/**	\brief	This is channel-A ADC converter */
	ADC_MOD_A,
	/** \brief	This is channel-B ADC converter */
	ADC_MOD_B,
	/**	\brief	Defines the total no of ADC converters supported */
	ADC_TOT_MOD
 }QADC_Module_ID;
 
/**	\typedef	QADC_Drv_Status
 *	\brief		This enum typedefine provides various status of ADC driver 
				module */ 
 typedef enum
 {
	/**	This indicates that driver is not initialized */
	ADC_DRV_UNINIT,
	/**	This indicates that driver is initialized but not ready for conversion */
	ADC_DRV_INIT,
	/** Unknown ADC converter state */
	ADC_UNKNOWN_STATE,
 }QADC_Drv_Status;

/**	\typedef	QADC_DrvOpStatus
 *	\brief		This enum typedefine provides various operation states 
				if the ADC driver */ 
typedef enum
{
	/** Driver operation failed due to unknwon reason */
	ADC_OP_FAIL,
	/** Driver operation success */
	ADC_OP_SUCCESS,
	/** Driver operation failed due to invalid ADC converter module ID */
	ADC_OP_INVALID_MOD_ID,
	/** Driver operation failed due to invalid CCW ID */
	ADC_OP_INVALID_CCW_ID,
	/** Driver operation failed due to invalid user provided data */
	ADC_OP_INVALID_DATA,
	/** Driver operation failed as the current state of driver does not support the requested action */
	ADC_OP_UNSUPP_STATE,
	/** Read operation failed, as the read format provided is not supported */
	ADC_OP_UNSUPP_RD_FMT,
}QADC_DrvOpStatus;;

/**	\typedef	QADC_ConvMod_CCWs
 *	\brief		This enum typedefine provides all the CCW IDs 
				registered for both ADC converter module A and B */ 
 typedef enum 
 {
	/* ADC converter module CCWs */
	ADC_MOD_CCW_0,
	ADC_MOD_CCW_1,
	ADC_MOD_CCW_2,
	ADC_MOD_CCW_3,
	ADC_MOD_CCW_4,
	ADC_MOD_CCW_5,
	ADC_MOD_CCW_6,
	ADC_MOD_CCW_7,
	ADC_MOD_CCW_8,
	ADC_MOD_CCW_9,
	ADC_MOD_CCW_10,
	ADC_MOD_CCW_11,
	ADC_MOD_CCW_12,
	ADC_MOD_CCW_13,
	ADC_MOD_CCW_14,
	ADC_MOD_CCW_15,
	ADC_MOD_CCW_16,
	ADC_MOD_CCW_17,
	ADC_MOD_CCW_18,
	ADC_MOD_CCW_19,
	ADC_MOD_CCW_20,
	ADC_MOD_CCW_21,
	ADC_MOD_CCW_22,
	ADC_MOD_CCW_23,
	ADC_MOD_CCW_24,
	ADC_MOD_CCW_25,
	ADC_MOD_CCW_26,
	ADC_MOD_CCW_27,
	ADC_MOD_CCW_28,
	ADC_MOD_CCW_29,
	ADC_MOD_CCW_30,
	ADC_MOD_CCW_31,
	ADC_MOD_CCW_32,
	ADC_MOD_CCW_33,
	ADC_MOD_CCW_34,
	ADC_MOD_CCW_35,
	ADC_MOD_CCW_36,
	ADC_MOD_CCW_37,
	ADC_MOD_CCW_38,
	ADC_MOD_CCW_39,
	ADC_MOD_CCW_40,
	ADC_MOD_CCW_41,
	ADC_MOD_CCW_42,
	ADC_MOD_CCW_43,
	ADC_MOD_CCW_44,
	ADC_MOD_CCW_45,
	ADC_MOD_CCW_46,
	ADC_MOD_CCW_47,
	ADC_MOD_CCW_48,
	ADC_MOD_CCW_49,
	ADC_MOD_CCW_50,
	ADC_MOD_CCW_51,
	ADC_MOD_CCW_52,
	ADC_MOD_CCW_53,
	ADC_MOD_CCW_54,
	ADC_MOD_CCW_55,
	ADC_MOD_CCW_56,
	ADC_MOD_CCW_57,
	ADC_MOD_CCW_58,
	ADC_MOD_CCW_59,
	ADC_MOD_CCW_60,
	ADC_MOD_CCW_61,
	ADC_MOD_CCW_62,
	ADC_MOD_CCW_63,
	ADC_MOD_CCW_TOT,
 }QADC_ConvMod_CCWs;

/* Structure used to hold all configuration of each ADC module converter */
/** \brief	This structure holds all conversion configuration of any ADC 
			converter module */
typedef struct
{
	/** ADC converter module ID */
	QADC_Module_ID ModID;
	/** ADC converter operational mode */
	QADC_OpMode OpMode;
	/** ADC converter register access */
	QADC_RegAcc RegAcc;
	/** ADC converter external Mux selection */
	QADC_ExtMuxSel ExtMux;
	/** External triggering type*/
	QADC_ExtTrig_Sel ExtTrigType;
	/** Single scan enable/ disable */
	QADC_Q1_SingleScanOpt Q1_SingleScan;
	/** Queue-1 operation mode */
	ADC_Q1_OpModes Q1_OpMode;
	/** Queue-2 single scan enable/ disable */
	QADC_Q2_SingleScanOpt Q2_SingleScan;
	/** Queue-2 operation mode */
	ADC_Q2_OpModes Q2_OpMode;
	/** Queue-2 resume position*/
	QADC_Q2_ResumePos Q2_ResumeType;
	/** ADC converter Queue-1 priority */
	U8 u8Q1IntPrio;
	/** ADC converter Queue-2 priority */
	U8 u8Q2IntPrio;
	/** Presclar clock high time for ADC conversion */
	U8 u8PresclClkHighTime;
	/** Presclar clock low time for ADC conversion */
	U8 u8PresclClkLowTime;
	/** Queue-2 CCW begin position in the total queue */
	U8 u8Q2Begin;
}QADC_ConvModConf;

/** \brief	Application callback function prototype to call after 
			ADC conversion complete */
typedef void (*ConvComplCB)(QADC_ConvMod_CCWs CCW_ID, U16 u16Data);

/** \brief	This enum provides mask settings for calling application 
			call-back functions. */
typedef enum
{
	/** This will disable to call the application registered call-back 
		function afetr ADC conversion completes */
	ADC_CB_NONE = 0x0000,
	/** This mask will work if you have enabled Pause settings for this 
		CCW. As the queue execution will enter into pause state after 
		execution of the CCW, the MCU will call the registered 
		call-back function for this CCW */
	ADC_CB_PAUSE_MSK = 0x8000,
	/** This mask will enable to call the application registered 
		call-back function for this CCW, after the whole ADC CCW 
		queue execution will complete */
	ADC_CB_COMPL_MSK = 0x4000,
	/** This mask will enable to call the application registered 
		call-back function after the pause and queue completion */
	ADC_CB_PAUSE_COMPL_MASK = 0xC000,
}ADC_ApplCBMasks;

/**	\brief	This enum holds different interrupt status supported */
typedef enum
{
	/** QADC Queue-1 completion interrupt status */
	ADC_Q1_COMPL_STATUS,
	/** QADC Queue-1 pause interrupt status */
	ADC_Q1_PAUSE_STATUS,
	/** QADC Queue-2 completion interrupt status */
	ADC_Q2_COMPL_STATUS,
	/** QADC Queue-2 pause interrupt status */
	ADC_Q2_PAUSE_STATUS,
	/** QADC Queue-1 overrun interrupt status */
	ADC_Q1_TRIG_OVERRUN,
	/** QADC Queue-2 overrun interrupt status */
	ADC_Q2_TRIG_OVERRUN,
}ADC_INT_Type;


/**	\brief	This structure defines the data for each CCW and the 
			Application call-back function for that conversion */
typedef struct
{
	/** Call-back function to be called after this CCW conversion is complete */
	ConvComplCB ApplCB;
	/**	This defines the ADC converter module ID */
	QADC_Module_ID ModID;
	/** Defines the CCW ID (CCW location), where the CCW will be stored */
	QADC_ConvMod_CCWs CCW_ID;
	/** If Pause value is provided, the queue execution will be 
		suspended and will be resumed after getting a new trigger 
		signal. If the CCW is in Queue-1, the execution will 
		resume from the next CCW. If this CCW is in Queue-2, 
		the resume position will be depend on the resume position 
		defined in the ADC converter module configuration */
	QADC_CCW_PauseVal PauseVal;
	/** Amplifier bypass selection for the analog channel in 
		legacy mode of operation. FO renhanced mode of oepration, 
		this value is not used */
	QADC_CCW_AmplBypassVal BypassVal;
	/**	This is used for reference selection in enhanced mode of operation */
	QADC_CCW_RefVal RefVal;
	/** ADC conversion time for this CCW */
	ADC_InputSampleTime SampTime;
	/** Callback mask setting. Use masks values from the enum 
		"ADC_ApplCBMasks" */
	ADC_ApplCBMasks ApplCB_Mask;
	/** Analog input channel No.	\n
		- 0 to 3, 48 to 59				=> (If Ext MUX is not selected)
		- 0 to 31, 48 to 51, 55 to 59	=> (If Ext MUX is selected)
		- 4 to 31 (if Ext MUX is not selected), 32 to 47	=> The low 
										reference level (VRL) is converted
		- 60 to 62			=> Sample amplifier will not be used. The value
								of VRL, VRH, or (VRH  VRL)/2 is placed 
								directly into the converter
		- 63				=> End of queue
	*/
	U8 u8ChannelNo;
}QADC_CCW_ApplData;


/**	\brief	This structure defines the data for reading the result of 
			any ADC conversion */
typedef struct
{
	/** ADc converter module ID */
	QADC_Module_ID ModID;
	/** ADC converter CCW ID (queue Index) */
	QADC_ConvMod_CCWs QueueID;
	/** Result read format requested */
	QADC_ResultFormat ReadFormat;
	/** Buffer to get theresult value from the ADC driver */
	U16 u16Data;
}QADC_CCW_ApplResultData;



/******************************************************************************
API Prototypes
******************************************************************************/
 QADC_DrvOpStatus ADC_Init(QADC_ConvModConf* ModConf);
 QADC_DrvOpStatus ADC_AddCCW(QADC_CCW_ApplData* CCW_Conf);
 QADC_DrvOpStatus ADC_Read (QADC_CCW_ApplResultData* ResultData);
 void ADC_ConvStop(QADC_Module_ID ModID);
 void ADC_Off(QADC_Module_ID ModID);

/* Interrupt status */
Bool ADC_InterruptStatus(QADC_Module_ID ModID, ADC_INT_Type IntType);
void ADC_ClearInterrupt(QADC_Module_ID ModID, ADC_INT_Type IntType);

 QADC_IO_OpStatus QADC_PortConfig(QADC_Module_ID ModID, QADC_Ports PortID, QADC_PortConfigType Type);
 QADC_IO_OpStatus QADC_PortSet(QADC_Module_ID ModID, QADC_Ports PortID, QADC_PortVal Value);
 QADC_IO_OpStatus QADC_PortGet(QADC_Module_ID ModID, QADC_Ports PortID, QADC_PortVal* Value);
#endif