/* GIMP - The GNU Image Manipulation Program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * GIMP PSD Plug-in
 * Copyright 2007 by John Marshall
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* ----- Known Layer Resource Block Types -----
  All layer resources not otherwise handled, including unknown types
  are dropped with a warning.

 * Adjustment layer IDs * 
  PSD_LADJ_LEVEL          "levl"        -       * Adjustment layer - levels (PS4) *  
  PSD_LADJ_CURVE          "curv"        -       * Adjustment layer - curves (PS4) * 
  PSD_LADJ_BRIGHTNESS     "brit"        -       * Adjustment layer - brightness contrast (PS4) * 
  PSD_LADJ_BALANCE        "blnc"        -       * Adjustment layer - color balance (PS4) * 
  PSD_LADJ_HUE            "hue "        -       * Adjustment layer - old hue saturation (PS4) * 
  PSD_LADJ_HUE2           "hue2"        -       * Adjustment layer - hue saturation (PS5) * 
  PSD_LADJ_SELECTIVE      "selc"        -       * Adjustment layer - selective color (PS4) * 
  PSD_LADJ_THRESHOLD      "thrs"        -       * Adjustment layer - threshold (PS4) * 
  PSD_LADJ_INVERT         "nvrt"        -       * Adjustment layer - invert (PS4) * 
  PSD_LADJ_POSTERIZE      "post"        -       * Adjustment layer - posterize (PS4) * 

 * Fill Layer IDs * 
  PSD_LFIL_SOLID          "SoCo"        -       * Solid color sheet setting (PS6) * 
  PSD_LFIL_PATTERN        "PtFl"        -       * Pattern fill setting (PS6) * 
  PSD_LFIL_GRADIENT       "GdFl"        -       * Gradient fill setting (PS6) * 

 * Effects Layer IDs * 
  PSD_LFX_FX              "lrFX"        -       * Effects layer info (PS5) * 
  PSD_LFX_FX2             "lfx2"        -       * Object based effects layer info (PS6) * 

 * Type Tool Layers * 
  PSD_LTYP_TYPE           "tySh"        -       * Type tool layer (PS5) * 
  PSD_LTYP_TYPE2          "TySh"        -       * Type tool object setting (PS6) * 

 * Layer Properties * 
  PSD_LPRP_UNICODE        "luni"     Loaded     * Unicode layer name (PS5) * 
  PSD_LPRP_SOURCE         "lnsr"     Loaded     * Layer name source setting (PS6) * 
  PSD_LPRP_ID             "lyid"     Loaded     * Layer ID (PS5) * 
  PSD_LPRP_BLEND_CLIP     "clbl"        -       * Blend clipping elements (PS6) * 
  PSD_LPRP_BLEND_INT      "infx"        -       * Blend interior elements (PS6) * 
  PSD_LPRP_KNOCKOUT       "knko"        -       * Knockout setting (PS6) * 
  PSD_LPRP_PROTECT        "lspf"        -       * Protected setting (PS6) * 
  PSD_LPRP_COLOR          "lclr"        -       * Sheet color setting (PS6) * 
  PSD_LPRP_REF_POINT      "fxrp"        -       * Reference point (PS6) * 

 * Vector mask * 
  PSD_LMSK_VMASK          "vmsk"        -       * Vector mask setting (PS6) * 

 * Parasites * 
  PSD_LPAR_ANNOTATE       "Anno"        -       * Annotation (PS6) * 

 * Other * 
  PSD_LOTH_PATTERN        "Patt"        -       * Patterns (PS6) * 
  PSD_LOTH_GRADIENT       "grdm"        -       * Gradient settings (PS6) * 
  PSD_LOTH_SECTION        "lsct"        -       * Section divider setting (PS6) * 
  PSD_LOTH_RESTRICT       "brst"        -       * Channel blending restirction setting (PS6) * 
  PSD_LOTH_FOREIGN_FX     "ffxi"        -       * Foreign effect ID (PS6) * 
  PSD_LOTH_PATT_DATA      "shpa"        -       * Pattern data (PS6) * 
  PSD_LOTH_META_DATA      "shmd"        -       * Meta data setting (PS6) * 
  PSD_LOTH_LAYER_DATA     "layr"        -       * Layer data (PS6) * 

 * Effects layer resource IDs * 
  PSD_LFX_COMMON          "cmnS"        -       * Effects layer - common state (PS5) * 
  PSD_LFX_DROP_SDW        "dsdw"        -       * Effects layer - drop shadow (PS5) * 
  PSD_LFX_INNER_SDW       "isdw"        -       * Effects layer - inner shadow (PS5) * 
  PSD_LFX_OUTER_GLW       "oglw"        -       * Effects layer - outer glow (PS5) * 
  PSD_LFX_INNER_GLW       "iglw"        -       * Effects layer - inner glow (PS5) * 
  PSD_LFX_BEVEL           "bevl"        -       * Effects layer - bevel (PS5) * 
*/

#include "config.h"

#include <string.h>
#include <errno.h>

#include <glib/gstdio.h>
#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>

#include "psd.h"
#include "psd-layer-res.h"

#include "libgimp/stdplugins-intl.h"


/*  Local function prototypes  */
static gint     load_resource_unknown           (const PSDlayerres     *res_a,
                                                 const gint32           layer_id,
                                                 FILE                   *f);

static gint     load_resource_ladj              (const PSDlayerres     *res_a,
                                                 const gint32           layer_id,
                                                 FILE                   *f);

static gint     load_resource_lfil              (const PSDlayerres     *res_a,
                                                 const gint32           layer_id,
                                                 FILE                   *f);

static gint     load_resource_lfx               (const PSDlayerres     *res_a,
                                                 const gint32           layer_id,
                                                 FILE                   *f);

static gint     load_resource_ltyp              (const PSDlayerres     *res_a,
                                                 const gint32           layer_id,
                                                 FILE                   *f);

static gint     load_resource_luni              (const PSDlayerres     *res_a,
                                                 const gint32           layer_id,
                                                 FILE                   *f);

static gint     load_resource_lyid              (const PSDlayerres     *res_a,
                                                 const gint32           layer_id,
                                                 FILE                   *f);

/* Public Functions */
gint
get_layer_resource_header (PSDlayerres *res_a, FILE *f)
{
  if (fread (res_a->sig, 4, 1, f) < 1
      || fread (res_a->key, 4, 1, f) < 1
      || fread (&res_a->data_len, 4, 1, f) < 1)
    {
      g_message (_("Error reading layer resource block header"));
      return -1;
    }
  res_a->data_len = GUINT32_FROM_BE (res_a->data_len);
  res_a->data_start = ftell (f);

  IFDBG(2) g_debug ("Sig: %.4s, key: %.4s, start: %d, len: %d",
                     res_a->sig, res_a->key, res_a->data_start, res_a->data_len);

  return 0;
}

gint
load_layer_resource (PSDlayerres *res_a, const gint32 layer_id, FILE *f)
{
  gint  pad;

  /* Set file position to start of layer resource data block */
  if (fseek (f, res_a->data_start, SEEK_SET) < 0)
    {
      g_message (_("Error setting file position"));
      return -1;
    }

   /* Process layer resource blocks */
  if (memcmp (res_a->sig, "8BIM", 4) != 0)
      g_message (_("Unknown layer resource signature %.4s"), res_a->sig);

  if (memcmp (res_a->key, PSD_LADJ_LEVEL, 4) == 0
      || memcmp (res_a->key, PSD_LADJ_CURVE, 4) == 0
      || memcmp (res_a->key, PSD_LADJ_BRIGHTNESS, 4) == 0
      || memcmp (res_a->key, PSD_LADJ_BALANCE, 4) == 0
      || memcmp (res_a->key, PSD_LADJ_HUE, 4) == 0
      || memcmp (res_a->key, PSD_LADJ_HUE2, 4) == 0
      || memcmp (res_a->key, PSD_LADJ_SELECTIVE, 4) == 0
      || memcmp (res_a->key, PSD_LADJ_THRESHOLD, 4) == 0
      || memcmp (res_a->key, PSD_LADJ_INVERT, 4) == 0
      || memcmp (res_a->key, PSD_LADJ_LEVEL, 4) == 0)
        load_resource_ladj (res_a, layer_id, f);

  else if (memcmp (res_a->key, PSD_LFIL_SOLID, 4) == 0
      || memcmp (res_a->key, PSD_LFIL_PATTERN, 4) == 0
      || memcmp (res_a->key, PSD_LFIL_GRADIENT, 4) == 0)
        load_resource_lfil (res_a, layer_id, f);

  else if (memcmp (res_a->key, PSD_LFX_FX, 4) == 0
      || memcmp (res_a->key, PSD_LFX_FX2, 4) == 0)
        load_resource_lfx (res_a, layer_id, f);

  else if (memcmp (res_a->key, PSD_LTYP_TYPE, 4) == 0
      || memcmp (res_a->key, PSD_LTYP_TYPE, 4) == 0)
        load_resource_ltyp (res_a, layer_id, f);

  else if (memcmp (res_a->key, PSD_LPRP_UNICODE, 4) == 0)
        load_resource_luni (res_a, layer_id, f);
        
  else if (memcmp (res_a->key, PSD_LPRP_ID, 4) == 0)
        load_resource_lyid (res_a, layer_id, f);

  else
    load_resource_unknown (res_a, layer_id, f);

  /* Layer blocks are null padded to even length */
  if (res_a->data_len % 2 == 0)
    pad = 0;
  else
    pad = 1;

  /* Set file position to end of layer resource block */
  if (fseek (f, res_a->data_start + res_a->data_len + pad, SEEK_SET) < 0)
    {
      g_message (_("Error setting file position"));
      return -1;
    }
  
  return 0;
}

/* Private Functions */

static gint
load_resource_unknown (const PSDlayerres *res_a, const gint32 layer_id, FILE *f)
{
  IFDBG(2) g_debug ("Process unknown layer resource block: %.4s", res_a->key);

  return 0;
}

static gint
load_resource_ladj (const PSDlayerres *res_a, const gint32 layer_id, FILE *f)
{
  /* Load adjustment layer */

  IFDBG(2) g_debug ("Process layer resource block %.4s: Adjustment layer", res_a->key);
 
  g_message (_("Warning:\n"
               "The image file contains adjustment layers. "
               "These are not supported by the GIMP and will "
               "be dropped. An empty layer will be included "
               "as a placeholder"));

  return 0;
}

static gint
load_resource_lfil (const PSDlayerres *res_a, const gint32 layer_id, FILE *f)
{
  /* Load fill layer */

  IFDBG(2) g_debug ("Process layer resource block %.4s: Fill layer", res_a->key);
  
  g_message (_("Warning:\n"
               "The image file contains fill layers. "
               "These are not supported by the GIMP and will "
               "be rasterized."));

  return 0;
}

static gint
load_resource_lfx (const PSDlayerres *res_a, const gint32 layer_id, FILE *f)
{
  /* Load layer effects */

  IFDBG(2) g_debug ("Process layer resource block %.4s: Layer effects", res_a->key);
 
  g_message (_("Warning:\n"
               "The image file contains layer effects. "
               "These are not supported by the GIMP and will "
               "be dropped."));

  return 0;
}

static gint
load_resource_ltyp (const PSDlayerres *res_a, const gint32 layer_id, FILE *f)
{
  /* Load type tool layer */

  IFDBG(2) g_debug ("Process layer resource block %.4s: Type tool layer", res_a->key);
 
  g_message (_("Warning:\n"
               "The image file contains type tool layers. "
               "These are not supported by the GIMP and will "
               "be dropped."));

  return 0;
}

static gint
load_resource_luni (const PSDlayerres *res_a, const gint32 layer_id, FILE *f)
{
  /* Load layer name in unicode */
  
  IFDBG(2) g_debug ("Process layer resource block luni: Unicode Name");

  return 0;
}

static gint
load_resource_lyid (const PSDlayerres *res_a, const gint32 layer_id, FILE *f)
{
  /* Load layer id (tattoo) */

  IFDBG(2) g_debug ("Process layer resource block lyid: Layer ID");

  return 0;
}
