/* GIMP Plug-in DeNoise
 * Copyright (C) 2007 Roland Simmen
 * All Rights Reserved.
 * 
 * This plugin is derived from the template provided by Michael Natterer
 * 
 * GIMP Plug-in Template
 * Copyright (C) 2000  Michael Natterer <mitch@gimp.org> (the "Author").
 * All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of the Author of the
 * Software shall not be used in advertising or otherwise to promote the
 * sale, use or other dealings in this Software without prior written
 * authorization from the Author.
 */

#include <memory.h>
#include <math.h>
#include <limits.h>

#include "denoise_constants.h"
#include "denoise_render.h"
#include "plugin-intl.h"

#define RENDER_INLINE    inline      // use inline for performance

#ifdef RENDER_8BIT
  #define CONTRAST_MAX     UCHAR_MAX   // colors are in 8bit
#endif
#ifdef RENDER_16BIT
  #define CONTRAST_MAX     USHRT_MAX   // colors are in 16bit
#endif

#ifdef RENDER_8BIT
typedef guchar       PixelColor;     // a 8 bit pixel color
#endif

typedef PixelColor  *PixelRow;       // a row of pixels with colors
typedef PixelRow    *PixelRowArray;  // an array of pixel rows

typedef gdouble      WeightArray[NOISE_MAX+1];                     
typedef gdouble      WeightMatrix[2*RADIUS_MAX+1][2*RADIUS_MAX+1]; 


static void init_weight_contrast (gint        noise, 
                                  WeightArray fContrast);
static void init_weight_gauss (gint           radius, 
                               WeightMatrix   fGauss);


static void init_mem    (PixelRowArray    *row_in,
                         PixelRow         *row_out,
                         gint              width,
                         gint              channels,
                         gint              radius);

static void free_mem    (PixelRowArray  *row_in,
                         PixelRow       *row_out,
                         gint            radius);

RENDER_INLINE
static void process_row (gint              i,
                         PixelRowArray     row_in,
                         PixelRow          row_out,
                         gint              radius,
                         gint              width,
                         gint              height,
                         gint              channels,
                         gint              colors);

RENDER_INLINE
static void analyze_row (PixelRow  row_in_i,
                         PixelRow  row_out,
                         gint      noiseDark,
                         gint      noiseLight,
                         gint      width,
                         gint      channels,
                         gint      colors);

RENDER_INLINE
static gdouble luminance (gint        colors,
                          PixelColor  apixel[]);

RENDER_INLINE
static void init_load_row (GimpPixelRgn     *rgn_in,
                           PixelRowArray     row,
                           gint              x1,
                           gint              y1,
                           gint              radius,
                           gint              width,
                           gint              height);

RENDER_INLINE
static void update_row  (GimpPixelRgn     *rgn_in,
                         gint              i,
                         PixelRowArray     row,
                         gint              x1,
                         gint              y1,
                         gint              radius,
                         gint              width,
                         gint              height);

static WeightArray    fWeightContrastDark = {0.0};  // weight function dark noise
static WeightArray    fWeightContrastLight = {0.0}; // weight function light noise
static WeightMatrix   fWeightGauss = {{0.0}};       // gaussian blur matrix

/*  Public functions  */

extern
void   denoise_render (PlugInVals  *data)
/***
 * Render the picture
 * 
 * in : data = handle to plugin values
 */
{
  GimpDrawable   *drawable = data->drawable; // handle to drawable
  GimpPreview    *preview  = data->preview;  // handle to preview
  RenderParams   *vals     = &(data->vals);  // render parameters
  
  gint            i = 0;           // tile row index
  gint            channels = 0;    // number of channels
  gint            colors = 0;      // number of colors
  gint            x1 = 0;          // x-axis of upper left corner
  gint            y1 = 0;          // y-axis of upper left corner
  gint            x2 = 0;          // x-axis of lower right corner
  gint            y2 = 0;          // y-axis of lower right corner
  gint            width = 0;       // width  
  gint            height = 0;      // heigt 
  GimpPixelRgn    rgn_in = {0};    // region for input region
  GimpPixelRgn    rgn_out = {0};   // region for output region
  PixelRowArray   row_in = NULL;   // array of input tile rows
  PixelRow        row_out = NULL;  // output tile row
          
  /* intialize the progess bar */
  gimp_progress_init (STR_DENOISE_PROGRESS);

  /* initialize the weight function */
  init_weight_contrast(data->vals.sigmaNoiseDark, fWeightContrastDark);
  init_weight_contrast(data->vals.sigmaNoiseLight, fWeightContrastLight);
  init_weight_gauss(data->vals.radius, fWeightGauss);
  
  /* for analysis, set the median weight to zero */
  if (vals->checkNoise && (preview != NULL))
  {
  	fWeightGauss[data->vals.radius][data->vals.radius] = 0.0;
  }

  /* Gets upper left and lower right coordinates,
   * and layers number in the image */
  if (preview != NULL)
  {
    gimp_preview_get_position (preview, &x1, &y1);
    gimp_preview_get_size (preview, &width, &height);
  }
  else
  {
    gimp_drawable_mask_bounds (drawable->drawable_id,
                               &x1, &y1,
                               &x2, &y2);
    width = x2 - x1;
    height = y2 - y1;
  }
   
  /* get the number of channels (1 byte per channel) */
  channels = gimp_drawable_bpp (drawable->drawable_id);
  
  /* get the number of colors*/
  if (gimp_drawable_is_rgb (drawable->drawable_id))
  {
    colors = 3;
  }
  else if (gimp_drawable_is_gray (drawable->drawable_id))
  {
    colors = 1;
  }

  /* allocate a big enough tile cache */
  gimp_tile_cache_ntiles (2 * (drawable->width / gimp_tile_width () + 1));

  /* Initialises two PixelRgns, one to read original data,
   * and the other to write output data. That second one will
   * be merged at the end by calling gimp_drawable_merge_shadow() */
  gimp_pixel_rgn_init (&rgn_in,
                       drawable,
                       x1, y1,
                       width, height,
                       FALSE, FALSE);
  gimp_pixel_rgn_init (&rgn_out,
                       drawable,
                       x1, y1,
                       width, height,
                       preview == NULL, TRUE);

  /* allocate memory for input and output tile rows */
  init_mem (&row_in, &row_out, width, channels, vals->radius);

  /* load the pixels to input tile rows */
  init_load_row (&rgn_in, row_in, x1, y1, vals->radius, width, height);

  for (i = 0; i < height; i++)
    {
      /* To be done for each tile row */
      process_row (i,
                   row_in,
                   row_out,
                   vals->radius,
                   width,
                   height,
                   channels,
                   colors);
      
      if (vals->checkNoise && (preview != NULL))
      {
        analyze_row (row_in[vals->radius],
                     row_out,
                     vals->sigmaNoiseDark,
                     vals->sigmaNoiseLight,
                     width,
                     channels,
                     colors);
      }
      
      gimp_pixel_rgn_set_row (&rgn_out, row_out, x1, i + y1, width);
      
      /* shift tile rows to append the new one to the end */
      update_row (&rgn_in, i, row_in, x1, y1, vals->radius, width, height);
               
      if ((preview == NULL) && (i % 10 == 0))
        gimp_progress_update ((gdouble) i / (gdouble) height);
    }

  /* Free memory */
  free_mem (&row_in, &row_out, vals->radius);

  /*  Update the modified region */
  if (preview != NULL)
    {
      gimp_drawable_preview_draw_region (GIMP_DRAWABLE_PREVIEW (preview),
                                         &rgn_out);
    }
  else
    {
      gimp_drawable_flush (drawable);
      gimp_drawable_merge_shadow (drawable->drawable_id, TRUE);
      gimp_drawable_update (drawable->drawable_id,
                            x1, y1,
                            width, height);
    }
}

/*  Private functions  */

static void
init_weight_contrast (gint         noise, 
                      WeightArray  fContrast)
/***
 * Intitialize the contrast weight function
 * 
 * in     : noise     = average noise level
 * in&out : fContrast = weight function for contrast to initialize
 */
{
  gint    i = 0;
  gdouble sigma = (gdouble)noise;  

  for (i = 0; i < (sizeof(WeightArray)/sizeof(gdouble)); i++)
  {
    fContrast[i] = exp(- (gdouble)(i * i) / (sigma * sigma));
  }
}


static void
init_weight_gauss (gint         radius, 
                   WeightMatrix fGauss)
/***
 * Intitialize the gaussian blur weight function
 * 
 * in     : radius = blur radius
 * in&out : fGauss = gaussian blur weight function
 */
{
  gint    i = 0;
  gint    j = 0;
  gdouble r2 = 0.0;
  gdouble sigma = (gdouble)radius / sqrt(log(255.0));
  
  for (i = -radius; i <= radius; i++)
  {
    for (j = -radius; j <= radius; j++)
    {
      r2 = (gdouble)(i*i + j*j);
      fWeightGauss[radius + i][radius + j] = exp(- r2 / (sigma * sigma));
    }
  }
}


static void
init_mem (PixelRowArray  *row_in,
          PixelRow       *row_out,
          gint            width,
          gint            channels,
          gint            radius)
/***
 * Intitialize memory for row buffers
 * 
 * in  : width    = picture width
 *       channels = number of channels
 *       radius   = blur radius
 * out : row_in   = buffer for input tile rows
 *       row_out  = buffer for resulting tile row
 */
{
  gint  ii = 0;
  gint  lenRow = width * channels;

  /* Allocate enough memory for row_in and row_out */
  *row_in = g_new (PixelRow, 2 * radius + 1);

  for (ii = 0; ii < 2 * radius + 1; ii++)
  {
    (*row_in)[ii] = g_new (PixelColor, lenRow);
  }

  *row_out = g_new (PixelColor, lenRow);
}


static void
free_mem (PixelRowArray  *row_in,
          PixelRow       *row_out,
          gint            radius)
/***
 * Free memory for row buffers
 * 
 * in     : radius   = blur radius
 * in&out : row_in   = input tile row buffer as input 
 *                     and freed memory as output
 *          row_out  = output tile row buffer as input
 *                     and freed memory as output
 */
{
  gint  ii = 0;
 
  /* Free allocated  memory for row_in and row_out */
  for (ii = 0; ii < 2 * radius + 1; ii++)
  {
    g_free ((*row_in)[ii]);
  }

  g_free (*row_in);
  g_free (*row_out);
}


RENDER_INLINE
static void
process_row (gint           i,
             PixelRowArray  row_in,
             PixelRow       row_out,
             gint           radius,
             gint           width,
             gint           height,
             gint           channels,
             gint           colors)
/***
 * Process a single row
 * 
 * in  : i        = row index
 *       row_in   = input tile row buffer
 *       row_out  = resulting tile row buffer
 *       radius   = blur radius
 *       width    = picture width
 *       height   = picture height
 *       channels = number of channels
 *       colors   = number of colors
 */
{
  gint     j      = 0; // column index
  gint     jc     = 0; // column index in row
  gint     k = 0;      // color index
  gint     ii = 0;     // row index in convolution matrix
  gint     jj = 0;     // column index in convolution matrix
  gint     jjc = 0;    // column index in row for convolution
  gint     bottom = MAX(radius - i, 0);                    // bottom boundary for ii
  gint     top    = MIN(height - i - 1, radius) + radius;  // top boundary for ii
  gint     left  = 0;                                      // left boundary for jj
  gint     right = 0;                                      // right boundary for ii
  
  gint     delta = 0;  // difference in color intensity scaled to max noise level
  gdouble  fdelta = (gdouble)NOISE_MAX / (gdouble)CONTRAST_MAX / (gdouble)colors; 
   
  WeightMatrix  weightDark = {{0.0}};  // weights for convolution matrix for dark noise
  WeightMatrix  weightLight = {{0.0}}; // weights for convolution matrix for light noise
  
  gdouble  sumDark = 0.0;         // sum of weighted colors for dark noise
  gdouble  sumWeightDark = 0.0;   // sum of the weights for dark noise
  gdouble  sumLight = 0.0;        // sum of weighted colors for light noise 
  gdouble  sumWeightLight = 0.0;  // sum of the weights for light noise
  gdouble  lum = 0.0;             // luminance
  
  for (j = 0; j < width; j++)
  {
    left  = MAX(radius - j, 0);
    right = MIN(width - j - 1, radius) + radius; 
    jc    = channels * j;
      
    /* fill the weight matrix */
    memcpy(&weightDark, &fWeightGauss, sizeof(WeightMatrix));
    memcpy(&weightLight, &fWeightGauss, sizeof(WeightMatrix));

    sumWeightDark = 0.0;
    sumWeightLight = 0.0;
    
    jjc = channels * (j - radius + left);
    
    for (jj = left; jj <= right; jj++, jjc += channels)
    {   
      for (ii = bottom; ii <= top; ii++)
      {        
        for (k = 0; k < colors; k++)
        {
          delta = (gint)(fdelta * (gdouble)(row_in[ii][jjc + k] - row_in[radius][jc + k]));            
          if (delta < 0) delta = -delta;
          
          weightDark[ii][jj] *= fWeightContrastDark[delta];
          weightLight[ii][jj] *= fWeightContrastLight[delta];
        }

        sumWeightDark += weightDark[ii][jj];
        sumWeightLight += weightLight[ii][jj];
      }
    }
    
    /* get the luminance */
    lum = luminance (colors, (row_in[radius])+jc);

    /* For each color, compute the average of the pixels */
    for (k = 0; k < colors; k++)
    {
      sumDark = 0.0;
      sumLight = 0.0;
      
      jjc = channels * (j - radius + left);
      
      for (jj = left; jj <= right; jj++, jjc += channels)
      {   
        for (ii = bottom; ii <= top; ii++)
        {
          sumDark += weightDark[ii][jj] * (gdouble)row_in[ii][jjc + k];
          sumLight += weightLight[ii][jj] * (gdouble)row_in[ii][jjc + k];
        }
      }

      row_out[jc + k] = (PixelColor)((1.-lum) * sumDark / sumWeightDark + 
                                     lum * sumLight / sumWeightLight + 0.5);
    }
    
    /* Copy the remaining channels, e.g. the alpha channel */
    for (k = colors; k < channels; k++)
    {
      row_out[jc + k] = row_in[radius][jc + k];
    }    
  }
}


RENDER_INLINE
static void
analyze_row (PixelRow       row_in_i,
             PixelRow       row_out,
             gint           noiseDark,
             gint           noiseLight,
             gint           width,
             gint           channels,
             gint           colors)
/***
 * Analyze a single row
 * 
 * in  : row_in     = input tile row buffer at index i
 *       row_out    = output tile row buffer
 *       radius     = blur radius
 *       noiseDark  = average dark noise level
 *       noiseLight = average light noise level
 *       width      = picture width
 *       channels   = number of channels
 *       colors     = number of colors
 */
{
  gint     j      = 0; // column index
  gint     jc     = 0; // column index in row
  gint     k = 0;      // color index
  
  gdouble  lum = 0.0;     // luminance
  gint     meanNoise = 0; // noise weighted by luminance
  gint     delta = 0;     // difference in color intensity scaled to max noise level  
  gdouble  fdelta = (gdouble)NOISE_MAX / (gdouble)CONTRAST_MAX;
  
  for (j = 0; j < width; j++)
  {
    jc    = channels * j;

    /* get the luminance */
    lum = luminance (colors, row_in_i + jc);

    /* get the mean noise */
    meanNoise = (gint)((1.-lum) * (gdouble)noiseDark +  lum * (gdouble)noiseLight);

    for (k = 0; k < colors; k++)
    {
      /* get the absolute difference between the average and the actual value */
      delta = (gint)(fdelta * (gdouble)(row_out[jc + k] - row_in_i[jc + k]));
      if (delta < 0) delta = -delta;
      
      /* if delta is less than noise, set grey, otherwise true color */
      if (delta < meanNoise) row_out[jc + k] = CONTRAST_MAX / 2;
      else                   row_out[jc + k] = row_in_i[jc + k];
    }
    
    /* Copy the remaining channels, e.g. the alpha channel */
    for (k = colors; k < channels; k++)
    {
      row_out[jc + k] = row_in_i[jc + k];
    }    
  }
}


RENDER_INLINE
static gdouble luminance (gint     colors, 
                          guchar   apixel[])
/***
 * Get the luminance of a single pixel
 * 
 * in     : colors = number of colors
 *          apixel = pixel to get luminance from
 * return : luminance of the pixel
 */
{
  gdouble  lum = 0; // luminance

  if (colors == 1)
  {
    lum = apixel[0];
  }
  else if (colors == 3)
  {
    lum = GIMP_RGB_INTENSITY(apixel[0], apixel[1], apixel[2]);
  }   

  return (lum / (gdouble)CONTRAST_MAX);
}


RENDER_INLINE
static void
init_load_row (GimpPixelRgn  *rgn_in,
               PixelRowArray  row_in,
               gint           x1,
               gint           y1,
               gint           radius,
               gint           width,
               gint           height)
/***
 * Initial load of rows row_in[radius], row_in[radius+1] ...
 * 
 * in  : rgn_in     = pixel region input
 *       row_in     = input tile row buffer
 *       x1         = x-coordinate of top left cornet
 *       y1         = y-coordinate of top left corner
 *       radius     = blur radius
 *       width      = picture width
 *       height     = picture height
 */
{
  gint ii = 0;
 
  for (ii = 0; ii <= MIN(height - 1, radius); ii++)
  {
    gimp_pixel_rgn_get_row (rgn_in, row_in[radius + ii], x1, y1 + ii, width);
  }
}


RENDER_INLINE
static void
update_row (GimpPixelRgn  *rgn_in,
            gint           i,
            PixelRowArray  row_in,
            gint           x1,
            gint           y1,
            gint           radius,
            gint           width,
            gint           height)
/***
 * Drop all rows row_in[i+1] to row_in[i] and fill top row
 * 
 * in  : rgn_in     = pixel region input
 *       i          = row index
 *       row_in     = input tile row buffer
 *       x1         = x-coordinate of top left cornet
 *       y1         = y-coordinate of top left corner
 *       radius     = blur radius
 *       width      = picture width
 *       height     = picture height
 */
{
  gint    ii = 0;
  gint    iiTop = 2 * radius;
  guchar *tmp_row = row_in[0];
 
  /* Circular shift of row_in[i+1] to row_in[i] and row_in[0] to row_in[iTop] */
  tmp_row = row_in[0];
  for (ii = 0; ii < iiTop; ii++) {
    row_in[ii] = row_in[ii + 1];
  } 
  
  row_in[iiTop] = tmp_row;
 
  /* Get tile row at i+1 into top row_in[iiTop] */
  gimp_pixel_rgn_get_row (rgn_in,
                          row_in[iiTop],
                          x1, y1 + MIN(i + 1 + radius, height - 1),
                          width);
}
