/* GIMP Plug-in DeNoise
 * Copyright (C) 2007 Roland Simmen
 * All Rights Reserved.
 * 
 * This plugin is derived from the template provided by Michael Natterer
 * 
 * GIMP Plug-in Template
 * Copyright (C) 2000  Michael Natterer <mitch@gimp.org> (the "Author").
 * All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of the Author of the
 * Software shall not be used in advertising or otherwise to promote the
 * sale, use or other dealings in this Software without prior written
 * authorization from the Author.
 */

#include <string.h>
#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>

#include "denoise_constants.h"
#include "denoise_interface.h"
#include "denoise_render.h"
#include "plugin-intl.h"

/*  Local function prototypes  */

static void   query (void);
static void   run   (const gchar      *name,
                     gint              nparams,
                     const GimpParam  *param,
                     gint             *nreturn_vals,
                     GimpParam       **return_vals);


/*  Local variables  */

const RenderParams default_vals =
{
  3,      // blurring radius
  10,     // sigma of noise level for dark
  10,     // sigma of noise level for light
  TRUE,   // check box for preview
  TRUE,   // check box for locking ratio
  FALSE   // check box for check noise
};


GimpPlugInInfo PLUG_IN_INFO =
{
  NULL,  /* init_proc  */
  NULL,  /* quit_proc  */
  query, /* query_proc */
  run,   /* run_proc   */
};

MAIN ()

static void
query (void)
{
  static GimpParamDef args[] =
  {
    { GIMP_PDB_INT32,    "run_mode",     "Interactive, non-interactive"    },
    { GIMP_PDB_IMAGE,    "image",        "Input image"                     },
    { GIMP_PDB_DRAWABLE, "drawable",     "Input drawable"                  },
    { GIMP_PDB_INT32,    "radius",       "Radius"                          },
    { GIMP_PDB_INT32,    "noise_dark",   "Dark Noise Level"                },
    { GIMP_PDB_INT32,    "noise_light",  "Light Noise Level"               },
    { GIMP_PDB_INT32,    "with_preview", "With Preview"                    },
    { GIMP_PDB_INT32,    "check_noise",  "Check Noise"                     },
    { GIMP_PDB_INT32,    "lock_ratio",   "Lock Noise Ratio"                }
  };

  gimp_install_procedure (STR_DENOISE_PROCEDURE_NAME,
                          STR_DENOISE_BLURB,
                          STR_DENOISE_HELP,
                          STR_DENOISE_AUTHOR,
                          STR_DENOISE_COPYRIGHT,
                          STR_DENOISE_DATE,
                          N_(STR_DENOISE_MENU),
                          STR_DENOISE_IMAGE_TYPE,
                          GIMP_PLUGIN,
                          G_N_ELEMENTS (args), 
                          0,
                          args, 
                          NULL);

  gimp_plugin_menu_register (STR_DENOISE_PROCEDURE_NAME, STR_DENOISE_MENUPATH);
}

static void
run (const gchar      *name,
     gint              n_params,
     const GimpParam  *param,
     gint             *nreturn_vals,
     GimpParam       **return_vals)
{
  static GimpParam   values[1];
  PlugInVals         data = {0};
  GimpRunMode        run_mode;
  GimpPDBStatusType  status = GIMP_PDB_SUCCESS;

  *nreturn_vals = 1;
  *return_vals  = values;

  run_mode = param[0].data.d_int32;
  data.image_ID = param[1].data.d_int32;
  data.drawable = gimp_drawable_get (param[2].data.d_drawable);
  data.preview  = NULL;

  /*  Initialize with default values  */
  data.vals     = default_vals;

  if (strcmp (name, STR_DENOISE_PROCEDURE_NAME) == 0)
    {
      switch (run_mode)
        {
        case GIMP_RUN_NONINTERACTIVE:
          if (n_params != 9)
            {
              status = GIMP_PDB_CALLING_ERROR;
            }
          else
            {
              data.vals.radius          = param[3].data.d_int32;
              data.vals.sigmaNoiseDark  = param[4].data.d_int32;
              data.vals.sigmaNoiseLight = param[5].data.d_int32;
              data.vals.withPreview     = param[6].data.d_int32;
              data.vals.checkNoise      = FALSE;
              data.vals.lockRatio       = param[8].data.d_int32;
            }
          break;

        case GIMP_RUN_INTERACTIVE:
          /*  Possibly retrieve data  */
          gimp_get_data (STR_DENOISE_DATA_KEY_VALS, &data.vals);

          if (! dlg_denoise (&data))
            {
              status = GIMP_PDB_CANCEL;
            }
          break;

        case GIMP_RUN_WITH_LAST_VALS:
          /*  Possibly retrieve data  */
          gimp_get_data (STR_DENOISE_DATA_KEY_VALS, &data.vals);
          break;

        default:
          break;
        }
    }
  else
    {
      status = GIMP_PDB_CALLING_ERROR;
    }

  if (status == GIMP_PDB_SUCCESS)
    {
      if (gimp_drawable_is_rgb (data.drawable->drawable_id) ||
          gimp_drawable_is_gray (data.drawable->drawable_id))
        {
        
          data.preview = NULL;
          denoise_render (&data);
      
          if (run_mode != GIMP_RUN_NONINTERACTIVE)
            gimp_displays_flush ();

          if (run_mode == GIMP_RUN_INTERACTIVE)
            {
              gimp_set_data (STR_DENOISE_DATA_KEY_VALS, &data.vals, sizeof (RenderParams));
            }

          gimp_drawable_detach (data.drawable);
        }
      else
        {
          g_message (_("Cannot operate on indexed color images."));
          status = GIMP_PDB_EXECUTION_ERROR;
        }
        
    }

  values[0].type = GIMP_PDB_STATUS;
  values[0].data.d_status = status;
  
  return;
}
