/* GIMP Plug-in DeNoise
 * Copyright (C) 2007 Roland Simmen
 * All Rights Reserved.
 * 
 * This plugin is derived from the template provided by Michael Natterer
 * 
 * GIMP Plug-in Template
 * Copyright (C) 2000-2004  Michael Natterer <mitch@gimp.org> (the "Author").
 * All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of the Author of the
 * Software shall not be used in advertising or otherwise to promote the
 * sale, use or other dealings in this Software without prior written
 * authorization from the Author.
 */

#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>
#include <stdlib.h>

#include "denoise_constants.h"
#include "denoise_interface.h"
#include "denoise_render.h"
#include "plugin-intl.h"

/* type definitions */
typedef struct _NoiseSpinButtonParams
{
  PlugInVals    *data;                        // handle to render data
  GtkWidget     *spinbutton_dark_noise;       // spin button for dark noise
  GtkObject     *spinbutton_adj_dark_noise;   // adjustment for dark noise
  GtkWidget     *spinbutton_light_noise;      // spin button for light noise
  GtkObject     *spinbutton_adj_light_noise;  // adjustment for light noise
  gdouble       ratio;
} NoiseSpinButtonParams;


/*  Local function prototypes  */
static void  change_dark_noise_callback  (GtkWidget    *widget,
                                          gpointer      data);
static void  change_light_noise_callback  (GtkWidget    *widget,
                                           gpointer      data);

/*  Public functions  */

extern 
gboolean  dlg_denoise (PlugInVals  *data)
/***
 * A dialog to get the render params from user
 * 
 * in : data = handle to plugin values
 */
{
  GtkWidget              *dialog = NULL;          // dialog handle
  GtkWidget              *main_vbox = NULL;       // vertical box
  GtkWidget              *main_hbox = NULL;       // horizontal box
  GtkWidget              *table = NULL;           // table to organize widgets
  GtkWidget              *frame = NULL;           // frame 
  GtkWidget              *label = NULL;           // label 
  GtkWidget              *spinbutton = NULL;      // spin button
  GtkObject              *spinbutton_adj = NULL;  // adjustment
  GtkWidget              *alignment = NULL;       // alignment widget
  GtkWidget              *checkbutton;            // check button
  NoiseSpinButtonParams   noise = {0};            // spin button parameters for noise
  gboolean                run = FALSE;

  gimp_ui_init (STR_DENOISE_NAME, TRUE);

  /* create new dialog */
  dialog = gimp_dialog_new (_(STR_DENOISE_DLG_TITLE), 
                            STR_DENOISE_DLG_ROLE,
                            NULL, 0,
                            gimp_standard_help_func, 
                            STR_DENOISE_HELP,
                            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                            GTK_STOCK_OK,     GTK_RESPONSE_OK,
                            NULL);
                            
  /* create a container 'vbox' to organize widgets in a column */
  main_vbox = gtk_vbox_new (FALSE, 6);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), main_vbox);

  /* create a preview of the drawable */
  data->preview = gimp_drawable_preview_new(data->drawable, &(data->vals.withPreview));
  gtk_box_pack_start (GTK_BOX (main_vbox), (GtkWidget *)data->preview, TRUE, TRUE, 0);
  
  g_signal_connect_swapped (data->preview, "invalidated",
                            G_CALLBACK (denoise_render),
                            data);
                            
  /* create a frame for the radius and set the border width */
  frame = gtk_frame_new (NULL);
  gtk_container_set_border_width (GTK_CONTAINER (frame), 6);
  gtk_box_pack_start (GTK_BOX (main_vbox), frame, TRUE, TRUE, 0);
                   
  label = gtk_label_new (_(STR_DENOISE_DLG_FRAME_RADIUS));
  gtk_frame_set_label_widget (GTK_FRAME (frame), label);
  gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
                      
  /* create an alignment widget and add it to the frame */
  alignment = gtk_alignment_new (0.0, 0.0, 0, 0);
  gtk_container_add (GTK_CONTAINER (frame), alignment);
  gtk_alignment_set_padding (GTK_ALIGNMENT (alignment), 6, 6, 6, 6);

  /* create a container 'hbox' to organize widgets in a row within the alignment */
  main_hbox = gtk_hbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (alignment), main_hbox);

  /* create a text label for the radius value input */
  label = gtk_label_new_with_mnemonic (_(STR_DENOISE_DLG_RADIUS));
  gtk_box_pack_start (GTK_BOX (main_hbox), label, FALSE, FALSE, 6);                      
  gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_RIGHT);

  /* create a spin button for the radius */
  spinbutton = gimp_spin_button_new (&spinbutton_adj, 
                                     data->vals.radius, 
                                     1, RADIUS_MAX, 1, 1, 1, 5, 0);                                     
  gtk_box_pack_start (GTK_BOX (main_hbox), spinbutton, FALSE, FALSE, 0);
                               
  g_signal_connect_swapped (spinbutton_adj, "value_changed",
                            G_CALLBACK (gimp_preview_invalidate),
                            data->preview);

  g_signal_connect (spinbutton_adj, "value_changed",
                    G_CALLBACK (gimp_int_adjustment_update),
                    &(data->vals.radius));

  /* create a frame for the noise level and set the border width */
  frame = gtk_frame_new (NULL);
  gtk_container_set_border_width (GTK_CONTAINER (frame), 6);
  gtk_box_pack_start (GTK_BOX (main_vbox), frame, TRUE, TRUE, 0);
                   
  /* create a label for the frame and add it to the frame */
  label = gtk_label_new (_(STR_DENOISE_DLG_FRAME_NOISE));
  gtk_frame_set_label_widget (GTK_FRAME (frame), label);
  gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
                      
  /* create an alignment widget and add it to the frame */
  alignment = gtk_alignment_new (0.0, 0.0, 0, 0);
  gtk_container_add (GTK_CONTAINER (frame), alignment);
  gtk_alignment_set_padding (GTK_ALIGNMENT (alignment), 6, 6, 6, 6);

  /* create a table to organize noise widgets in a table */
  table = gtk_table_new (2, 4, FALSE);
  gtk_table_set_col_spacings (GTK_TABLE (table), 6);
  gtk_table_set_row_spacings (GTK_TABLE (table), 6);
  
  gtk_container_add (GTK_CONTAINER (alignment), table);
  
  /* initialize noise button params */
  noise.data   = data;
  noise.ratio  = (gdouble)data->vals.sigmaNoiseLight / (gdouble)data->vals.sigmaNoiseDark;
  
  /* create a spin button for the dark noise */
  noise.spinbutton_dark_noise = gimp_spin_button_new (
                                         &noise.spinbutton_adj_dark_noise, 
                                         data->vals.sigmaNoiseDark, 
                                         1, NOISE_MAX, 1, 1, 1, 5, 0);
                                     
  gimp_table_attach_aligned (GTK_TABLE (table), 0, 0,
                             _(STR_DENOISE_DLG_NOISE_DARK), 0.0, 0.5,
                             noise.spinbutton_dark_noise, 1, TRUE);

  g_signal_connect (noise.spinbutton_dark_noise, "value_changed",
                    G_CALLBACK (change_dark_noise_callback),
                    &(noise));
                                      
  /* create a spin button for the light noise */
  noise.spinbutton_light_noise = gimp_spin_button_new (
                                          &noise.spinbutton_adj_light_noise, 
                                          data->vals.sigmaNoiseLight, 
                                          1, NOISE_MAX, 1, 1, 1, 5, 0);
                                     
  gimp_table_attach_aligned (GTK_TABLE (table), 0, 1,
                             _(STR_DENOISE_DLG_NOISE_LIGHT), 0.0, 0.5,
                             noise.spinbutton_light_noise, 1, TRUE);

  g_signal_connect (noise.spinbutton_light_noise, "value_changed",
                    G_CALLBACK (change_light_noise_callback),
                    &(noise));

  /* create a check button to toggle check noise or not */
  checkbutton = gtk_check_button_new_with_label(_(STR_DENOISE_DLG_NOISE_CHECK));
  gimp_table_attach_aligned (GTK_TABLE (table), 2, 0,
                             NULL, 0.0, 0.5,
                             checkbutton, 1, TRUE);
  
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbutton), 
                               data->vals.checkNoise);
  
  g_signal_connect_swapped (GTK_OBJECT(checkbutton), "clicked",
                            G_CALLBACK (gimp_preview_invalidate),
                            data->preview);
                            
  g_signal_connect (checkbutton, "clicked",
                    G_CALLBACK (gimp_toggle_button_update),
                    &(data->vals.checkNoise));

  /* create a check button to toggle lock ratio or not */
  checkbutton = gtk_check_button_new_with_label(_(STR_DENOISE_DLG_NOISE_RATIOLOCK));
  gimp_table_attach_aligned (GTK_TABLE (table), 2, 1,
                             NULL, 0.0, 0.5,
                             checkbutton, 1, TRUE);
  
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbutton), 
                               data->vals.lockRatio);
  
  g_signal_connect_swapped (GTK_OBJECT(checkbutton), "clicked",
                            G_CALLBACK (gimp_preview_invalidate),
                            data->preview);
                            
  g_signal_connect (checkbutton, "clicked",
                    G_CALLBACK (gimp_toggle_button_update),
                    &(data->vals.lockRatio));

  /* show all widgets */                                     
  gtk_widget_show_all (dialog);

  /* render with the changed values */
  denoise_render (data);

  run = (gimp_dialog_run (GIMP_DIALOG (dialog)) == GTK_RESPONSE_OK);

  gtk_widget_destroy (dialog);

  return run;
}


static void
change_dark_noise_callback (GtkWidget *spinbutton,
                            gpointer   data)
/***
 * Call back when the light noise value has changed
 * 
 * in : spinbutton = spin button for dark noise
 *      data       = noise data 
 */
{
  NoiseSpinButtonParams  *params = (NoiseSpinButtonParams *)data;
  RenderParams           *vals = &(params->data->vals);

  /* get the noise value fromspinbutton */
  vals->sigmaNoiseDark = gtk_spin_button_get_value_as_int ((GtkSpinButton *)spinbutton);
  
  if (vals->lockRatio && params->ratio > 0.0)
  {
    /* if noise ratio has been locked, update the light noise */    
    vals->sigmaNoiseLight = (gint)((gdouble)vals->sigmaNoiseDark * params->ratio + 0.5);
    gtk_adjustment_set_value ((GtkAdjustment *)params->spinbutton_adj_light_noise,
                              (gdouble)(vals->sigmaNoiseLight));
  }
  else if (!vals->lockRatio && vals->sigmaNoiseDark)
  {
    /* update the noise ratio if not locked and render */
    params->ratio = (gdouble)vals->sigmaNoiseLight / (gdouble)vals->sigmaNoiseDark;
  }
  
  /* render with new noise params */
  denoise_render(params->data);
}


static void
change_light_noise_callback (GtkWidget *spinbutton,
                             gpointer   data)
/***
 * Call back when the light noise value has changed
 * 
 * in : spinbutton = spin button for light noise
 *      data       = noise data 
 */
{
  NoiseSpinButtonParams  *params = (NoiseSpinButtonParams *)data;
  RenderParams           *vals = &(params->data->vals);

  /* get the noise value fromspinbutton */
  vals->sigmaNoiseLight = gtk_spin_button_get_value_as_int ((GtkSpinButton *)spinbutton);
  
  if (vals->lockRatio && params->ratio > 0.0)
  {
    /* if noise ratio has been locked, update the dark noise
     * do not render because update of the dark noise button
     * will cause rendering in the call back for dark noise */    
    vals->sigmaNoiseDark = (gint)((gdouble)vals->sigmaNoiseLight / params->ratio + 0.5);
    gtk_adjustment_set_value ((GtkAdjustment *)params->spinbutton_adj_dark_noise,
                              (gdouble)(vals->sigmaNoiseDark));
  }
  else if (!vals->lockRatio && vals->sigmaNoiseDark)
  {
    /* update the noise ratio if not locked and render */
    params->ratio = (gdouble)vals->sigmaNoiseLight / (gdouble)vals->sigmaNoiseDark;
    denoise_render (params->data);
  }
  else
  {
    /* render per default */
    denoise_render (params->data);
  }
}
