
#include "config.h"

#include <string.h>

#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

#include "webx_main.h"
#include "webx_source.h"
#include "webx_dialog.h"

#include "libgimp/stdplugins-intl.h"


static void     query (void);
static void     run   (const gchar      *name,
                       gint              nparams,
                       const GimpParam  *param,
                       gint             *nreturn_vals,
                       GimpParam       **return_vals);
static void     webx_run (gint32 image_ID,
                          gint32 drawable_ID);

const GimpPlugInInfo PLUG_IN_INFO =
  {
    NULL,  /* init_proc  */
    NULL,  /* quit_proc  */
    query, /* query_proc */
    run,   /* run_proc   */
  };


MAIN ()

     static void
query (void)
{
  static GimpParamDef args[] =
    {
      { GIMP_PDB_INT32,      "run-mode",    "Interactive, non-interactive" },
      { GIMP_PDB_IMAGE,      "image",       "Input image (unused)" },
      { GIMP_PDB_DRAWABLE,   "drawable",    "Input drawable" }
    };

  gimp_install_procedure (PLUG_IN_PROC,
			  N_("Save for Web"),
			  "Optimize image for web.",
			  "Aurimas Juska",
			  "Aurimas Juska",
			  "0.1",
			  N_("Save for Web..."),
			  "RGB*, GRAY*, INDEXED*",
			  GIMP_PLUGIN,
			  G_N_ELEMENTS (args), 0,
			  args, NULL);

  gimp_plugin_menu_register (PLUG_IN_PROC, "<Image>/Filters/Web");
}

static void
run (const gchar      *name,
     gint              nparams,
     const GimpParam  *param,
     gint             *nreturn_vals,
     GimpParam       **return_vals)
{
  static GimpParam   values[1];
  gint32             image_ID;
  gint32             drawable_ID;
  GimpPDBStatusType  status   = GIMP_PDB_SUCCESS;
  GimpRunMode        run_mode;

  run_mode = param[0].data.d_int32;
  image_ID = param[1].data.d_int32;
  drawable_ID = param[2].data.d_int32;

  values[0].type = GIMP_PDB_STATUS;
  values[0].data.d_status = status;

  INIT_I18N ();

  *nreturn_vals = 1;
  *return_vals = values;

  if (run_mode == GIMP_RUN_INTERACTIVE)
    {
      webx_run (image_ID, drawable_ID);
    }

  values[0].data.d_status = status;
}

static void
webx_pixbuf_destroy (guchar *pixels, gpointer data)
{
  g_free (pixels);
}

GdkPixbuf*
webx_drawable_to_pixbuf (gint32 layer)
{
  gint             width;
  gint             height;
  gint             bpp;
  guchar          *buf;
  GimpPixelRgn     pixel_rgn;
  GimpDrawable    *drawable;
  GdkPixbuf       *pixbuf;

  width = gimp_drawable_width (layer);
  height = gimp_drawable_height (layer);
  bpp = gimp_drawable_bpp (layer);
  drawable = gimp_drawable_get (layer);
  gimp_pixel_rgn_init (&pixel_rgn, drawable, 0, 0, width, height, FALSE, FALSE);
  buf = g_malloc (width * height * bpp);
  gimp_pixel_rgn_get_rect (&pixel_rgn, buf, 0, 0, width, height);
  gimp_drawable_detach (drawable);

  pixbuf = gdk_pixbuf_new_from_data (buf, GDK_COLORSPACE_RGB,
				     gimp_drawable_has_alpha (layer), 8,
				     width, height, width * bpp,
				     webx_pixbuf_destroy, NULL);
  return pixbuf;
}

GdkPixbuf*
webx_image_to_pixbuf (gint32 image)
{
  gint32           layer;

  layer = gimp_image_merge_visible_layers (image, GIMP_CLIP_TO_IMAGE);
  if (! gimp_drawable_is_rgb (layer))
    gimp_image_convert_rgb (image);
  return webx_drawable_to_pixbuf (layer);
}


static void
webx_run (gint32 image_ID, gint32 drawable_ID)
{
  gimp_ui_init (PLUG_IN_BINARY, FALSE);
  webx_source_init (image_ID, drawable_ID);
  webx_dialog_run ();
  webx_source_cleanup ();
}
