
#include "config.h"

#include <string.h>

#include <glib.h>
#include <glib/gstdio.h>
#include <gtk/gtk.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>

#include "webx_main.h"
#include "webx_source.h"
#include "webx_dialog.h"
#include "webx_jpeg.h"
#include "webx_png24.h"
#include "webx_png8.h"
#include "webx_gif.h"

#include "libgimp/stdplugins-intl.h"

#define WEBX_DRAG_HAND      (1<<0)
#define WEBX_DRAG_LEFT      (1<<1)
#define WEBX_DRAG_RIGHT     (1<<2)
#define WEBX_DRAG_TOP       (1<<3)
#define WEBX_DRAG_BOTTOM    (1<<4)

#define WEBX_HANDLE_SIZE        4


#define WEBX_DEFAULT_ZOOMLEVEL  4

const static gdouble webx_zoomlevels[] = {
  8.0, 4.0, 2.0, 1.5,
  1.0,
  0.5, 0.25, 0.125,
};

static WebxDialog      webx_dlg;


/* Draw an outline outside target image (which can be cropped)
 * and crop handles. */
static void
webx_draw_target_outline (GtkWidget     *widget,
                          GdkRectangle  *rect,
                          gboolean       filled)
{
  gdk_gc_set_function (widget->style->fg_gc[GTK_STATE_NORMAL], GDK_INVERT);

  gdk_draw_rectangle (widget->window,
		      widget->style->fg_gc[GTK_STATE_NORMAL],
		      filled,
		      rect->x-1, rect->y-1,
		      rect->width+1, rect->height+1);

  gdk_draw_rectangle (widget->window,
		      widget->style->fg_gc[GTK_STATE_NORMAL],
		      FALSE,
		      rect->x - WEBX_HANDLE_SIZE - 2,
		      rect->y - WEBX_HANDLE_SIZE - 1,
		      WEBX_HANDLE_SIZE, WEBX_HANDLE_SIZE);
  gdk_draw_rectangle (widget->window,
		      widget->style->fg_gc[GTK_STATE_NORMAL],
		      FALSE,
		      rect->x+rect->width + 1,
		      rect->y - WEBX_HANDLE_SIZE - 2,
		      WEBX_HANDLE_SIZE, WEBX_HANDLE_SIZE);
  gdk_draw_rectangle (widget->window,
		      widget->style->fg_gc[GTK_STATE_NORMAL],
		      FALSE,
		      rect->x + rect->width + 1,
		      rect->y + rect->height + 1,
		      WEBX_HANDLE_SIZE, WEBX_HANDLE_SIZE);
  gdk_draw_rectangle (widget->window,
		      widget->style->fg_gc[GTK_STATE_NORMAL],
		      FALSE,
		      rect->x - WEBX_HANDLE_SIZE - 2,
		      rect->y + rect->height + 1,
		      WEBX_HANDLE_SIZE, WEBX_HANDLE_SIZE);

  gdk_gc_set_function (widget->style->fg_gc[GTK_STATE_NORMAL], GDK_COPY);
}

/* We want to center view if image is smaller than preview window.
 * This routine returns image's rectangle in view window coordinates. */
static void
webx_view_get_rect (GdkRectangle    *rect)
{
  gint        width;
  gint        height;
  gint        view_width;
  gint        view_height;

  width = webx_background_get_width () * webx_dlg.zoom;
  height = webx_background_get_height () * webx_dlg.zoom;
    
  view_width = webx_dlg.optimized.view_widget->allocation.width;
  view_height = webx_dlg.optimized.view_widget->allocation.height;

  if (width + WEBX_VIEW_PADDING * 2 < view_width)
    rect->x = (view_width - width) / 2;
  else
    rect->x = WEBX_PADDING;
  if (height + WEBX_VIEW_PADDING*2 < view_height)
    rect->y = (view_height - height) / 2;
  else
    rect->y = WEBX_PADDING;
  rect->width = width;
  rect->height = height;
}

/* Returns rectangle of target image in view widget coordinates. */
static void
webx_view_get_target_rect (GdkRectangle     *rect)
{
  GdkRectangle    bg_rect;
  GdkRectangle    target_rect;

  webx_view_get_rect (&bg_rect);
  webx_source_get_rect (&target_rect);
  target_rect.x *= webx_dlg.zoom;
  target_rect.y *= webx_dlg.zoom;
  target_rect.width *= webx_dlg.zoom;
  target_rect.height *= webx_dlg.zoom;
  target_rect.x += bg_rect.x;
  target_rect.y += bg_rect.y;
  *rect = target_rect;
}

static gboolean
webx_view_expose (GtkWidget         *widget,
                  GdkEventExpose    *event,
                  WebxView          *view)
{
  GdkPixbuf      *pixbuf;
  GdkRectangle    bg_rect;
  GdkRectangle    target_rect;
  GdkRectangle    clipbox;
  GdkPixbuf      *src_pixbuf;

  webx_view_get_rect (&bg_rect);
  src_pixbuf = webx_background_get_pixbuf ();
  if (!src_pixbuf && !view->pixbuf)
    {
      /* nothing to draw except outline */
      webx_draw_target_outline (widget, &bg_rect, TRUE);
      return TRUE;
    }

  webx_view_get_target_rect (&target_rect);

  /* Draw background if some cropping has been done or 
   * target is recreated and cannot be drawn. */
  if ( (bg_rect.width != target_rect.width
	|| bg_rect.height != target_rect.height
	|| !view->pixbuf)
       && src_pixbuf
       && gdk_rectangle_intersect (&event->area, &bg_rect, &clipbox) )
    {
      pixbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, FALSE, 8,
			       clipbox.width, clipbox.height);
      gdk_pixbuf_composite_color (src_pixbuf, pixbuf,
				  0, 0, clipbox.width, clipbox.height,
				  bg_rect.x - clipbox.x, bg_rect.y - clipbox.y,
				  webx_dlg.zoom, webx_dlg.zoom,
				  GDK_INTERP_TILES, 255,
				  clipbox.x, clipbox.y,
				  16, 0xaaaaaa, 0x555555);

      gdk_draw_pixbuf (widget->window,
		       widget->style->fg_gc[GTK_STATE_NORMAL],
		       pixbuf,
		       0, 0, clipbox.x, clipbox.y,
		       clipbox.width, clipbox.height,
		       GDK_RGB_DITHER_NORMAL,
		       clipbox.x, clipbox.y);

      g_object_unref (pixbuf);
    }

  /* Draw target pixbuf */
  if (view->pixbuf &&
      gdk_rectangle_intersect (&event->area, &target_rect, &clipbox))
    {
      pixbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, FALSE, 8,
			       clipbox.width, clipbox.height);
      gdk_pixbuf_composite_color (view->pixbuf, pixbuf,
				  0, 0, clipbox.width, clipbox.height,
				  target_rect.x - clipbox.x,
				  target_rect.y - clipbox.y,
				  webx_dlg.zoom, webx_dlg.zoom,
				  GDK_INTERP_TILES, 255,
				  clipbox.x, clipbox.y,
				  16, 0xaaaaaa, 0x555555);

      gdk_draw_pixbuf (widget->window,
		       widget->style->fg_gc[GTK_STATE_NORMAL],
		       pixbuf,
		       0, 0, clipbox.x, clipbox.y,
		       clipbox.width, clipbox.height,
		       GDK_RGB_DITHER_NORMAL,
		       clipbox.x, clipbox.y);

      g_object_unref (pixbuf);

      webx_draw_target_outline (widget, &target_rect, FALSE);
    }
  else
    {
      /* If we are creating new preview, just invert background
       * so that user won't think this is the resulting image. */
      webx_draw_target_outline (widget, &target_rect, TRUE);
    }

    

  return TRUE;
}

/* Determine what do we have under mouse pointer. */
static gint
webx_view_get_drag_type (WebxView       *view,
                         gint            x,
                         gint            y)
{
  GdkRectangle    target_rect;
  gint            drag_type = 0;

  webx_view_get_target_rect (&target_rect);
    
  if (x < target_rect.x - WEBX_HANDLE_SIZE*2
      || y < target_rect.y - WEBX_HANDLE_SIZE*2
      || x > target_rect.x + target_rect.width + WEBX_HANDLE_SIZE*2
      || y > target_rect.y + target_rect.height + WEBX_HANDLE_SIZE*2)
    return WEBX_DRAG_HAND;
        
  if (x < target_rect.x)
    drag_type |= WEBX_DRAG_LEFT;
  else if (x > target_rect.x + target_rect.width)
    drag_type |= WEBX_DRAG_RIGHT;
  if (y < target_rect.y)
    drag_type |= WEBX_DRAG_TOP;
  else if (y > target_rect.y + target_rect.height)
    drag_type |= WEBX_DRAG_BOTTOM;
        
  if (! drag_type)
    return WEBX_DRAG_HAND;

  return drag_type;
}

/* Scroll view. */
static void
webx_view_hand_scroll (WebxView     *view,
                       gdouble       delta_x,
                       gdouble       delta_y)
{
  GtkAdjustment   *hadj;
  GtkAdjustment   *vadj;
  gdouble          new_x, new_y;
  gdouble          max_x, max_y;

  hadj = gtk_scrolled_window_get_hadjustment (GTK_SCROLLED_WINDOW (view->scroll_widget));
  vadj = gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW (view->scroll_widget));

  new_x = hadj->value - delta_x;
  new_y = vadj->value - delta_y;

  max_x = hadj->upper - hadj->page_size;
  max_y = vadj->upper - vadj->page_size;
  if (new_x > max_x)
    new_x = max_x;
  if (new_y > max_y)
    new_y = max_y;

  gtk_adjustment_set_value (GTK_ADJUSTMENT (hadj), new_x);
  gtk_adjustment_set_value (GTK_ADJUSTMENT (vadj), new_y);
}

/* Change crop settings by dragging crop handles. */
static void
webx_crop_handle_drag (WebxView    *view,
                       gint         delta_x,
                       gint         delta_y)
{
  GdkRectangle    crop;
  GdkRectangle    clipbox;

  delta_x /= webx_dlg.zoom;
  delta_y /= webx_dlg.zoom;

  crop = webx_dlg.drag_crop_rect;
   
  if (webx_dlg.drag_mode & WEBX_DRAG_LEFT)
    {
      crop.x += delta_x;
      crop.width -= delta_x;
    }
  else if (webx_dlg.drag_mode & WEBX_DRAG_RIGHT)
    {
      crop.width += delta_x;
    }
  if (webx_dlg.drag_mode & WEBX_DRAG_TOP)
    {
      crop.y += delta_y;
      crop.height -= delta_y;
    }
  else if (webx_dlg.drag_mode & WEBX_DRAG_BOTTOM)
    {
      crop.height += delta_y;
    }

  if (webx_source_crop (crop.width, crop.height,
			crop.x, crop.y, FALSE))
    {
      /* crop modifies original too. */
      g_object_unref (webx_dlg.original.pixbuf);
      webx_dlg.original.pixbuf = NULL;
      webx_dialog_queue_update ();

      clipbox.x = 0;
      clipbox.y = 0;
      clipbox.width = webx_background_get_width () * webx_dlg.zoom
	+ WEBX_VIEW_PADDING * 2;
      clipbox.height = webx_background_get_height () * webx_dlg.zoom
	+ WEBX_VIEW_PADDING * 2;
      if (webx_dlg.original.view_widget->allocation.width > clipbox.width)
	clipbox.width = webx_dlg.original.view_widget->allocation.width;
      if (webx_dlg.original.view_widget->allocation.height > clipbox.height)
	clipbox.height = webx_dlg.original.view_widget->allocation.height;
      gdk_window_invalidate_rect (webx_dlg.original.view_widget->window,
				  &clipbox, FALSE);
      gdk_window_invalidate_rect (webx_dlg.optimized.view_widget->window,
				  &clipbox, FALSE);
    }
}

static void
webx_view_update_cursor (WebxView  *view,
			 gint       x,
			 gint       y)
{
  gint            drag_type;
  GdkCursor      *cursor;
  GdkCursorType   cursor_type;

  drag_type = webx_view_get_drag_type (view, x, y);
  switch (drag_type)
    {
    case WEBX_DRAG_LEFT:
      cursor_type = GDK_LEFT_SIDE;
      break;
    case WEBX_DRAG_RIGHT:
      cursor_type = GDK_RIGHT_SIDE;
      break;
    case WEBX_DRAG_TOP:
      cursor_type = GDK_TOP_SIDE;
      break;
    case WEBX_DRAG_BOTTOM:
      cursor_type = GDK_BOTTOM_SIDE;
      break;
    case (WEBX_DRAG_LEFT|WEBX_DRAG_TOP):
      cursor_type = GDK_TOP_LEFT_CORNER;
      break;
    case (WEBX_DRAG_RIGHT|WEBX_DRAG_TOP):
      cursor_type = GDK_TOP_RIGHT_CORNER;
      break;
    case (WEBX_DRAG_LEFT|WEBX_DRAG_BOTTOM):
      cursor_type = GDK_BOTTOM_LEFT_CORNER;
      break;
    case (WEBX_DRAG_RIGHT|WEBX_DRAG_BOTTOM):
      cursor_type = GDK_BOTTOM_RIGHT_CORNER;
      break;
    default:
      cursor_type = GDK_CROSSHAIR;
      break;
    }

  if (view->cursor_type != cursor_type)
    {
      view->cursor_type = cursor_type;
      cursor = gdk_cursor_new (cursor_type);
      gdk_window_set_cursor (view->view_widget->window, cursor);
      gdk_cursor_unref (cursor);
    }
}

static void
webx_view_button_press (GtkWidget       *widget,
                        GdkEventButton  *event,
                        WebxView        *view)
{
  webx_dlg.drag_view = view;
  webx_dlg.drag_mode = webx_view_get_drag_type (view, event->x, event->y);
  webx_dlg.drag_start_x = event->x_root;
  webx_dlg.drag_start_y = event->y_root;
  webx_source_get_rect (&webx_dlg.drag_crop_rect);
}

static void
webx_view_button_release (GtkWidget       *widget,
                          GdkEventButton  *event,
                          WebxView        *view)
{
  webx_dlg.drag_mode = 0;
}

static void
webx_view_motion_notify (GtkWidget       *widget,
                         GdkEventMotion  *event,
                         WebxView        *view)
{
  gint     x, y;
  gint     x_root, y_root;
  gint     delta_x, delta_y;

  if (event->is_hint)
    {
      gdk_window_get_pointer (event->window, &x, &y, NULL);
      gdk_window_get_origin (event->window, &x_root, &y_root);
      x_root += x;
      y_root += y;
    }
  else
    {
      x = event->x;
      y = event->y;
      x_root = event->x_root;
      y_root = event->y_root;
    }

  delta_x = x_root - webx_dlg.drag_start_x;
  delta_y = y_root - webx_dlg.drag_start_y;

  if (! webx_dlg.drag_mode)
    {
      webx_view_update_cursor (view, x, y);
    }
  else if (webx_dlg.drag_mode & WEBX_DRAG_HAND)
    {
      webx_view_hand_scroll (view, delta_x, delta_y);
      webx_dlg.drag_start_x = x_root;
      webx_dlg.drag_start_y = y_root;
    }
  else
    {
      webx_crop_handle_drag (view, delta_x, delta_y);
    }
}

/* Changes file format. */
static void
webx_dialog_set_format (gint    new_format)
{
  gint i;
  webx_dlg.format = new_format;
  for (i = 0; i < WEBX_MAXFORMATS; i++)
    {
      if (i != new_format)
	gtk_widget_hide (webx_dlg.toolboxes[i]);
    }
  switch (webx_dlg.format)
    {
    case WEBX_FMT_JPEG:
      gtk_widget_show (webx_dlg.toolboxes[WEBX_FMT_JPEG]);
      webx_jpeg_toolbox_update (webx_dlg.toolboxes[WEBX_FMT_JPEG]);
      gtk_label_set_text (GTK_LABEL (webx_dlg.file_fmt_label),
			  _("JPEG"));
      break;
    case WEBX_FMT_PNG8:
      gtk_widget_show (webx_dlg.toolboxes[WEBX_FMT_PNG8]);
      webx_png8_toolbox_update (webx_dlg.toolboxes[WEBX_FMT_PNG8]);
      gtk_label_set_text (GTK_LABEL (webx_dlg.file_fmt_label),
			  _("PNG-8"));
      break;
    case WEBX_FMT_PNG24:
      gtk_widget_show (webx_dlg.toolboxes[WEBX_FMT_PNG24]);
      webx_png24_toolbox_update (webx_dlg.toolboxes[WEBX_FMT_PNG24]);
      gtk_label_set_text (GTK_LABEL (webx_dlg.file_fmt_label),
			  _("PNG-24"));
      break;
    case WEBX_FMT_GIF:
      gtk_widget_show (webx_dlg.toolboxes[WEBX_FMT_GIF]);
      webx_gif_toolbox_update (webx_dlg.toolboxes[WEBX_FMT_GIF]);
      gtk_label_set_text (GTK_LABEL (webx_dlg.file_fmt_label),
			  _("GIF"));
      break;
    default:
      g_assert_not_reached ();
      break;
    }
}

static void
webx_format_combo_changed (GimpIntComboBox  *combo,
                           gpointer          data)
{
  gint    new_format;

  if (gimp_int_combo_box_get_active (combo, &new_format)
      && webx_dlg.format != new_format)
    {
      webx_dialog_set_format (new_format);
      webx_dialog_queue_update ();
    }
}

static GtkWidget*
webx_view_new (WebxView     *view)
{
  GtkWidget       *vbox;
  GtkWidget       *scrollbox;

  view->frame_widget = gtk_vbox_new (FALSE, 0);
  vbox = view->frame_widget;

  scrollbox = gtk_scrolled_window_new (webx_dlg.hadj, webx_dlg.vadj);
  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrollbox),
				       GTK_SHADOW_IN);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrollbox),
                                  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  view->scroll_widget = scrollbox;
  gtk_box_pack_start (GTK_BOX (vbox), scrollbox, TRUE, TRUE, 4);
  view->view_widget = gtk_drawing_area_new ();
  gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (scrollbox),
					 view->view_widget);
  gtk_widget_add_events (view->view_widget,  GDK_POINTER_MOTION_MASK
			 | GDK_POINTER_MOTION_HINT_MASK
			 | GDK_BUTTON_PRESS_MASK
			 | GDK_BUTTON_RELEASE_MASK);
  g_signal_connect (view->view_widget, "expose-event",
		    G_CALLBACK (webx_view_expose), view);
  g_signal_connect (view->view_widget, "button_press_event",
		    G_CALLBACK (webx_view_button_press), view);
  g_signal_connect (view->view_widget, "button_release_event",
		    G_CALLBACK (webx_view_button_release), view);
  g_signal_connect (view->view_widget, "motion_notify_event",
		    G_CALLBACK (webx_view_motion_notify), view);

  return view->frame_widget;
}

static void
webx_zoom_combo_changed (GimpIntComboBox  *combo,
			 gpointer          data)
{
  gint            index;
  gdouble         newzoom;
  gint            width;
  gint            height;
  GdkRectangle    clipbox;
    
  if (! gimp_int_combo_box_get_active (combo, &index))
    return;
        
  if (index < 0 || index >= G_N_ELEMENTS (webx_zoomlevels))
    return;
    
  newzoom = webx_zoomlevels[index];
  if (newzoom == webx_dlg.zoom)
    return;

  webx_dlg.zoom = newzoom;

  width = webx_background_get_width () * webx_dlg.zoom
    + WEBX_VIEW_PADDING * 2;
  height = webx_background_get_height () * webx_dlg.zoom
    + WEBX_VIEW_PADDING * 2;
  gtk_widget_set_size_request (webx_dlg.original.view_widget,
			       width, height);
  gtk_widget_set_size_request (webx_dlg.optimized.view_widget,
			       width, height);
  clipbox.x = 0;
  clipbox.y = 0;
  clipbox.width = width;
  clipbox.height = height;
  gdk_window_invalidate_rect (webx_dlg.original.view_widget->window,
			      &clipbox, FALSE);
  gdk_window_invalidate_rect (webx_dlg.optimized.view_widget->window,
			      &clipbox, FALSE);
}

static void
webx_resize_width_changed (GtkSpinButton *spinbutton,
                           gpointer       user_data)
{
  gint            width;
  gint            height;
  gboolean        chain;
  GdkRectangle    clipbox;
    
  if (webx_dlg.stop_recursion > 0)
    return;

  webx_dlg.stop_recursion++;

  width = gtk_spin_button_get_value (GTK_SPIN_BUTTON (webx_dlg.resize_width));
  height = gtk_spin_button_get_value (GTK_SPIN_BUTTON (webx_dlg.resize_height));
  chain = gimp_chain_button_get_active (GIMP_CHAIN_BUTTON (webx_dlg.resize_chain));
  if (chain)
    {
      gint    scaled_height;
      scaled_height = (gdouble)width / webx_dlg.resize_aspect;
      if (scaled_height != height)
        {
	  gtk_spin_button_set_value (GTK_SPIN_BUTTON (webx_dlg.resize_height),
				     scaled_height);
	  height = gtk_spin_button_get_value (GTK_SPIN_BUTTON (webx_dlg.resize_height));
        }
    }

  if (webx_source_resize (width, height))
    {
      g_object_unref (webx_dlg.original.pixbuf);
      webx_dlg.original.pixbuf = NULL;
      webx_dialog_queue_update ();

      clipbox.x = 0;
      clipbox.y = 0;
      clipbox.width = webx_background_get_width () * webx_dlg.zoom
	+ WEBX_VIEW_PADDING * 2;
      clipbox.height = webx_background_get_height () * webx_dlg.zoom
	+ WEBX_VIEW_PADDING * 2;
      if (webx_dlg.original.view_widget->allocation.width > clipbox.width)
	clipbox.width = webx_dlg.original.view_widget->allocation.width;
      if (webx_dlg.original.view_widget->allocation.height > clipbox.height)
	clipbox.height = webx_dlg.original.view_widget->allocation.height;
      gdk_window_invalidate_rect (webx_dlg.original.view_widget->window,
				  &clipbox, FALSE);
      gdk_window_invalidate_rect (webx_dlg.optimized.view_widget->window,
				  &clipbox, FALSE);
    }

  webx_dlg.stop_recursion--;
}

static void
webx_resize_height_changed (GtkSpinButton *spinbutton,
                            gpointer       user_data)
{
  gint            width;
  gint            height;
  gboolean        chain;
  GdkRectangle    clipbox;
    
  if (webx_dlg.stop_recursion > 0)
    return;

  webx_dlg.stop_recursion++;

  width = gtk_spin_button_get_value (GTK_SPIN_BUTTON (webx_dlg.resize_width));
  height = gtk_spin_button_get_value (GTK_SPIN_BUTTON (webx_dlg.resize_height));
  chain = gimp_chain_button_get_active (GIMP_CHAIN_BUTTON (webx_dlg.resize_chain));
  if (chain)
    {
      gint    scaled_width;
      scaled_width = (gdouble)height * webx_dlg.resize_aspect;
      if (scaled_width != width)
        {
	  gtk_spin_button_set_value (GTK_SPIN_BUTTON (webx_dlg.resize_width),
				     scaled_width);
	  width = gtk_spin_button_get_value (GTK_SPIN_BUTTON (webx_dlg.resize_width));
        }
    }

  if (webx_source_resize (width, height))
    {
      g_object_unref (webx_dlg.original.pixbuf);
      webx_dlg.original.pixbuf = NULL;
      webx_dialog_queue_update ();
        
      clipbox.x = 0;
      clipbox.y = 0;
      clipbox.width = webx_background_get_width () * webx_dlg.zoom
	+ WEBX_VIEW_PADDING * 2;
      clipbox.height = webx_background_get_height () * webx_dlg.zoom
	+ WEBX_VIEW_PADDING * 2;
      if (webx_dlg.original.view_widget->allocation.width > clipbox.width)
	clipbox.width = webx_dlg.original.view_widget->allocation.width;
      if (webx_dlg.original.view_widget->allocation.height > clipbox.height)
	clipbox.height = webx_dlg.original.view_widget->allocation.height;
      gdk_window_invalidate_rect (webx_dlg.original.view_widget->window,
				  &clipbox, FALSE);
      gdk_window_invalidate_rect (webx_dlg.optimized.view_widget->window,
				  &clipbox, FALSE);
    }

  webx_dlg.stop_recursion--;
}

static void
webx_crop_value_changed (GtkSpinButton *spinbutton,
                         gpointer       user_data)
{
  gint            offsx;
  gint            offsy;
  gint            width;
  gint            height;
  GdkRectangle    rect;
  gboolean        clip_offs = FALSE;
    
  GdkRectangle    clipbox;

  if (webx_dlg.stop_recursion > 0)
    return;

  webx_dlg.stop_recursion++;
    
  offsx = gtk_spin_button_get_value (GTK_SPIN_BUTTON (webx_dlg.crop_offsx));
  offsy = gtk_spin_button_get_value (GTK_SPIN_BUTTON (webx_dlg.crop_offsy));
  width = gtk_spin_button_get_value (GTK_SPIN_BUTTON (webx_dlg.crop_width));
  height = gtk_spin_button_get_value (GTK_SPIN_BUTTON (webx_dlg.crop_height));
    
  if (user_data)
    clip_offs = TRUE;
   
  if (webx_source_crop (width, height, offsx, offsy, clip_offs))
    {
      g_object_unref (webx_dlg.original.pixbuf);
      webx_dlg.original.pixbuf = NULL;
      webx_dialog_queue_update ();

      clipbox.x = 0;
      clipbox.y = 0;
      clipbox.width = webx_background_get_width () * webx_dlg.zoom
	+ WEBX_VIEW_PADDING * 2;
      clipbox.height = webx_background_get_height () * webx_dlg.zoom
	+ WEBX_VIEW_PADDING * 2;
      if (webx_dlg.original.view_widget->allocation.width > clipbox.width)
	clipbox.width = webx_dlg.original.view_widget->allocation.width;
      if (webx_dlg.original.view_widget->allocation.height > clipbox.height)
	clipbox.height = webx_dlg.original.view_widget->allocation.height;
      gdk_window_invalidate_rect (webx_dlg.original.view_widget->window,
				  &clipbox, FALSE);
      gdk_window_invalidate_rect (webx_dlg.optimized.view_widget->window,
				  &clipbox, FALSE);
    }

  webx_source_get_rect (&rect);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (webx_dlg.crop_offsx), rect.x);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (webx_dlg.crop_offsy), rect.y);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (webx_dlg.crop_width), rect.width);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (webx_dlg.crop_height), rect.height);
        
  webx_dlg.stop_recursion--;
}

static gboolean
webx_save_to_file (const gchar  *file_name)
{
  gboolean    saved;

  switch (webx_dlg.format)
    {
    case WEBX_FMT_JPEG:
      saved = webx_jpeg_save (file_name);
      break;
    case WEBX_FMT_PNG8:
      saved = webx_png8_save (file_name);
      break;
    case WEBX_FMT_PNG24:
      saved = webx_png24_save (file_name);
      break;
    case WEBX_FMT_GIF:
      saved = webx_gif_save (file_name);
      break;
    default:
      g_assert_not_reached ();
    }
    
  return saved;
}

static gboolean
webx_save_dialog (void)
{
  GtkWidget  *dialog;
  gchar       file_name[1024];
  gchar      *ext;
  gboolean    saved = FALSE;

  dialog = gtk_file_chooser_dialog_new (_("Save Image"),
					GTK_WINDOW (webx_dlg.dialog),
					GTK_FILE_CHOOSER_ACTION_SAVE,
					GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
					NULL);
  gtk_file_chooser_set_do_overwrite_confirmation (GTK_FILE_CHOOSER (dialog),
						  TRUE);

  switch (webx_dlg.format)
    {
    case WEBX_FMT_JPEG:
      ext = "jpg";
      break;
    case WEBX_FMT_PNG8:
    case WEBX_FMT_PNG24:
      ext = "png";
      break;
    case WEBX_FMT_GIF:
      ext = "gif";
      break;
    default:
      g_assert_not_reached ();
    }
  g_snprintf (file_name, sizeof (file_name),
	      "%s.%s",
	      gimp_image_get_name (webx_source_get_rgb (NULL)), ext);
  gtk_file_chooser_set_current_name (GTK_FILE_CHOOSER (dialog), file_name);

  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
    {
      gchar *filename;

      filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
      saved = webx_save_to_file (filename);
      if (! saved)
	g_message (_("Failed to save the file!")); 
      g_free (filename);
    }

  gtk_widget_destroy (dialog);
  return saved;
}

static void
webx_dialog_response (GtkWidget *widget,
                      gint       response_id,
                      gpointer   data)
{
  switch (response_id)
    {
    case GTK_RESPONSE_OK:
      if (webx_save_dialog ())
	gtk_widget_destroy (GTK_WIDGET (widget));
      break;

    default:
      gtk_widget_destroy (GTK_WIDGET (widget));
      break;
    }
}

void
webx_dialog_init (void)
{
  GtkWidget    *dialog_vbox;
  GtkWidget    *main_hpaned;
  GtkWidget    *main_vbox;
  GtkWidget    *options_vbox;
  GtkWidget    *resize_expander;
  GtkWidget    *crop_expander;
  GtkWidget    *vbox;
  GtkWidget    *table;
  GtkWidget    *label;
  GtkWidget    *scrolledwnd;
  GtkWidget    *statusbar;
  GtkWidget    *view;
  GtkWidget    *spinbtn;
  GtkWidget    *chainbtn;
  const gchar  *zoom_labels[G_N_ELEMENTS (webx_zoomlevels)];

  memset (&webx_dlg, 0, sizeof(webx_dlg));
  webx_dlg.zoom = 1.0;

  webx_dlg.dialog = gimp_dialog_new (_("Save for Web"), PLUG_IN_BINARY,
				     NULL, 0,
				     gimp_standard_help_func, PLUG_IN_PROC,

				     GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				     GTK_STOCK_SAVE,   GTK_RESPONSE_OK,

				     NULL);

  gtk_dialog_set_alternative_button_order (GTK_DIALOG (webx_dlg.dialog),
					   GTK_RESPONSE_OK,
					   GTK_RESPONSE_CANCEL,
					   -1);

  gimp_window_set_transient (GTK_WINDOW (webx_dlg.dialog));

  gtk_window_set_default_size (GTK_WINDOW(webx_dlg.dialog), 800, 600);

  g_signal_connect (webx_dlg.dialog, "response",
                    G_CALLBACK (webx_dialog_response), NULL);
  g_signal_connect (webx_dlg.dialog, "destroy",
		    G_CALLBACK (gtk_main_quit), NULL);

  dialog_vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (webx_dlg.dialog)->vbox),
		     dialog_vbox);

  main_hpaned = gtk_hpaned_new ();
  gtk_box_pack_start (GTK_BOX (dialog_vbox), main_hpaned, TRUE, TRUE, 0);

  /*
   * TOOLBOX
   */

  scrolledwnd = gtk_scrolled_window_new (NULL, NULL);
  gtk_paned_add1 (GTK_PANED (main_hpaned), scrolledwnd);
  gtk_widget_set_size_request (scrolledwnd, 220, -1);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwnd),
                                  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

  options_vbox = gtk_vbox_new (FALSE, 0);
  gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (scrolledwnd),
                                         options_vbox);
  vbox = gtk_vbox_new (FALSE, 4);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), 8);

  gtk_box_pack_start (GTK_BOX (options_vbox), vbox, FALSE, FALSE, 0);

  webx_dlg.format_combo = gimp_int_combo_box_new (_("JPEG"), WEBX_FMT_JPEG,
						  _("PNG-8"), WEBX_FMT_PNG8,
						  _("PNG-24"), WEBX_FMT_PNG24,
						  _("GIF"), WEBX_FMT_GIF,
						  NULL);

  gtk_box_pack_start (GTK_BOX (vbox), webx_dlg.format_combo,
		      FALSE, FALSE, 4);

  webx_dlg.toolboxes[WEBX_FMT_JPEG] = webx_jpeg_toolbox_new ();
  gtk_box_pack_start (GTK_BOX (vbox), webx_dlg.toolboxes[WEBX_FMT_JPEG],
		      FALSE, FALSE, 4);
  webx_dlg.toolboxes[WEBX_FMT_PNG8] = webx_png8_toolbox_new ();
  gtk_box_pack_start (GTK_BOX (vbox), webx_dlg.toolboxes[WEBX_FMT_PNG8],
		      FALSE, FALSE, 4);
  webx_dlg.toolboxes[WEBX_FMT_PNG24] = webx_png24_toolbox_new ();
  gtk_box_pack_start (GTK_BOX (vbox), webx_dlg.toolboxes[WEBX_FMT_PNG24],
		      FALSE, FALSE, 4);
  webx_dlg.toolboxes[WEBX_FMT_GIF] = webx_gif_toolbox_new ();
  gtk_box_pack_start (GTK_BOX (vbox), webx_dlg.toolboxes[WEBX_FMT_GIF],
		      FALSE, FALSE, 4);

  /* RESIZE */

  resize_expander = gtk_expander_new (_("Resize"));
  gtk_box_pack_start (GTK_BOX (options_vbox), resize_expander, FALSE, TRUE, 0);
  
  webx_dlg.resize_aspect = (gdouble) webx_background_get_width () /
    webx_background_get_height ();
  table = gtk_table_new (2, 3, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 8);
  gtk_container_add (GTK_CONTAINER (resize_expander), table);
  label = gtk_label_new (_("Width:"));
  gtk_table_attach (GTK_TABLE (table), label,
		    0, 1, 0, 1,
		    0, 0, 0, 0);
  spinbtn = gtk_spin_button_new_with_range (1, WEBX_MAX_DIMENSION, 1);
  webx_dlg.resize_width = spinbtn;
  gtk_table_attach (GTK_TABLE (table), spinbtn,
		    1, 2, 0, 1,
		    0, 0, 0, 0);
  g_signal_connect (spinbtn, "value-changed",
                    G_CALLBACK (webx_resize_width_changed), NULL);
  label = gtk_label_new (_("Height:"));
  gtk_table_attach (GTK_TABLE (table), label,
		    0, 1, 1, 2,
		    GTK_SHRINK, GTK_SHRINK, 0, 0);
  spinbtn = gtk_spin_button_new_with_range (1, WEBX_MAX_DIMENSION, 1);
  webx_dlg.resize_height = spinbtn;
  gtk_table_attach (GTK_TABLE (table), spinbtn,
		    1, 2, 1, 2,
		    0, 0, 0, 0);
  g_signal_connect (spinbtn, "value-changed",
                    G_CALLBACK (webx_resize_height_changed), NULL);
  chainbtn = gimp_chain_button_new (GIMP_CHAIN_RIGHT);
  webx_dlg.resize_chain = chainbtn;
  gtk_table_attach (GTK_TABLE (table), chainbtn,
		    2, 3, 0, 2,
		    0, 0, 0, 0);
  gimp_chain_button_set_active (GIMP_CHAIN_BUTTON (chainbtn), TRUE);

  /* CROP */

  crop_expander = gtk_expander_new (_("Crop"));
  gtk_box_pack_start (GTK_BOX (options_vbox), crop_expander, FALSE, TRUE, 0);
  
  table = gtk_table_new (2, 5, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 8);
  gtk_container_add (GTK_CONTAINER (crop_expander), table);
  
  label = gtk_label_new (_("X Offset:"));
  gtk_table_attach (GTK_TABLE (table), label,
		    0, 1, 0, 1,
		    GTK_SHRINK, GTK_SHRINK, 0, 0);
  spinbtn = gtk_spin_button_new_with_range (0, WEBX_MAX_DIMENSION-1, 1);
  webx_dlg.crop_offsx = spinbtn;
  gtk_table_attach (GTK_TABLE (table), spinbtn,
		    1, 2, 0, 1,
		    0, 0, 0, 0);
  g_signal_connect (spinbtn, "value-changed",
                    G_CALLBACK (webx_crop_value_changed),
                    NULL);
  label = gtk_label_new (_("Y Offset:"));
  gtk_table_attach (GTK_TABLE (table), label,
		    0, 1, 1, 2,
		    GTK_SHRINK, GTK_SHRINK, 0, 0);
  spinbtn = gtk_spin_button_new_with_range (0, WEBX_MAX_DIMENSION-1, 1);
  webx_dlg.crop_offsy = spinbtn;
  gtk_table_attach (GTK_TABLE (table), spinbtn,
		    1, 2, 1, 2,
		    0, 0, 0, 0);
  g_signal_connect (spinbtn, "value-changed",
                    G_CALLBACK (webx_crop_value_changed),
                    NULL);
  label = gtk_label_new (_("Width:"));
  gtk_table_attach (GTK_TABLE (table), label,
		    0, 1, 2, 3,
		    GTK_SHRINK, GTK_SHRINK, 0, 0);
  spinbtn = gtk_spin_button_new_with_range (1, WEBX_MAX_DIMENSION, 1);
  webx_dlg.crop_width = spinbtn;
  gtk_table_attach (GTK_TABLE (table), spinbtn,
		    1, 2, 2, 3,
		    0, 0, 0, 0);
  g_signal_connect (spinbtn, "value-changed",
                    G_CALLBACK (webx_crop_value_changed),
                    &webx_dlg.crop_rect.width);
  label = gtk_label_new (_("Height:"));
  gtk_table_attach (GTK_TABLE (table), label,
		    0, 1, 3, 4,
		    GTK_SHRINK, GTK_SHRINK, 0, 0);
  spinbtn = gtk_spin_button_new_with_range (1, WEBX_MAX_DIMENSION, 1);
  webx_dlg.crop_height = spinbtn;
  gtk_table_attach (GTK_TABLE (table), spinbtn,
		    1, 2, 3, 4,
		    0, 0, 0, 0);
  g_signal_connect (spinbtn, "value-changed",
                    G_CALLBACK (webx_crop_value_changed),
                    &webx_dlg.crop_rect.height);
  
  /*
   * CREATE VIEWS
   */

  webx_dlg.hadj = GTK_ADJUSTMENT (gtk_adjustment_new (0.0, 0.0, 100.0,
						      10.0, 10.0, 10.0));
  webx_dlg.vadj = GTK_ADJUSTMENT (gtk_adjustment_new (0.0, 0.0, 100.0,
						      10.0, 10.0, 10.0));

  main_vbox = gtk_vbox_new (FALSE, 0);
  gtk_paned_add2 (GTK_PANED (main_hpaned), main_vbox);
  if ((gdouble)webx_background_get_width ()/webx_background_get_height ()
      > 1024.0 / 768.0)
    vbox = gtk_vbox_new (TRUE, 0);
  else
    vbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (main_vbox), vbox, TRUE, TRUE, 0);
  view = webx_view_new (&webx_dlg.original);
  gtk_box_pack_start (GTK_BOX (vbox), view, TRUE, TRUE, 0);
  view = webx_view_new (&webx_dlg.optimized);
  gtk_box_pack_start (GTK_BOX (vbox), view, TRUE, TRUE, 0);
  
  statusbar = gtk_hbox_new (FALSE, 0);
  gtk_container_set_border_width (GTK_CONTAINER (statusbar), 8);
  gtk_box_pack_start (GTK_BOX (main_vbox), statusbar, FALSE, FALSE, 0);
  zoom_labels[0] = "800%";
  zoom_labels[1] = "400%";
  zoom_labels[2] = "200%";
  zoom_labels[3] = "150%";
  zoom_labels[4] = "100%";
  zoom_labels[5] = "50%";
  zoom_labels[6] = "25%";
  zoom_labels[7] = "13%";
  webx_dlg.zoom_combo =
    gimp_int_combo_box_new_array (G_N_ELEMENTS (zoom_labels),
				  zoom_labels);
  gtk_box_pack_end (GTK_BOX (statusbar), webx_dlg.zoom_combo, FALSE, FALSE, 0);
  webx_dlg.file_fmt_label = gtk_label_new ("");
  gtk_box_pack_start (GTK_BOX (statusbar), webx_dlg.file_fmt_label,
		      FALSE, FALSE, 8);
  webx_dlg.dimensions_label = gtk_label_new ("");
  gtk_box_pack_start (GTK_BOX (statusbar), webx_dlg.dimensions_label,
		      FALSE, FALSE, 8);
  webx_dlg.file_size_label = gtk_label_new ("");
  gtk_box_pack_start (GTK_BOX (statusbar), webx_dlg.file_size_label,
		      FALSE, FALSE, 8);

  gtk_widget_show_all (dialog_vbox);

  /* Shut up unnecessary messages and progress of PDB calls on
   * temporary images. */
  webx_dlg.progress_bar = gimp_progress_bar_new ();
  gtk_box_pack_start (GTK_BOX (options_vbox), webx_dlg.progress_bar,
		      FALSE, FALSE, 8);
}

static void
webx_dialog_invalidate_all (void)
{
  webx_jpeg_invalidate ();
  webx_png8_invalidate ();
  webx_png24_invalidate ();
  webx_gif_invalidate ();
}

static void
webx_dialog_optimized_update ()
{
  GdkRectangle    clipbox;

  if (webx_dlg.optimized.pixbuf)
    g_object_unref (webx_dlg.optimized.pixbuf);
  switch (webx_dlg.format)
    {
    case WEBX_FMT_JPEG:
      webx_dlg.optimized.pixbuf = webx_jpeg_generate_preview ();
      break;
    case WEBX_FMT_PNG8:
      webx_dlg.optimized.pixbuf = webx_png8_generate_preview ();
      break;
    case WEBX_FMT_PNG24:
      webx_dlg.optimized.pixbuf = webx_png24_generate_preview ();
      break;
    case WEBX_FMT_GIF:
      webx_dlg.optimized.pixbuf = webx_gif_generate_preview ();
      break;
    default:
      g_assert_not_reached ();
      break;
    }
    
  webx_view_get_target_rect (&clipbox);
  gdk_window_invalidate_rect (webx_dlg.optimized.view_widget->window,
			      &clipbox, FALSE);
}

static void
webx_dialog_views_update ()
{
  gint32          layer;
  GdkRectangle    clipbox;

  if (webx_dlg.original.pixbuf)
    g_object_unref (webx_dlg.original.pixbuf);

  webx_source_get_rgb (&layer);
  webx_dlg.original.pixbuf = webx_drawable_to_pixbuf (layer);
  webx_dialog_optimized_update ();

  clipbox.x = 0;
  clipbox.y = 0;
  clipbox.width = webx_background_get_width () * webx_dlg.zoom
    + WEBX_VIEW_PADDING * 2;
  clipbox.height = webx_background_get_height () * webx_dlg.zoom
    + WEBX_VIEW_PADDING * 2;
  gtk_widget_set_size_request (webx_dlg.original.view_widget,
			       clipbox.width, clipbox.height);
  gtk_widget_set_size_request (webx_dlg.optimized.view_widget,
			       clipbox.width, clipbox.height);
        
  if (webx_dlg.original.view_widget->allocation.width > clipbox.width)
    clipbox.width = webx_dlg.original.view_widget->allocation.width;
  if (webx_dlg.original.view_widget->allocation.height > clipbox.height)
    clipbox.height = webx_dlg.original.view_widget->allocation.height;
  gdk_window_invalidate_rect (webx_dlg.original.view_widget->window,
			      &clipbox, FALSE);
  gdk_window_invalidate_rect (webx_dlg.optimized.view_widget->window,
			      &clipbox, FALSE);
}

static void
webx_dialog_resize_crop_update ()
{
  gchar           temp[256];
  GdkRectangle    rect;
  gint            bg_width, bg_height;
    
  webx_dlg.stop_recursion++;
    
  webx_source_get_rect (&rect);

  bg_width = webx_background_get_width ();
  bg_height = webx_background_get_height ();
  webx_dlg.resize_aspect = (gdouble) bg_width / bg_height;
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (webx_dlg.resize_width),
			     bg_width);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (webx_dlg.resize_height),
			     bg_height);
                                
  webx_source_get_rect (&rect);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (webx_dlg.crop_offsx), rect.x);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (webx_dlg.crop_offsy), rect.y);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (webx_dlg.crop_width), rect.width);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON (webx_dlg.crop_height), rect.height);
    
  webx_dlg.stop_recursion--;

  g_snprintf (temp, sizeof (temp), "%d x %d", rect.width, rect.height);
  gtk_label_set_text (GTK_LABEL (webx_dlg.dimensions_label), temp);
}

static void
webx_dialog_update ()
{
  if (webx_source_update ())
    {
      /* resize/crop changed. update everything */
      webx_dialog_resize_crop_update ();
      webx_dialog_invalidate_all ();
      webx_dialog_views_update ();
    }
  else
    {
      /* update optimized image */
      webx_dialog_optimized_update ();
    }

  if (! webx_dlg.optimized.pixbuf)
    g_message (_("Failed to create preview!"));
}

static gboolean
webx_timeout_check_updates (gpointer data)
{
  static gint     last_update = 0;
    
  if (webx_dlg.update_count)
    {
      if (webx_dlg.update_count != last_update)
        {
	  /* user has changed something; wait */
	  last_update = webx_dlg.update_count;
        }
      else
        {
	  webx_dialog_update ();
	  webx_dlg.update_count = 0;
	  last_update = 0;
        }
    }
  return TRUE;
}

void
webx_dialog_queue_update (void)
{
  GdkRectangle    clipbox;

  if (webx_dlg.optimized.pixbuf)
    {
      g_object_unref (webx_dlg.optimized.pixbuf);
      webx_dlg.optimized.pixbuf = NULL;

      gtk_label_set_text (GTK_LABEL (webx_dlg.file_size_label),
			  _("File size: calculating..."));

      webx_view_get_rect (&clipbox);
      gdk_window_invalidate_rect (webx_dlg.optimized.view_widget->window,
				  &clipbox, FALSE);
    }

  webx_dlg.update_count++;
}

static void
webx_initial_update ()
{
  webx_dialog_set_format (webx_dlg.format);
  gimp_int_combo_box_connect (GIMP_INT_COMBO_BOX (webx_dlg.format_combo),
			      webx_dlg.format,
			      G_CALLBACK (webx_format_combo_changed),
			      NULL);
  gimp_int_combo_box_connect (GIMP_INT_COMBO_BOX (webx_dlg.zoom_combo),
			      WEBX_DEFAULT_ZOOMLEVEL,
			      G_CALLBACK (webx_zoom_combo_changed),
			      NULL);

  g_timeout_add (250, webx_timeout_check_updates, NULL);
}

void
webx_dialog_cleanup ()
{
  if (webx_dlg.original.pixbuf)
    {
      g_object_unref (webx_dlg.original.pixbuf);
      webx_dlg.original.pixbuf = NULL;
    }
  if (webx_dlg.optimized.pixbuf)
    {
      g_object_unref (webx_dlg.optimized.pixbuf);
      webx_dlg.optimized.pixbuf = NULL;
    }
}

gboolean
webx_dialog_run ()
{
  webx_jpeg_init ();
  webx_png8_init ();
  webx_png24_init ();
  webx_gif_init ();
  webx_dialog_init ();

  /* this is a place where settings could be loaded */

  webx_initial_update ();
  webx_dialog_resize_crop_update ();
  gtk_widget_show (webx_dlg.dialog);
  webx_dialog_views_update ();

  gtk_main ();

  webx_dialog_cleanup ();
  webx_jpeg_cleanup ();
  webx_png8_cleanup ();
  webx_png24_cleanup ();
  webx_gif_cleanup ();
  return TRUE;
}

glong
webx_dialog_update_file_size (const gchar *file_name)
{
  FILE    *fp;
  glong    file_size;
  gchar    temp[256];

  fp = g_fopen (file_name, "rb");
  if (fp)
    {
      fseek (fp, 0, SEEK_END);
      file_size = ftell (fp);
      fclose (fp);
    }
  else
    {
      file_size = 0;
    }
    
  g_snprintf (temp, sizeof (temp),
	      _("File size: %02.01f kB"),
	      (gdouble)file_size / 1024.0);
  gtk_label_set_text (GTK_LABEL (webx_dlg.file_size_label), temp);

  return file_size;
}
