#include "TagData.h"

#include <id3v2tag.h>
#include <id3v2frame.h>
#include <id3v2header.h>
#include <id3v1tag.h>
#include <attachedpictureframe.h>
#include <textidentificationframe.h>
#include <relativevolumeframe.h>
#include <audioproperties.h>
#include <apetag.h>
#include <xiphcomment.h>
#include <vorbisfile.h>
#include <mpegfile.h>
#include <flacfile.h>
#include <mpcfile.h>
#include <fileref.h>
#include <tag.h>

#include <string.h>
#include <stdio.h>
#include <sys/stat.h>


struct _TagData
{
   TagLib::FileRef          * file;
   TagLib::ID3v2::Tag       * id3v2;
   TagLib::ID3v1::Tag       * id3v1;
   TagLib::APE::Tag         * ape;
   TagLib::Ogg::XiphComment * xiph;
   char                     * mime;
   int                        mtime;
};


/* LOAD UP THE FILE'S TAG
 * Search each file type individually so we can get individual tags for
 * custom tagging.
 */
extern "C" TagData * tag_data_load (char *url)
{
   TagData * data;
   TagLib::String s = url;
   int mtime;
   struct stat buf;
   if (!stat (url, &buf))
      mtime = (int) buf.st_mtime;

   
   /* FIXME: Using filename to find media type. GStreamer probe instead?
    */
   if(s.size() > 4) {
      if(s.substr(s.size() - 4, 4).upper() == ".OGG")
      {
         TagLib::Vorbis::File * f = new TagLib::Vorbis::File(url);
         if (! f->isValid ())
            return NULL;
         data = (TagData*) malloc (sizeof (TagData));
         data->file = new TagLib::FileRef (f);
         data->id3v2 = NULL;
         data->id3v1 = NULL;
         data->ape   = NULL;
         data->xiph  = f->tag ();
         data->mime  = "application/ogg";
         data->mtime = mtime;
         return data;
      }
      if(s.substr(s.size() - 4, 4).upper() == ".MP3")
      {
         TagLib::MPEG::File * f = new TagLib::MPEG::File(url);
         if (! f->isValid ())
            return NULL;
         data = (TagData*) malloc (sizeof (TagData));
         data->file = new TagLib::FileRef (f);
         data->id3v2 = f->ID3v2Tag ();
         data->id3v1 = f->ID3v1Tag ();
         data->ape   = f->APETag ();
         data->xiph  = NULL;
         data->mime  = "audio/mpeg";
         data->mtime = mtime;
         return data;
      }
      if(s.substr(s.size() - 5, 5).upper() == ".FLAC")
      {
         TagLib::FLAC::File * f = new TagLib::FLAC::File(url);
         if (! f->isValid ())
            return NULL;
         data = (TagData*) malloc (sizeof (TagData));
         data->file = new TagLib::FileRef (f);
         data->id3v2 = f->ID3v2Tag ();
         data->id3v1 = f->ID3v1Tag ();
         data->ape   = NULL;
         data->xiph  = f->xiphComment ();
         data->mime  = "audio/x-flac";
         data->mtime = mtime;
         return data;
      }
      if(s.substr(s.size() - 4, 4).upper() == ".MPC")
      {
         TagLib::MPC::File * f = new TagLib::MPC::File(url);
         if (! f->isValid ())
            return NULL;
         data = (TagData*) malloc (sizeof (TagData));
         data->file = new TagLib::FileRef (f);
         data->id3v2 = NULL;
         data->id3v1 = f->ID3v1Tag ();
         data->ape   = f->APETag ();
         data->xiph  = NULL;
         data->mime  = "audio/x-musepack";
         data->mtime = mtime;
         return data;
      }
   }
   return NULL;
}


/* SAVE CHANGES TO TAG
 */
extern "C" bool tag_data_save (TagData * data)
{
   return data->file->save ();
}


/* FREE CHANGES
 * FIXME: Is all memory freed, check for leaks.
 */
extern "C" void tag_data_free (TagData * data)
{
   delete data->file;
   free (data);
}


/* GET/SET SONG TITLE
 */
extern "C" const char * tag_data_get_title (TagData * data)
{
   return ::strdup(data->file->tag ()->title ().toCString (1));
}
extern "C" void tag_data_set_title (TagData * data, char * val)
{
   data->file->tag ()->setTitle (TagLib::String(val, TagLib::String::UTF8));
}


/* GET/SET ARTIST
 */
extern "C" const char * tag_data_get_artist (TagData * data)
{
   return ::strdup(data->file->tag ()->artist ().toCString (1));
}
extern "C" void tag_data_set_artist (TagData * data, char * val)
{
   data->file->tag ()->setArtist (TagLib::String(val, TagLib::String::UTF8));
}


/* GET/SET ALBUM
 */
extern "C" const char * tag_data_get_album (TagData * data)
{
   return ::strdup(data->file->tag ()->album ().toCString (1));
}
extern "C" void tag_data_set_album (TagData * data, char * val)
{
   data->file->tag ()->setAlbum (TagLib::String(val, TagLib::String::UTF8));
}


/* GET/SET COMMENT
 */
extern "C" const char * tag_data_get_comment (TagData * data)
{
   return ::strdup(data->file->tag ()->comment ().toCString (1));
}
extern "C" void tag_data_set_comment (TagData * data, char * val)
{
   data->file->tag ()->setComment (TagLib::String(val, TagLib::String::UTF8));
}


/* GET/SET GENRE
 */
extern "C" const char * tag_data_get_genre (TagData * data)
{
   return ::strdup(data->file->tag ()->genre ().toCString (1));
}
extern "C" void tag_data_set_genre (TagData * data, char * val)
{
   data->file->tag ()->setGenre (TagLib::String(val, TagLib::String::UTF8));
}


/* GET/SET YEAR
 */
extern "C" uint tag_data_get_year (TagData * data)
{
   return data->file->tag ()->year ();
}
extern "C" void tag_data_set_year (TagData * data, uint val)
{
   data->file->tag ()->setYear (val);
}


/* GET/SET SONG TRACK NUMBER/TOTAL TRACKS
 */
extern "C" uint tag_data_get_track (TagData * data)
{
   return data->file->tag ()->track ();
}
extern "C" uint tag_data_get_track_count (TagData * data)
{
   if (data->xiph && !data->xiph->fieldListMap()["TRACKTOTAL"].isEmpty())
      return data->xiph->fieldListMap()["TRACKTOTAL"].front().toInt();
   if (data->ape && !data->ape->itemListMap()["TRACKTOTAL"].isEmpty())
      return data->ape->itemListMap()["TRACKTOTAL"].toString().toInt();
   if (data->id3v2)
   {
      int i;
      if (data->id3v2->frameListMap()["TRCK"].isEmpty())
         return 0;
      i = data->id3v2->frameListMap()["TRCK"].front()->toString().find ("/");
      if (i == -1)
         return 0;
      return data->id3v2->frameListMap()["TRCK"].front()->toString().substr(i+1).toInt();
   }
   return 0;
}
extern "C" void tag_data_set_track (TagData * data, uint track, uint track_count)
{
   if (data->xiph)
   {
      data->xiph->setTrack (track);
      if(track_count == 0)
         data->xiph->removeField("TRACKTOTAL");
      else
         data->xiph->addField("TRACKTOTAL", TagLib::String::number(track_count));
   }
   if (data->ape)
   {
      data->ape->setTrack (track);
      if(track_count == 0)
         data->ape->removeItem("TRACKTOTAL");
      else
         data->ape->addValue("TRACKTOTAL", TagLib::String::number(track_count), true);
   }
   if (data->id3v2)
   {
      if (track == 0 || track_count == 0)
         data->id3v2->setTrack(track);
      else
      {
         TagLib::ID3v2::FrameList l = data->id3v2->frameListMap()["TRCK"];
         if(!l.isEmpty())
            l.front()->setText(TagLib::String::number(track) + "/" + TagLib::String::number(track_count));
         else
         {
            TagLib::ID3v2::TextIdentificationFrame *f = new TagLib::ID3v2::TextIdentificationFrame("TRCK", TagLib::String::UTF8);
            data->id3v2->addFrame (f);
            f->setText(TagLib::String::number(track) + "/" + TagLib::String::number(track_count));
         }
      }
   }
   if (data->id3v1)
      data->id3v1->setTrack(track);
}


/* GET/SET PERFORMERS
 */
extern "C" const char * tag_data_get_performers (TagData * data)
{
   if (data->xiph && !data->xiph->fieldListMap()["PERFORMERS"].isEmpty())
      return ::strdup(data->xiph->fieldListMap()["PERFORMERS"].front().toCString(1));
   if (data->ape && !data->ape->itemListMap()["PERFORMERS"].isEmpty())
      return ::strdup(data->ape->itemListMap()["PERFORMERS"].toString().toCString(1));
   if (data->id3v2 && !data->id3v2->frameListMap()["TPE2"].isEmpty())
      return ::strdup(data->id3v2->frameListMap()["TPE2"].front()->toString().toCString(1));
   return ::strdup("");
}
extern "C" void tag_data_set_performers (TagData * data, char * val)
{
   if (data->xiph)
   {
      if(!val || !(val[0]))
         data->xiph->removeField("PERFORMERS");
      else
         data->xiph->addField("PERFORMERS", TagLib::String(val, TagLib::String::UTF8));
   }
   if (data->ape)
   {
      if(!val || !(val[0]))
         data->ape->removeItem("PERFORMERS");
      else
         data->ape->addValue("PERFORMERS", TagLib::String(val, TagLib::String::UTF8), true);
   }
   if (data->id3v2)
   {
      if(!val || !(val[0]))
         data->id3v2->removeFrames("TPE2");
      else
      {
         TagLib::ID3v2::FrameList l = data->id3v2->frameListMap()["TPE2"];
         if(!l.isEmpty())
            l.front()->setText(TagLib::String(val, TagLib::String::UTF8));
         else
         {
            TagLib::ID3v2::TextIdentificationFrame *f = new TagLib::ID3v2::TextIdentificationFrame("TPE2", TagLib::String::UTF8);
            data->id3v2->addFrame (f);
            f->setText(TagLib::String(val, TagLib::String::UTF8));
         }
      }
   }
}


/* GET/SET DISC NUMBER
 * FIXME: Include number of discs in set? #/#
 */
extern "C" uint tag_data_get_disc_number (TagData * data)
{
   if (data->xiph && !data->xiph->fieldListMap()["DISCNUMBER"].isEmpty())
      return data->xiph->fieldListMap()["DISCNUMBER"].front().toInt();
   if (data->ape && !data->ape->itemListMap()["DISCNUMBER"].isEmpty())
      return data->ape->itemListMap()["DISCNUMBER"].toString().toInt();
   if (data->id3v2)
   {
      int i;
      if (data->id3v2->frameListMap()["TPOS"].isEmpty())
         return 0;
      return data->id3v2->frameListMap()["TPOS"].front()->toString().toInt();
   }
   return 0;
}
extern "C" void tag_data_set_disc_number (TagData * data, uint val)
{
   if (data->xiph)
   {
      if(val == 0)
         data->xiph->removeField("DISCNUMBER");
      else
         data->xiph->addField("DISCNUMBER", TagLib::String::number(val));
   }
   if (data->ape)
   {
      if(val == 0)
         data->ape->removeItem("DISCNUMBER");
      else
         data->ape->addValue("DISCNUMBER", TagLib::String::number(val), true);
   }
   if (data->id3v2)
   {
      TagLib::ID3v2::FrameList l = data->id3v2->frameListMap()["TPOS"];
      if(!l.isEmpty())
         l.front()->setText(TagLib::String::number(val));
      else
      {
         TagLib::ID3v2::TextIdentificationFrame *f = new TagLib::ID3v2::TextIdentificationFrame("TPOS", TagLib::String::UTF8);
         data->id3v2->addFrame (f);
         f->setText(TagLib::String::number(val));
      }
   }
}


/* GET SONG LENGTH
 */
extern "C" uint tag_data_get_duration (TagData * data)
{
   return data->file->audioProperties()->length();
}

/* FIXME: I have no clue what the documentation is saying on this one.

/* GET REPLAY GAIN
 */
extern "C" float tag_data_get_gain (TagData * data)
{

   if (data->xiph && !data->xiph->fieldListMap()["REPLAYGAIN_ALBUM_GAIN"].isEmpty())
      return atof(data->xiph->fieldListMap()["REPLAYGAIN_ALBUM_GAIN"].front().toCString(1));
   if (data->xiph && !data->xiph->fieldListMap()["REPLAYGAIN_TRACK_GAIN"].isEmpty())
      return atof(data->xiph->fieldListMap()["REPLAYGAIN_TRACK_GAIN"].front().toCString(1));
   if (data->xiph && !data->xiph->fieldListMap()["RG_AUDIOPHILE"].isEmpty())
      return atof(data->xiph->fieldListMap()["RG_AUDIOPHILE"].front().toCString(1));
   if (data->xiph && !data->xiph->fieldListMap()["RG_RADIO"].isEmpty())
      return atof(data->xiph->fieldListMap()["RG_RADIO"].front().toCString(1));
      
      
   if (data->id3v2 && !data->id3v2->frameListMap()["RVA2"].isEmpty())
   {
      TagLib::ID3v2::FrameList::ConstIterator it;
      TagLib::ID3v2::FrameList l = data->id3v2->frameListMap()["RVA2"];
      for(it=l.begin(); it!=l.end(); ++it)
      {
         TagLib::ID3v2::RelativeVolumeFrame * rva = dynamic_cast<TagLib::ID3v2::RelativeVolumeFrame*>(*it);
         if (rva->channelType () == TagLib::ID3v2::RelativeVolumeFrame::MasterVolume) 
            return rva->volumeAdjustment ();
      }
   }
   return 0.0;
}


/* GET PEAK
 */
extern "C" float tag_data_get_peak (TagData * data)
{
   if (data->xiph && !data->xiph->fieldListMap()["REPLAYGAIN_ALBUM_PEAK"].isEmpty())
      return atof(data->xiph->fieldListMap()["REPLAYGAIN_ALBUM_PEAK"].front().toCString(1));
   if (data->xiph && !data->xiph->fieldListMap()["REPLAYGAIN_TRACK_PEAK"].isEmpty())
      return atof(data->xiph->fieldListMap()["REPLAYGAIN_TRACK_PEAK"].front().toCString(1));
   if (data->xiph && !data->xiph->fieldListMap()["RG_PEAK"].isEmpty())
      return atof(data->xiph->fieldListMap()["RG_PEAK"].front().toCString(1));
   return 0.0;
}


/* GET ALBUM COVER
 */
extern "C" const char * tag_data_get_cover (TagData * data, uint * len)
{
   TagLib::ID3v2::AttachedPictureFrame * pic = NULL;
   if (!data->id3v2 || data->id3v2->frameListMap()["APIC"].isEmpty())
   {
      *len = 0;
      return NULL;
   }
   
   TagLib::ID3v2::FrameList::ConstIterator it;
   TagLib::ID3v2::FrameList l = data->id3v2->frameListMap()["APIC"];
   for(it=l.begin(); it!=l.end(); ++it)
   {
      TagLib::ID3v2::AttachedPictureFrame * t = dynamic_cast<TagLib::ID3v2::AttachedPictureFrame*>(*it);
      if (pic == NULL)
         pic = t;
      if (t->type () == TagLib::ID3v2::AttachedPictureFrame::FrontCover) 
      {
         pic = t;
         break;
      }
   }
   if (pic == NULL)
   {
      *len = 0;
      return NULL;
   }
   *len = pic->picture ().size ();
   const char * out = (const char *) malloc (*len);
   memcpy ((void *) out, pic->picture ().data(), *len);
   return out;
}


/* GET MIME TYPE
 */
extern "C" const char * tag_data_get_mime (TagData * data)
{
   return ::strdup(data->mime);
}


/* GET MTIME
 */
extern "C" int tag_data_get_mtime (TagData * data)
{
   return data->mtime;
}
