#include "GSBridge.h"
#include "GSBridgeDelegate.h"

#import <gst/gst.h>
#import <gst/video/video.h>

GST_DEBUG_CATEGORY_STATIC (debug_category);
#define GST_CAT_DEFAULT debug_category

@implementation GSBridge {
    UIView *view;
    NSString *source;
    
    GMainLoop *loop;
    
    GstElement *pipeline;
    GstElement *sink;
    
    gint64 position, duration;
    
    gboolean isLive;
    gboolean isInitialized;
}

@synthesize delegate;

-(id) init:(NSString *)source view:(UIView *)view {
    if (self = [super init]) {
        self->source = source;
        self->view = view;
        self->position = GST_CLOCK_TIME_NONE;
        self->duration = GST_CLOCK_TIME_NONE;
        
    #if DEBUG
        GST_DEBUG_CATEGORY_INIT (debug_category, "App", 0, "iOS");
        gst_debug_set_threshold_for_name("App", GST_LEVEL_LOG);
    #endif
        
        /* Start the bus monitoring task */
        dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
            [self run];
        });
    }
    
    return self;
}

-(void) play {
    isLive = gst_element_set_state(pipeline, GST_STATE_PLAYING) == GST_STATE_CHANGE_NO_PREROLL;
}

-(void) pause {
    isLive = gst_element_set_state(pipeline, GST_STATE_PAUSED) == GST_STATE_CHANGE_NO_PREROLL;
}

-(void) stop {
    gst_element_send_event(pipeline, gst_event_new_eos());
}

-(BOOL) isPlaying {
    GstState state;
    GstState pending;
    
    gst_element_get_state(pipeline, &state, &pending, GST_CLOCK_TIME_NONE);
    
    return state == GST_STATE_PLAYING;
}

-(UIImage*) takeSnapshot {
    GstElement *sink = self->sink;
    
    GstSample *videobuffer;
    g_object_get(G_OBJECT(sink), "last-sample", &videobuffer, NULL);
    
    if (videobuffer) {
        GstCaps *caps = gst_sample_get_caps(videobuffer);
        
        if (!caps) {
            return NULL;
        }
        
        gint width, height;
        GstStructure *s = gst_caps_get_structure(caps, 0);
        /* we need to get the final caps on the buffer to get the size */
        gboolean res;
        res = gst_structure_get_int (s, "width", &width);
        res |= gst_structure_get_int (s, "height", &height);
        if (!res) {
            return NULL;
        }
        
        GstMapInfo map;
        GstBuffer *snapbuffer = gst_sample_get_buffer(videobuffer);
        if (snapbuffer && gst_buffer_map (snapbuffer, &map, GST_MAP_READ)) {
            CGDataProviderRef provider = CGDataProviderCreateWithData(NULL, map.data, height * width * 4, NULL);
            
            CGColorSpaceRef colorSpaceRef = CGColorSpaceCreateDeviceRGB();
            CGBitmapInfo bitmapInfo = kCGBitmapByteOrderDefault;
            CGColorRenderingIntent renderingIntent = kCGRenderingIntentDefault;
            
            CGImageRef imageRef = CGImageCreate(width, height, 8, 4 * 8, width * 4, colorSpaceRef, bitmapInfo, provider, NULL, NO, renderingIntent);
            
            UIImage *uiImage = [UIImage imageWithCGImage:imageRef];
            CGColorSpaceRelease(colorSpaceRef);
            CGImageRelease(imageRef);
            
            return uiImage;
        }
        
        return NULL;
    }
    
    return NULL;
}

-(gint) run {
    pipeline = gst_pipeline_new("pipeline");
    gst_element_add_property_deep_notify_watch(pipeline, NULL, true);
    
    GstElement *src = gst_element_factory_make("rtspsrc", NULL);
    g_object_set(G_OBJECT(src), "location", [source UTF8String], NULL);
    g_object_set(G_OBJECT(src), "latency", 50, NULL);
    
    GstElement *depay = gst_element_factory_make("rtph264depay", NULL);
    g_signal_connect(G_OBJECT(src), "pad-added", G_CALLBACK(depay_add_pad), depay);
    
    GstElement *decode = gst_element_factory_make("avdec_h264", NULL);
    sink = gst_element_factory_make("glimagesink", NULL);
    
    if (src && sink) {
        gst_video_overlay_set_window_handle(GST_VIDEO_OVERLAY(sink), (guintptr) (id) view);
        
    } else {
        GST_ERROR("Could not retrieve video sink\n");
        return -1;
    }
    
    /* add elements */
    gst_bin_add_many(GST_BIN(pipeline), src, depay, decode, sink, NULL);
    
    /* link elements */
    if(!gst_element_link_many(depay, decode, sink, NULL)) {
        GST_ERROR("Failed to link src to sink!\n");
        return -1;
    }

//    GstCaps *caps = gst_caps_new_empty_simple("video/x-raw");
//    gst_caps_set_simple(caps, "width", G_TYPE_INT, 1920, nil);
//    gst_caps_set_simple(caps, "height", G_TYPE_INT, 1080, nil);
//    gst_caps_unref(caps);
//
//    gboolean link_ok = gst_element_link_filtered(src, sink, caps);
//    if (!link_ok) {
//        g_print("Could not link caps");
//        return -1;
//    }
    
    GMainContext *context = g_main_context_new();
    g_main_context_push_thread_default(context);
    
    /* Register a function that GLib will call 4 times per second */
    GSource *timeoutSource = g_timeout_source_new(250);
    g_source_set_callback(timeoutSource, (GSourceFunc) refresh, (__bridge void *) self, NULL);
    g_source_attach(timeoutSource, context);
    g_source_unref(timeoutSource);
    
    loop = g_main_loop_new(context, FALSE);
    
    GstBus *bus = gst_element_get_bus(pipeline);
    GSource *busSource = gst_bus_create_watch(bus);
    g_source_set_callback(busSource, (GSourceFunc) gst_bus_async_signal_func, NULL, NULL);
    g_source_attach(busSource, context);
    g_source_unref(busSource);
    g_signal_connect(G_OBJECT(bus), "message::error", G_CALLBACK(error), (__bridge void *) self);
    g_signal_connect(G_OBJECT(bus), "message::state-changed", G_CALLBACK(state_changed), (__bridge void *) self);
    g_signal_connect(G_OBJECT(bus), "message::clock-lost", G_CALLBACK(clock_lost), (__bridge void *) self);
    g_signal_connect(G_OBJECT(bus), "message::eos", G_CALLBACK(eos), (__bridge void *) self);
    gst_object_unref(bus);
    
    checkIsInitialized(self);
    g_main_loop_run(loop); // Running
    g_main_loop_unref(loop); // Exiting
    
    g_main_context_pop_thread_default(context);
    g_main_context_unref(context);

    gst_element_set_state(pipeline, GST_STATE_NULL);
    gst_object_unref(pipeline);
    
    return 0;
}

static void checkIsInitialized(GSBridge *self) {
    GMainLoop *loop = self->loop;
    gboolean isInitialized = self->isInitialized;
    id delegate = self->delegate;
    
    if (isInitialized || !loop)
        return;
    
    if (delegate && [delegate respondsToSelector:@selector(initialized)]) {
        [delegate initialized];
        
    } else {
        GST_INFO("Initialized");
    }
    
    isInitialized = true;
}

static gboolean refresh(GSBridge *self) {
    GstElement *pipeline = self->pipeline;
    gint64 position = self->position;
    gint64 duration = self->duration;
    id delegate = self->delegate;

    GstState current, pending;
    gst_element_get_state(pipeline, &current, &pending, GST_CLOCK_TIME_NONE);
    
    if (!self || !pipeline || current < GST_STATE_PAUSED)
        return TRUE;
        
    if (!GST_CLOCK_TIME_IS_VALID(duration)) {
        gst_element_query_position(pipeline, GST_FORMAT_TIME, &position);
        gst_element_query_duration(pipeline, GST_FORMAT_TIME, &duration);
    }
    
    if (delegate && [delegate respondsToSelector:@selector(positionChangedTo:duration:)]) {
        [delegate positionChangedTo:position duration:duration];
        
    } else {
        GST_INFO("Position: %ld, Duration: %ld", position / GST_MSECOND, duration / GST_MSECOND);
    }
    
    return TRUE;
}

static void state_changed(GstBus *bus, GstMessage *msg, GSBridge *self) {
    GstElement *pipeline = self->pipeline;
    id delegate = self->delegate;
    
    GstState old, new, pending;
    gst_message_parse_state_changed(msg, &old, &new, &pending);
    
    /* Only pay attention to messages coming from the pipeline, not its children */
    if (GST_MESSAGE_SRC(msg) != GST_OBJECT(pipeline))
        return;
    
    if (delegate && [delegate respondsToSelector:@selector(stateChanged:)]) {
        [delegate stateChanged:new];
        
    } else {
        GST_INFO("State changed to: %i", new);
    }
}

static void depay_add_pad(GstElement *element, GstPad *pad, gpointer data) {
    gchar *name = gst_pad_get_name(pad);
    GstElement *depay = GST_ELEMENT(data);
    
    if(!gst_element_link_pads(element, name, depay, "sink")) {
        GST_ERROR("Failed to link elements\n");
    }
    
    g_free(name);
}

static void clock_lost(GstBus *bus, GstMessage *msg, GSBridge *self) {
    GstElement *pipeline = self->pipeline;
    
    gst_element_set_state(pipeline, GST_STATE_PAUSED);
    gst_element_set_state(pipeline, GST_STATE_PLAYING);
}

static void eos(GstBus *bus, GstMessage *msg, GSBridge *self) {
    GMainLoop *loop = self->loop;
    GstElement *pipeline = self->pipeline;
    gboolean isLive = self->isLive;
    
    isLive = gst_element_set_state(pipeline, GST_STATE_NULL) == GST_STATE_CHANGE_NO_PREROLL;
    
    if (loop)
        g_main_loop_quit(loop);
}

static void error(GstBus *bus, GstMessage *msg, GSBridge *self) {
    GstElement *pipeline = self->pipeline;
    id delegate = self->delegate;

    GError *err;
    gchar *debug_info;
    gst_message_parse_error (msg, &err, &debug_info);
    
    gchar *error_source = g_strdup_printf("%s", GST_OBJECT_NAME(msg->src));
    gchar *error_message = g_strdup_printf("%s", err->message);
    
    if (delegate && [delegate respondsToSelector:@selector(errorThrown:code:message:)]) {
        NSString *source = [NSString stringWithUTF8String:error_source];
        NSString *message = [NSString stringWithUTF8String:error_message];
        
        [delegate errorThrown:err->code source:source message:message];
        
    } else {
        GST_INFO("Error %i %s: %s", err->code, error_source, error_message);
    }
    
    g_clear_error(&err);
    g_free(debug_info);
    g_free(error_source);
    g_free(error_message);
    
    gst_element_set_state(pipeline, GST_STATE_NULL);
}

@end
