<?php
/**
 * 2007-2016 maofree
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Attribution-NonCommercial-NoDerivatives 4.0 International (CC BY-NC-ND 4.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://creativecommons.org/licenses/by-nc-nd/4.0/
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to maofreedeveloper@gmail.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * @author    maofree <maofreedeveloper@gmail.com>
 * @link      http://www.maofree-developer.com
 * @copyright 2007-2016 maofree
 * @license   http://creativecommons.org/licenses/by-nc-nd/4.0/  Attribution-NonCommercial-NoDerivatives 4.0 International (CC BY-NC-ND 4.0)
 */

if (!defined('_PS_VERSION_')) {
    exit;
}

if (!ini_get('safe_mode')) {
    if (function_exists('set_time_limit')) {
        @set_time_limit(10000);
    }

    $memory_limit = @ini_get('memory_limit');
    if (
        Tools::substr($memory_limit, -1) != 'G' &&
        (
            (Tools::substr($memory_limit, -1) == 'M' && Tools::substr($memory_limit, 0, -1) < 256) ||
            is_numeric($memory_limit) && ((int)$memory_limit < 262144)
        )
    ) {
        @ini_set('memory_limit', '256M');
    }
} else {
    @ini_set('max_execution_time', '10000');
}

@ignore_user_abort();

require_once dirname(__FILE__).'/classes/MaofreeComparatorClass.php';
require_once dirname(__FILE__).'/classes/MaofreeDatafeedClass.php';

class MaofreeComparator extends Module
{
    const MAIN_TABLE = 'mao_comparator';
    const CRONJOBS_TABLE = 'mao_comparator_cronjobs';
    const TAXONOMY_LANG_TABLE = 'mao_comparator_taxonomy_lang';
    const TAXONOMY_COUNTRY_TABLE = 'mao_comparator_taxonomy_country';

    protected static $cache_links;
    protected static $cache_tracks;
    protected static $cache_settings;

    public function __construct()
    {
        $this->name = 'maofreecomparator';
        $this->tab = 'smart_shopping';
        $this->version = '16.3.0';
        $this->author = 'maofree';
        $this->need_instance = 0;
        $this->is_configurable = 1;
        $this->token = Tools::substr(Tools::encrypt($this->name), 0, 10);
        $this->bootstrap = true;
        $this->ps_versions_compliancy = array('min' => '1.6', 'max' => _PS_VERSION_);

        parent::__construct();

        $this->displayName = $this->l('Comparators on your website');
        $this->description = $this->l('Creates and manages data feed for many comparators');
    }

    public function install()
    {
        $success = (
            parent::install() &&
            MaofreeComparatorClass::createMainTable(self::MAIN_TABLE) &&
            MaofreeComparatorClass::initializationMainTable(self::MAIN_TABLE, Shop::getContextListShopID()) &&
            $this->registerHook('displayHeader') &&
            $this->registerHook('displayFooter') &&
            $this->registerHook('displayOrderConfirmation') &&
            $this->registerHook('actionObjectLanguageAddAfter') &&
            $this->registerHook('actionCategoryDelete') &&
            $this->registerHook('actionCarrierUpdate') &&
            Configuration::updateGlobalValue('MAO_COMPARATOR_TAXONOMY_LANG', (int) Configuration::get('PS_LANG_DEFAULT')) &&
            Configuration::updateGlobalValue('MAO_COMPARATOR_TAXONOMY_COUNTRY', (int) Configuration::get('PS_COUNTRY_DEFAULT')) &&
            $this->installTab() &&
            $this->installFixtures()
        );

        $this->moduleClearCache();

        return $success;
    }

    public function uninstall()
    {
        $this->moduleClearCache();

        $settings = unserialize(Configuration::get('MAO_COMPARATOR_SETTINGS'));

        if (
            $settings['removetable'] && (
                !MaofreeComparatorClass::removeTable(self::MAIN_TABLE) ||
                !Configuration::deleteByName('MAO_COMPARATOR_SETTINGS') ||
                !Configuration::deleteByName('MAO_COMPARATOR_ONECRONJOB') ||
                !Configuration::deleteByName('MAO_COMPARATOR_COMPANYDESC') ||
                !Configuration::deleteByName('MAO_COMPARATOR_CARRIERSZONE')
            ) ||
            !Configuration::deleteByName('MAO_COMPARATOR_TAXONOMY_LANG') ||
            !Configuration::deleteByName('MAO_COMPARATOR_TAXONOMY_COUNTRY') ||
            !parent::uninstall() ||
            !$this->uninstallTab() ||
            !MaofreeComparatorClass::updateAutoIncrement()
        ) {
            return false;
        }

        return true;
    }

    public function installTab()
    {
        $tab = new Tab();
        $tab->active = 1;
        $tab->class_name = 'AdminMaofreeComparator';
        $tab->name = array();
        foreach (Language::getLanguages(true) as $lang) {
            $id_lang = (int) $lang['id_lang'];
            $iso_code = Tools::strtolower($lang['iso_code']);

            if ($iso_code == 'it') {
                $tab->name[$id_lang] = 'Comparatori';
            } elseif ($iso_code == 'es') {
                $tab->name[$id_lang] = 'Comparadores';
            } elseif ($iso_code == 'pt' || $iso_code == 'br') {
                $tab->name[$id_lang] = 'Comparadores';
            } elseif ($iso_code == 'de') {
                $tab->name[$id_lang] = 'Komparatoren';
            } elseif ($iso_code == 'fr') {
                $tab->name[$id_lang] = 'Comparateurs';
            } else {
                $tab->name[$id_lang] = $this->l('Comparators');
            }
        }
        $tab->id_parent = (int) Tab::getIdFromClassName('AdminPriceRule');
        $tab->module = $this->name;

        return $tab->add();
    }

    public function uninstallTab()
    {
        $id_tab = (int) Tab::getIdFromClassName('AdminMaofreeComparator');
        if ($id_tab) {
            $tab = new Tab($id_tab);

            return $tab->delete();
        } else {
            return false;
        }
    }

    private function variableInitialization()
    {
        $abbreviation = '';
        $shop_name = Configuration::get('PS_SHOP_NAME');
        $words = explode(' ', $shop_name);

        if (count($words) > 1) {
            foreach ($words as $index => $word) {
                if ($index < 3) {
                    $abbreviation .= $word[0];
                }
            }
        } else {
            $abbreviation .= Tools::substr($shop_name, 0, 3);
        }

        $success = (
            Configuration::updateGlobalValue('MAO_COMPARATOR_ONECRONJOB', 1) &&
            Configuration::updateValue(
                'MAO_COMPARATOR_SETTINGS',
                serialize(array(
                    'description' => 'short',
                    'products_cycle' => 100,
                    'company_abbreviation' => $abbreviation,
                    'removetable' => 0,
                ))
            )
        );

        return $success;
    }

    public function hookActionObjectLanguageAddAfter($params)
    {
        return $this->installFixture((int) $params['object']->id);
    }

    protected function installFixtures()
    {
        $languages = Language::getLanguages(false);

        foreach ($languages as $lang) {
            if (!$this->installFixture((int) $lang['id_lang'])) {
                return false;
            }
        }

        return true;
    }

    protected function installFixture($id_lang)
    {
        $scanned_directory = $directory_list = $values = array();

        if (Configuration::get('MAO_COMPARATOR_COMPANYDESC', $id_lang) === false) {
            $values['MAO_COMPARATOR_COMPANYDESC'][(int) $id_lang] = '';

            Configuration::updateValue('MAO_COMPARATOR_COMPANYDESC', $values['MAO_COMPARATOR_COMPANYDESC']);
        }

        $directory = _PS_MODULE_DIR_.$this->name.'/comparators';
        $scanned_directory = array_diff(scandir($directory), array('..', '.', 'index.php'));

        if (is_array($scanned_directory) && count($scanned_directory)) {
            foreach ($scanned_directory as $file) {
                if (is_dir($directory.'/'.$file)) {
                    $directory_list[] = $file;
                }
            }

            if (is_array($directory_list) && count($directory_list)) {
                foreach ($directory_list as $comparator_name) {
                    require_once dirname(__FILE__).'/comparators/'.$comparator_name.'/'.$comparator_name.'.php';

                    $comparator_obj = new $comparator_name();

                    if (
                        is_object($comparator_obj) &&
                        isset($comparator_obj->taxonomy) &&
                        $comparator_obj->taxonomy &&
                        !MaofreeComparatorClass::addTaxonomyFiles($comparator_name, $directory)
                    ) {
                        return false;
                    }
                }
            }
        }

        return true;
    }

    public function getContent()
    {
        $multishop = Configuration::getGlobalValue('PS_MULTISHOP_FEATURE_ACTIVE');

        if (
            $multishop &&
            Shop::getTotalShops(false) > 1 &&
            (Shop::getContextShopGroupID(true) === null || Shop::getContextShopID(true) === null)
        ) {
            // this condition is necessary to avoid problems about context and to load the right configuration variables where $id_shop and $id_shop_group can have null values, in this mode they always have integer values
            return $this->displayError($this->l('You are using the multistore feature, to use this module it is necessary that you select a shop (do not choose generic selections: all shops or a group shop).'));
        } else {
            // necessary to avoid problems in the mode MultiStore ($id_shop_group = Shop::getContextShopGroupID(true) and $id_shop = Shop::getContextShopID(true))
            if (Configuration::get('MAO_COMPARATOR_SETTINGS') === false) {
                $this->variableInitialization();
            }
            if (Configuration::get('MAO_COMPARATOR_COMPANYDESC', (int) $this->context->language->id) === false) {
                $this->installFixture((int) $this->context->language->id);
            }
            if (Configuration::get('MAO_COMPARATOR_CARRIERSZONE') === false) {
                Configuration::updateValue('MAO_COMPARATOR_CARRIERSZONE', '');
            }

            $output = $this->maofreeLogo();

            $with_errors = false;
            $comparator_name = '';
            $id_shop = (int) $this->context->shop->id;
            $id_lang = (int) $this->context->language->id;
            $settings = unserialize(Configuration::get('MAO_COMPARATOR_SETTINGS'));

            if (Tools::isSubmit('submitGeneralSettings')) {
                $description = Tools::getValue('description');
                $link_position = Tools::getValue('link_position');
                $products_cycle = Tools::getValue('products_cycle');
                $company_abbreviation = Tools::getValue('company_abbreviation');
                $onecronjob = Tools::getValue('onecronjob');
                $removetable = Tools::getValue('removetable');
                $hookposition = Tools::getValue('hookposition');

                if (!Validate::isUnsignedInt($products_cycle)) {
                    $with_errors = true;

                    $output .= $this->displayError($this->l('Invalid number in = Products processed at each cycle'));
                } elseif ($products_cycle && $products_cycle < 2) {
                    $with_errors = true;

                    $output .= $this->displayError($this->l('The limit is too low'));
                } elseif (!Validate::isName($company_abbreviation)) {
                    $with_errors = true;

                    $output .= $this->displayError($this->l('Invalid text in = Company\'s abbreviation'));
                } elseif ($company_abbreviation && Tools::strlen($company_abbreviation) > 10) {
                    $with_errors = true;

                    $output .= $this->displayError($this->l('The abbreviation is too long'));
                } else {
                    $settings['description'] = $description;
                    $settings['products_cycle'] = (int) $products_cycle;
                    $settings['company_abbreviation'] = $company_abbreviation;
                    $settings['removetable'] = (int) $removetable;

                    Configuration::updateValue('MAO_COMPARATOR_SETTINGS', serialize($settings));
                    Configuration::updateGlobalValue('MAO_COMPARATOR_ONECRONJOB', (int) $onecronjob);

                    $shop_list_ID = $multishop ? Shop::getContextListShopID() : null;
                    $hookidleftcolumn = (int) Hook::getIdByName('displayLeftColumn');
                    $hookidrightcolumn = (int) Hook::getIdByName('displayRightColumn');
                    $hookidhome = (int) Hook::getIdByName('displayHome');
                    $hookidfooter = (int) Hook::getIdByName('displayFooter');
                    $hook_on_left = $this->isRegisteredInHook('displayLeftColumn');
                    $hook_on_right = $this->isRegisteredInHook('displayRightColumn');
                    $hook_on_home = $this->isRegisteredInHook('displayHome');
                    $hook_on_footer = $this->isRegisteredInHook('displayFooter');

                    $homepage = in_array('home', $link_position);
                    $footer = in_array('footer', $link_position);
                    $side = in_array('side', $link_position);

                    if ($homepage && !$hook_on_home) {
                        $this->registerHook('displayHome', $shop_list_ID);
                    } elseif (!$homepage && $hook_on_home) {
                        $this->unregisterHook($hookidhome, $shop_list_ID);
                    }

                    if ($footer && !$hook_on_footer) {
                        $this->registerHook('displayFooter', $shop_list_ID);
                    } elseif (!$footer && $hook_on_footer) {
                        $this->unregisterHook($hookidfooter, $shop_list_ID);
                    }

                    if (!$side) {
                        if ($hook_on_left) {
                            $this->unregisterHook($hookidleftcolumn, $shop_list_ID);
                        }
                        if ($hook_on_right) {
                            $this->unregisterHook($hookidrightcolumn, $shop_list_ID);
                        }
                    } else {
                        if ($hookposition == 'displayLeftColumn') {
                            if (!$hook_on_left) {
                                $this->registerHook($hookposition, $shop_list_ID);
                            }

                            if ($hook_on_right) {
                                $this->unregisterHook($hookidrightcolumn, $shop_list_ID);
                            }
                        } elseif ($hookposition == 'displayRightColumn') {
                            if (!$hook_on_right) {
                                $this->registerHook($hookposition, $shop_list_ID);
                            }

                            if ($hook_on_left) {
                                $this->unregisterHook($hookidleftcolumn, $shop_list_ID);
                            }
                        } else {
                            if (!$hook_on_left) {
                                $this->registerHook('displayLeftColumn', $shop_list_ID);
                            }

                            if (!$hook_on_right) {
                                $this->registerHook('displayRightColumn', $shop_list_ID);
                            }
                        }
                    }

                    $desc_error = 0;
                    $values = $values['MAO_COMPARATOR_COMPANYDESC'] = array();
                    $languages = Language::getLanguages(false);

                    foreach ($languages as $lang) {
                        $company_description = Tools::getValue('company_description_'.$lang['id_lang']);

                        if (Validate::isCleanHtml($company_description)) {
                            $values['MAO_COMPARATOR_COMPANYDESC'][$lang['id_lang']] = $company_description;
                        } else {
                            ++$desc_error;
                        }
                    }

                    if ($desc_error) {
                        $with_errors = true;

                        $output .= $this->displayError($this->l('The description must be a message.'));
                    } else {
                        Configuration::updateValue('MAO_COMPARATOR_COMPANYDESC', $values['MAO_COMPARATOR_COMPANYDESC']);
                    }

                    $this->moduleClearCache();

                    $output .= $this->displayConfirmation($this->l('Settings updated'));
                }
            } elseif (Tools::isSubmit('submitCarriersAssociation')) {
                $carriers_zone = array();

                foreach (Zone::getZones(true) as $zone) {
                    $id_zone = (int) $zone['id_zone'];

                    if (Tools::getValue('carrier_zone_'.$id_zone)) {
                        $carriers_zone[] = $id_zone.'#'.(int) Tools::getValue('carrier_zone_'.$id_zone);
                    }
                }

                Configuration::updateValue('MAO_COMPARATOR_CARRIERSZONE', count($carriers_zone) ? implode(',', $carriers_zone) : '');

                $output .= $this->displayConfirmation($this->l('Settings updated'));
            } elseif (Tools::isSubmit('submitAddNewFeedToCronJob')) {
                $is_accepted = false;
                $comparator_name = Tools::strtolower(Tools::getValue('feed_name'));
                $feed_tax = Tools::getValue('feed_tax');
                $feed_currency = Tools::getValue('feed_currency');
                $feed_language = Tools::getValue('feed_language');
                $feed_country = Tools::getValue('feed_country');
                $feed_type = Tools::strtolower(Tools::getValue('feed_type'));

                if (isset($comparator_name) && $comparator_name) {
                    require_once dirname(__FILE__).'/comparators/'.$comparator_name.'/'.$comparator_name.'.php';
                    $comparator_obj = new $comparator_name();

                    if (isset($comparator_obj->datafeed_type) && is_array($comparator_obj->datafeed_type) && count($comparator_obj->datafeed_type)) {
                        $datafeed_type_to_check = array_diff($comparator_obj->datafeed_type, array('html'));

                        if (in_array($feed_type, $datafeed_type_to_check)) {
                            $is_accepted = true;
                        }
                    }

                    if ($is_accepted) {
                        $cronjob = MaofreeComparatorClass::getCronJobs(
                            self::CRONJOBS_TABLE,
                            $id_shop,
                            false,
                            'name',
                            $comparator_name,
                            (int) $feed_tax,
                            (int) $feed_currency,
                            (int) $feed_language,
                            (int) $feed_country,
                            $feed_type
                        );

                        if (!isset($cronjob) || !$cronjob) {
                            MaofreeComparatorClass::newCronJob(
                                self::CRONJOBS_TABLE,
                                $id_shop,
                                $comparator_name,
                                (int) $feed_tax,
                                (int) $feed_currency,
                                (int) $feed_language,
                                (int) $feed_country,
                                $feed_type
                            );

                            $output .= $this->displayConfirmation($this->l('A new data feed will be managed by a single Cron Job'));
                        } else {
                            $output .= $this->displayError($this->l('This data feed is already managed by the Global Cron Job'));
                        }
                    } else {
                        $output .= $this->displayError($this->l('The selected comparator cannot manage data feed with extension in').' '.Tools::strtoupper($feed_type));
                    }
                } else {
                    $output .= $this->displayError($this->l('Before you need to active at least one comparator'));
                }
            } elseif (Tools::isSubmit('statusComparatorList') && Tools::getValue('name')) {
                $comparator_name = Tools::strtolower(Tools::getValue('name'));
                $comparator = MaofreeComparatorClass::getValuesFromTable(self::MAIN_TABLE, $id_shop, $comparator_name);
                $active = $comparator['active'];

                MaofreeComparatorClass::updateValue(
                    self::MAIN_TABLE,
                    $id_shop,
                    $comparator_name,
                    'active',
                    (int) (!$active)
                );

                $this->moduleClearCache();

                $output .= $this->displayConfirmation(Tools::strtoupper($comparator_name).' '.(!$active ? $this->l('enabled') : $this->l('disabled')));
            } elseif (Tools::isSubmit('submitComparatorEnable') && Tools::getValue('name')) {
                $comparator_name = Tools::strtolower(Tools::getValue('name'));
                $active = Tools::getValue('active');

                MaofreeComparatorClass::updateValue(
                    self::MAIN_TABLE,
                    $id_shop,
                    $comparator_name,
                    'active',
                    (int) $active
                );

                $this->moduleClearCache();

                $output .= $this->displayConfirmation(Tools::strtoupper($comparator_name).' '.($active ? $this->l('enabled') : $this->l('disabled')));
            } elseif (Tools::isSubmit('deleteComparatorList') && Tools::getValue('name')) {
                $comparator_name = Tools::strtolower(Tools::getValue('name'));
                require_once dirname(__FILE__).'/comparators/'.$comparator_name.'/'.$comparator_name.'.php';
                $comparator = MaofreeComparatorClass::getValuesFromTable(self::MAIN_TABLE, $id_shop, $comparator_name);
                $comparator_obj = new $comparator_name();

                if (isset($comparator_obj->datafeed_type) && is_array($comparator_obj->datafeed_type) && count($comparator_obj->datafeed_type)) {
                    $datafeed_type_to_check = array_diff($comparator_obj->datafeed_type, array('html'));
                    // taxonomy folder and data will not be removed to avoid to remove previous taxonomy associations
                    foreach ($datafeed_type_to_check as $extension) {
                        $directory = _PS_MODULE_DIR_.$this->name.'/comparators/'.$comparator_name.'/'.$extension.'/';

                        if (file_exists($directory)) {
                            Tools::deleteDirectory($directory);
                        }
                    }
                }

                MaofreeComparatorClass::removeRowFromTable(self::MAIN_TABLE, $id_shop, $comparator_name);

                $this->moduleClearCache();

                $output .= $this->displayConfirmation(Tools::strtoupper($comparator_name).' '.$this->l('has been removed from "mao_comparator" table and also all its data feed files. The taxonomy data was preserved'));
            } elseif (Tools::isSubmit('submitComparatorSettings') && Tools::getValue('name')) {
                $errors_num = 0;
                $comparator_name = Tools::strtolower(Tools::getValue('name'));
                $comparator = MaofreeComparatorClass::getValuesFromTable(self::MAIN_TABLE, $id_shop, $comparator_name);

                if (
                    file_exists(dirname(__FILE__).'/comparators/'.$comparator_name.'/'.$comparator_name.'.php') &&
                    is_array($comparator) &&
                    count($comparator)
                ) {
                    $on_stock_only = (int) Tools::getValue($comparator_name.'_on_stock_only');
                    $combinations = Tools::getIsset($comparator_name.'_combinations') ? (int) Tools::getValue($comparator_name.'_combinations') : $comparator['combinations'];
                    $limits = Tools::getIsset($comparator_name.'_limits') ? (int) Tools::getValue($comparator_name.'_limits') : $comparator['limits'];
                    $tracking = Tools::getIsset($comparator_name.'_tracking') ? (int) Tools::getValue($comparator_name.'_tracking') : $comparator['tracking'];
                    $add_link = (int) Tools::getValue($comparator_name.'_add_link');
                    $brand_in_title = (int) Tools::getValue($comparator_name.'_brand_in_title');
                    $compression_type = Tools::getIsset($comparator_name.'_compression_type') ? Tools::getValue($comparator_name.'_compression_type') : $comparator['compression_type'];
                    $extra_images = Tools::getIsset($comparator_name.'_extra_images') ? (int) Tools::getValue($comparator_name.'_extra_images') : $comparator['extra_images'];
                    $image_type = Tools::getValue($comparator_name.'_image_type');
                    $price_min = Tools::getValue($comparator_name.'_price_min');
                    $price_max = Tools::getValue($comparator_name.'_price_max');
                    $quantity_min = Tools::getValue($comparator_name.'_quantity_min');
                    $products_number = Tools::getValue($comparator_name.'_products_number');
                    $fee = Tools::getIsset($comparator_name.'_fee') ? (int) Tools::getValue($comparator_name.'_fee') : $comparator['fee'];

                    $extra_fields = $extra_fields_value = $extra_fields_unit = $extra_fields_title = $extra_fields_value_set = $extra_fields_unit_set = array();

                    require_once dirname(__FILE__).'/comparators/'.$comparator_name.'/'.$comparator_name.'.php';
                    $comparator_obj = new $comparator_name();

                    if (isset($comparator['extra_fields_keys']) && $comparator['extra_fields_keys']) {
                        $extra_fields = explode(',', $comparator['extra_fields_keys']);
                        $extra_fields_title = $comparator_obj->extra_fields_title;

                        if ($extra_fields) {
                            for ($i = 0; $i < count($extra_fields); ++$i) {
                                $extra_fields_value = Tools::getValue($comparator_name.'_'.$extra_fields[$i]);

                                if (isset($comparator['extra_fields_units']) && $comparator['extra_fields_units']) {
                                    $extra_fields_unit = Tools::getValue($comparator_name.'_'.$extra_fields[$i].'_unit');
                                }

                                if (!Validate::isString($extra_fields_value)) {
                                    ++$errors_num;
                                    $output .= $this->displayError($this->l('Invalid string in the field =').' "'.$extra_fields_title[$i].'"');
                                } else {
                                    $extra_fields_value_set[] = trim($extra_fields_value);

                                    if (isset($comparator['extra_fields_units']) && $comparator['extra_fields_units']) {
                                        $extra_fields_unit_set[] = $extra_fields_unit;
                                    }
                                }
                            }

                            if (!$errors_num) {
                                MaofreeComparatorClass::updateValue(
                                    self::MAIN_TABLE,
                                    $id_shop,
                                    $comparator_name,
                                    'extra_fields_values',
                                    implode('#', $extra_fields_value_set)
                                );

                                if (isset($comparator['extra_fields_units']) && $comparator['extra_fields_units']) {
                                    MaofreeComparatorClass::updateValue(
                                        self::MAIN_TABLE,
                                        $id_shop,
                                        $comparator_name,
                                        'extra_fields_units',
                                        implode(',', $extra_fields_unit_set)
                                    );
                                }
                            }
                        }
                    }

                    if (!Validate::isUnsignedFloat(str_replace(',', '.', $price_min))) {
                        ++$errors_num;
                        $output .= $this->displayError($this->l('Invalid number in the field = Minimum price'));
                    } elseif (!Validate::isUnsignedFloat(str_replace(',', '.', $price_max))) {
                        ++$errors_num;
                        $output .= $this->displayError($this->l('Invalid number in the field = Maximum price'));
                    } elseif (!Validate::isUnsignedInt($quantity_min)) {
                        ++$errors_num;
                        $output .= $this->displayError($this->l('Invalid number in the field = Minimum quantity'));
                    } elseif (!Validate::isUnsignedInt($products_number)) {
                        ++$errors_num;
                        $output .= $this->displayError($this->l('Invalid number in the field = Number of products'));
                    } elseif (!Validate::isUnsignedFloat($fee)) {
                        ++$errors_num;
                        $output .= $this->displayError($this->l('Invalid number in the field = Additional fee'));
                    } elseif (!$image_type) {
                        ++$errors_num;
                        $output .= $this->displayError($this->l('Image type cannot be void'));
                    }

                    if (!$errors_num) {
                        MaofreeComparatorClass::updateValue(self::MAIN_TABLE, $id_shop, $comparator_name, 'on_stock_only', $on_stock_only);
                        MaofreeComparatorClass::updateValue(self::MAIN_TABLE, $id_shop, $comparator_name, 'combinations', $combinations);
                        MaofreeComparatorClass::updateValue(self::MAIN_TABLE, $id_shop, $comparator_name, 'limits', $limits);
                        MaofreeComparatorClass::updateValue(self::MAIN_TABLE, $id_shop, $comparator_name, 'tracking', $tracking);
                        MaofreeComparatorClass::updateValue(self::MAIN_TABLE, $id_shop, $comparator_name, 'add_link', $add_link);
                        MaofreeComparatorClass::updateValue(self::MAIN_TABLE, $id_shop, $comparator_name, 'brand_in_title', $brand_in_title);
                        MaofreeComparatorClass::updateValue(self::MAIN_TABLE, $id_shop, $comparator_name, 'compression_type', $compression_type);
                        MaofreeComparatorClass::updateValue(self::MAIN_TABLE, $id_shop, $comparator_name, 'extra_images', $extra_images);
                        MaofreeComparatorClass::updateValue(self::MAIN_TABLE, $id_shop, $comparator_name, 'image_type', $image_type);
                        MaofreeComparatorClass::updateValue(self::MAIN_TABLE, $id_shop, $comparator_name, 'price_min', (float) $price_min);
                        MaofreeComparatorClass::updateValue(self::MAIN_TABLE, $id_shop, $comparator_name, 'price_max', (float) $price_max);
                        MaofreeComparatorClass::updateValue(self::MAIN_TABLE, $id_shop, $comparator_name, 'quantity_min', $quantity_min);
                        MaofreeComparatorClass::updateValue(self::MAIN_TABLE, $id_shop, $comparator_name, 'products_number', $products_number);
                        MaofreeComparatorClass::updateValue(self::MAIN_TABLE, $id_shop, $comparator_name, 'fee', (float) $fee);
                    }
                }

                if (!$errors_num) {
                    $this->moduleClearCache();

                    $output .= $this->displayConfirmation($this->l('The settings of this comparator have been updated'));
                } else {
                    $with_errors = true;
                }
            } elseif (Tools::isSubmit('submitAnalyticsComparatorSettings') && Tools::getValue('name')) {
                $errors_num = 0;
                $comparator_name = Tools::strtolower(Tools::getValue('name'));
                $comparator = MaofreeComparatorClass::getValuesFromTable(self::MAIN_TABLE, $id_shop, $comparator_name);

                if (
                    file_exists(dirname(__FILE__).'/comparators/'.$comparator_name.'/'.$comparator_name.'.php') &&
                    is_array($comparator) &&
                    count($comparator)
                ) {
                    $analytics_params = array();
                    $analytics = (int) Tools::getValue($comparator_name.'_analytics');
                    $analytics_utm_source = Tools::getValue($comparator_name.'_analytics_utm_source');
                    $analytics_utm_medium = Tools::getValue($comparator_name.'_analytics_utm_medium');
                    $analytics_utm_campaign = Tools::getValue($comparator_name.'_analytics_utm_campaign');

                    if (
                        $analytics_utm_source && !Validate::isUnixName($analytics_utm_source) ||
                        $analytics_utm_medium && !Validate::isUnixName($analytics_utm_medium) ||
                        $analytics_utm_campaign && !Validate::isUnixName($analytics_utm_campaign)
                    ) {
                        ++$errors_num;
                    } else {
                        $analytics_params[] = trim($analytics_utm_source);
                        $analytics_params[] = trim($analytics_utm_medium);
                        $analytics_params[] = trim($analytics_utm_campaign);

                        MaofreeComparatorClass::updateValue(
                            self::MAIN_TABLE,
                            $id_shop,
                            $comparator_name,
                            'analytics',
                            $analytics
                        );
                        MaofreeComparatorClass::updateValue(
                            self::MAIN_TABLE,
                            $id_shop,
                            $comparator_name,
                            'analytics_params',
                            (count($analytics_params) && implode('', $analytics_params) == '' ? '' : implode(',', $analytics_params))
                        );
                    }
                }

                if (!$errors_num) {
                    $output .= $this->displayConfirmation($this->l('The analytics parameters have been updated'));
                } else {
                    $with_errors = true;

                    $output .= $this->displayError($this->l('Some analytics parameters are not valid'));
                }
            } elseif (Tools::isSubmit('submitUrlComparatorSettings') && Tools::getValue('name')) {
                $errors_num = 0;
                $comparator_name = Tools::strtolower(Tools::getValue('name'));
                $comparator = MaofreeComparatorClass::getValuesFromTable(self::MAIN_TABLE, $id_shop, $comparator_name);

                if (
                    file_exists(dirname(__FILE__).'/comparators/'.$comparator_name.'/'.$comparator_name.'.php') &&
                    is_array($comparator) &&
                    count($comparator)
                ) {
                    $website_url = array();
                    $add_link = (int) Tools::getValue($comparator_name.'_add_link');

                    require_once dirname(__FILE__).'/comparators/'.$comparator_name.'/'.$comparator_name.'.php';
                    $comparator_obj = new $comparator_name();

                    foreach ($comparator_obj->website_url as $k => $url) {
                        $item_url = Tools::getValue($comparator_name.'_website_url_'.$k);

                        if (!Validate::isUrl($item_url)) {
                            ++$errors_num;
                        } else {
                            $website_url[] = !$item_url || $item_url == $url ? '' : $item_url;
                        }
                    }

                    if (!$errors_num) {
                        MaofreeComparatorClass::updateValue(
                            self::MAIN_TABLE,
                            $id_shop,
                            $comparator_name,
                            'add_link',
                            $add_link
                        );
                        MaofreeComparatorClass::updateValue(
                            self::MAIN_TABLE,
                            $id_shop,
                            $comparator_name,
                            'website_url',
                            (count($website_url) && implode('', $website_url) == '' ? '' : implode('@', $website_url))
                        );
                    }
                }

                if (!$errors_num) {
                    $this->moduleClearCache();

                    $output .= $this->displayConfirmation($this->l('The settings of external links have been updated'));
                } else {
                    $with_errors = true;

                    $output .= $this->displayError($this->l('Some URL are not valid'));
                }
            } elseif (Tools::isSubmit('submitComparatorSelection') && Tools::getValue('name')) {
                $ok = true;
                $comparator_name = Tools::strtolower(Tools::getValue('name'));
                $selectall = Tools::getValue('selectall');
                $selectcat = Tools::getValue('selectcate');
                $selectmanu = Tools::getValue('selectmanu');
                $selectmanual = Tools::getValue('selectmanual');
                $manual_selection_type = Tools::getValue('manual_selection_type');

                MaofreeComparatorClass::updateValue(
                    self::MAIN_TABLE,
                    $id_shop,
                    $comparator_name,
                    'selected_all',
                    $selectall
                );
                MaofreeComparatorClass::updateValue(
                    self::MAIN_TABLE,
                    $id_shop,
                    $comparator_name,
                    'selected_products_manually_type',
                    $manual_selection_type
                );

                MaofreeComparatorClass::updateValue(
                    self::MAIN_TABLE,
                    $id_shop,
                    $comparator_name,
                    'selected_manufacturers',
                    empty($selectmanu) ? '' : implode(',', $selectmanu)
                );

                MaofreeComparatorClass::updateValue(
                    self::MAIN_TABLE,
                    $id_shop,
                    $comparator_name,
                    'selected_categories',
                    empty($selectcat) ? '' : implode(',', $selectcat)
                );

                if (empty($selectmanual)) {
                    MaofreeComparatorClass::updateValue(
                        self::MAIN_TABLE,
                        $id_shop,
                        $comparator_name,
                        'selected_products_manually',
                        ''
                    );
                } else {
                    $data = $error = $refnotexist = $productnotexist = array();
                    $id_product = 0;
                    $selectmanual = str_replace(array('\r', '\n', ' '), '', $selectmanual);
                    $selectmanual_array = explode(',', $selectmanual);

                    foreach ($selectmanual_array as $i) {
                        if ($manual_selection_type == 'id') {
                            $product = new Product($i);

                            if (Validate::isUnsignedId($i) && Validate::isLoadedObject($product)) {
                                $data[] = $i;
                            } elseif (!Validate::isUnsignedId($i)) {
                                $error[] = $i;
                            } elseif (Validate::isUnsignedId($i) && !Validate::isLoadedObject($product)) {
                                $productnotexist[] = $i;
                            }
                        } else {
                            $id_product = (int) MaofreeComparatorClass::existsRefInDatabase($i);

                            if (Validate::isReference($i) && $id_product) {
                                $data[] = $id_product;
                            } elseif (!Validate::isReference($i)) {
                                $error[] = $i;
                            } elseif (Validate::isReference($i) && !$id_product) {
                                $refnotexist[] = $i;
                            }
                        }
                    }

                    if (empty($error) && empty($productnotexist) && empty($refnotexist)) {
                        MaofreeComparatorClass::updateValue(
                            self::MAIN_TABLE,
                            $id_shop,
                            $comparator_name,
                            'selected_products_manually',
                            (count($data) ? implode(',', $data) : '')
                        );
                    }
                    if ($error) {
                        $output .= $this->displayError($this->l('These values have not been accepted').': '.implode(', ', $error));
                        $ok = false;
                    }
                    if ($productnotexist) {
                        $output .= $this->displayError($this->l('There are no products with these ID in the "product" table of your database').': '.implode(', ', $productnotexist));
                        $ok = false;
                    }
                    if ($refnotexist) {
                        $output .= $this->displayError($this->l('There are no products with these references in the "product" table of your database').': '.implode(', ', $refnotexist));
                        $ok = false;
                    }
                }

                if ($ok) {
                    $output .= $this->displayConfirmation($this->l('SCROLL DOWN THIS PAGE to see all selected products for this comparator.').' '.$this->l('Now you can exclude some products from its data feed'));
                } else {
                    $with_errors = true;
                }
            } elseif (Tools::isSubmit('submitComparatorProducts') && Tools::getValue('name')) {
                $comparator_name = Tools::strtolower(Tools::getValue('name'));
                $pageNumber = (int) Tools::getValue('ComparatorProducts_page');
                $nbProducts = (int) Tools::getValue('ComparatorProducts_pagination');
                $removeCheckBox = Tools::getValue('removeCheckBox');
                $comparator = MaofreeComparatorClass::getValuesFromTable(self::MAIN_TABLE, $id_shop, $comparator_name);
                $removed_products = $comparator['removed_products'] ? explode(',', $comparator['removed_products']) : array();

                if (is_array($removed_products) && count($removed_products)) {
                    foreach ($removed_products as $key => $value) {
                        if ($value > (($pageNumber - 1) * $nbProducts) && $value <= ($pageNumber * $nbProducts)) {
                            unset($removed_products[$key]);
                        }
                    }

                    if (is_array($removeCheckBox) && count($removeCheckBox)) {
                        $removed_products = array_merge($removed_products, $removeCheckBox);
                    }

                    if (is_array($removed_products) && count($removed_products)) {
                        sort($removed_products);
                        $removed_products_id = implode(',', $removed_products);
                    } else {
                        $removed_products_id = '';
                    }
                } else {
                    $removed_products_id = (is_array($removeCheckBox) && count($removeCheckBox) ? implode(',', $removeCheckBox) : '');
                }

                MaofreeComparatorClass::updateValue(
                    self::MAIN_TABLE,
                    $id_shop,
                    $comparator_name,
                    'removed_products',
                    $removed_products_id
                );

                if ($removed_products_id) {
                    $output .= $this->displayConfirmation($this->l('All selected products will not be present into the data feed associated to this comparator'));
                } else {
                    $output .= $this->displayConfirmation($this->l('All products chosen from your selections will be present into the data feed associated to this comparator'));
                }
            } elseif (Tools::isSubmit('updateUrlTaxonomyFiles') && Tools::getValue('name')) {
                $comparator_name = Tools::strtolower(Tools::getValue('name'));
                $taxonomy_files_id = (int) Tools::getValue('taxonomy_files_id');
                require_once dirname(__FILE__).'/comparators/'.$comparator_name.'/'.$comparator_name.'.php';

                $comparator_obj = new $comparator_name();

                if (
                    isset($comparator_obj->taxonomy) &&
                    $comparator_obj->taxonomy &&
                    !MaofreeComparatorClass::addTaxonomyFiles(
                        $comparator_name,
                        dirname(__FILE__).'/comparators',
                        $taxonomy_files_id,
                        true
                    )
                ) {
                    $output .= $this->displayError($this->l('There was an error while downloading the taxonomy files from the site of this comparator'));
                } else {
                    $output .= $this->displayConfirmation($this->l('Taxonomy files successfully updated'));
                }
            } elseif (Tools::isSubmit('activeTaxonomyAssociation') && Tools::getValue('name')) {
                $comparator_name = Tools::strtolower(Tools::getValue('name'));
                $comparator = MaofreeComparatorClass::getValuesFromTable(self::MAIN_TABLE, $id_shop, $comparator_name);
                $taxonomy = $comparator['taxonomy'];

                MaofreeComparatorClass::updateValue(self::MAIN_TABLE, $id_shop, $comparator_name, 'taxonomy', (int) (!$taxonomy));

                $output .= $this->displayConfirmation($this->l('Taxonomy').' '.(!$taxonomy ? $this->l('enabled') : $this->l('disabled')));
            } elseif (
                (Tools::isSubmit('submitLanguageUpdateNoUrlTaxonomy') || Tools::isSubmit('submitCountryUpdateNoUrlTaxonomy')) &&
                Tools::getValue('name')
            ) {
                $comparator_name = Tools::strtolower(Tools::getValue('name'));
                $taxonomy_files_id = (int) Tools::getValue('taxonomy_files_id');

                if (Tools::isSubmit('submitLanguageUpdateNoUrlTaxonomy')) {
                    $taxonomy_id_lang = (int) Tools::getValue('taxonomy_id_lang');
                    $taxonomy_language = new Language($taxonomy_id_lang);
                    $iso_code = $taxonomy_language->language_code;
                } else {
                    $taxonomy_id_country = (int) Tools::getValue('taxonomy_id_country');
                    $taxonomy_country = new Country($taxonomy_id_country, $id_lang);
                    $iso_code = $taxonomy_country->iso_code;
                }

                require_once dirname(__FILE__).'/comparators/'.$comparator_name.'/'.$comparator_name.'.php';

                $comparator_obj = new $comparator_name();

                if (
                    isset($comparator_obj->taxonomy) &&
                    $comparator_obj->taxonomy &&
                    isset($comparator_obj->taxonomy_files) &&
                    is_array($comparator_obj->taxonomy_files) &&
                    count($comparator_obj->taxonomy_files) &&
                    isset($taxonomy_files_id) &&
                    isset($comparator_obj->taxonomy_files[$taxonomy_files_id]) &&
                    MaofreeComparatorClass::addTaxonomyFiles($comparator_name, dirname(__FILE__).'/comparators', $taxonomy_files_id)
                ) {
                    $taxonomy_file_info = $comparator_obj->taxonomy_files[$taxonomy_files_id];
                    $orig_file_ext = $taxonomy_file_info['orig_file_ext'];

                    if ($_FILES['taxonomy_file']['error'] === UPLOAD_ERR_OK && $_FILES['taxonomy_file']['name']) {
                        if (
                            strpos($_FILES['taxonomy_file']['name'], '.'.$orig_file_ext) === false &&
                            strpos($_FILES['taxonomy_file']['name'], '.'.Tools::strtoupper($orig_file_ext)) === false
                        ) {
                            $with_errors = true;

                            $output .= $this->displayError(sprintf($this->l('Accepted only file with extension equal to .%s'), $orig_file_ext));
                        } else {
                            $directory = dirname(__FILE__).'/comparators'.'/'.$comparator_name.'/taxonomy/';
                            $filename = $comparator_obj->nameTaxonomyFile($taxonomy_file_info, $iso_code);

                            if ($orig_file_ext == 'txt') {
                                if (file_exists($directory.$filename)) {
                                    @unlink($directory.$filename);
                                }

                                if (!move_uploaded_file($_FILES['taxonomy_file']['tmp_name'], $directory.$filename)) {
                                    $with_errors = true;

                                    $output .= $this->displayError($this->l('There is an error to copy this file into the taxonomy folder'));
                                }
                            } elseif ($orig_file_ext == 'csv') {
                                if (!method_exists($comparator_obj, 'extractCsvData')) {
                                    $with_errors = true;

                                    $output .= $this->displayError($this->l('The main php file of this comparator does not contain "extractCsvData" method to manage csv files'));
                                } elseif (!move_uploaded_file($_FILES['taxonomy_file']['tmp_name'], $directory.'temp.csv')) {
                                    $with_errors = true;

                                    $output .= $this->displayError($this->l('There is an error to copy this file into the taxonomy folder'));
                                } elseif (!$csv_data = $comparator_obj->extractCsvData('temp.csv')) {
                                    $with_errors = true;

                                    $output .= $this->displayError($this->l('There is an error to extract data from the csv file'));
                                } elseif ($csv_data && !file_put_contents($directory.$filename, "\xEF\xBB\xBF".$csv_data, LOCK_EX)) {
                                    $with_errors = true;

                                    $output .= $this->displayError($this->l('There is an error to copy the data into the txt file'));
                                }

                                if (file_exists($directory.'temp.csv')) {
                                    @unlink($directory.'temp.csv');
                                }
                            }

                            if (!$with_errors) {
                                if (Tools::isSubmit('submitLanguageUpdateNoUrlTaxonomy')) {
                                    Configuration::updateGlobalValue('MAO_COMPARATOR_TAXONOMY_LANG', (int) $taxonomy_id_lang);
                                } else {
                                    Configuration::updateGlobalValue('MAO_COMPARATOR_TAXONOMY_COUNTRY', (int) $taxonomy_id_country);
                                }

                                $output .= $this->displayConfirmation($this->l('The file has been loaded correctly'));
                            }
                        }
                    } else {
                        $with_errors = true;

                        $output .= $this->displayError($this->l('There is an error to upload this file'));
                    }
                } else {
                    $with_errors = true;

                    $output .= $this->displayError($this->l('There is an error in the main php file of this comparator or to create its taxonomy folder'));
                }
            } elseif (Tools::isSubmit('submitLanguageTaxonomySelection') && Tools::getValue('name')) {
                $comparator_name = Tools::strtolower(Tools::getValue('name'));
                $taxonomy_id_lang = Tools::getValue('taxonomy_id_lang');

                Configuration::updateGlobalValue('MAO_COMPARATOR_TAXONOMY_LANG', (int) $taxonomy_id_lang);

                $output .= $this->displayConfirmation($this->l('Language selected. Now, from the next block, you can add or change some category associations'));
            } elseif (Tools::isSubmit('submitLanguageTaxonomyAssociation') && Tools::getValue('name')) {
                $comparator_name = Tools::strtolower(Tools::getValue('name'));
                $taxonomy_id_lang = (int) Tools::getValue('taxonomy_id_lang');
                $taxonomy_field = Tools::getValue('taxonomy_field');
                $taxonomy_form = Tools::getValue('taxonomy_form');
                $taxonomy_files_id = Tools::getValue('taxonomy_files_id');

                require_once dirname(__FILE__).'/comparators/'.$comparator_name.'/'.$comparator_name.'.php';

                $id_shop = (int) $this->context->shop->id;
                $id_root_cat = (int) Configuration::get('PS_ROOT_CATEGORY');
                $id_home_cat = (int) Configuration::get('PS_HOME_CATEGORY');
                $categories = Category::getSimpleCategories($taxonomy_id_lang);
                $comparator_obj = new $comparator_name();
                $taxonomy_language = new Language($taxonomy_id_lang);
                $taxonomy = $comparator_obj->readTaxonomyFile($taxonomy_language->language_code, $taxonomy_files_id);
                $taxonomy_files = $comparator_obj->taxonomy_files[$taxonomy_files_id];
                $taxonomy_type = $taxonomy_files['type'];

                foreach ($categories as $category) {
                    $id_category = (int) $category['id_category'];

                    if ($id_category == $id_root_cat || $id_category == $id_home_cat) {
                        continue;
                    }

                    $taxonomy_text = trim(Tools::getValue('inputString-'.$id_category));

                    if (
                        $taxonomy_text &&
                        (!Validate::isString($taxonomy_text) || $taxonomy_form != 'text' && !in_array($taxonomy_text, $taxonomy))
                    ) {
                        $with_errors = true;

                        $cat_name = MaofreeComparatorClass::getPath(
                            $id_category,
                            $id_shop,
                            $taxonomy_id_lang,
                            ' > ',
                            $id_root_cat,
                            $id_home_cat
                        );

                        $output .= $this->displayError($this->l('Taxonomy not correct for').': '.$cat_name);
                    } else {
                        MaofreeComparatorClass::updateTaxonomyData(
                            self::TAXONOMY_LANG_TABLE,
                            $id_shop,
                            $taxonomy_id_lang,
                            $id_category,
                            $comparator_name,
                            $taxonomy_type,
                            $taxonomy_field,
                            $taxonomy_text
                        );

                        if (!$taxonomy_text) {
                            $taxonomy_data = MaofreeComparatorClass::getAllTaxonomyData(
                                $id_shop,
                                $taxonomy_id_lang,
                                null,
                                $id_category,
                                $comparator_name,
                                $taxonomy_type
                            );

                            if (
                                is_array($taxonomy_data) &&
                                $taxonomy_data &&
                                !$taxonomy_data['main_taxonomy'] &&
                                !$taxonomy_data['secondary_taxonomy'] &&
                                !$taxonomy_data['extra_taxonomy']
                            ) {
                                MaofreeComparatorClass::removeTaxonomyData(
                                    self::TAXONOMY_LANG_TABLE,
                                    $id_shop,
                                    $taxonomy_id_lang,
                                    $id_category,
                                    $comparator_name,
                                    $taxonomy_type
                                );
                            }
                        }
                    }
                }

                if (!$with_errors) {
                    $lang = new Language((int) $taxonomy_id_lang);
                    $language_code = $lang->language_code;
                    $language_code_array = explode('-', $language_code);
                    $language_code = Tools::strtolower($language_code_array[0]).'-'.Tools::strtoupper($language_code_array[1]);

                    $output .= $this->displayConfirmation($this->l('Taxonomy updated for the language').' '.$language_code);
                }
            } elseif (Tools::isSubmit('submitCountryTaxonomySelection') && Tools::getValue('name')) {
                $comparator_name = Tools::strtolower(Tools::getValue('name'));
                $taxonomy_id_country = Tools::getValue('taxonomy_id_country');

                Configuration::updateGlobalValue('MAO_COMPARATOR_TAXONOMY_COUNTRY', (int) $taxonomy_id_country);

                $output .= $this->displayConfirmation($this->l('Country selected. Now, from the next block, you can add or change some category associations'));
            } elseif (Tools::isSubmit('submitCountryTaxonomyAssociation') && Tools::getValue('name')) {
                $comparator_name = Tools::strtolower(Tools::getValue('name'));
                $taxonomy_id_country = (int) Tools::getValue('taxonomy_id_country');
                $taxonomy_field = Tools::getValue('taxonomy_field');
                $taxonomy_form = Tools::getValue('taxonomy_form');
                $taxonomy_files_id = Tools::getValue('taxonomy_files_id');

                require_once dirname(__FILE__).'/comparators/'.$comparator_name.'/'.$comparator_name.'.php';

                $id_shop = (int) $this->context->shop->id;
                $id_root_cat = (int) Configuration::get('PS_ROOT_CATEGORY');
                $id_home_cat = (int) Configuration::get('PS_HOME_CATEGORY');
                $categories = Category::getSimpleCategories($id_lang);
                $comparator_obj = new $comparator_name();
                $taxonomy_country = new Country($taxonomy_id_country, $id_lang);
                $taxonomy = $comparator_obj->readTaxonomyFile($taxonomy_country->iso_code, $taxonomy_files_id);
                $taxonomy_files = $comparator_obj->taxonomy_files[$taxonomy_files_id];
                $taxonomy_type = $taxonomy_files['type'];

                foreach ($categories as $category) {
                    $id_category = (int) $category['id_category'];

                    if ($id_category == $id_root_cat || $id_category == $id_home_cat) {
                        continue;
                    }

                    $taxonomy_text = trim(Tools::getValue('inputString-'.$id_category));

                    if (
                        $taxonomy_text &&
                        (!Validate::isString($taxonomy_text) || $taxonomy_form != 'text' && !in_array($taxonomy_text, $taxonomy))
                    ) {
                        $with_errors = true;

                        $cat_name = MaofreeComparatorClass::getPath($id_category, $id_shop, $id_lang, ' > ', $id_root_cat, $id_home_cat);

                        $output .= $this->displayError($this->l('Taxonomy not correct for').': '.$cat_name);
                    } else {
                        MaofreeComparatorClass::updateTaxonomyData(
                            self::TAXONOMY_COUNTRY_TABLE,
                            $id_shop,
                            $taxonomy_id_country,
                            $id_category,
                            $comparator_name,
                            $taxonomy_type,
                            $taxonomy_field,
                            $taxonomy_text
                        );

                        if (!$taxonomy_text) {
                            $taxonomy_data = MaofreeComparatorClass::getAllTaxonomyData(
                                $id_shop,
                                null,
                                $taxonomy_id_country,
                                $id_category,
                                $comparator_name,
                                $taxonomy_type
                            );

                            if (
                                is_array($taxonomy_data) &&
                                $taxonomy_data &&
                                !$taxonomy_data['main_taxonomy'] &&
                                !$taxonomy_data['secondary_taxonomy'] &&
                                !$taxonomy_data['extra_taxonomy']
                            ) {
                                MaofreeComparatorClass::removeTaxonomyData(
                                    self::TAXONOMY_COUNTRY_TABLE,
                                    $id_shop,
                                    $taxonomy_id_country,
                                    $id_category,
                                    $comparator_name,
                                    $taxonomy_type
                                );
                            }
                        }
                    }
                }

                if (!$with_errors) {
                    $output .= $this->displayConfirmation($this->l('Taxonomy updated for the country').' = '.Tools::strtoupper($taxonomy_country->name));
                }
            } elseif (Tools::isSubmit('updateComparatorList')) {
                if (MaofreeComparatorClass::initializationMainTable(self::MAIN_TABLE, Shop::getContextListShopID())) {
                    $output .= $this->displayConfirmation($this->l('"mao_comparator" table updated.'));
                } else {
                    $output .= $this->displayError($this->l('There was an error while updating the comparators of "mao_comparator" table'));
                }

                $this->moduleClearCache();
            } elseif (Tools::isSubmit('submitFinalStep') && Tools::getValue('filename_updated')) {
                $file = Tools::getValue('filename_updated');
                $comparator_name_array = explode('-', $file);
                $comparator_name = Tools::strtolower($comparator_name_array[0]);

                if (Tools::getValue('filename_errors') != 'all_is_ok') {
                    $output .= $this->displayError($this->l('There was an error while updating of the file').' "'.$file.'"');
                } else {
                    $output .= $this->displayConfirmation($this->l('The file').' "'.$file.'" '.$this->l('has been processed correctly'));
                }
            } elseif (Tools::isSubmit('submitBulkdeleteTxtList') && Tools::getValue('TxtListBox')) {
                $files_checked = $comparator_name_array = array();
                $files_checked = Tools::getValue('TxtListBox');

                if (is_array($files_checked) && count($files_checked)) {
                    foreach ($files_checked as $file) {
                        $comparator_name_array = explode('-', $file);
                        $comparator_name = Tools::strtolower($comparator_name_array[0]);
                        $directory = _PS_MODULE_DIR_.$this->name.'/comparators/'.$comparator_name.'/txt/';

                        if (file_exists($directory.$file)) {
                            @unlink($directory.$file);
                        }
                    }

                    MaofreeComparatorClass::removeStaticFolder($comparator_name, 'txt');

                    $output .= $this->displayConfirmation($this->l('Deleted all selected files'));
                }
            } elseif (Tools::isSubmit('deleteTxtList') && Tools::getValue('filename')) {
                $file = Tools::getValue('filename');
                $comparator_name_array = explode('-', $file);
                $comparator_name = Tools::strtolower($comparator_name_array[0]);
                $directory = _PS_MODULE_DIR_.$this->name.'/comparators/'.$comparator_name.'/txt/';

                if (file_exists($directory.$file)) {
                    @unlink($directory.$file);
                }

                MaofreeComparatorClass::removeStaticFolder($comparator_name, 'txt');

                $output .= $this->displayConfirmation($this->l('The file').' "'.$file.'" '.$this->l('has been successfully deleted'));
            } elseif (Tools::isSubmit('submitBulkdeleteCsvList') && Tools::getValue('CsvListBox')) {
                $files_checked = $comparator_name_array = array();
                $files_checked = Tools::getValue('CsvListBox');

                if (is_array($files_checked) && count($files_checked)) {
                    foreach ($files_checked as $file) {
                        $comparator_name_array = explode('-', $file);
                        $comparator_name = Tools::strtolower($comparator_name_array[0]);
                        $directory = _PS_MODULE_DIR_.$this->name.'/comparators/'.$comparator_name.'/csv/';

                        if (file_exists($directory.$file)) {
                            @unlink($directory.$file);
                        }
                    }

                    MaofreeComparatorClass::removeStaticFolder($comparator_name, 'csv');

                    $output .= $this->displayConfirmation($this->l('Deleted all selected files'));
                }
            } elseif (Tools::isSubmit('deleteCsvList') && Tools::getValue('filename')) {
                $file = Tools::getValue('filename');
                $comparator_name_array = explode('-', $file);
                $comparator_name = Tools::strtolower($comparator_name_array[0]);
                $directory = _PS_MODULE_DIR_.$this->name.'/comparators/'.$comparator_name.'/csv/';

                if (file_exists($directory.$file)) {
                    @unlink($directory.$file);
                }

                MaofreeComparatorClass::removeStaticFolder($comparator_name, 'csv');

                $output .= $this->displayConfirmation($this->l('The file').' "'.$file.'" '.$this->l('has been successfully deleted'));
            } elseif (Tools::isSubmit('submitBulkdeleteXmlList') && Tools::getValue('XmlListBox')) {
                $files_checked = $comparator_name_array = array();
                $files_checked = Tools::getValue('XmlListBox');

                if (is_array($files_checked) && count($files_checked)) {
                    foreach ($files_checked as $file) {
                        $comparator_name_array = explode('-', $file);
                        $comparator_name = Tools::strtolower($comparator_name_array[0]);
                        $directory = _PS_MODULE_DIR_.$this->name.'/comparators/'.$comparator_name.'/xml/';

                        if (file_exists($directory.$file)) {
                            @unlink($directory.$file);
                        }
                    }

                    MaofreeComparatorClass::removeStaticFolder($comparator_name, 'xml');

                    $output .= $this->displayConfirmation($this->l('Deleted all selected files'));
                }
            } elseif (Tools::isSubmit('deleteXmlList') && Tools::getValue('filename')) {
                $file = Tools::getValue('filename');
                $comparator_name_array = explode('-', $file);
                $comparator_name = Tools::strtolower($comparator_name_array[0]);
                $directory = _PS_MODULE_DIR_.$this->name.'/comparators/'.$comparator_name.'/xml/';

                if (file_exists($directory.$file)) {
                    @unlink($directory.$file);
                }

                MaofreeComparatorClass::removeStaticFolder($comparator_name, 'xml');

                $output .= $this->displayConfirmation($this->l('The file').' "'.$file.'" '.$this->l('has been successfully deleted'));
            } elseif (Tools::isSubmit('submitBulkenableCronJobsList') && Tools::getValue('CronJobsListBox')) {
                $files_checked = Tools::getValue('CronJobsListBox');

                if (is_array($files_checked) && count($files_checked)) {
                    foreach ($files_checked as $id_cronjob) {
                        MaofreeComparatorClass::setStatusCronJob(
                            self::CRONJOBS_TABLE,
                            (int) $id_cronjob,
                            1
                        );
                    }

                    $output .= $this->displayConfirmation($this->l('Enabled all selected cronjobs'));
                }
            } elseif (Tools::isSubmit('submitBulkdisableCronJobsList') && Tools::getValue('CronJobsListBox')) {
                $files_checked = Tools::getValue('CronJobsListBox');

                if (is_array($files_checked) && count($files_checked)) {
                    foreach ($files_checked as $id_cronjob) {
                        MaofreeComparatorClass::setStatusCronJob(
                            self::CRONJOBS_TABLE,
                            (int) $id_cronjob,
                            0
                        );
                    }

                    $output .= $this->displayConfirmation($this->l('Disabled all selected cronjobs'));
                }
            } elseif (Tools::isSubmit('submitBulkdeleteCronJobsList') && Tools::getValue('CronJobsListBox')) {
                $files_checked = Tools::getValue('CronJobsListBox');

                if (is_array($files_checked) && count($files_checked)) {
                    foreach ($files_checked as $id_cronjob) {
                        MaofreeComparatorClass::removeCronJob(
                            self::CRONJOBS_TABLE,
                            (int) $id_cronjob
                        );
                    }

                    $output .= $this->displayConfirmation($this->l('Deleted all selected cronjobs'));
                }
            } elseif (Tools::isSubmit('removeDataAndFiles')) {
                $success = (
                    MaofreeComparatorClass::removeTable(self::MAIN_TABLE) &&
                    MaofreeComparatorClass::createMainTable(self::MAIN_TABLE) &&
                    MaofreeComparatorClass::initializationMainTable(self::MAIN_TABLE, Shop::getContextListShopID())
                );

                if ($success) {
                    $dir_to_remove = array('taxonomy', 'csv', 'txt', 'xml');
                    $comparators_on_table = MaofreeComparatorClass::getComparators(self::MAIN_TABLE, $id_shop);

                    if (is_array($comparators_on_table) && count($comparators_on_table)) {
                        foreach ($comparators_on_table as $comparator) {
                            $comparator_name = $comparator['name'];

                            require_once dirname(__FILE__).'/comparators/'.$comparator_name.'/'.$comparator_name.'.php';

                            $comparator_obj = new $comparator_name();

                            foreach ($dir_to_remove as $dir) {
                                $directory = _PS_MODULE_DIR_.$this->name.'/comparators/'.$comparator['name'].'/'.$dir;

                                if (file_exists($directory)) {
                                    if ($dir != 'taxonomy') {
                                        if (!Tools::deleteDirectory($directory)) {
                                            $with_errors = true;

                                            $output .= $this->displayError($this->l('There was an error while removing of the directory').': /comparators/'.$comparator['name'].'/'.$dir);
                                        }
                                    } elseif (
                                        isset($comparator_obj->taxonomy) &&
                                        $comparator_obj->taxonomy &&
                                        isset($comparator_obj->taxonomy_files)
                                    ) {
                                        $scanned_directory = array_diff(scandir($directory), array('..', '.', 'index.php'));

                                        if (is_array($scanned_directory) && count($scanned_directory)) {
                                            $file_removed = 0;
                                            $taxonomy_files = $comparator_obj->taxonomy_files;

                                            foreach ($taxonomy_files as $row) {
                                                if (isset($row['filename']) && $row['filename'] && isset($row['url']) && $row['url']) {
                                                    foreach ($scanned_directory as $file) {
                                                        if (strpos($file, $row['filename']) !== false && !unlink($directory.'/'.$file)) {
                                                            $with_errors = true;

                                                            $output .= $this->displayError($this->l('There was an error while removing of the file').': /comparators/'.$comparator['name'].'/'.$dir.'/'.$file);
                                                        } else {
                                                            ++$file_removed;
                                                        }
                                                    }
                                                }
                                            }

                                            // scan again to see if the folder is void to remove it
                                            if ($file_removed) {
                                                $scanned_directory = array_diff(scandir($directory), array('..', '.', 'index.php'));
                                            }
                                        }

                                        if (is_array($scanned_directory) && !count($scanned_directory)) {
                                            if (!Tools::deleteDirectory($directory)) {
                                                $with_errors = true;

                                                $output .= $this->displayError($this->l('There was an error while removing of the directory').': /comparators/'.$comparator['name'].'/'.$dir);
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                } else {
                    $output .= $this->displayError($this->l('There was an error while removing of the tables'));
                }

                $this->moduleClearCache();

                if (!$with_errors) {
                    $output .= $this->displayConfirmation($this->l('Data and files of all shops have been successfully deleted'));
                }
            }

            if (!$comparator_name && Tools::getValue('name')) {
                $comparator_name = Tools::strtolower(Tools::getValue('name'));
            }

            if (
                Tools::isSubmit('updateGeneralSettings') ||
                Tools::isSubmit('submitGeneralSettings') ||
                Tools::isSubmit('submitCarriersAssociation') ||
                Tools::isSubmit('submitBulkenableCronJobsList') ||
                Tools::isSubmit('submitBulkdisableCronJobsList') ||
                Tools::isSubmit('submitBulkdeleteCronJobsList') ||
                Tools::isSubmit('submitAddNewFeedToCronJob')
            ) {
                $output .= $this->renderSettingsForm();

                if (Configuration::getGlobalValue('MAO_COMPARATOR_ONECRONJOB')) {
                    $output .= $this->renderCronJobForm();
                    $output .= $this->renderCronJobList();
                }
            } elseif (
                (
                    Tools::isSubmit('submitComparatorEnable') ||
                    Tools::isSubmit('viewComparatorList') ||
                    Tools::isSubmit('updateUrlTaxonomyFiles') ||
                    Tools::isSubmit('activeTaxonomyAssociation') ||
                    Tools::isSubmit('submitBulkdeleteTxtList') ||
                    Tools::isSubmit('deleteTxtList') ||
                    Tools::isSubmit('submitBulkdeleteCsvList') ||
                    Tools::isSubmit('deleteCsvList') ||
                    Tools::isSubmit('submitBulkdeleteXmlList') ||
                    Tools::isSubmit('deleteXmlList') ||
                    Tools::isSubmit('submitFinalStep')
                ) &&
                $comparator_name
            ) {
                require_once dirname(__FILE__).'/comparators/'.$comparator_name.'/'.$comparator_name.'.php';

                if (isset($this->context->cookie->{'ComparatorProducts_page'})) {
                    unset($this->context->cookie->{'ComparatorProducts_page'});
                }

                if (isset($this->context->cookie->{'ComparatorProducts_pagination'})) {
                    unset($this->context->cookie->{'ComparatorProducts_pagination'});
                }

                $comparator = MaofreeComparatorClass::getValuesFromTable(self::MAIN_TABLE, $id_shop, $comparator_name);
                $comparator_obj = new $comparator_name();

                $output .= '<h2 style="color: #FF8C00; font-weight: bold; text-transform: uppercase;">* '.Tools::strtoupper($comparator_name).' *</h2>';
                $output .= $this->displayComparatorErrors($comparator);

                if (
                    isset($comparator_obj->company_abbreviation) &&
                    $comparator_obj->company_abbreviation &&
                    !$settings['company_abbreviation']
                ) {
                    $output .= $this->displayAlertCompanyAbbr();
                }

                if (
                    isset($comparator_obj->company_description) &&
                    $comparator_obj->company_description &&
                    (
                        Configuration::get('MAO_COMPARATOR_COMPANYDESC', $id_lang) === false ||
                        !Configuration::get('MAO_COMPARATOR_COMPANYDESC', $id_lang)
                    )
                ) {
                    $output .= $this->displayAlertCompanyDesc();
                }

                $output .= $this->renderComparatorEnableForm($comparator);
                $output .= $this->renderComparatorSettingsList($comparator);
                $output .= $this->renderComparatorSelectionList($comparator);

                if (
                    isset($comparator_obj->taxonomy) &&
                    $comparator_obj->taxonomy &&
                    isset($comparator_obj->taxonomy_files)
                ) {
                    $output .= $this->renderTaxonomyAssociationList($comparator);
                }

                $output .= $this->displayComparatorDatafeedLink($comparator);

                if (
                    isset($comparator_obj->datafeed_type) &&
                    is_array($comparator_obj->datafeed_type)
                ) {
                    if (in_array('html', $comparator_obj->datafeed_type)) {
                        $output .= $this->displayDynamicLastCheck($comparator);
                    }

                    foreach ($comparator_obj->datafeed_type as $datafeed_type) {
                        if ($datafeed_type == 'txt') {
                            $output .= $this->renderTxtList($comparator);
                        } elseif ($datafeed_type == 'csv') {
                            $output .= $this->renderCsvList($comparator);
                        } elseif ($datafeed_type == 'xml') {
                            $output .= $this->renderXmlList($comparator);
                        }
                    }
                }
            } elseif (
                (
                    Tools::isSubmit('updateComparatorSettings') ||
                    Tools::isSubmit('submitComparatorSettings') ||
                    Tools::isSubmit('submitAnalyticsComparatorSettings') ||
                    Tools::isSubmit('submitUrlComparatorSettings')
                ) &&
                $comparator_name
            ) {
                $output .= $this->displayIntroInfo();
                $output .= $this->renderComparatorSettingsForm();
            } elseif (
                (
                    Tools::isSubmit('updateComparatorSelection') ||
                    Tools::isSubmit('submitComparatorSelection') ||
                    Tools::isSubmit('submitComparatorProducts') ||
                    Tools::isSubmit('submitComparatorProductsPagination')
                ) &&
                $comparator_name
            ) {
                $comparator = MaofreeComparatorClass::getValuesFromTable(self::MAIN_TABLE, $id_shop, $comparator_name);

                $output .= $this->renderComparatorSelectionForm();
                $output .= $this->displayComparatorProductsList($comparator);
            } elseif (
                (
                    Tools::isSubmit('updateLanguageNoUrlTaxonomyFiles') ||
                    Tools::isSubmit('submitLanguageUpdateNoUrlTaxonomy') && $with_errors
                ) &&
                $comparator_name) {
                require_once dirname(__FILE__).'/comparators/'.$comparator_name.'/'.$comparator_name.'.php';

                $comparator = MaofreeComparatorClass::getValuesFromTable(self::MAIN_TABLE, $id_shop, $comparator_name);

                $output .= $this->displayLanguageUpdateNoUrlTaxonomyFilesForm($comparator);
            } elseif (
                (
                    Tools::isSubmit('updateCountryNoUrlTaxonomyFiles') ||
                    Tools::isSubmit('submitCountryUpdateNoUrlTaxonomy') && $with_errors
                ) &&
                $comparator_name) {
                require_once dirname(__FILE__).'/comparators/'.$comparator_name.'/'.$comparator_name.'.php';

                $comparator = MaofreeComparatorClass::getValuesFromTable(self::MAIN_TABLE, $id_shop, $comparator_name);

                $output .= $this->displayCountryUpdateNoUrlTaxonomyFilesForm($comparator);
            } elseif (
                (
                    Tools::isSubmit('updateTaxonomyAssociation') ||
                    Tools::isSubmit('submitLanguageTaxonomySelection') ||
                    Tools::isSubmit('submitLanguageTaxonomyAssociation') ||
                    Tools::isSubmit('submitCountryTaxonomySelection') ||
                    Tools::isSubmit('submitCountryTaxonomyAssociation') ||
                    Tools::isSubmit('submitLanguageUpdateNoUrlTaxonomy') ||
                    Tools::isSubmit('submitCountryUpdateNoUrlTaxonomy')
                ) &&
                $comparator_name
            ) {
                require_once dirname(__FILE__).'/comparators/'.$comparator_name.'/'.$comparator_name.'.php';

                $comparator = MaofreeComparatorClass::getValuesFromTable(self::MAIN_TABLE, $id_shop, $comparator_name);
                $comparator_obj = new $comparator_name();
                $taxonomy_files_id = (int) Tools::getValue('taxonomy_files_id');
                $taxonomy_files = $comparator_obj->taxonomy_files[$taxonomy_files_id];

                if (
                    !isset($taxonomy_files['form']) ||
                    !isset($taxonomy_files['field']) ||
                    !isset($taxonomy_files['type'])
                ) {
                    $output .= $this->displayError(sprintf($this->l('Some arguments are missing in the variable %s of this comparator'), '"$taxonomy_files"'));
                } else {
                    $file_is_present = false;
                    MaofreeComparatorClass::addTaxonomyFiles($comparator_name, dirname(__FILE__).'/comparators', $taxonomy_files_id);

                    $scanned_directory = array_diff(scandir(dirname(__FILE__).'/comparators'.'/'.$comparator_name.'/taxonomy'), array('..', '.', 'index.php'));

                    if ($taxonomy_files['type'] == 'language') {
                        $taxonomy_language = new Language((int) Configuration::getGlobalValue('MAO_COMPARATOR_TAXONOMY_LANG'));
                        $taxonomy_iso_code_array = explode('-', $taxonomy_language->language_code);
                        $taxonomy_iso_code = Tools::strtolower($taxonomy_iso_code_array[0]).'-'.Tools::strtoupper($taxonomy_iso_code_array[1]);

                        if (is_array($scanned_directory) && count($scanned_directory)) {
                            foreach ($scanned_directory as $file) {
                                if (
                                    strpos($file, $taxonomy_files['filename']) !== false &&
                                    $comparator_obj->bestTaxonomyIsoCode($taxonomy_iso_code, $taxonomy_files['type']) &&
                                    strpos($file, $comparator_obj->bestTaxonomyIsoCode($taxonomy_iso_code, $taxonomy_files['type'])) !== false
                                ) {
                                    $file_is_present = true;
                                }
                            }
                        }

                        $output .= $this->displayLanguageTaxonomySelectionForm($comparator);

                        if ($taxonomy_files['form'] == 'text' || $taxonomy_files['filename'] && $file_is_present) {
                            $output .= $this->displayLanguageSummaryTaxonomyAssociation($comparator, $taxonomy_files);
                            $output .= $this->displayLanguageTaxonomyAssociationForm($comparator, $taxonomy_files);
                        } elseif ($comparator_obj->bestTaxonomyIsoCode($taxonomy_iso_code, $taxonomy_files['type'])) {
                            $output .= $this->displayAlertFileNoPresent($taxonomy_files['type']);
                            $output .= $this->displayLanguageUpdateNoUrlTaxonomyFilesForm($comparator);
                        }
                    } else {
                        $taxonomy_country = new Country((int) Configuration::getGlobalValue('MAO_COMPARATOR_TAXONOMY_COUNTRY'), $id_lang);

                        if (is_array($scanned_directory) && count($scanned_directory)) {
                            foreach ($scanned_directory as $file) {
                                if (
                                    strpos($file, $taxonomy_files['filename']) !== false &&
                                    $comparator_obj->bestTaxonomyIsoCode($taxonomy_country->iso_code, $taxonomy_files['type']) &&
                                    strpos($file, $comparator_obj->bestTaxonomyIsoCode($taxonomy_country->iso_code, $taxonomy_files['type'])) !== false
                                ) {
                                    $file_is_present = true;
                                }
                            }
                        }

                        $output .= $this->displayCountryTaxonomySelectionForm($comparator);

                        if ($taxonomy_files['form'] == 'text' || $taxonomy_files['filename'] && $file_is_present) {
                            $output .= $this->displayCountrySummaryTaxonomyAssociation($comparator, $taxonomy_files);
                            $output .= $this->displayCountryTaxonomyAssociationForm($comparator, $taxonomy_files);
                        } elseif ($comparator_obj->bestTaxonomyIsoCode($taxonomy_country->iso_code, $taxonomy_files['type'])) {
                            $output .= $this->displayAlertFileNoPresent($taxonomy_files['type']);
                            $output .= $this->displayCountryUpdateNoUrlTaxonomyFilesForm($comparator);
                        }
                    }
                }
            } else {
                $output .= $this->displayUpdateTable();
                $output .= $this->renderSettingsList($settings);
                $output .= $this->renderComparatorsList();
            }

            return $output;
        }
    }

    private function maofreeLogo()
    {
        $output = '
        <style type="text/css">
            #mylogo { margin-bottom: 10px; background: none; border: none }
            #mylogo a img { padding: 1px; border: 1px solid skyBlue; }
        </style>

        <script type="text/javascript">
            var mao_ajax_url = "'.$this->context->link->getAdminLink('AdminModules').'&configure='.$this->name.'";
        </script>

        <fieldset id="mylogo">
            <a href="http://www.maofree-developer.com" target="_blank">
                <img src="'.$this->_path.'views/img/mylogo.png" alt="maofree\'s website" title="'.$this->l('Do you need some help or other modules? Visit my website (click here)').'" />
            </a>
        </fieldset>';

        return $output;
    }

    private function displayIntroInfo()
    {
        return $this->display(__FILE__, 'mao_intro_info.tpl');
    }

    private function displayAlertCompanyAbbr()
    {
        return $this->display(__FILE__, 'mao_alert_companyabbr.tpl');
    }

    private function displayAlertCompanyDesc()
    {
        return $this->display(__FILE__, 'mao_alert_companydesc.tpl');
    }

    private function displayUpdateTable()
    {
        $this->smarty->assign(
            'mao_comparator_action_url',
            AdminController::$currentIndex.'&configure='.$this->name.'&updateComparatorList'.'&token='.Tools::getAdminTokenLite('AdminModules')
        );

        return $this->display(__FILE__, 'mao_update_table.tpl');
    }

    private function displayAlertFileNoPresent($type)
    {
        if ($type == 'language') {
            return $this->displayError($this->l('The taxonomy file is not present for this language').'. '.$this->l('Require the right file to this comparator (ask to its employees)').'. '.$this->l('Then upload it from your pc').'.');
        } else {
            return $this->displayError($this->l('The taxonomy file is not present for this country').'. '.$this->l('Require the right file to this comparator (ask to its employees)').'. '.$this->l('Then upload it from your pc').'.');
        }
    }

    private function renderSettingsList($settings)
    {
        $generalsettings = array();
        $link_position_text = '';
        $description = $settings['description'];
        $hook_on_left = $this->isRegisteredInHook('displayLeftColumn');
        $hook_on_right = $this->isRegisteredInHook('displayRightColumn');
        $hook_on_home = $this->isRegisteredInHook('displayHome');
        $hook_on_footer = $this->isRegisteredInHook('displayFooter');

        if ($hook_on_left || $hook_on_right) {
            $link_position_text .= $this->l('Side block').', ';
        }
        if ($hook_on_home) {
            $link_position_text .= $this->l('Home block').', ';
        }
        if ($hook_on_footer) {
            $link_position_text .= $this->l('Footer').', ';
        }

        if ($link_position_text) {
            $link_position_text = Tools::substr($link_position_text, 0, -2);
        } else {
            $link_position_text = $this->l('Nowhere');
        }

        if ($description == 'short') {
            $description_text = $this->l('Short description');
        } elseif ($description == 'long') {
            $description_text = $this->l('Long description');
        } else {
            $description_text = $this->l('Short + long description');
        }

        $generalsettings[] = array(
            'name' => $this->name,
            'link_position' => $link_position_text,
            'description' => $description_text,
            'products_cycle' => $settings['products_cycle'],
            'company_abbreviation' => $settings['company_abbreviation'],
            'company_description' => (Configuration::get('MAO_COMPARATOR_COMPANYDESC', (int) $this->context->language->id) === false || !Configuration::get('MAO_COMPARATOR_COMPANYDESC', (int) $this->context->language->id)) ? '--' : Configuration::get('MAO_COMPARATOR_COMPANYDESC', (int) $this->context->language->id),
            'carriers_zone' => Configuration::get('MAO_COMPARATOR_CARRIERSZONE') ? 1 : 0,
            'onecronjob' => (int) Configuration::getGlobalValue('MAO_COMPARATOR_ONECRONJOB'),
            'removetable' => $settings['removetable'],
        );

        $fields_list = array(
            'link_position' => array(
                'title' => $this->l('Links position'),
                'type' => 'text',
                'align' => 'center',
            ),
            'description' => array(
                'title' => $this->l('Description type'),
                'type' => 'text',
                'align' => 'center',
            ),
            'products_cycle' => array(
                'title' => $this->l('Products per cycle'),
                'type' => 'text',
                'align' => 'center',
            ),
            'company_abbreviation' => array(
                'title' => $this->l('Company\'s abbreviation'),
                'type' => 'text',
                'align' => 'center',
            ),
            'company_description' => array(
                'title' => $this->l('Company\'s description'),
                'type' => 'text',
                'align' => 'center',
            ),
            'carriers_zone' => array(
                'title' => $this->l('Carriers association'),
                'type' => 'action_icon',
                'align' => 'center',
            ),
            'onecronjob' => array(
                'title' => $this->l('Global Cron Job'),
                'type' => 'action_icon',
                'align' => 'center',
            ),
            'removetable' => array(
                'title' => $this->l('Remove table'),
                'type' => 'action_icon',
                'align' => 'center',
            ),
        );

        $helper = new HelperList();
        $helper->shopLinkType = '';
        $helper->simple_header = true;
        $helper->identifier = 'name';
        $helper->actions = array('edit');
        $helper->table = 'GeneralSettings';
        $helper->imageType = 'jpg';
        $helper->show_toolbar = false;
        $helper->module = $this;
        $helper->title = '<i class="icon-cogs"></i> <span style="color: royalblue">'.$this->l('General Settings').'</span>, <span style="color: orangered">'.$this->l('Carriers Association').'</span> '.$this->l('and').' <span style="color: green">'.$this->l('Global Cron Job').'</span>';
        $helper->token = Tools::getAdminTokenLite('AdminModules');
        $helper->currentIndex = AdminController::$currentIndex.'&configure='.$this->name;

        return $helper->generateList($generalsettings, $fields_list);
    }

    private function renderSettingsForm()
    {
        $fields = array();

        $positions = array(
            array('id_position' => 'side', 'position' => $this->l('Side block')),
            array('id_position' => 'home', 'position' => $this->l('Home block')),
            array('id_position' => 'footer', 'position' => $this->l('Footer')),
        );

        $hooks = array(
            array('id_hook' => 'displayLeftColumn', 'hook' => 'displayLeftColumn'),
            array('id_hook' => 'displayRightColumn', 'hook' => 'displayRightColumn'),
            array('id_hook' => 'both', 'hook' => $this->l('On both sides')),
        );

        $descriptions = array(
            array('id_description' => 'short', 'description' => $this->l('Short description')),
            array('id_description' => 'long', 'description' => $this->l('Long description')),
            array('id_description' => 'full', 'description' => $this->l('Short + long description')),
        );

        $fields_form_1 = array(
            'form' => array(
                'legend' => array(
                    'title' => $this->l('General Settings'),
                    'icon' => 'icon-cogs',
                ),
                'input' => array(
                    array(
                        'type' => 'select',
                        'label' => $this->l('Position of comparator links').':',
                        'name' => 'link_position[]',
                        'id' => 'positions',
                        'desc' => $this->l('Here you can decide where to show, on your website, the links of the comparators. Normally they request to add a link that redirects to their homepage'),
                        'multiple' => true,
                        'options' => array(
                            'query' => $positions,
                            'id' => 'id_position',
                            'name' => 'position',
                        ),
                    ),
                    array(
                        'type' => 'select',
                        'label' => $this->l('Position of the side block').':',
                        'name' => 'hookposition',
                        'desc' => $this->l('If you choose the "On both sides" option, remember not to show both blocks in a page. This option can be useful for example when you have the left side on category pages and the right side on product pages. In this way you will increase the visibility of this block on more pages'),
                        'options' => array(
                            'query' => $hooks,
                            'id' => 'id_hook',
                            'name' => 'hook',
                        ),
                    ),
                    array(
                        'type' => 'select',
                        'label' => $this->l('Description type').':',
                        'name' => 'description',
                        'desc' => $this->l('The descriptions are related to the product descriptions'),
                        'options' => array(
                            'query' => $descriptions,
                            'id' => 'id_description',
                            'name' => 'description',
                        ),
                    ),
                    array(
                        'type' => 'text',
                        'label' => $this->l('Products processed at each cycle').':',
                        'name' => 'products_cycle',
                        'autocomplete' => false,
                        'class' => 'fixed-width-xl',
                        'desc' => $this->l('You can decrease this value if you got a timeout error of the server, otherwise you can leave as is. It is used only to make static files'),
                        'suffix' => '<i class="icon-retweet"></i>',
                    ),
                    array(
                        'type' => 'text',
                        'label' => $this->l('Company\'s abbreviation').':',
                        'name' => 'company_abbreviation',
                        'autocomplete' => false,
                        'class' => 'fixed-width-xl',
                        'desc' => $this->l('For some comparators, like google shopping, it is necessary to set an abbreviation of your company that will be used to make unique each identifier. Use only 2-5 letters'),
                        'suffix' => '<i class="icon-home"></i>',
                    ),
                    array(
                        'type' => 'textarea',
                        'label' => $this->l('Company\'s description').':',
                        'name' => 'company_description',
                        'desc' => $this->l('Some comparators like Google Shopping require a little description about your company for its xml data feed'),
                        'lang' => true,
                        'cols' => 40,
                        'rows' => 10,
                    ),
                    array(
                        'type' => 'switch',
                        'is_bool' => true,
                        'label' => $this->l('Only a single Cron Job').':',
                        'name' => 'onecronjob',
                        'desc' => $this->l('All static files will be created only by one Cron Job simplifying their management. Scroll down this page to configure it'),
                        'values' => array(
                            array(
                                'id' => 'active_on',
                                'value' => 1,
                                'label' => $this->l('Enabled'),
                            ),
                            array(
                                'id' => 'active_off',
                                'value' => 0,
                                'label' => $this->l('Disabled'),
                            ),
                        ),
                    ),
                    array(
                        'type' => 'switch',
                        'is_bool' => true,
                        'label' => $this->l('Remove table').':',
                        'name' => 'removetable',
                        'desc' => $this->l('Uninstalling this module, the table "mao_comparator" will be removed with its data'),
                        'values' => array(
                            array(
                                'id' => 'active_on',
                                'value' => 1,
                                'label' => $this->l('Enabled'),
                            ),
                            array(
                                'id' => 'active_off',
                                'value' => 0,
                                'label' => $this->l('Disabled'),
                            ),
                        ),
                    ),
                ),
                'submit' => array(
                    'title' => $this->l('Save'),
                    'class' => 'btn btn-default pull-right',
                    'name' => 'submitGeneralSettings',
                ),
                'buttons' => array(
                    array(
                        'href' => AdminController::$currentIndex.'&configure='.$this->name.'&token='.Tools::getAdminTokenLite('AdminModules'),
                        'title' => $this->l('Back'),
                        'icon' => 'process-icon-back',
                    ),
                ),
            ),
        );

        foreach (Zone::getZones(true) as $zone) {
            $carriers = array();
            $id_zone = (int) $zone['id_zone'];
            $carriers[] = array('id_carrier' => 0, 'carrier' => '--');

            foreach (Carrier::getCarriers((int) $this->context->language->id, true, false, $id_zone, null, ALL_CARRIERS) as $carrier) {
                $carriers[] = array('id_carrier' => (int) $carrier['id_carrier'], 'carrier' => $carrier['name']);
            }

            $fields[] =
                array(
                    'type' => 'select',
                    'label' => $zone['name'].':',
                    'name' => 'carrier_zone_'.$id_zone,
                    'class' => 'fixed-width-xxl',
                    'options' => array(
                        'query' => $carriers,
                        'id' => 'id_carrier',
                        'name' => 'carrier',
                    ),
                );
        }

        $fields_form_2 = array(
            'form' => array(
                'legend' => array(
                    'title' => $this->l('Carriers Association by Zone'),
                    'icon' => 'icon-truck',
                ),
                'description' => $this->l('From here you can associate the carriers for each used zones'),
                'input' => $fields,
                'submit' => array(
                    'title' => $this->l('Save'),
                    'class' => 'btn btn-default pull-right',
                    'name' => 'submitCarriersAssociation',
                ),
                'buttons' => array(
                    array(
                        'href' => AdminController::$currentIndex.'&configure='.$this->name.'&token='.Tools::getAdminTokenLite('AdminModules'),
                        'title' => $this->l('Back'),
                        'icon' => 'process-icon-back',
                    ),
                ),
            ),
        );

        $helper = new HelperForm();
        $helper->name_controller = $this->name;
        $lang = new Language((int) Configuration::get('PS_LANG_DEFAULT'));
        $helper->default_form_language = $lang->id;
        $helper->module = $this;
        $helper->toolbar_scroll = true;
        $helper->allow_employee_form_lang = Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG') ? Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG') : 0;
        $helper->identifier = $this->identifier;
        $helper->currentIndex = AdminController::$currentIndex.'&configure='.$this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');
        $helper->tpl_vars = array(
            'fields_value' => $this->getConfigFieldsValues1(),
            'languages' => $this->context->controller->getLanguages(),
            'id_language' => $this->context->language->id,
        );

        return $helper->generateForm(array($fields_form_1, $fields_form_2));
    }

    private function getConfigFieldsValues1()
    {
        $carriers_zone_array = $carrier_zone = array();
        $settings = unserialize(Configuration::get('MAO_COMPARATOR_SETTINGS'));

        if (Configuration::get('MAO_COMPARATOR_CARRIERSZONE')) {
            $carriers_zone_array = explode(',', Configuration::get('MAO_COMPARATOR_CARRIERSZONE'));

            foreach ($carriers_zone_array as $row) {
                $zone_array = explode('#', $row);

                $carrier_zone[$zone_array[0]] = $zone_array[1];
            }
        }

        $link_position = array();
        $hook_on_left = $this->isRegisteredInHook('displayLeftColumn');
        $hook_on_right = $this->isRegisteredInHook('displayRightColumn');
        $hook_on_home = $this->isRegisteredInHook('displayHome');
        $hook_on_footer = $this->isRegisteredInHook('displayFooter');

        if ($hook_on_left && $hook_on_right) {
            $hookposition = 'both';
        } elseif ($hook_on_left) {
            $hookposition = 'displayLeftColumn';
        } else {
            $hookposition = 'displayRightColumn';
        }

        if ($hook_on_left || $hook_on_right) {
            $link_position[] = 'side';
        }
        if ($hook_on_home) {
            $link_position[] = 'home';
        }
        if ($hook_on_footer) {
            $link_position[] = 'footer';
        }

        $fields = array();
        $languages = Language::getLanguages(false);

        $fields = array(
            'hookposition' => $hookposition,
            'link_position[]' => $link_position,
            'description' => Tools::getValue('description', $settings['description']),
            'products_cycle' => Tools::getValue('products_cycle', $settings['products_cycle']),
            'company_abbreviation' => Tools::getValue('company_abbreviation', $settings['company_abbreviation']),
            'onecronjob' => Tools::getValue('onecronjob', (int) Configuration::getGlobalValue('MAO_COMPARATOR_ONECRONJOB')),
            'removetable' => Tools::getValue('removetable', $settings['removetable']),
        );

        foreach ($languages as $lang) {
            $fields['company_description'][$lang['id_lang']] = Tools::getValue(
                'company_description_'.$lang['id_lang'],
                Configuration::get('MAO_COMPARATOR_COMPANYDESC', $lang['id_lang'])
            );
        }

        foreach (Zone::getZones(true) as $zone) {
            $id_zone = (int) $zone['id_zone'];

            $fields['carrier_zone_'.$id_zone] = Tools::getValue(
                'carrier_zone_'.$id_zone,
                count($carrier_zone) && isset($carrier_zone[$id_zone]) && $carrier_zone[$id_zone] ? $carrier_zone[$id_zone] : false
            );
        }

        return $fields;
    }

    private function renderCronJobForm()
    {
        $id_shop = (int) $this->context->shop->id;
        $languages = $currencies = $countries = array();
        $comparators = MaofreeComparatorClass::getComparatorName(self::MAIN_TABLE, $id_shop);
        $default_datafeed_url = Tools::getShopDomainSsl(true, true).$this->_path.'cron/cron.php?token='.$this->token;
        $default_datafeed_crontab = dirname(__FILE__).'/cron/cron.php?token='.$this->token;

        if (is_array($comparators) && count($comparators)) {
            foreach ($comparators as $key => $row) {
                $comparators[$key]['up_name'] = Tools::strtoupper($row['name']);
            }
        }

        foreach (Currency::getCurrencies(false, true) as $currency) {
            $currencies[] = array('id_currency' => $currency['id_currency'], 'currency' => $currency['name']);
        }

        foreach (Language::getLanguages(true) as $lang) {
            $languages[] = array('id_language' => $lang['id_lang'], 'language' => $lang['name']);
        }

        foreach (Country::getCountries((int) $this->context->language->id, true) as $country) {
            $countries[] = array('id_country' => $country['id_country'], 'country' => $country['name']);
        }

        $types = array(
            array('id_type' => 'txt', 'type' => 'TXT'),
            array('id_type' => 'csv', 'type' => 'CSV'),
            array('id_type' => 'xml', 'type' => 'XML'),
        );

        $description = '
			<h4>'.$this->l('Informations').' :</h4>
			<ul style="padding-left: 10px; padding-bottom: 5px">
				<li>'.$this->l('From here you can add all wanted data feed that will be managed from a single Cron Job, according to the settings of each comparator').'.</li>
				<li>'.$this->l('Remember that each combination corresponds to a single file').'.</li>
				<li>'.$this->l('The data saved in this form will be added in the table here below').'.</li>
			</ul>
            <br />
            <h4>* '.$this->l('URL to execute a Cron Job task, activatable from the control panel of your hosting (scheduled task)').':</h4>
            <p>'.$this->l('To keep the data updated, you need to add a new web Cron Job task, inserting the green URL here below, to the scheduled task page of your hosting, then set the frequency you want, possibly at night').'.</p><br />
            <span style="color: green; font-weight: bold;">'.$default_datafeed_url.'</span><br /><br />
            <h4>* '.$this->l('In the field of scheduled tasks (Web Cron Job) on your cPanel or Plesk not use the "CURL" command, but "WGET", as shown below').':</h4>
            <em style="color: orangered;">wget -O /dev/null --no-check-certificate --max-redirect=100000 "<b>'.$this->l('put here the green URL written above within these quotes').'</b>"</em><br /><br />
            <h4 style="color: silver;">* '.$this->l('Instruction to execute a cron from crontab of your server every day at 03:00 (only for expert)').':</h4>
            <span class="default_datafeed_crontab" style="color: silver; font-weight: bold;">0 3 * * * php -d '.$default_datafeed_crontab.'</span>';

        $fields_form_1 = array(
            'form' => array(
                'legend' => array(
                    'title' => $this->l('New data feed nanaged by a single Cron Job'),
                    'icon' => 'icon-calendar',
                ),
                'description' => $description,
                'input' => array(
                    array(
                        'type' => 'select',
                        'label' => $this->l('Active comparator').':',
                        'name' => 'feed_name',
                        'options' => array(
                            'query' => $comparators,
                            'id' => 'name',
                            'name' => 'up_name',
                        ),
                    ),
                    array(
                        'type' => 'select',
                        'label' => $this->l('Currency').':',
                        'name' => 'feed_currency',
                        'options' => array(
                            'query' => $currencies,
                            'id' => 'id_currency',
                            'name' => 'currency',
                        ),
                    ),
                    array(
                        'type' => 'select',
                        'label' => $this->l('Language').':',
                        'name' => 'feed_language',
                        'options' => array(
                            'query' => $languages,
                            'id' => 'id_language',
                            'name' => 'language',
                        ),
                    ),
                    array(
                        'type' => 'select',
                        'label' => $this->l('Country').':',
                        'name' => 'feed_country',
                        'options' => array(
                            'query' => $countries,
                            'id' => 'id_country',
                            'name' => 'country',
                        ),
                    ),
                    array(
                        'type' => 'select',
                        'label' => $this->l('File extension').':',
                        'name' => 'feed_type',
                        'options' => array(
                            'query' => $types,
                            'id' => 'id_type',
                            'name' => 'type',
                        ),
                    ),
                    array(
                        'type' => 'switch',
                        'is_bool' => true,
                        'label' => $this->l('Tax included').':',
                        'name' => 'feed_tax',
                        'values' => array(
                            array(
                                'id' => 'active_on',
                                'value' => 1,
                                'label' => $this->l('Enabled'),
                            ),
                            array(
                                'id' => 'active_off',
                                'value' => 0,
                                'label' => $this->l('Disabled'),
                            ),
                        ),
                    ),
                ),
                'submit' => array(
                    'title' => $this->l('Add to Cron Job'),
                    'class' => 'btn btn-default pull-right',
                    'name' => 'submitAddNewFeedToCronJob',
                ),
                'buttons' => array(
                    array(
                        'href' => AdminController::$currentIndex.'&configure='.$this->name.'&token='.Tools::getAdminTokenLite('AdminModules'),
                        'title' => $this->l('Back'),
                        'icon' => 'process-icon-back',
                    ),
                ),
            ),
        );

        $helper = new HelperForm();
        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $lang = new Language((int) Configuration::get('PS_LANG_DEFAULT'));
        $helper->default_form_language = $lang->id;
        $helper->module = $this;
        $helper->allow_employee_form_lang = Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG') ? Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG') : 0;
        $helper->identifier = $this->identifier;
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false).'&configure='.$this->name.'&tab_module='.$this->tab.'&module_name='.$this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');
        $helper->tpl_vars = array(
            'fields_value' => $this->getConfigFieldsValues9(),
            'languages' => $this->context->controller->getLanguages(),
            'id_language' => $this->context->language->id,
        );

        return $helper->generateForm(array($fields_form_1));
    }

    private function getConfigFieldsValues9()
    {
        return array(
            'feed_name' => Tools::getValue('feed_name'),
            'feed_tax' => Tools::getValue('feed_tax', (int) Configuration::get('PS_TAX')),
            'feed_currency' => Tools::getValue('feed_currency', (int) Configuration::get('PS_CURRENCY_DEFAULT')),
            'feed_language' => Tools::getValue('feed_language', (int) Configuration::get('PS_LANG_DEFAULT')),
            'feed_country' => Tools::getValue('feed_country', (int) Configuration::get('PS_COUNTRY_DEFAULT')),
            'feed_type' => Tools::getValue('feed_type'),
        );
    }

    private function renderCronJobList()
    {
        $id_shop = (int) $this->context->shop->id;
        $cronjobs = MaofreeComparatorClass::getCronJobs(self::CRONJOBS_TABLE, $id_shop);
        $this->context->controller->addCSS($this->_path.'views/css/cronjobs_style.css', 'all');
        $this->context->controller->addJS($this->_path.'views/js/maofree_cronjobs.js');

        foreach ($cronjobs as $key => $row) {
            $id_cronjob = (int) $row['id_'.self::CRONJOBS_TABLE];
            $id_currency = (int) $row['id_currency'];
            $id_lang = (int) $row['id_lang'];
            $id_country = (int) $row['id_country'];
            $type = $row['type'];
            $comparator = Tools::strtolower($row['name']);
            $currency = new Currency($id_currency);
            $language = new Language($id_lang);
            $country = new Country($id_country, $id_lang);

            $cronjobs[$key]['name'] = Tools::strtoupper($row['name']);
            $cronjobs[$key]['name_color'] = '#1E90FF';
            $cronjobs[$key]['currency'] = $currency->name;
            $cronjobs[$key]['language'] = $language->name;
            $cronjobs[$key]['country'] = $country->name;
            $cronjobs[$key]['type'] = Tools::strtoupper($row['type']);
            $cronjobs[$key]['type_color'] = '#FF4500';
            $cronjobs[$key]['active'] = $id_cronjob;
            $cronjobs[$key]['icon_name'] = $row['active'] ? 'check' : 'remove';
            $cronjobs[$key]['remove'] = $id_cronjob;
        }

        $fields_list = array(
            'name' => array(
                'title' => $this->l('Comparator'),
                'type' => 'text',
                'align' => 'left',
                'color' => 'name_color',
                'search' => false,
                'orderby' => false,
            ),
            'currency' => array(
                'title' => $this->l('Currency'),
                'type' => 'text',
                'align' => 'center',
                'search' => false,
                'orderby' => false,
            ),
            'language' => array(
                'title' => $this->l('Language'),
                'type' => 'text',
                'align' => 'center',
                'search' => false,
                'orderby' => false,
            ),
            'country' => array(
                'title' => $this->l('Country'),
                'type' => 'text',
                'align' => 'center',
                'search' => false,
                'orderby' => false,
            ),
            'type' => array(
                'title' => $this->l('File extension'),
                'type' => 'text',
                'align' => 'center',
                'color' => 'type_color',
                'search' => false,
                'orderby' => false,
            ),
            'tax' => array(
                'title' => $this->l('Tax included'),
                'type' => 'action_icon',
                'align' => 'center',
                'search' => false,
                'orderby' => false,
            ),
            'active' => array(
                'title' => $this->l('Status'),
                'type' => 'active_cronjob',
                'iconame' => 'icon_name',
                'align' => 'center',
                'class' => 'fixed-width-xl',
                'search' => false,
                'orderby' => false,
            ),
            'remove' => array(
                'title' => $this->l('Remove'),
                'type' => 'remove_cronjob',
                'align' => 'center',
                'class' => 'fixed-width-xl',
                'search' => false,
                'orderby' => false,
            ),
        );

        $helper = new HelperList();
        $helper->shopLinkType = '';
        $helper->simple_header = false;
        $helper->no_link = true;
        $helper->listTotal = count($cronjobs);
        $helper->identifier = 'active';
        $helper->table = 'CronJobsList';
        $helper->imageType = 'jpg';
        $helper->show_toolbar = true;
        $helper->module = $this;
        $helper->title = '<i class="icon-files-o"></i> '.$this->l('Data feed list associated to the Global Cron Job');
        $helper->toolbar_btn['a'] = array(
            'desc' => '',
        );
        $helper->bulk_actions = array(
            'enable' => array(
                'text' => $this->l('Enable selected'),
                'icon' => 'icon-check',
                'confirm' => $this->l('Active selected items?'),
            ),
            'disable' => array(
                'text' => $this->l('Disable selected'),
                'icon' => 'icon-remove',
                'confirm' => $this->l('Disable selected items?'),
            ),
            'delete' => array(
                'text' => $this->l('Delete selected'),
                'icon' => 'icon-trash',
                'confirm' => $this->l('Delete selected items forever?'),
            ),
        );
        $helper->token = Tools::getAdminTokenLite('AdminModules');
        $helper->currentIndex = AdminController::$currentIndex.'&configure='.$this->name;

        return $helper->generateList($cronjobs, $fields_list);
    }

    private function renderComparatorsList()
    {
        $id_shop = (int) $this->context->shop->id;
        $comparators = MaofreeComparatorClass::getComparators(self::MAIN_TABLE, $id_shop);

        foreach ($comparators as $key => $row) {
            $errors_text = '';
            $comparator_name = $row['name'];
            require_once dirname(__FILE__).'/comparators/'.$comparator_name.'/'.$comparator_name.'.php';

            $comparator_obj = new $comparator_name();

            if ($row['type_error'] != 'all_is_ok') {
                if ($row['type_error'] == 'product_error' && $row['product_id_errors'] && is_string($row['product_id_errors'])) {
                    $errors_text = $this->l('Problem about required fields for').' ID: '.str_replace(',', ', ', $row['product_id_errors']);
                } elseif ($row['type_error'] == 'database_error') {
                    $errors_text = $this->l('Missing data in "mao_comparator" table');
                } elseif ($row['type_error'] == 'disabled_error') {
                    $errors_text = $this->l('First active this comparator from this module');
                } elseif ($row['type_error'] == 'file_error') {
                    $errors_text = $this->l('The main file is not present in the folder');
                } elseif ($row['type_error'] == 'parameter_error') {
                    $errors_text = $this->l('Some URL parameters are not active or correct');
                } elseif ($row['type_error'] == 'nocarrier_error') {
                    $errors_text = $this->l('There is not a carrier associated for selected zone');
                } elseif ($row['type_error'] == 'activecarrier_error') {
                    $errors_text = $this->l('The carrier associated for selected zone is not active');
                } elseif ($row['type_error'] == 'zonecarrier_error') {
                    $errors_text = $this->l('The carriers association needs to be updated. The current carrier is no more associated for selected zone');
                } elseif ($row['type_error'] == 'noproducts_error') {
                    $errors_text = $this->l('There are no data');
                } elseif ($row['type_error'] == 'companydesc_error') {
                    $errors_text = $this->l('Missing a little description of your company');
                } elseif ($row['type_error'] == 'creationdir_error') {
                    $errors_text = $this->l('No folder where to place the data feed');
                } elseif ($row['type_error'] == 'removeprevfile_error') {
                    $errors_text = $this->l('Problem to remove the data feed more dated');
                } elseif ($row['type_error'] == 'creationtempfile_error') {
                    $errors_text = $this->l('Problem to create the temporary file');
                } elseif ($row['type_error'] == 'compression_error') {
                    $errors_text = $this->l('Problem to compress the temporary file');
                } elseif ($row['type_error'] == 'removetempfile_error') {
                    $errors_text = $this->l('Problem to remove the temporary file');
                }
            } else {
                $errors_text = $this->l('All is ok');
            }

            $comparators[$key]['name'] = Tools::strtoupper($comparator_name);
            $comparators[$key]['name_color'] = '#1E90FF';
            $comparators[$key]['marketplace'] = is_object($comparator_obj) && isset($comparator_obj->marketplace) && $comparator_obj->marketplace ? 1 : 0;
            $comparators[$key]['errors'] = $errors_text;
            $comparators[$key]['errors_color'] = ($row['type_error'] != 'all_is_ok' ? '#D9534F' : '#499931');
            $comparators[$key]['last_check'] = ($row['last_check'] ? date('g:ia, F j, Y', strtotime($row['last_check'])) : '--');
        }

        $fields_list = array(
            'name' => array(
                'title' => $this->l('Name'),
                'type' => 'text',
                'align' => 'left',
                'color' => 'name_color',
                'search' => false,
                'orderby' => false,
            ),
            'combinations' => array(
                'title' => $this->l('Combinations'),
                'type' => 'action_icon',
                'align' => 'center',
                'search' => false,
                'orderby' => false,
            ),
            'limits' => array(
                'title' => $this->l('With limits'),
                'type' => 'action_icon',
                'align' => 'center',
                'search' => false,
                'orderby' => false,
            ),
            'add_link' => array(
                'title' => $this->l('Show links'),
                'hint' => $this->l('Show links to its websites'),
                'type' => 'action_icon',
                'align' => 'center',
                'search' => false,
                'orderby' => false,
            ),
            'tracking' => array(
                'title' => $this->l('Tracking'),
                'type' => 'action_icon',
                'align' => 'center',
                'search' => false,
                'orderby' => false,
            ),
            'taxonomy' => array(
                'title' => $this->l('Taxonomy'),
                'type' => 'action_icon',
                'align' => 'center',
                'search' => false,
                'orderby' => false,
            ),
            'marketplace' => array(
                'title' => $this->l('Marketplace'),
                'hint' => $this->l('The products are sold on its website'),
                'type' => 'action_icon',
                'align' => 'center',
                'search' => false,
                'orderby' => false,
            ),
            'errors' => array(
                'title' => $this->l('Errors'),
                'type' => 'text',
                'align' => 'center',
                'color' => 'errors_color',
                'maxlength' => 30,
                'search' => false,
                'orderby' => false,
            ),
            'datafeed_type' => array(
                'title' => $this->l('Data feed'),
                'type' => 'text',
                'align' => 'center',
                'search' => false,
                'orderby' => false,
            ),
            'last_check' => array(
                'title' => $this->l('Last check or file creation'),
                'type' => 'text',
                'align' => 'center',
                'search' => false,
                'orderby' => false,
            ),
            'active' => array(
                'title' => $this->l('Status'),
                'type' => 'bool',
                'active' => 'status',
                'align' => 'center',
                'class' => 'fixed-width-sm',
                'search' => false,
                'orderby' => false,
            ),
        );

        $helper = new HelperList();
        $helper->shopLinkType = '';
        $helper->simple_header = false;
        $helper->listTotal = count($comparators);
        $helper->identifier = 'name';
        $helper->actions = array('view', 'delete');
        $helper->table = 'ComparatorList';
        $helper->imageType = 'jpg';
        $helper->no_link = true;
        $helper->show_toolbar = false;
        $helper->module = $this;
        $helper->title = '<i class="icon-sitemap"></i> '.$this->l('List of comparators');
        $helper->toolbar_btn = array(
            'clean' => array(
                'desc' => $this->l('Reset all data and remove all files'),
                'imgclass' => 'eraser',
                'href' => AdminController::$currentIndex.'&configure='.$this->name.'&removeDataAndFiles&token='.Tools::getAdminTokenLite('AdminModules'),
                'js' => 'return confirm(\''.$this->l('Are you sure you want to reset the comparator data and remove all data feeds?').'\');',
            ),
        );
        $helper->token = Tools::getAdminTokenLite('AdminModules');
        $helper->currentIndex = AdminController::$currentIndex.'&configure='.$this->name;

        return $helper->generateList($comparators, $fields_list);
    }

    private function displayComparatorErrors($comparator)
    {
        if ($comparator['type_error'] != 'all_is_ok') {
            $this->smarty->assign(array(
                'mao_comparator_type_error' => $comparator['type_error'],
                'mao_comparator_id_errors' => ($comparator['type_error'] == 'product_error' && $comparator['product_id_errors'] ? str_replace(',', ', ', $comparator['product_id_errors']) : ''),
            ));

            return $this->display(__FILE__, 'mao_errors.tpl');
        } else {
            return;
        }
    }

    private function renderComparatorEnableForm($comparator)
    {
        $fields_form_1 = array(
            'form' => array(
                'legend' => array(
                    'title' => $this->l('Give permission to access the data'),
                    'icon' => 'icon-power-off',
                ),
                'input' => array(
                    array(
                        'type' => 'switch',
                        'is_bool' => true,
                        'label' => $this->l('Active').':',
                        'name' => 'active',
                        'desc' => $this->l('Allow this comparator to access the data feed that you have prepared'),
                        'values' => array(
                            array(
                                'id' => 'active_on',
                                'value' => 1,
                                'label' => $this->l('Enabled'),
                            ),
                            array(
                                'id' => 'active_off',
                                'value' => 0,
                                'label' => $this->l('Disabled'),
                            ),
                        ),
                    ),
                ),
                'submit' => array(
                    'title' => $this->l('Save'),
                    'class' => 'btn btn-default pull-right',
                    'name' => 'submitComparatorEnable',
                ),
                'buttons' => array(
                    array(
                        'href' => AdminController::$currentIndex.'&configure='.$this->name.'&token='.Tools::getAdminTokenLite('AdminModules'),
                        'title' => $this->l('Back'),
                        'icon' => 'process-icon-back',
                    ),
                ),
            ),
        );

        $helper = new HelperForm();
        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $lang = new Language((int) Configuration::get('PS_LANG_DEFAULT'));
        $helper->default_form_language = $lang->id;
        $helper->module = $this;
        $helper->allow_employee_form_lang = Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG') ? Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG') : 0;
        $helper->identifier = $this->identifier;
        $helper->currentIndex = AdminController::$currentIndex.'&configure='.$this->name.'&name='.Tools::strtoupper($comparator['name']);
        $helper->token = Tools::getAdminTokenLite('AdminModules');
        $helper->tpl_vars = array(
            'fields_value' => $this->getConfigFieldsValues2($comparator),
            'languages' => $this->context->controller->getLanguages(),
            'id_language' => $this->context->language->id,
        );

        return $helper->generateForm(array($fields_form_1));
    }

    private function getConfigFieldsValues2($comparator)
    {
        return array(
            'active' => Tools::getValue('active', $comparator['active']),
        );
    }

    private function renderComparatorSettingsList($comparator)
    {
        $settings = array();
        $comparator_name = $comparator['name'];
        require_once dirname(__FILE__).'/comparators/'.$comparator_name.'/'.$comparator_name.'.php';

        $comparator_obj = new $comparator_name();

        $settings[] = array(
            'name' => Tools::strtoupper($comparator_name),
            'on_stock_only' => $comparator['on_stock_only'],
            'combinations' => $comparator['combinations'],
            'limits' => $comparator['limits'],
            'tracking' => $comparator['tracking'],
            'marketplace' => is_object($comparator_obj) && isset($comparator_obj->marketplace) && $comparator_obj->marketplace ? 1 : 0,
            'add_link' => $comparator['add_link'],
            'brand_in_title' => $comparator['brand_in_title'],
            'analytics_params' => $comparator['analytics_params'],
            'compression_type' => $comparator['compression_type'] == 'none' ? $this->l('None') : $comparator['compression_type'],
            'extra_fields' => ($comparator['extra_fields_keys'] ? count(explode(',', $comparator['extra_fields_keys'])) : $this->l('No')),
            'extra_images' => ($comparator['extra_images'] > 0 ? (int) $comparator['extra_images'] : $this->l('No')),
            'image_type' => ImageType::getFormatedName($comparator['image_type']),
            'products_number' => ($comparator['products_number'] > 0 ? (int) $comparator['products_number'] : $this->l('All')),
            'fee' => Tools::ps_round($comparator['fee'], 2),
            'badge_warning' => true,
        );

        $fields_list1 = array(
            'on_stock_only' => array(
                'title' => $this->l('On stock only'),
                'type' => 'action_icon',
                'align' => 'center',
            ),
            'combinations' => array(
                'title' => $this->l('Combinations'),
                'type' => 'action_icon',
                'align' => 'center',
            ),
            'limits' => array(
                'title' => $this->l('With limits'),
                'type' => 'action_icon',
                'align' => 'center',
            ),
            'tracking' => array(
                'title' => $this->l('Tracking'),
                'type' => 'action_icon',
                'align' => 'center',
            ),
        );

        if (is_object($comparator_obj) && isset($comparator_obj->marketplace) && $comparator_obj->marketplace) {
            $fields_list2 = array(
                'marketplace' => array(
                    'title' => $this->l('Marketplace'),
                    'hint' => $this->l('The products are sold on its website'),
                    'type' => 'action_icon',
                    'align' => 'center',
                    'suffix' => '%',
                ),
            );
        } else {
            $fields_list2 = array();
        }

        $fields_list3 = array(
            'add_link' => array(
                'title' => $this->l('Show its links'),
                'hint' => $this->l('Show the links of its websites in the front-office'),
                'type' => 'action_icon',
                'align' => 'center',
            ),
            'brand_in_title' => array(
                'title' => $this->l('Brand in title'),
                'hint' => $this->l('Add the manufacturer in the product name'),
                'type' => 'action_icon',
                'align' => 'center',
            ),
            'analytics_params' => array(
                'title' => $this->l('Statistical parameters'),
                'hint' => $this->l('For Google Analytics will be added 3 parameters to each product URL'),
                'type' => 'action_icon',
                'align' => 'center',
            ),
        );

        if (is_object($comparator_obj) && isset($comparator_obj->marketplace) && $comparator_obj->marketplace) {
            $fields_list4 = array(
                'fee' => array(
                    'title' => $this->l('Fee'),
                    'hint' => $this->l('Additional fee'),
                    'type' => 'text',
                    'align' => 'center',
                ),
            );
        } else {
            $fields_list4 = array();
        }

        $fields_list5 = array(
            'compression_type' => array(
                'title' => $this->l('Compression type'),
                'type' => 'text',
                'align' => 'center',
            ),
            'extra_fields' => array(
                'title' => $this->l('Extra fields'),
                'type' => 'text',
                'align' => 'center',
                'badge_warning' => $comparator['extra_fields_keys'] ? true : false,
            ),
            'extra_images' => array(
                'title' => $this->l('Extra images'),
                'type' => 'text',
                'align' => 'center',
                'badge_warning' => $comparator['extra_images'] > 0 ? true : false,
            ),
            'image_type' => array(
                'title' => $this->l('Image type'),
                'type' => 'text',
                'align' => 'center',
            ),
            'products_number' => array(
                'title' => $this->l('Limit to'),
                'type' => 'text',
                'align' => 'center',
                'badge_warning' => $comparator['products_number'] > 0 ? true : false,
            ),
        );

        $fields_list = array_merge($fields_list1, $fields_list2, $fields_list3, $fields_list4, $fields_list5);

        $helper = new HelperList();
        $helper->shopLinkType = '';
        $helper->simple_header = true;
        $helper->identifier = 'name';
        $helper->actions = array('edit');
        $helper->table = 'ComparatorSettings';
        $helper->imageType = 'jpg';
        $helper->no_link = true;
        $helper->show_toolbar = false;
        $helper->module = $this;
        $helper->title = '<i class="icon-cogs"></i> '.$this->l('Comparator Settings');
        $helper->token = Tools::getAdminTokenLite('AdminModules');
        $helper->currentIndex = AdminController::$currentIndex.'&configure='.$this->name.'&name='.Tools::strtoupper($comparator_name);

        return $helper->generateList($settings, $fields_list);
    }

    private function renderComparatorSettingsForm()
    {
        $comparator_name = Tools::strtolower(Tools::getValue('name'));

        if (file_exists(dirname(__FILE__).'/comparators/'.$comparator_name.'/'.$comparator_name.'.php')) {
            require_once dirname(__FILE__).'/comparators/'.$comparator_name.'/'.$comparator_name.'.php';

            $id_shop = (int) $this->context->shop->id;
            $id_lang = (int) $this->context->language->id;
            $default_currency = new Currency((int) Configuration::get('PS_CURRENCY_DEFAULT'));
            $comparator = MaofreeComparatorClass::getValuesFromTable(self::MAIN_TABLE, $id_shop, $comparator_name);

            $extra_description = '';
            $comparator_obj = new $comparator_name();

            if (isset($comparator_obj->datafeed_compatibility) && $comparator_obj->datafeed_compatibility) {
                $extra_description .= '<p><span style="font-weight: bold;">'.$this->l('You can use its data feed for other comparators').'</span>:<br />'.(is_array($comparator_obj->datafeed_compatibility) ? implode(', ', $comparator_obj->datafeed_compatibility) : $comparator_obj->datafeed_compatibility).'</p>';
            }

            if (isset($comparator_obj->extra_info) && $comparator_obj->extra_info) {
                $extra_description .= '<p>'.$comparator_obj->extra_info.'</p>';
            }

            $compressions = array(array('id_compression' => 'none', 'compression' => $this->l('no compression')));
            if (
                isset($comparator_obj->compression_type) &&
                is_array($comparator_obj->compression_type) &&
                $comparator_obj->compression_type
            ) {
                $accepted_type = array('zip', 'gzip', 'bzip2');

                foreach ($comparator_obj->compression_type as $type) {
                    if (in_array($type, $accepted_type)) {
                        $compressions[] = array('id_compression' => $type, 'compression' => $type);
                    }
                }
            }

            $extraimages = array(array('id_extraimage' => 0, 'extraimage' => 0));
            if (isset($comparator_obj->extra_images) && $comparator_obj->extra_images) {
                for ($i = 1; $i <= $comparator_obj->extra_images; ++$i) {
                    $extraimages[] = array('id_extraimage' => $i, 'extraimage' => $i);
                }
            }

            $imagetypes = $my_imagetypes_array = $sel_imagetypes_array = array();
            $theme_name = $this->context->shop->theme_name;
            $imagetypes_array = ImageType::getImagesTypes('products');
            foreach ($imagetypes_array as $type) {
                if (strstr($type['name'], $theme_name)) {
                    $sel_imagetypes_array['name'] = $type['name'];
                }
            }
            // to show only image types of the current theme
            if (count($sel_imagetypes_array)) {
                $imagetypes_array = $sel_imagetypes_array;
            }
            foreach ($imagetypes_array as $type) {
                if (strstr($type['name'], $theme_name)) {
                    $name = str_replace(array('_'.$theme_name, $theme_name.'_'), '', $type['name']);
                } else {
                    $name_array = explode('_', $type['name']);
                    $name = $name_array[0];
                }

                if (!in_array($name, $my_imagetypes_array)) {
                    $my_imagetypes_array[] = $name;
                    $imagetypes[] = array('id_imagetype' => $name, 'imagetype' => $type['name']);
                }
            }

            $fields_form_1_input = array(
                array(
                    'type' => 'switch',
                    'is_bool' => true,
                    'label' => $this->l('On stock only').':',
                    'name' => $comparator_name.'_on_stock_only',
                    'values' => array(
                        array(
                            'id' => 'active_on',
                            'value' => 1,
                            'label' => $this->l('Enabled'),
                        ),
                        array(
                            'id' => 'active_off',
                            'value' => 0,
                            'label' => $this->l('Disabled'),
                        ),
                    ),
                ),
                array(
                    'type' => 'switch',
                    'is_bool' => true,
                    'disabled' => (!$comparator_obj->combinations ? true : false),
                    'label' => $this->l('Product combinations').':',
                    'name' => $comparator_name.'_combinations',
                    'desc' => $this->l('If the data feed contains many products, it would be better to disable this option.').(!$comparator_obj->combinations ? ' '.$this->l('(no combination for this comparator)') : ''),
                    'values' => array(
                        array(
                            'id' => 'active_on',
                            'value' => 1,
                            'label' => $this->l('Enabled'),
                        ),
                        array(
                            'id' => 'active_off',
                            'value' => 0,
                            'label' => $this->l('Disabled'),
                        ),
                    ),
                ),
                array(
                    'type' => 'switch',
                    'is_bool' => true,
                    'disabled' => (!$comparator_obj->limits ? true : false),
                    'label' => $this->l('Comparator limits').':',
                    'name' => $comparator_name.'_limits',
                    'desc' => $this->l('Remove the default limits imposed by the comparator. In some cases, like for free plans, there are some limits on the number of items or on the size of the data feed.').(!$comparator_obj->limits ? ' '.$this->l('(no limit for this comparator)') : ''),
                    'values' => array(
                        array(
                            'id' => 'active_on',
                            'value' => 1,
                            'label' => $this->l('Enabled'),
                        ),
                        array(
                            'id' => 'active_off',
                            'value' => 0,
                            'label' => $this->l('Disabled'),
                        ),
                    ),
                ),
                array(
                    'type' => 'switch',
                    'is_bool' => true,
                    'disabled' => (!isset($comparator_obj->tracking) || !$comparator_obj->tracking || isset($comparator_obj->tracking_required) && $comparator_obj->tracking_required ? true : false),
                    'label' => $this->l('Tracking').':',
                    'name' => $comparator_name.'_tracking',
                    'desc' => $this->l('Tracking is used for various reasons such as: to track orders placed in the store, to send an evaluation questionnaire, to calculate its fee, for statistical data.').(!isset($comparator_obj->tracking) || !$comparator_obj->tracking ? ' '.$this->l('(no tracking for this comparator)') : '').(isset($comparator_obj->tracking_required) && $comparator_obj->tracking_required ? ' '.$this->l('(tracking required by this comparator)') : ''),
                    'values' => array(
                        array(
                            'id' => 'active_on',
                            'value' => 1,
                            'label' => $this->l('Enabled'),
                        ),
                        array(
                            'id' => 'active_off',
                            'value' => 0,
                            'label' => $this->l('Disabled'),
                        ),
                    ),
                ),
                array(
                    'type' => 'switch',
                    'is_bool' => true,
                    'label' => $this->l('Brand in the product name').':',
                    'name' => $comparator_name.'_brand_in_title',
                    'desc' => $this->l('Enabling this option you can improve the seo inside the comparator research, but not all comparators permit it. The brand will be added at the end'),
                    'values' => array(
                        array(
                            'id' => 'active_on',
                            'value' => 1,
                            'label' => $this->l('Enabled'),
                        ),
                        array(
                            'id' => 'active_off',
                            'value' => 0,
                            'label' => $this->l('Disabled'),
                        ),
                    ),
                ),
                array(
                    'type' => 'select',
                    'disabled' => (!$compressions ? true : false),
                    'label' => $this->l('Compression type of data feed').':',
                    'name' => $comparator_name.'_compression_type',
                    'desc' => $this->l('With static data feed, with extensions like .txt, .csv and .xml, it is possible to add a compression to reduce their size (ask first to this comparator if they accept this solution)'),
                    'options' => array(
                        'query' => $compressions,
                        'id' => 'id_compression',
                        'name' => 'compression',
                    ),
                ),
                array(
                    'type' => 'select',
                    'disabled' => (count($extraimages) == 1 ? true : false),
                    'label' => $this->l('Extra images').':',
                    'name' => $comparator_name.'_extra_images',
                    'desc' => $this->l('Some comparators accept more than one product image, here you can set the number of extra images to pass to this comparator'),
                    'options' => array(
                        'query' => $extraimages,
                        'id' => 'id_extraimage',
                        'name' => 'extraimage',
                    ),
                ),
                array(
                    'type' => 'select',
                    'label' => $this->l('Image type').':',
                    'name' => $comparator_name.'_image_type',
                    'desc' => $this->l('Choose which type of image you want to use for the products').'. ('.$this->l('default').' "'.ImageType::getFormatedName('thickbox').'")',
                    'options' => array(
                        'query' => $imagetypes,
                        'id' => 'id_imagetype',
                        'name' => 'imagetype',
                    ),
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Minimum price').':',
                    'name' => $comparator_name.'_price_min',
                    'desc' => $this->l('Minimum price which must have a product to be accepted in the data feed. (No limits with 0.00)'),
                    'autocomplete' => false,
                    'class' => 'fixed-width-xl',
                    'prefix' => $default_currency->sign,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Maximum price').':',
                    'name' => $comparator_name.'_price_max',
                    'desc' => $this->l('Maximum price which must have a product to be accepted in the data feed. (No limits with 0.00)'),
                    'autocomplete' => false,
                    'class' => 'fixed-width-xl',
                    'prefix' => $default_currency->sign,
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Minimum quantity').':',
                    'name' => $comparator_name.'_quantity_min',
                    'desc' => $this->l('Minimum quantity which must have a product to be accepted in the data feed. (No limits with 0)'),
                    'autocomplete' => false,
                    'class' => 'fixed-width-xl',
                    'prefix' => '<i class="icon-shopping-cart"></i>',
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Number of products').':',
                    'name' => $comparator_name.'_products_number',
                    'desc' => $this->l('Maximum number of products to insert into the data feed. Remember that for a comparator each combination is considered as a product. (All products with 0)'),
                    'autocomplete' => false,
                    'class' => 'fixed-width-xl',
                    'prefix' => '<i class="icon-sort-amount-desc"></i>',
                ),
            );

            $fields_form_2_input = array();
            if (is_object($comparator_obj) && isset($comparator_obj->marketplace) && $comparator_obj->marketplace) {
                $fields_form_2_input[] =
                    array(
                        'type' => 'text',
                        'label' => $this->l('Additional fee').':',
                        'name' => $comparator_name.'_fee',
                        'desc' => $this->l('Here you can set a percentage that will be added to the prices of each product and combination present in the data feed of this comparator').'. <strong>('.$this->l('eg.').' 4.00)</strong>',
                        'autocomplete' => false,
                        'class' => 'fixed-width-md',
                        'prefix' => '%',
                    );
            }

            $fields_form_3_input = array();
            $extra_fields = $extra_fields_value = $extra_fields_unit = $extra_fields_unit_set = $extra_fields_title =
            $extra_fields_description = $extra_fields_prefix = $extra_fields_fieldlength = $product_features = $product_attributes = array();

            if (isset($comparator['extra_fields_keys']) && $comparator['extra_fields_keys']) {
                $extra_fields = explode(',', $comparator['extra_fields_keys']);
            }
            // the follow condition is ok when there is only one extra field
            if (isset($comparator['extra_fields_values']) && $extra_fields) {
                $extra_fields_value = explode('#', $comparator['extra_fields_values']);
            }
            if (isset($comparator['extra_fields_units']) && $comparator['extra_fields_units']) {
                $extra_fields_unit = explode(',', $comparator['extra_fields_units']);
            }

            if ($extra_fields) {
                if ($extra_fields_unit) {
                    $extra_fields_unit_set = $comparator_obj->extra_fields_unit_set;
                }
                $extra_fields_title = $comparator_obj->extra_fields_title;
                if (
                    isset($comparator_obj->extra_fields_description) &&
                    is_array($comparator_obj->extra_fields_description) &&
                    $comparator_obj->extra_fields_description
                ) {
                    $extra_fields_description = $comparator_obj->extra_fields_description;
                }
                if (
                    isset($comparator_obj->extra_fields_prefix) &&
                    is_array($comparator_obj->extra_fields_prefix) &&
                    $comparator_obj->extra_fields_prefix
                ) {
                    $extra_fields_prefix = $comparator_obj->extra_fields_prefix;
                }
                if (
                    isset($comparator_obj->extra_fields_fieldlength) &&
                    is_array($comparator_obj->extra_fields_fieldlength) &&
                    $comparator_obj->extra_fields_fieldlength
                ) {
                    $extra_fields_fieldlength = $comparator_obj->extra_fields_fieldlength;
                }
                if (
                    isset($comparator_obj->product_features) &&
                    $comparator_obj->product_features &&
                    isset($comparator_obj->extra_fields_type) &&
                    $comparator_obj->extra_fields_type &&
                    in_array('feature', $comparator_obj->extra_fields_type)
                ) {
                    $product_features = Feature::getFeatures($id_lang);
                }
                if (
                    isset($comparator_obj->product_attributes) &&
                    $comparator_obj->product_attributes &&
                    isset($comparator_obj->extra_fields_type) &&
                    $comparator_obj->extra_fields_type &&
                    in_array('attribute', $comparator_obj->extra_fields_type)
                ) {
                    $product_attributes = AttributeGroup::getAttributesGroups($id_lang);
                }

                for ($i = 0; $i < count($extra_fields); ++$i) {
                    $output = '';

                    if (
                        isset($comparator_obj->extra_fields_type) &&
                        $comparator_obj->extra_fields_type &&
                        isset($comparator_obj->extra_fields_type[$i]) &&
                        (
                            $comparator_obj->extra_fields_type[$i] == 'radio' ||
                            $comparator_obj->extra_fields_type[$i] == 'feature' ||
                            $comparator_obj->extra_fields_type[$i] == 'attribute'
                        )
                    ) {
                        if (
                            $comparator_obj->extra_fields_type[$i] == 'radio' &&
                            is_numeric($extra_fields_value[$i]) &&
                            ($extra_fields_value[$i] == 0 || $extra_fields_value[$i] == 1)
                        ) {
                            $output .= '<span class="switch prestashop-switch fixed-width-lg">
                                            <input type="radio"'.($extra_fields_value[$i] ==  1 ? ' checked="checked"' : '').' value="1" id="'.$comparator_name.'_'.$extra_fields[$i].'_on" name="'.$comparator_name.'_'.$extra_fields[$i].'">
                                            <label for="'.$comparator_name.'_'.$extra_fields[$i].'_on">'.$this->l('Yes').'</label>
                                            <input type="radio"'.($extra_fields_value[$i] ==  0 ? ' checked="checked"' : '').' value="0" id="'.$comparator_name.'_'.$extra_fields[$i].'_off" name="'.$comparator_name.'_'.$extra_fields[$i].'">
                                            <label for="'.$comparator_name.'_'.$extra_fields[$i].'_off">'.$this->l('No').'</label>
                                            <a class="slide-button btn"></a>
                                        </span>';
                        } elseif ($comparator_obj->extra_fields_type[$i] == 'feature') {
                            $output .= ' <select id="'.$comparator_name.'_'.$extra_fields[$i].'" class="fixed-width-xl" name="'.$comparator_name.'_'.$extra_fields[$i].'">
                                <option value=""'.($extra_fields_value[$i] == '' ? ' selected="selected"' : '').'> -- </option>';

                            foreach ($product_features as $feature) {
                                $output .= '<option value="'.((int) $feature['id_feature']).'"'.((int) $feature['id_feature'] == $extra_fields_value[$i] ? ' selected="selected"' : '').'> '.$feature['name'].' </option>';
                            }

                            $output .= '</select>';
                        } elseif ($comparator_obj->extra_fields_type[$i] == 'attribute') {
                            $output .= ' <select id="'.$comparator_name.'_'.$extra_fields[$i].'" class="fixed-width-xl" name="'.$comparator_name.'_'.$extra_fields[$i].'">
                                <option value=""'.($extra_fields_value[$i] == '' ? ' selected="selected"' : '').'> -- </option>';

                            foreach ($product_attributes as $attribute) {
                                $output .= '<option value="'.((int) $attribute['id_attribute_group']).'"'.((int) $attribute['id_attribute_group'] == $extra_fields_value[$i] ? ' selected="selected"' : '').'> '.$attribute['name'].' </option>';
                            }

                            $output .= '</select>';
                        }
                    } else {
                        if (isset($extra_fields_prefix[$i]) && $extra_fields_prefix[$i]) {
                            $output .= '<div class="input-group"><span class="input-group-addon"> '.$extra_fields_prefix[$i].' </span>';
                        }

                        $output .= '<input type="text" autocomplete="off"'.(isset($extra_fields_fieldlength[$i]) && $extra_fields_fieldlength[$i] == 'nolimit' ? '' : 'class="fixed-width-xl"').' style="display: inline-block;" name="'.$comparator_name.'_'.$extra_fields[$i].'" value="'.$extra_fields_value[$i].'" />';

                        if ($extra_fields_unit && $extra_fields_unit[$i]) {
                            $output .= ' <select class="fixed-width-xl" style="display: inline-block;" name="'.$comparator_name.'_'.$extra_fields[$i].'_unit'.'">';

                            foreach ($extra_fields_unit_set[$i] as $unit) {
                                $output .= '<option value="'.$unit.'"'.($extra_fields_unit[$i] ==  $unit ? ' selected="selected"' : '').'> '.$unit.' </option>';
                            }

                            $output .= '</select>';
                        }

                        if (isset($extra_fields_prefix[$i]) && $extra_fields_prefix[$i]) {
                            $output .= '</div>';
                        }
                    }

                    $fields_form_3_input[] =
                        array(
                            'type' => 'html',
                            'id' => $comparator_name.'_'.$extra_fields[$i],
                            'label' => $extra_fields_title[$i].':',
                            'name' => $output,
                            'desc' => (isset($extra_fields_description[$i]) ? $extra_fields_description[$i] : ''),
                        );
                }
            }

            $fields_form_input = array_merge($fields_form_1_input, $fields_form_2_input, $fields_form_3_input);

            $fields_form_1 = array(
                'form' => array(
                    'legend' => array(
                        'title' => $this->l('Settings for').' <span style="color: red;">'.Tools::strtoupper($comparator_name).'</span>',
                        'icon' => 'icon-cogs',
                    ),
                    'description' => ($extra_description ? $extra_description : ''),
                    'input' => $fields_form_input,
                    'submit' => array(
                        'title' => $this->l('Save'),
                        'class' => 'btn btn-default pull-right',
                        'name' => 'submitComparatorSettings',
                    ),
                    'buttons' => array(
                        array(
                            'href' => AdminController::$currentIndex.'&configure='.$this->name.'&name='.Tools::strtoupper($comparator_name).'&viewComparatorList&token='.Tools::getAdminTokenLite('AdminModules'),
                            'title' => $this->l('Back'),
                            'icon' => 'process-icon-back',
                        ),
                    ),
                ),
            );

            $fields_form_2 = $default_analytics_params = array();

            if (isset($comparator_obj->analytics_params) && $comparator_obj->analytics_params) {
                $default_analytics_params = explode('&', $comparator_obj->analytics_params);

                $utm_source = explode('=', $default_analytics_params[0]);
                $utm_medium = explode('=', $default_analytics_params[1]);
                $utm_campaign = explode('=', $default_analytics_params[2]);

                $fields_form_2 = array(
                    'form' => array(
                        'legend' => array(
                            'title' => $this->l('Parameters for Google analytics'),
                            'icon' => 'icon-bar-chart',
                        ),
                        'description' => $this->l('Adds three parameters "utm_source, utm_medium, utm_campaign" to each product url to track clicks coming from this comparator. (no spaces allowed only - or _)').'<br />'.$this->l('Default parameters').': <b>'.$comparator_obj->analytics_params.'</b>',
                        'input' => array(
                            array(
                                'type' => 'switch',
                                'is_bool' => true,
                                'label' => $this->l('Add analytics parameters').':',
                                'name' => $comparator_name.'_analytics',
                                'desc' => $this->l('In Google Analytics you will also need to add some settings to allow the tracking of this comparator'),
                                'values' => array(
                                    array(
                                        'id' => 'active_on',
                                        'value' => 1,
                                        'label' => $this->l('Enabled'),
                                    ),
                                    array(
                                        'id' => 'active_off',
                                        'value' => 0,
                                        'label' => $this->l('Disabled'),
                                    ),
                                ),
                            ),
                            array(
                                'type' => 'text',
                                'label' =>  $this->l('Value for'). ' "<b>utm_source</b>"'.':',
                                'name' => $comparator_name.'_analytics_utm_source',
                                'desc' => $this->l('Leaving blank the value will be').' = <b>'.trim($utm_source[1]).'</b>',
                                'autocomplete' => false,
                                'class' => 'fixed-width-xxl',
                            ),
                            array(
                                'type' => 'text',
                                'label' =>  $this->l('Value for'). ' "<b>utm_medium</b>"'.':',
                                'name' => $comparator_name.'_analytics_utm_medium',
                                'desc' => $this->l('Leaving blank the value will be').' = <b>'.trim($utm_medium[1]).'</b>',
                                'autocomplete' => false,
                                'class' => 'fixed-width-xxl',
                            ),
                            array(
                                'type' => 'text',
                                'label' =>  $this->l('Value for'). ' "<b>utm_campaign</b>"'.':',
                                'name' => $comparator_name.'_analytics_utm_campaign',
                                'desc' => $this->l('Leaving blank the value will be').' = <b>'.trim($utm_campaign[1]).'</b>',
                                'autocomplete' => false,
                                'class' => 'fixed-width-xxl',
                            ),
                        ),
                        'submit' => array(
                            'title' => $this->l('Save'),
                            'class' => 'btn btn-default pull-right',
                            'name' => 'submitAnalyticsComparatorSettings',
                        ),
                        'buttons' => array(
                            array(
                                'href' => AdminController::$currentIndex.'&configure='.$this->name.'&name='.Tools::strtoupper($comparator_name).'&viewComparatorList&token='.Tools::getAdminTokenLite('AdminModules'),
                                'title' => $this->l('Back'),
                                'icon' => 'process-icon-back',
                            ),
                        ),
                    ),
                );
            }

            $fields_form_4_input = array(
                array(
                    'type' => 'switch',
                    'is_bool' => true,
                    'label' => $this->l('Add links in the front-office').':',
                    'name' => $comparator_name.'_add_link',
                    'desc' => $this->l('Normally it is required to add one or more links on the front-office, which redirect to the comparator\'s websites'),
                    'values' => array(
                        array(
                            'id' => 'active_on',
                            'value' => 1,
                            'label' => $this->l('Enabled'),
                        ),
                        array(
                            'id' => 'active_off',
                            'value' => 0,
                            'label' => $this->l('Disabled'),
                        ),
                    ),
                ),
            );

            $fields_form_5_input = $website_url = array();

            if ($comparator['website_url']) {
                $website_url_saved = explode('@', $comparator['website_url']);

                foreach ($comparator_obj->website_url as $k => $url) {
                    $website_url[] = $website_url_saved[$k] ? $website_url_saved[$k] : $url;
                }
            } elseif (isset($comparator_obj->website_url) && is_array($comparator_obj->website_url) && $comparator_obj->website_url) {
                $website_url = $comparator_obj->website_url;
            }

            if (is_array($website_url) && count($website_url)) {
                foreach ($website_url as $k => $url) {
                    $fields_form_5_input[] =
                        array(
                            'type' => 'text',
                            'label' => $this->l('URL to').' '.$comparator_obj->name[$k].':',
                            'name' => $comparator_name.'_website_url_'.$k,
                            'autocomplete' => false,
                            'prefix' => '<i class="icon-external-link"></i>',
                        );
                }
            }

            $fields_form_input1 = array_merge($fields_form_4_input, $fields_form_5_input);

            $fields_form_3 = array(
                'form' => array(
                    'legend' => array(
                        'title' => $this->l('External links to this comparator'),
                        'icon' => 'icon-link',
                    ),
                    'description' => $this->l('From here you can enable and edit the URLs that redirect to the sites of this comparator'),
                    'input' => $fields_form_input1,
                    'submit' => array(
                        'title' => $this->l('Save'),
                        'class' => 'btn btn-default pull-right',
                        'name' => 'submitUrlComparatorSettings',
                    ),
                    'buttons' => array(
                        array(
                            'href' => AdminController::$currentIndex.'&configure='.$this->name.'&name='.Tools::strtoupper($comparator_name).'&viewComparatorList&token='.Tools::getAdminTokenLite('AdminModules'),
                            'title' => $this->l('Back'),
                            'icon' => 'process-icon-back',
                        ),
                    ),
                ),
            );

            $helper = new HelperForm();
            $helper->name_controller = $this->name;
            $lang = new Language((int) Configuration::get('PS_LANG_DEFAULT'));
            $helper->default_form_language = $lang->id;
            $helper->module = $this;
            $helper->toolbar_scroll = true;
            $helper->allow_employee_form_lang = Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG') ? Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG') : 0;
            $helper->identifier = $this->identifier;
            $helper->currentIndex = AdminController::$currentIndex.'&configure='.$this->name.'&name='.Tools::strtoupper($comparator_name);
            $helper->token = Tools::getAdminTokenLite('AdminModules');
            $helper->tpl_vars = array(
                'fields_value' => $this->getConfigFieldsValues3($comparator, $default_analytics_params, $website_url),
                'languages' => $this->context->controller->getLanguages(),
                'id_language' => $this->context->language->id,
            );

            if ($fields_form_2) {
                return $helper->generateForm(array($fields_form_1, $fields_form_2, $fields_form_3));
            } else {
                return $helper->generateForm(array($fields_form_1, $fields_form_3));
            }
        } else {
            return '
                <div class="alert alert-danger">
                    <span style="font-weight: bold;">'.$this->l('The main file to manage the data feed of this comparator is missing').'</span>:<br />
                    '.dirname(__FILE__).'/comparators/'.$comparator_name.'/'.$comparator_name.'.php'.'
                </div>
                <div class="col-lg-12">
                    <a class="btn btn-default" href="#" onclick="window.history.back()"><i class="process-icon-back"></i> '.$this->l('Back').'</a>
                </div>';
        }
    }

    private function getConfigFieldsValues3($comparator, $default_analytics_params, $website_url)
    {
        $comparator_name = $comparator['name'];
        $analytics_utm_source = $analytics_utm_medium = $analytics_utm_campaign = '';
        $analytics_params = $comparator['analytics_params'] ? explode(',', $comparator['analytics_params']) : array();

        if ($default_analytics_params && $analytics_params) {
            $analytics_utm_source = $analytics_params[0];
            $analytics_utm_medium = $analytics_params[1];
            $analytics_utm_campaign = $analytics_params[2];
        }

        $data = array(
            $comparator_name.'_on_stock_only' => Tools::getValue($comparator_name.'_on_stock_only', $comparator['on_stock_only']),
            $comparator_name.'_combinations' => $comparator['combinations'],
            $comparator_name.'_limits' => $comparator['limits'],
            $comparator_name.'_tracking' => $comparator['tracking'],
            $comparator_name.'_add_link' => Tools::getValue($comparator_name.'_add_link', $comparator['add_link']),
            $comparator_name.'_brand_in_title' => Tools::getValue($comparator_name.'_brand_in_title', $comparator['brand_in_title']),
            $comparator_name.'_analytics' => $comparator['analytics'],
            $comparator_name.'_analytics_utm_source' => $analytics_utm_source,
            $comparator_name.'_analytics_utm_medium' => $analytics_utm_medium,
            $comparator_name.'_analytics_utm_campaign' => $analytics_utm_campaign,
            $comparator_name.'_compression_type' => Tools::getValue($comparator_name.'_compression_type', $comparator['compression_type']),
            $comparator_name.'_extra_images' => Tools::getValue($comparator_name.'_extra_images', $comparator['extra_images']),
            $comparator_name.'_image_type' => Tools::getValue($comparator_name.'_image_type', $comparator['image_type']),
            $comparator_name.'_price_min' => Tools::getValue($comparator_name.'_price_min', $comparator['price_min']),
            $comparator_name.'_price_max' => Tools::getValue($comparator_name.'_price_max', $comparator['price_max']),
            $comparator_name.'_quantity_min' => Tools::getValue($comparator_name.'_quantity_min', $comparator['quantity_min']),
            $comparator_name.'_products_number' => Tools::getValue($comparator_name.'_products_number', $comparator['products_number']),
            $comparator_name.'_fee' => Tools::getValue($comparator_name.'_fee', $comparator['fee']),
        );

        if (is_array($website_url) && count($website_url)) {
            foreach ($website_url as $k => $url) {
                $data[$comparator_name.'_website_url_'.$k] = Tools::getValue($comparator_name.'_website_url_'.$k, $url);
            }
        }

        return $data;
    }

    private function renderComparatorSelectionList($comparator)
    {
        $selected_categories = $selected_manufacturers = '';
        $selection = $category_name = $manufacturer_name = array();
        $comparator_name = $comparator['name'];

        $categories_array = $comparator['selected_categories'] ? explode(',', $comparator['selected_categories']) : array();
        if (count($categories_array)) {
            foreach ($categories_array as $row) {
                $category = new Category($row, (int) $this->context->language->id);

                if (Validate::isLoadedObject($category)) {
                    $category_name[] = $category->name;
                }
            }

            if (count($category_name)) {
                $selected_categories = implode(', ', $category_name);
            }
        }

        $manufacturers_array = $comparator['selected_manufacturers'] ? explode(',', $comparator['selected_manufacturers']) : array();
        if (count($manufacturers_array)) {
            foreach ($manufacturers_array as $row) {
                $manufacturer = new Manufacturer($row, (int) $this->context->language->id);

                if (Validate::isLoadedObject($manufacturer)) {
                    $manufacturer_name[] = $manufacturer->name;
                }
            }

            if (count($manufacturer_name)) {
                $selected_manufacturers = implode(', ', $manufacturer_name);
            }
        }

        $selection[] = array(
            'name' => Tools::strtoupper($comparator_name),
            'selected_all' => $comparator['selected_all'],
            'selected_categories' => ($selected_categories ? $selected_categories : $this->l('No categories selected')),
            'selected_manufacturers' => ($selected_manufacturers ? $selected_manufacturers : $this->l('No manufacturers selected')),
            'selected_products_manually_type' => ($comparator['selected_products_manually_type'] == 'id' ? $this->l('by product ID') : $this->l('by product reference')),
            'selected_products_manually' => ($comparator['selected_products_manually'] ? str_replace(',', ', ', $comparator['selected_products_manually']) : '--'),
            'removed_products' => count($comparator['removed_products'] ? explode(',', $comparator['removed_products']) : array()),
            'badge_warning' => true,
        );

        $fields_list = array(
            'selected_all' => array(
                'title' => $this->l('All catalog'),
                'type' => 'action_icon',
                'align' => 'center',
            ),
            'selected_categories' => array(
                'title' => $this->l('Selected categories'),
                'type' => 'text',
                'align' => 'center',
                'maxlength' => 25,
            ),
            'selected_manufacturers' => array(
                'title' => $this->l('Selected manufacturers'),
                'type' => 'text',
                'align' => 'center',
                'maxlength' => 25,
            ),
            'selected_products_manually_type' => array(
                'title' => $this->l('Manual selection'),
                'type' => 'text',
                'align' => 'center',
            ),
            'selected_products_manually' => array(
                'title' => $this->l('Selected items manually'),
                'type' => 'text',
                'align' => 'center',
                'maxlength' => 25,
            ),
            'removed_products' => array(
                'title' => $this->l('Excluded products'),
                'type' => 'text',
                'align' => 'center',
                'badge_warning' => true,
            ),
        );

        $helper = new HelperList();
        $helper->shopLinkType = '';
        $helper->simple_header = true;
        $helper->identifier = 'name';
        $helper->actions = array('edit');
        $helper->table = 'ComparatorSelection';
        $helper->imageType = 'jpg';
        $helper->no_link = true;
        $helper->show_toolbar = false;
        $helper->module = $this;
        $helper->title = '<i class="icon-search"></i> '.$this->l('Products Selection');
        $helper->token = Tools::getAdminTokenLite('AdminModules');
        $helper->currentIndex = AdminController::$currentIndex.'&configure='.$this->name;

        return $helper->generateList($selection, $fields_list);
    }

    private function renderComparatorSelectionForm()
    {
        $id_shop = (int) $this->context->shop->id;
        $comparator_name = Tools::strtolower(Tools::getValue('name'));
        $comparator = MaofreeComparatorClass::getValuesFromTable(self::MAIN_TABLE, $id_shop, $comparator_name);
        $manufacturers = Manufacturer::getManufacturers(false, 0, true);

        $description = '
            <ul style="padding-left: 10px; padding-bottom: 5px">
                <li>'.$this->l('Will be selected only the products orderable and visible everywhere').'.</li>
                <li>'.$this->l('The result of this research is represented by the union of selections indicated by you (categories, manufacturers and ID)').'.</li>
            </ul>';

        $selection = array(
            array('id_selection' => 'id', 'selection' => $this->l('by Product ID')),
            array('id_selection' => 'reference', 'selection' => $this->l('by Product Reference')),
        );

        $fields_form_1 = array(
            'form' => array(
                'legend' => array(
                    'title' => $this->l('Products selection for').' <span style="color: red;">'.$comparator_name.'</span>',
                    'icon' => 'icon-search-plus',
                ),
                'description' => $description,
                'input' => array(
                    array(
                        'type' => 'switch',
                        'is_bool' => true,
                        'label' => $this->l('All products of catalog').':',
                        'name' => 'selectall',
                        'values' => array(
                            array(
                                'id' => 'active_on',
                                'value' => 1,
                                'label' => $this->l('Enabled'),
                            ),
                            array(
                                'id' => 'active_off',
                                'value' => 0,
                                'label' => $this->l('Disabled'),
                            ),
                        ),
                    ),
                    array(
                        'type' => 'categories',
                        'label' => $this->l('Categories to select').':',
                        'name' => 'selectcate',
                        'tree' => array(
                            'use_search' => true,
                            'id' => 'selectcategories',
                            'use_checkbox' => true,
                            'full_tree' => true,
                            'selected_categories' => $comparator['selected_categories'] ? explode(',', $comparator['selected_categories']) : array(),
                        ),
                    ),
                    array(
                        'type' => 'select',
                        'label' => $this->l('Manufacturers to select').':',
                        'name' => 'selectmanu[]',
                        'id' => 'manufacturers',
                        'desc' => $this->l('(Hold CTRL to select multiples)'),
                        'multiple' => true,
                        'size' => 20,
                        'options' => array(
                            'query' => $manufacturers,
                            'id' => 'id_manufacturer',
                            'name' => 'name',
                        ),
                    ),
                    array(
                        'type' => 'select',
                        'label' => $this->l('Type of manual selection').':',
                        'name' => 'manual_selection_type',
                        'options' => array(
                            'query' => $selection,
                            'id' => 'id_selection',
                            'name' => 'selection',
                        ),
                    ),
                    array(
                        'type' => 'textarea',
                        'label' => $this->l('Items').':',
                        'name' => 'selectmanual',
                        'desc' => $this->l('Write all the references or the IDs of products that you want to insert in the comparator, separated by a comma'),
                        'cols' => 40,
                        'rows' => 5,
                    ),
                ),
                'submit' => array(
                    'title' => $this->l('Save'),
                    'class' => 'btn btn-default pull-right',
                    'name' => 'submitComparatorSelection',
                ),
                'buttons' => array(
                    array(
                        'href' => AdminController::$currentIndex.'&configure='.$this->name.'&name='.Tools::strtoupper($comparator_name).'&viewComparatorList&token='.Tools::getAdminTokenLite('AdminModules'),
                        'title' => $this->l('Back'),
                        'icon' => 'process-icon-back',
                    ),
                ),
            ),
        );

        $helper = new HelperForm();
        $helper->name_controller = $this->name;
        $lang = new Language((int) Configuration::get('PS_LANG_DEFAULT'));
        $helper->default_form_language = $lang->id;
        $helper->module = $this;
        $helper->toolbar_scroll = true;
        $helper->allow_employee_form_lang = Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG') ? Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG') : 0;
        $helper->identifier = $this->identifier;
        $helper->currentIndex = AdminController::$currentIndex.'&configure='.$this->name.'&name='.Tools::strtoupper($comparator_name);
        $helper->token = Tools::getAdminTokenLite('AdminModules');
        $helper->tpl_vars = array(
            'fields_value' => $this->getConfigFieldsValues4($comparator),
            'languages' => $this->context->controller->getLanguages(),
            'id_language' => $this->context->language->id,
        );

        return $helper->generateForm(array($fields_form_1));
    }

    private function getConfigFieldsValues4($comparator)
    {
        $selected_products_manually = $comparator['selected_products_manually'];
        $selected_products_manually_type = $comparator['selected_products_manually_type'];

        if ($selected_products_manually_type == 'reference' && $selected_products_manually) {
            $selected_products_array = array();
            $selected_products = explode(',', $selected_products_manually);

            foreach ($selected_products as $row) {
                $product = new Product($row);

                if ($product->reference) {
                    $selected_products_array[] = $product->reference;
                }
            }

            if ($selected_products_array) {
                $selected_products_manually = implode(',', $selected_products_array);
            }
        }

        return array(
            'selectall' => Tools::getValue('selectall', $comparator['selected_all']),
            'selectmanu[]' => $comparator['selected_manufacturers'] ? explode(',', $comparator['selected_manufacturers']) : array(),
            'manual_selection_type' => Tools::getValue('manual_selection_type', $selected_products_manually_type),
            'selectmanual' => $selected_products_manually,
        );
    }

    private function displayComparatorProductsList($comparator)
    {
        $output = '';
        $pagination = array(500, 1000, 1500, 2000);
        $pageNumber = 1;
        $nbProducts = $pagination[0];

        if (Tools::isSubmit('submitComparatorProductsPagination')) {
            if (
                isset($this->context->cookie->{'ComparatorProducts_pagination'}) &&
                $this->context->cookie->{'ComparatorProducts_pagination'} != (int) Tools::getValue('ComparatorProducts_pagination')
            ) {
                $pageNumber = 1;
            } else {
                $pageNumber = (int) Tools::getValue('ComparatorProducts_page');
            }
            $nbProducts = (int) Tools::getValue('ComparatorProducts_pagination');

            $this->context->cookie->{'ComparatorProducts_page'} = $pageNumber;
            $this->context->cookie->{'ComparatorProducts_pagination'} = $nbProducts;
        } else {
            if (isset($this->context->cookie->{'ComparatorProducts_page'})) {
                $pageNumber = (int) $this->context->cookie->{'ComparatorProducts_page'};
            }

            if (isset($this->context->cookie->{'ComparatorProducts_pagination'})) {
                $nbProducts = (int) $this->context->cookie->{'ComparatorProducts_pagination'};
            }
        }

        $id_shop = (int) $this->context->shop->id;
        $id_lang = (int) $this->context->language->id;
        $id_root_cat = (int) Configuration::get('PS_ROOT_CATEGORY');
        $id_home_cat = (int) Configuration::get('PS_HOME_CATEGORY');
        $comparator_name = $comparator['name'];
        $selectall = (int) $comparator['selected_all'];
        $selectcat = $comparator['selected_categories'];
        $selectmanu = $comparator['selected_manufacturers'];
        $selectmanual = $comparator['selected_products_manually'];
        $stock_management = (int) Configuration::get('PS_STOCK_MANAGEMENT'); // Get stock management condition
        $allow_out_of_stock_ordering = (int) Configuration::get('PS_ORDER_OUT_OF_STOCK'); // Get allow out of stock ordering
        $totalproducts = MaofreeComparatorClass::getTotalProductsComparator(
            $id_shop,
            $id_lang,
            $selectall,
            $selectmanu,
            $selectcat,
            $selectmanual,
            '',
            false,
            false,
            $stock_management,
            $allow_out_of_stock_ordering,
            $id_root_cat,
            $id_home_cat,
            $nbProducts * ($pageNumber - 1),
            $nbProducts
        );

        if (is_array($totalproducts) && count($totalproducts)) {
            $id_product_array = array();
            $p = 0;
            $id_totalproducts = MaofreeComparatorClass::getIdProducts(
                $id_shop,
                $selectall,
                $selectmanu,
                $selectcat,
                $selectmanual,
                '',
                false,
                false,
                $stock_management,
                $allow_out_of_stock_ordering,
                $id_root_cat,
                $id_home_cat,
                null,
                null
            );
            $total_count = count($id_totalproducts);

            foreach ($id_totalproducts as $id) {
                $id_product_array[] = (int) $id['id_product'];
            }

            $id_product_array = array_diff($id_product_array, explode(',', $comparator['removed_products']));
            $active_count = count($id_product_array);
            $total_pages = max(1, ceil($total_count / $nbProducts));

            $output .= '
		<style type="text/css">
			.table tbody > tr.alt_row > td { background-color: #EDFDFD }
			.table tbody > tr.alt_row2 > td { background-color: #C0E0E0 }
			.table tbody > tr.alt_row3 > td { background-color: #D3D3D3 }

			#mao_ajax_running { display: none; z-index: 200000; background-color: #000; opacity: 0.5; height: 100%; width: 100%; position: fixed; top: 0; left: 0 }
			#mao_ajax_running img { position: absolute; top: 50%; left: 50% }
		</style>

		<script type="text/javascript">
			$(document).ready(function() {
				$("body").append("<div id=\'mao_ajax_running\'><img src=\''.$this->_path.'views/img/ajax-loader.gif\'></div>");
				$("#submitComparatorProducts").on("click", function() {
					$("#mao_ajax_running").css("display", "block");
				});

				$(".pagination-items-page").on("click",function(e){
					e.preventDefault();

					$("#ComparatorProducts-pagination").val($(this).data("items"));
					$("#submitComparatorProductsPagination").click();
				});

				$(".pagination-link").on("click",function(e){
					e.preventDefault();

					if (!$(this).parent().hasClass("disabled")) {
						$("#ComparatorProducts-page").val($(this).data("page"));
						$("#submitComparatorProductsPagination").click();
					}
				});
			});
		</script>

		<div class="clearfix">
			<div class="panel col-lg-12">
				<div class="panel-heading">
					<i class="icon-list"></i>
					'.$this->l('Products for its data feed').' <span class="badge">'.$total_count.'</span>
				</div>';

            if ($total_count > $active_count) {
                $output .= '
				<div class="alert alert-warning">
					'.$this->l('Products included').'&nbsp;&nbsp;&nbsp;<span class="badge badge-success">'.$active_count.'</span>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;--&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;'.$this->l('Products excluded').'&nbsp;&nbsp;&nbsp;<span class="badge badge-danger">'.($total_count - $active_count).'</span>
				</div>';
            }

            $output .= '
				<div class="alert alert-warning">
					'.$this->l('When you choose the products to enter is necessary to consider the performance of the server that hosts the website, like the timeout and the memory limit but also the maximum time allowed by the comparator to load the data feed in its systems. If you incur in errors, to solve them, try to decrease the number of products to put in data feed.').'
				</div>
				<div class="alert alert-info">
					'.$this->l('Here you can select the products that you do not want to include into the data feed').'
				</div>

				<form class="form-horizontal clearfix" action="'.Tools::safeOutput($_SERVER['REQUEST_URI']).'" method="post" autocomplete="off">
					<div class="table-responsive clearfix">
						<table class="table" style="text-align: center;">
							<thead>
								<tr height="40">
									<th style="width: 30px; border-left: 0"> </th>
									<th style="width: 40px; text-align: center;">'.$this->l('ID').'</th>
									<th style="width: 50px; text-align: center;">'.$this->l('Image').'</th>
									<th style="text-align: center;">'.$this->l('Name').'</th>
									<th style="text-align: center;" class="fixed-width-md">'.$this->l('Reference').'</th>
									<th style="text-align: center;" class="fixed-width-lg">'.$this->l('Manufacturer').'</th>
									'.($stock_management ? '<th style="width: 50px; text-align: center;">'.$this->l('Quantity').'</th>' : '').'
									<th style="width: 50px; text-align: center;">'.$this->l('Combinations').'</th>
								</tr>
							</thead>
							<tbody>';

            $numberow = 0;
            $removed_products_id_array = array();
            $urlink = $this->context->link->getAdminLink('AdminProducts');
            $removed_products_id = $comparator['removed_products'];
            $removed_products_id_array = $removed_products_id ? explode(',', $removed_products_id) : array();

            foreach ($totalproducts as $product) {
                $id_product = (int) $product['id_product'];
                $product_obj = new Product($id_product);
                $removed_product = in_array($id_product, $removed_products_id_array);
                $image = Product::getCover($id_product);
                $cover = $id_product.'-'.(int) $image['id_image'];
                if ($stock_management) {
                    $quantity = (int) StockAvailable::getQuantityAvailableByProduct($id_product, null, $id_shop);
                }

                $output .= '
								<tr'.($removed_product ? ' class="alt_row2"' : ($numberow++ % 2 ? ' class="alt_row"' : '')).'>
									<td><input type="checkbox" '.($removed_product ? 'checked="checked" ' : '').'value="'.$id_product.'" name="removeCheckBox[]" /></td>
									<td>'.$id_product.'</td>
									<td>
										<a title="'.$product['name'].'" target="_blank" href="'.$urlink.'&id_product='.$id_product.'&updateproduct">
											<img alt="'.$product['name'].'" src="'.(Image::getImages($id_lang, $id_product) ? $this->context->link->getImageLink($product['link_rewrite'], $cover, ImageType::getFormatedName('small')) : $this->_path.'views/img/noimage.jpg').'" height="50px" width="50px" />
										</a>
									</td>
									<td>
										<a title="'.$product['name'].'" target="_blank" href="'.$urlink.'&id_product='.$id_product.'&updateproduct">
											<p>'.Tools::substr($product['name'], 0, 150).'</p>
										</a>
									</td>
									<td>'.$product['reference'].'</td>
									<td>'.$product['manufacturer_name'].'</td>
									'.($stock_management ? '<td><span class="badge badge-'.($quantity ? 'success' : 'danger').'">'.$quantity.'</span></td>' : '').'
									<td><span class="badge badge-warning">'.(int) $product_obj->hasAttributes().'</span></td>
								</tr>';
            }

            $output .= '
								<input type="hidden" name="ComparatorProducts_pagination" value="'.$nbProducts.'" />
								<input type="hidden" name="ComparatorProducts_page" value="'.$pageNumber.'" />
								<input type="hidden" name="name" value="'.$comparator_name.'" />
							</tbody>
						</table>
					</div>

					<div class="panel-body">';

            if ($total_count > $pagination[0]) {
                $output .= '
						<div class="col-lg-6">
							<div class="pagination">
								'.$this->l('Display').'
								<button type="button" class="btn btn-default dropdown-toggle" data-toggle="dropdown">
									'.$nbProducts.'
									<i class="icon-caret-down"></i>
								</button>
								<ul class="dropdown-menu">';

                foreach ($pagination as $value) {
                    $output .= '
									<li>
										<a href="javascript:void(0);" class="pagination-items-page" data-items="'.(int) $value.'">'.(int) $value.'</a>
									</li>';
                }

                $output .= '
								</ul>
								/ '.$total_count.' '.$this->l('result(s)').'
							</div>
							<ul class="pagination pull-right">
								<li'.($pageNumber <= 1 ? ' class="disabled"' : '').'>
									<a href="javascript:void(0);" class="pagination-link" data-page="1">
										<i class="icon-double-angle-left"></i>
									</a>
								</li>
								<li'.($pageNumber <= 1 ? ' class="disabled"' : '').'>
									<a href="javascript:void(0);" class="pagination-link" data-page="'.($pageNumber - 1).'">
										<i class="icon-angle-left"></i>
									</a>
								</li>';

                while ($p++ < $total_pages) {
                    if ($p < $pageNumber - 2) {
                        $output .= '
								<li class="disabled">
									<a href="javascript:void(0);">&hellip;</a>
								</li>';

                        $p = $pageNumber - 3;
                    } elseif ($p > $pageNumber + 2) {
                        $output .= '
								<li class="disabled">
									<a href="javascript:void(0);">&hellip;</a>
								</li>';

                        $p = $total_pages;
                    } else {
                        $output .= '
								<li'.($p == $pageNumber ? ' class="active"' : '').'>
									<a href="javascript:void(0);" class="pagination-link" data-page="'.$p.'">'.$p.'</a>
								</li>';
                    }
                }

                $output .= '
								<li'.($pageNumber >= $total_pages ? ' class="disabled"' : '').'>
									<a href="javascript:void(0);" class="pagination-link" data-page="'.($pageNumber + 1).'">
										<i class="icon-angle-right"></i>
									</a>
								</li>
								<li'.($pageNumber >= $total_pages ? ' class="disabled"' : '').'>
									<a href="javascript:void(0);" class="pagination-link" data-page="'.$total_pages.'">
										<i class="icon-double-angle-right"></i>
									</a>
								</li>
							</ul>
						</div>';
            }

            $output .= '
						<div'.($total_count > $pagination[0] ? ' class="col-lg-6"' : '').'>
							<button style="font-size: 17px; white-space: normal; margin-top: 11px;" class="btn btn-default pull-right" type="submit" id="submitComparatorProducts" name="submitComparatorProducts" onclick="return confirm(\''.$this->l('Are you sure you want to proceed with this operation?').'\');"><i class="process-icon-save" style="display: inline"></i> '.$this->l('Update the data to be passed to this comparator').'</button>
						</div>
					</div>
				</form>

				<div class="panel-footer">
					<div class="col-lg-6">
						<a class="btn btn-default" href="'.AdminController::$currentIndex.'&configure='.$this->name.'&name='.Tools::strtoupper($comparator_name).'&viewComparatorList&token='.Tools::getAdminTokenLite('AdminModules').'"><i class="process-icon-back"></i> '.$this->l('Back').'</a>
					</div>

					<form style="display: none" action="'.Tools::safeOutput($_SERVER['REQUEST_URI']).'" method="post">
						<input type="hidden" id="ComparatorProducts-pagination" name="ComparatorProducts_pagination" value="'.$nbProducts.'" />
						<input type="hidden" id="ComparatorProducts-page" name="ComparatorProducts_page" value="'.$pageNumber.'" />
						<input type="hidden" name="name" value="'.$comparator_name.'" />

						<input type="submit" name="submitComparatorProductsPagination" id="submitComparatorProductsPagination" value="" />
					</form>
				</div>
			</div>
		</div>';
        } else {
            $output = '
		<div class="panel">'.$this->displayError($this->l('There are no products assigned to this comparator. For first thing you must go into the "products selection", to choose which products must be added to the data feed.')).'</div>

		<div>
			<a class="btn btn-default" href="'.AdminController::$currentIndex.'&configure='.$this->name.'&name='.Tools::strtoupper($comparator_name).'&viewComparatorList&token='.Tools::getAdminTokenLite('AdminModules').'"><i class="process-icon-back"></i> '.$this->l('Back').'</a>
		</div>';
        }

        return $output;
    }

    private function renderTaxonomyAssociationList($comparator)
    {
        $fields_list = $selection = array();
        $comparator_name = $comparator['name'];
        $comparator_obj = new $comparator_name();
        $directory = _PS_MODULE_DIR_.$this->name.'/comparators/'.$comparator_name.'/taxonomy';
        $taxonomy_files = $comparator_obj->taxonomy_files;

        foreach ($taxonomy_files as $key => $row) {
            $timestamp = 0;
            $taxonomy_url = false;
            $taxonomy_filename = '';

            if (isset($row['url']) && $row['url']) {
                $taxonomy_url = true;
            }
            if (isset($row['filename']) && $row['filename']) {
                $taxonomy_filename = $row['filename'];
            }

            if (file_exists($directory)) {
                $scanned_directory = array_diff(scandir($directory), array('..', '.', 'index.php'));

                if (is_array($scanned_directory) && count($scanned_directory)) {
                    if ($taxonomy_filename) {
                        foreach ($scanned_directory as $file) {
                            if (!strstr($file, '.txt') || strpos($file, $taxonomy_filename) === false) {
                                continue;
                            }

                            $timestamp = filemtime($directory.'/'.$file);

                            break;
                        }
                    }
                }
            }

            if ($row['filename'] && $taxonomy_url) {
                $updatefile_text = 'url';
                $updatefile_link = AdminController::$currentIndex.'&configure='.$this->name.'&name='.Tools::strtoupper($comparator_name).'&updateUrlTaxonomyFiles&taxonomy_files_id='.$key.'&token='.Tools::getAdminTokenLite('AdminModules');
            } elseif ($row['filename'] && !$taxonomy_url) {
                $updatefile_text = 'nourl';
                $updatefile_link = AdminController::$currentIndex.'&configure='.$this->name.'&name='.Tools::strtoupper($comparator_name).'&update'.($row['type'] == 'language' ? 'Language' : 'Country').'NoUrlTaxonomyFiles&taxonomy_files_id='.$key.'&token='.Tools::getAdminTokenLite('AdminModules');
            } else {
                $updatefile_text = 'nofile';
                $updatefile_link = '';
            }

            if ($row['form'] == 'ajax') {
                $field_text = $this->l('with ajax search');
            } elseif ($row['form'] == 'select') {
                $field_text = $this->l('with input select');
            } else {
                $field_text = $this->l('only with input text');
            }

            if ($row['type'] == 'language') {
                $type_text = $this->l('by languages');
            } else {
                $type_text = $this->l('by countries');
            }

            $selection[] = array(
                'taxonomy_files_id' => $key,
                'field' => $row['field'].'_taxonomy',
                'field_color' => '#FF4500',
                'type' => $type_text,
                'form' => $field_text,
                'date' => $timestamp ? date($this->context->language->date_format_full, $timestamp) : $this->l('Files no present'),
                'date_color' => '#1E90FF',
                'updatefile' => $updatefile_text,
                'urllink' => $updatefile_link,
            );

            $asso = array();

            if ($row['type'] == 'language') {
                $tot_asso_array = MaofreeComparatorClass::getAssoTaxonomyData(
                    self::TAXONOMY_LANG_TABLE,
                    (int) $this->context->shop->id,
                    $comparator_name,
                    $row['type'],
                    $row['field']
                );

                foreach ($tot_asso_array as $line) {
                    $language = new Language((int) $line['id_lang']);
                    $tot_count = MaofreeComparatorClass::getAssoNumTaxonomyData(
                        self::TAXONOMY_LANG_TABLE,
                        (int) $this->context->shop->id,
                        (int) $line['id_lang'],
                        $comparator_name,
                        $row['type'],
                        $row['field']
                    );

                    $asso[] = $language->name.'@'.$tot_count;
                }
            } else {
                $tot_asso_array = MaofreeComparatorClass::getAssoTaxonomyData(
                    self::TAXONOMY_COUNTRY_TABLE,
                    (int) $this->context->shop->id,
                    $comparator_name,
                    $row['type'],
                    $row['field']
                );

                foreach ($tot_asso_array as $line) {
                    $country = new Country((int) $line['id_country'], (int) $this->context->language->id);
                    $tot_count = MaofreeComparatorClass::getAssoNumTaxonomyData(
                        self::TAXONOMY_COUNTRY_TABLE,
                        (int) $this->context->shop->id,
                        (int) $line['id_country'],
                        $comparator_name,
                        $row['type'],
                        $row['field']
                    );

                    $asso[] = $country->name.'@'.$tot_count;
                }
            }

            $selection[$key]['taxonomy_asso'] = $asso ? implode('#', $asso) : '';
        }

        $fields_list = array(
            'field' => array(
                'title' => $this->l('Taxonomy field'),
                'type' => 'text',
                'align' => 'left',
                'class' => 'fixed-width-xl',
                'color' => 'field_color',
                'search' => false,
                'orderby' => false,
            ),
            'taxonomy_asso' => array(
                'title' => $this->l('Associations'),
                'hint' => $this->l('Associated elements'),
                'type' => 'text_complex',
                'text_noitem' => $this->l('No association currently'),
                'align' => 'center',
                'maxlength' => 70,
                'search' => false,
                'orderby' => false,
            ),
            'type' => array(
                'title' => $this->l('Associations type'),
                'type' => 'text',
                'align' => 'center',
                'search' => false,
                'orderby' => false,
            ),
            'form' => array(
                'title' => $this->l('Form type'),
                'type' => 'text',
                'align' => 'center',
                'search' => false,
                'orderby' => false,
            ),
            'date' => array(
                'title' => $this->l('Last change'),
                'type' => 'text',
                'align' => 'center',
                'class' => 'fixed-width-xl',
                'color' => 'date_color',
                'search' => false,
                'orderby' => false,
            ),
            'updatefile' => array(
                'title' => $this->l('Update taxonomy files'),
                'hint' => $this->l('From here you can update the taxonomy files, updating them automatically from this comparator or loading them from your pc (normally they are updated rarely)'),
                'type' => 'update_file',
                'text_url' => $this->l('Update the taxonomy file from the site of this comparator'),
                'text_nourl' => $this->l('Update the taxonomy file loading a new one from your pc (accepted only original files)'),
                'text_nofile' => $this->l('Files not required'),
                'hreflink' => 'urllink',
                'align' => 'center',
                'class' => 'fixed-width-xl',
                'search' => false,
                'orderby' => false,
            ),
        );

        $helper = new HelperList();
        $helper->shopLinkType = '';
        $helper->simple_header = false;
        $helper->listTotal = count($selection);
        $helper->identifier = 'taxonomy_files_id';
        $helper->actions = array('edit');
        $helper->table = 'TaxonomyAssociation';
        $helper->imageType = 'jpg';
        $helper->no_link = true;
        $helper->show_toolbar = false;
        $helper->module = $this;
        $helper->title = '<i class="icon-random"></i> '.$this->l('Taxonomy Summary');
        $helper->toolbar_btn = array(
            'active' => array(
                'desc' => ($comparator['taxonomy'] ? $this->l('Disable taxonomy for this comparator') : $this->l('Active taxonomy for this comparator')),
                'imgclass' => 'power',
                'href' => AdminController::$currentIndex.'&configure='.$this->name.'&name='.Tools::strtoupper($comparator_name).'&activeTaxonomyAssociation&token='.Tools::getAdminTokenLite('AdminModules'),
            ),
        );
        $helper->token = Tools::getAdminTokenLite('AdminModules');
        $helper->currentIndex = AdminController::$currentIndex.'&configure='.$this->name.'&name='.Tools::strtoupper($comparator_name);

        return $helper->generateList($selection, $fields_list);
    }

    private function displayLanguageUpdateNoUrlTaxonomyFilesForm($comparator)
    {
        $found_taxonomy = $disabled_languages = array();
        $comparator_name = $comparator['name'];
        $comparator_obj = new $comparator_name();
        $id_shop = (int) $this->context->shop->id;
        $languages = Language::getLanguages(true, $id_shop);
        $taxonomy_files_id = (int) Tools::getValue('taxonomy_files_id');
        $orig_file_ext = $comparator_obj->taxonomy_files[$taxonomy_files_id]['orig_file_ext'];

        foreach ($languages as $lang) {
            $language = new Language((int) $lang['id_lang']);
            $iso_code_array = explode('-', $lang['language_code']);
            $iso_code = Tools::strtolower($iso_code_array[0]).'-'.Tools::strtoupper($iso_code_array[1]);

            if (
                $comparator_obj->bestTaxonomyIsoCode(
                    $language->language_code,
                    $comparator_obj->taxonomy_files[$taxonomy_files_id]['type']
                )
            ) {
                $found_taxonomy[] = array('id_language' => (int) $lang['id_lang'], 'language' => $iso_code.' - '.$language->name);
            } else {
                $disabled_languages[] = '"'.$language->name.'"';
            }
        }

        if (count($found_taxonomy) != count($languages)) {
            $description = sprintf($this->l('There are "%d" languages that you cannot use for the taxonomy, because they are not managed by this comparator'), count($languages) - count($found_taxonomy)).':<br />'.implode(', ', $disabled_languages);
        } else {
            $description = '';
        }

        $fields_form_1 = array(
            'form' => array(
                'legend' => array(
                    'title' => $this->l('Upload a new taxonomy file for').' <span style="color: red;">'.Tools::strtoupper($comparator_name).'</span>',
                    'icon' => 'icon-upload',
                ),
                'description' => $description,
                'input' => array(
                    array(
                        'type' => 'file',
                        'label' => $this->l('File').':',
                        'desc' => $this->l('The file must be taken from this comparator (ask to its employees). It must be in the format').': <span style="color: red; font-weight: bold;">'.Tools::strtoupper($orig_file_ext).'</span>',
                        'name' => 'taxonomy_file',
                    ),
                    array(
                        'type' => 'select',
                        'label' => $this->l('For which language').'?',
                        'desc' => $this->l('Choose the language you want to associate to this file'),
                        'name' => 'taxonomy_id_lang',
                        'class' => 'fixed-width-xxl',
                        'options' => array(
                            'query' => $found_taxonomy,
                            'id' => 'id_language',
                            'name' => 'language',
                        ),
                    ),
                ),
                'submit' => array(
                    'title' => $this->l('Upload'),
                    'icon' => 'process-icon-upload',
                    'class' => 'btn btn-default pull-right',
                    'name' => 'submitLanguageUpdateNoUrlTaxonomy',
                ),
                'buttons' => array(
                    array(
                        'href' => AdminController::$currentIndex.'&configure='.$this->name.'&name='.Tools::strtoupper($comparator_name).'&viewComparatorList&token='.Tools::getAdminTokenLite('AdminModules'),
                        'title' => $this->l('Back'),
                        'icon' => 'process-icon-back',
                    ),
                ),
            ),
        );

        $helper = new HelperForm();
        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $lang = new Language((int) Configuration::get('PS_LANG_DEFAULT'));
        $helper->default_form_language = $lang->id;
        $helper->module = $this;
        $helper->allow_employee_form_lang = Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG') ? Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG') : 0;
        $helper->identifier = $this->identifier;
        $helper->currentIndex = AdminController::$currentIndex.'&configure='.$this->name.'&name='.Tools::strtoupper($comparator_name).'&taxonomy_files_id='.$taxonomy_files_id;
        $helper->token = Tools::getAdminTokenLite('AdminModules');
        $helper->tpl_vars = array(
            'fields_value' => $this->getConfigFieldsValues(),
            'languages' => $this->context->controller->getLanguages(),
            'id_language' => $this->context->language->id,
        );

        return $helper->generateForm(array($fields_form_1));
    }

    private function getConfigFieldsValues()
    {
        return array(
            'taxonomy_id_lang' => Tools::getValue('taxonomy_id_lang', (int) Configuration::getGlobalValue('MAO_COMPARATOR_TAXONOMY_LANG')),
        );
    }

    private function displayLanguageTaxonomySelectionForm($comparator)
    {
        $found_taxonomy = $disabled_languages = array();
        $comparator_name = $comparator['name'];
        $comparator_obj = new $comparator_name();
        $id_shop = (int) $this->context->shop->id;
        $languages = Language::getLanguages(true, $id_shop);
        $taxonomy_files_id = (int) Tools::getValue('taxonomy_files_id');

        foreach ($languages as $lang) {
            $language = new Language((int) $lang['id_lang']);
            $iso_code_array = explode('-', $lang['language_code']);
            $iso_code = Tools::strtolower($iso_code_array[0]).'-'.Tools::strtoupper($iso_code_array[1]);

            if ($comparator_obj->bestTaxonomyIsoCode($language->language_code, $comparator_obj->taxonomy_files[$taxonomy_files_id]['type'])) {
                $found_taxonomy[] = array('id_language' => (int) $lang['id_lang'], 'language' => $iso_code.' - '.$language->name);
            } else {
                $disabled_languages[] = '"'.$language->name.'"';
            }
        }

        if (count($found_taxonomy) != count($languages)) {
            $description = sprintf($this->l('There are "%d" languages that you cannot use for the taxonomy, because they are not managed by this comparator'), count($languages) - count($found_taxonomy)).':<br />'.implode(', ', $disabled_languages);
        } else {
            $description = '';
        }

        $fields_form_1 = array(
            'form' => array(
                'legend' => array(
                    'title' => $this->l('Select the language to use in the next block'),
                    'icon' => 'icon-flag-checkered',
                ),
                'description' => $description,
                'input' => array(
                    array(
                        'type' => 'select',
                        'label' => $this->l('Language').':',
                        'name' => 'taxonomy_id_lang',
                        'class' => 'fixed-width-xxl',
                        'options' => array(
                            'query' => $found_taxonomy,
                            'id' => 'id_language',
                            'name' => 'language',
                        ),
                    ),
                ),
                'submit' => array(
                    'title' => $this->l('Save'),
                    'class' => 'btn btn-default pull-right',
                    'name' => 'submitLanguageTaxonomySelection',
                ),
                'buttons' => array(
                    array(
                        'href' => AdminController::$currentIndex.'&configure='.$this->name.'&name='.Tools::strtoupper($comparator_name).'&viewComparatorList&token='.Tools::getAdminTokenLite('AdminModules'),
                        'title' => $this->l('Back'),
                        'icon' => 'process-icon-back',
                    ),
                ),
            ),
        );

        $helper = new HelperForm();
        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $lang = new Language((int) Configuration::get('PS_LANG_DEFAULT'));
        $helper->default_form_language = $lang->id;
        $helper->module = $this;
        $helper->allow_employee_form_lang = Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG') ? Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG') : 0;
        $helper->identifier = $this->identifier;
        $helper->currentIndex = AdminController::$currentIndex.'&configure='.$this->name.'&name='.Tools::strtoupper($comparator_name).'&taxonomy_files_id='.$taxonomy_files_id;
        $helper->token = Tools::getAdminTokenLite('AdminModules');
        $helper->tpl_vars = array(
            'fields_value' => $this->getConfigFieldsValues6(),
            'languages' => $this->context->controller->getLanguages(),
            'id_language' => $this->context->language->id,
        );

        return $helper->generateForm(array($fields_form_1));
    }

    private function getConfigFieldsValues6()
    {
        return array(
            'taxonomy_id_lang' => Tools::getValue('taxonomy_id_lang', (int) Configuration::getGlobalValue('MAO_COMPARATOR_TAXONOMY_LANG')),
        );
    }

    private function displayLanguageSummaryTaxonomyAssociation($comparator, $taxonomy_files)
    {
        $output = '';
        $comparator_name = $comparator['name'];
        $id_shop = (int) $this->context->shop->id;
        $id_lang = (int) Configuration::getGlobalValue('MAO_COMPARATOR_TAXONOMY_LANG');
        $taxonomy_files_id = (int) Tools::getValue('taxonomy_files_id');
        $categories = Category::getSimpleCategories($id_lang);
        $taxonomy_form = $taxonomy_files['form'];
        $taxonomy_field = $taxonomy_files['field'];
        $taxonomy_type = $taxonomy_files['type'];
        $active_count = count(
            MaofreeComparatorClass::getTaxonomyData(
                self::TAXONOMY_LANG_TABLE,
                $id_shop,
                $comparator_name,
                $taxonomy_type,
                $taxonomy_field,
                $id_lang,
                null
            )
        );

        if ($taxonomy_form == 'ajax') {
            $this->context->controller->addCSS($this->_path.'views/css/search_style.css', 'all');

            $output .= '
		<script type="text/javascript">
			var mao_comparator_name = "'.$comparator_name.'";
			var mao_id_lang = '.$id_lang.';
			var mao_id_country = '.(int) Configuration::get('PS_COUNTRY_DEFAULT').';
			var taxonomy_files_id = '.$taxonomy_files_id.';
		</script>
		<script type="text/javascript" src="'.$this->_path.'views/js/maofree_search.js"></script>';
        }

        $output .= '
		<div class="clearfix">
			<div class="panel col-lg-12">
				<div class="panel-heading">
					<i class="icon-bar-chart"></i>
					'.$this->l('Summary of').' <span style="color: red;">'.Tools::strtoupper($comparator_name).'</span> <img style="margin-left: 10px;" class="img-thumbnail" alt="" src="../img/tmp/lang_mini_'.$id_lang.'_'.$id_shop.'.jpg?t='.time().'" />
				</div>
				<div class="alert alert-warning">
					'.$this->l('Categories associated').'&nbsp;&nbsp;&nbsp;<span class="badge badge-success">'.$active_count.'</span>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;--&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;'.$this->l('Categories not associated').'&nbsp;&nbsp;&nbsp;<span class="badge badge-danger">'.((count($categories) - 1) - $active_count).'</span>
				</div>
			</div>
		</div>';

        return $output;
    }

    private function displayLanguageTaxonomyAssociationForm($comparator, $taxonomy_files)
    {
        $output = '';
        $all_taxonomy = array();
        $comparator_name = $comparator['name'];
        $id_shop = (int) $this->context->shop->id;
        $id_lang = (int) Configuration::getGlobalValue('MAO_COMPARATOR_TAXONOMY_LANG');
        $id_root_cat = (int) Configuration::get('PS_ROOT_CATEGORY');
        $id_home_cat = (int) Configuration::get('PS_HOME_CATEGORY');
        $categories = Category::getSimpleCategories($id_lang);
        $taxonomy_files_id = (int) Tools::getValue('taxonomy_files_id');
        $taxonomy_form = $taxonomy_files['form'];
        $taxonomy_field = $taxonomy_files['field'];
        $taxonomy_type = $taxonomy_files['type'];
        $back_url = AdminController::$currentIndex.'&configure='.$this->name.'&name='.Tools::strtoupper($comparator_name).'&viewComparatorList&token='.Tools::getAdminTokenLite('AdminModules');
        $comparator_obj = new $comparator_name();
        $taxonomy_language = new Language($id_lang);

        if ($taxonomy_form == 'ajax') {
            $last_field_icon = 'search';
        } elseif ($taxonomy_form == 'text') {
            $last_field_icon = 'pencil';
        } else {
            $last_field_icon = 'list';
            $all_taxonomy = $comparator_obj->readTaxonomyFile($taxonomy_language->language_code, $taxonomy_files_id);
        }

        $output .= '
		<form class="clearfix" action="'.Tools::safeOutput($_SERVER['REQUEST_URI']).'" method="post" autocomplete="off">
			<div class="panel col-lg-12">
				<div class="panel-heading">
					<i class="icon-list"></i>
					'.$this->l('Taxonomy associations of').' <span style="color: red;">'.Tools::strtoupper($comparator_name).'</span> <img style="margin-left: 10px;" class="img-thumbnail" alt="" src="../img/tmp/lang_mini_'.$id_lang.'_'.$id_shop.'.jpg?t='.time().'" />
				</div>';

        if ($taxonomy_form == 'ajax') {
            $text_for_form = $this->l('On each field, enter a few characters and then select the most appropriate category');
        } elseif ($taxonomy_form == 'select') {
            $text_for_form = $this->l('On each field, choose which option is the most appropriate text for the associate category');
        } else {
            $text_for_form = $this->l('On each field, enter a text to specify in the best possible way the category to which it belongs. Ask to this comparator which text you need to enter');
        }

        $output .= '
				<div class="alert alert-info">
					<ul>
						<li>'.$this->l('Normally it is very important and mandatory to associate categories of your site to those of the comparator. It is not necessary to associate all categories, but only those containing products in the data feed').'.</li>
						<li>'.$this->l('The number shows the total products present in the category, the subcategories are not considered').'.</li>
						<li><b>'.$text_for_form.'.</b> '.$this->l('You must do this only for the languages that you used to create the data feed').'.</li>';

        if (isset($comparator_obj->taxonomy_ifo_url) && $comparator_obj->taxonomy_ifo_url) {
            $output .= '<li>'.$this->l('You can get more informations on how to manage the taxonomy from here').' <a style="font-weight: bold;" target="_blank" href="'.$comparator_obj->taxonomy_ifo_url.'">push here</a></li>';
        }

        $output .= '
					</ul>
				</div>';

        foreach ($categories as $category) {
            $id_category = (int) $category['id_category'];
            $category_obj = new Category($id_category);

            if ($id_category == $id_root_cat || $id_category == $id_home_cat) {
                continue;
            }

            $products_tot = (int) $category_obj->getProducts($id_lang, null, null, null, null, true, true, false, 1, false);
            $cat_name = MaofreeComparatorClass::getPath($id_category, $id_shop, $id_lang, ' > ', $id_root_cat, $id_home_cat);
            $taxonomy = MaofreeComparatorClass::getTaxonomyData(
                self::TAXONOMY_LANG_TABLE,
                $id_shop,
                $comparator_name,
                $taxonomy_type,
                $taxonomy_field,
                $id_lang,
                $id_category
            );
            if (!isset($taxonomy) || !$taxonomy) {
                $cat_taxonomy = '';
            } else {
                $cat_taxonomy = $taxonomy[$taxonomy_field.'_taxonomy'];
            }

            $output .= '
				<div class="form-group maoblocksearch">
					<label class="control-label col-md-2 intro-inputString" style="font-weight: bold;">'.$this->l('Category of this shop').':</label>
					<div class="input-group col-md-10">
						<span style="display: inline-block; font-weight: bold; vertical-align: middle; color: tomato;">'.$cat_name.'</span>
						<span class="badge badge-warning" style="margin-left: 20px">'.$products_tot.'</span>
					</div>
					<div class="input-group col-xs-12">
						<span class="input-group-addon"><i class="icon-random"></i></span>';

            if ($taxonomy_form == 'ajax' || $taxonomy_form == 'text') {
                $output .= '<input type="text" value="'.$cat_taxonomy.'" name="inputString-'.$id_category.'" id="inputString-'.$id_category.'" class="inputString" autocomplete="off" />';
            } else {
                $output .= '<select autocomplete="off" id="inputString-'.$id_category.'" name="inputString-'.$id_category.'">
					<option value="">* '.$this->l('Select a taxonomy category').' *</option>';

                foreach ($all_taxonomy as $row) {
                    $output .= '<option value="'.$row.'"'.($row == $cat_taxonomy ? ' selected="selected"' : '').'>'.$row.'</option>';
                }

                $output .= '</select>';
            }

            $output .= '<span class="input-group-addon"><i class="icon-'.$last_field_icon.'"></i></span>
						<div id="suggestions-'.$id_category.'" class="suggestions"></div>
					</div>
					<hr />
				</div>';
        }

        $output .= '
				<input type="hidden" name="taxonomy_id_lang" value="'.$id_lang.'">
				<input type="hidden" name="taxonomy_field" value="'.$taxonomy_field.'">
				<input type="hidden" name="taxonomy_form" value="'.$taxonomy_form.'">
				<input type="hidden" name="taxonomy_files_id" value="'.$taxonomy_files_id.'">
				<input type="hidden" name="name" value="'.$comparator_name.'">
				<div class="panel-footer">
					<button class="btn btn-default pull-right" name="submitLanguageTaxonomyAssociation" id="module_form_submit_btn_2" value="1" type="submit">
						<i class="process-icon-save"></i> '.$this->l('Save').'
					</button>
					<a class="btn btn-default" href="'.$back_url.'">
						<i class="process-icon-back"></i> '.$this->l('Back').'
					</a>
				</div>
			</div>
		</form>';

        return $output;
    }

    private function displayCountryUpdateNoUrlTaxonomyFilesForm($comparator)
    {
        $found_taxonomy = $disabled_countries = array();
        $comparator_name = $comparator['name'];
        $comparator_obj = new $comparator_name();
        $countries = Country::getCountries((int) $this->context->language->id, true);
        $taxonomy_files_id = (int) Tools::getValue('taxonomy_files_id');
        $orig_file_ext = $comparator_obj->taxonomy_files[$taxonomy_files_id]['orig_file_ext'];

        foreach ($countries as $country) {
            $country_obj = new Country((int) $country['id_country'], (int) $this->context->language->id);

            if ($comparator_obj->bestTaxonomyIsoCode($country['iso_code'], $comparator_obj->taxonomy_files[$taxonomy_files_id]['type'])) {
                $found_taxonomy[] = array('id_country' => (int) $country['id_country'], 'country' => $country['iso_code'].' - '.$country_obj->name);
            } else {
                $disabled_countries[] = '"'.$country_obj->name.'"';
            }
        }

        if (count($found_taxonomy) != count($countries)) {
            $description = sprintf($this->l('There are "%d" countries that you cannot use for the taxonomy, because they are not managed by this comparator'), count($countries) - count($found_taxonomy)).'.<br />'.implode(', ', $disabled_countries);
        } else {
            $description = '';
        }

        $fields_form_1 = array(
            'form' => array(
                'legend' => array(
                    'title' => $this->l('Upload a new taxonomy file for').' <span style="color: red;">'.Tools::strtoupper($comparator_name).'</span>',
                    'icon' => 'icon-upload',
                ),
                'description' => $description,
                'input' => array(
                    array(
                        'type' => 'file',
                        'label' => $this->l('File').':',
                        'desc' => $this->l('The file must be taken from this comparator (ask to its employees). It must be in the format').': <span style="color: red; font-weight: bold;">'.Tools::strtoupper($orig_file_ext).'</span>',
                        'name' => 'taxonomy_file',
                    ),
                    array(
                        'type' => 'select',
                        'label' => $this->l('For which country').'?',
                        'desc' => $this->l('Choose the country you want to associate to this file'),
                        'name' => 'taxonomy_id_country',
                        'class' => 'fixed-width-xxl',
                        'options' => array(
                            'query' => $found_taxonomy,
                            'id' => 'id_country',
                            'name' => 'country',
                        ),
                    ),
                ),
                'submit' => array(
                    'title' => $this->l('Upload'),
                    'icon' => 'process-icon-upload',
                    'class' => 'btn btn-default pull-right',
                    'name' => 'submitCountryUpdateNoUrlTaxonomy',
                ),
                'buttons' => array(
                    array(
                        'href' => AdminController::$currentIndex.'&configure='.$this->name.'&name='.Tools::strtoupper($comparator_name).'&viewComparatorList&token='.Tools::getAdminTokenLite('AdminModules'),
                        'title' => $this->l('Back'),
                        'icon' => 'process-icon-back',
                    ),
                ),
            ),
        );

        $helper = new HelperForm();
        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $lang = new Language((int) Configuration::get('PS_LANG_DEFAULT'));
        $helper->default_form_language = $lang->id;
        $helper->module = $this;
        $helper->allow_employee_form_lang = Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG') ? Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG') : 0;
        $helper->identifier = $this->identifier;
        $helper->currentIndex = AdminController::$currentIndex.'&configure='.$this->name.'&name='.Tools::strtoupper($comparator_name).'&taxonomy_files_id='.$taxonomy_files_id;
        $helper->token = Tools::getAdminTokenLite('AdminModules');
        $helper->tpl_vars = array(
            'fields_value' => $this->getConfigFieldsValues8(),
            'languages' => $this->context->controller->getLanguages(),
            'id_language' => $this->context->language->id,
        );

        return $helper->generateForm(array($fields_form_1));
    }

    private function getConfigFieldsValues8()
    {
        return array(
            'taxonomy_id_country' => Tools::getValue('taxonomy_id_country', (int) Configuration::getGlobalValue('MAO_COMPARATOR_TAXONOMY_COUNTRY')),
        );
    }

    private function displayCountryTaxonomySelectionForm($comparator)
    {
        $found_taxonomy = $disabled_countries = array();
        $comparator_name = $comparator['name'];
        $comparator_obj = new $comparator_name();
        $countries = Country::getCountries((int) $this->context->language->id, true);
        $taxonomy_files_id = (int) Tools::getValue('taxonomy_files_id');

        foreach ($countries as $country) {
            $country_obj = new Country((int) $country['id_country'], (int) $this->context->language->id);

            if ($comparator_obj->bestTaxonomyIsoCode($country['iso_code'], $comparator_obj->taxonomy_files[$taxonomy_files_id]['type'])) {
                $found_taxonomy[] = array('id_country' => (int) $country['id_country'], 'country' => $country['iso_code'].' - '.$country_obj->name);
            } else {
                $disabled_countries[] = '"'.$country_obj->name.'"';
            }
        }

        if (count($found_taxonomy) != count($countries)) {
            $description = sprintf($this->l('There are "%d" countries that you cannot use for the taxonomy, because they are not managed by this comparator'), count($countries) - count($found_taxonomy)).'.<br />'.implode(', ', $disabled_countries);
        } else {
            $description = '';
        }

        $fields_form_1 = array(
            'form' => array(
                'legend' => array(
                    'title' => $this->l('Select the country to use in the next block'),
                    'icon' => 'icon-globe',
                ),
                'description' => $description,
                'input' => array(
                    array(
                        'type' => 'select',
                        'label' => $this->l('Country').':',
                        'name' => 'taxonomy_id_country',
                        'class' => 'fixed-width-xxl',
                        'options' => array(
                            'query' => $found_taxonomy,
                            'id' => 'id_country',
                            'name' => 'country',
                        ),
                    ),
                ),
                'submit' => array(
                    'title' => $this->l('Save'),
                    'class' => 'btn btn-default pull-right',
                    'name' => 'submitCountryTaxonomySelection',
                ),
                'buttons' => array(
                    array(
                        'href' => AdminController::$currentIndex.'&configure='.$this->name.'&name='.Tools::strtoupper($comparator_name).'&viewComparatorList&token='.Tools::getAdminTokenLite('AdminModules'),
                        'title' => $this->l('Back'),
                        'icon' => 'process-icon-back',
                    ),
                ),
            ),
        );

        $helper = new HelperForm();
        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $lang = new Language((int) Configuration::get('PS_LANG_DEFAULT'));
        $helper->default_form_language = $lang->id;
        $helper->module = $this;
        $helper->allow_employee_form_lang = Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG') ? Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG') : 0;
        $helper->identifier = $this->identifier;
        $helper->currentIndex = AdminController::$currentIndex.'&configure='.$this->name.'&name='.Tools::strtoupper($comparator_name).'&taxonomy_files_id='.$taxonomy_files_id;
        $helper->token = Tools::getAdminTokenLite('AdminModules');
        $helper->tpl_vars = array(
            'fields_value' => $this->getConfigFieldsValues7(),
            'languages' => $this->context->controller->getLanguages(),
            'id_language' => $this->context->language->id,
        );

        return $helper->generateForm(array($fields_form_1));
    }

    private function getConfigFieldsValues7()
    {
        return array(
            'taxonomy_id_country' => Tools::getValue('taxonomy_id_country', (int) Configuration::getGlobalValue('MAO_COMPARATOR_TAXONOMY_COUNTRY')),
        );
    }

    private function displayCountrySummaryTaxonomyAssociation($comparator, $taxonomy_files)
    {
        $output = '';
        $comparator_name = $comparator['name'];
        $id_shop = (int) $this->context->shop->id;
        $id_lang = (int) $this->context->language->id;
        $id_country = (int) Configuration::getGlobalValue('MAO_COMPARATOR_TAXONOMY_COUNTRY');
        $categories = Category::getSimpleCategories($id_lang);
        $taxonomy_files_id = (int) Tools::getValue('taxonomy_files_id');
        $taxonomy_form = $taxonomy_files['form'];
        $taxonomy_field = $taxonomy_files['field'];
        $taxonomy_type = $taxonomy_files['type'];
        $active_count = count(
            MaofreeComparatorClass::getTaxonomyData(
                self::TAXONOMY_COUNTRY_TABLE,
                $id_shop,
                $comparator_name,
                $taxonomy_type,
                $taxonomy_field,
                $id_country,
                null
            )
        );
        $taxonomy_country = new Country($id_country, $id_lang);

        if ($taxonomy_form == 'ajax') {
            $this->context->controller->addCSS($this->_path.'views/css/search_style.css', 'all');

            $output .= '
		<script type="text/javascript">
			var mao_comparator_name = "'.$comparator_name.'";
			var mao_id_lang = '.$id_lang.';
			var mao_id_country = '.$id_country.';
			var taxonomy_files_id = '.$taxonomy_files_id.';
		</script>
		<script type="text/javascript" src="'.$this->_path.'views/js/maofree_search.js"></script>';
        }

        $output .= '
		<div class="clearfix">
			<div class="panel col-lg-12">
				<div class="panel-heading">
					<i class="icon-area-chart"></i>
					'.$this->l('Summary of').' <span style="color: red;">'.Tools::strtoupper($comparator_name).'</span> <span class="badge">'.$taxonomy_country->name.'</span>
				</div>
				<div class="alert alert-warning">
					'.$this->l('Categories associated').'&nbsp;&nbsp;&nbsp;<span class="badge badge-success">'.$active_count.'</span>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;--&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;'.$this->l('Categories not associated').'&nbsp;&nbsp;&nbsp;<span class="badge badge-danger">'.((count($categories) - 1) - $active_count).'</span>
				</div>
			</div>
		</div>';

        return $output;
    }

    private function displayCountryTaxonomyAssociationForm($comparator, $taxonomy_files)
    {
        $output = '';
        $all_taxonomy = array();
        $comparator_name = $comparator['name'];
        $id_shop = (int) $this->context->shop->id;
        $id_lang = (int) $this->context->language->id;
        $id_country = (int) Configuration::getGlobalValue('MAO_COMPARATOR_TAXONOMY_COUNTRY');
        $id_root_cat = (int) Configuration::get('PS_ROOT_CATEGORY');
        $id_home_cat = (int) Configuration::get('PS_HOME_CATEGORY');
        $categories = Category::getSimpleCategories($id_lang);
        $taxonomy_files_id = (int) Tools::getValue('taxonomy_files_id');
        $taxonomy_form = $taxonomy_files['form'];
        $taxonomy_field = $taxonomy_files['field'];
        $taxonomy_type = $taxonomy_files['type'];
        $back_url = AdminController::$currentIndex.'&configure='.$this->name.'&name='.Tools::strtoupper($comparator_name).'&viewComparatorList&token='.Tools::getAdminTokenLite('AdminModules');
        $comparator_obj = new $comparator_name();
        $taxonomy_country = new Country($id_country, $id_lang);

        if ($taxonomy_form == 'ajax') {
            $last_field_icon = 'search';
        } elseif ($taxonomy_form == 'text') {
            $last_field_icon = 'pencil';
        } else {
            $last_field_icon = 'list';
            $all_taxonomy = $comparator_obj->readTaxonomyFile($taxonomy_country->iso_code, $taxonomy_files_id);
        }

        $output .= '
		<form class="clearfix" action="'.Tools::safeOutput($_SERVER['REQUEST_URI']).'" method="post" autocomplete="off">
			<div class="panel col-lg-12">
				<div class="panel-heading">
					<i class="icon-list"></i>
					'.$this->l('Taxonomy associations of').' <span style="color: red;">'.Tools::strtoupper($comparator_name).'</span> <span class="badge">'.$taxonomy_country->name.'</span>
				</div>';

        if ($taxonomy_form == 'ajax') {
            $text_for_form = $this->l('On each field, enter a few characters and then select the most appropriate category');
        } elseif ($taxonomy_form == 'select') {
            $text_for_form = $this->l('On each field, choose which option is the most appropriate text for the associate category');
        } else {
            $text_for_form = $this->l('On each field, enter a text to specify in the best possible way the category to which it belongs. Ask to this comparator which text you need to enter');
        }

        $output .= '
				<div class="alert alert-info">
					<ul>
						<li>'.$this->l('Normally it is very important and mandatory to associate categories of your site to those of the comparator. It is not necessary to associate all categories, but only those containing products in the data feed').'.</li>
						<li><b>'.$text_for_form.'.</b> '.$this->l('You must do this only for the countries that you used to create the data feed').'.</li>';

        if (isset($comparator_obj->taxonomy_ifo_url) && $comparator_obj->taxonomy_ifo_url) {
            $output .= '<li>'.$this->l('You can get more informations on how to manage the taxonomy from here').' <a style="font-weight: bold;" target="_blank" href="'.$comparator_obj->taxonomy_ifo_url.'">push here</a></li>';
        }

        $output .= '
					</ul>
				</div>';

        foreach ($categories as $category) {
            $id_category = (int) $category['id_category'];
            $category_obj = new Category($id_category);

            if ($id_category == $id_root_cat || $id_category == $id_home_cat) {
                continue;
            }

            $products_tot = (int) $category_obj->getProducts($id_lang, null, null, null, null, true, true, false, 1, false);
            $cat_name = MaofreeComparatorClass::getPath($id_category, $id_shop, $id_lang, ' > ', $id_root_cat, $id_home_cat);
            $taxonomy = MaofreeComparatorClass::getTaxonomyData(
                self::TAXONOMY_COUNTRY_TABLE,
                $id_shop,
                $comparator_name,
                $taxonomy_type,
                $taxonomy_field,
                $id_country,
                $id_category
            );
            if (!isset($taxonomy) || !$taxonomy) {
                $cat_taxonomy = '';
            } else {
                $cat_taxonomy = $taxonomy[$taxonomy_field.'_taxonomy'];
            }

            $output .= '
				<div class="form-group maoblocksearch">
					<label class="control-label col-md-2 intro-inputString" style="font-weight: bold;">'.$this->l('Category of this shop').':</label>
					<div class="input-group col-md-10">
						<span style="display: inline-block; font-weight: bold; vertical-align: middle; color: tomato;">'.$cat_name.'</span>
						<span class="badge badge-warning" style="margin-left: 20px">'.$products_tot.'</span>
					</div>
					<div class="input-group col-xs-12">
						<span class="input-group-addon"><i class="icon-random"></i></span>';

            if ($taxonomy_form == 'ajax' || $taxonomy_form == 'text') {
                $output .= '<input type="text" value="'.$cat_taxonomy.'" name="inputString-'.$id_category.'" id="inputString-'.$id_category.'" class="inputString" autocomplete="off" />';
            } else {
                $output .= '<select autocomplete="off" id="inputString-'.$id_category.'" name="inputString-'.$id_category.'">
					<option value="">* '.$this->l('Select a taxonomy category').' *</option>';

                foreach ($all_taxonomy as $row) {
                    $output .= '<option value="'.$row.'"'.($row == $cat_taxonomy ? ' selected="selected"' : '').'>'.$row.'</option>';
                }

                $output .= '</select>';
            }

            $output .= '<span class="input-group-addon"><i class="icon-'.$last_field_icon.'"></i></span>
					</div>
					<div id="suggestions-'.$id_category.'" class="suggestions"></div>
					<hr />
				</div>';
        }

        $output .= '
				<input type="hidden" name="taxonomy_id_country" value="'.$id_country.'">
				<input type="hidden" name="taxonomy_field" value="'.$taxonomy_field.'">
				<input type="hidden" name="taxonomy_form" value="'.$taxonomy_form.'">
				<input type="hidden" name="taxonomy_files_id" value="'.$taxonomy_files_id.'">
				<input type="hidden" name="name" value="'.$comparator_name.'">
				<div class="panel-footer">
					<button class="btn btn-default pull-right" name="submitCountryTaxonomyAssociation" id="module_form_submit_btn_2" value="1" type="submit">
						<i class="process-icon-save"></i> '.$this->l('Save').'
					</button>
					<a class="btn btn-default" href="'.$back_url.'">
						<i class="process-icon-back"></i> '.$this->l('Back').'
					</a>
				</div>
			</div>
		</form>';

        return $output;
    }

    private function displayComparatorDatafeedLink($comparator)
    {
        $comparator_name = $comparator['name'];

        require_once dirname(__FILE__).'/comparators/'.$comparator_name.'/'.$comparator_name.'.php';

        $comparator_obj = new $comparator_name();
        if (
            isset($comparator_obj->datafeed_type) &&
            is_array($comparator_obj->datafeed_type) &&
            count($comparator_obj->datafeed_type)
        ) {
            $datafeed_type = $comparator_obj->datafeed_type;
        } else {
            $datafeed_type = array('html');
        }
        if ($comparator['compression_type'] == 'none') {
            $compression_type = '';
        } elseif ($comparator['compression_type'] == 'gzip') {
            $compression_type = '.gz';
        } elseif ($comparator['compression_type'] == 'bzip2') {
            $compression_type = '.bz2';
        } else {
            $compression_type = '.zip';
        }
        $multishop = (int) Configuration::getGlobalValue('PS_MULTISHOP_FEATURE_ACTIVE');
        $settings = unserialize(Configuration::get('MAO_COMPARATOR_SETTINGS'));
        $onecronjob = Configuration::getGlobalValue('MAO_COMPARATOR_ONECRONJOB');
        $countries = Country::getCountries((int) $this->context->language->id, true);
        $id_shop_default = (int) Configuration::get('PS_SHOP_DEFAULT');
        $id_shop = (int) $this->context->shop->id;
        $id_lang = (int) Configuration::get('PS_LANG_DEFAULT');
        $id_country = (int) Configuration::get('PS_COUNTRY_DEFAULT');
        $id_currency = (int) Configuration::get('PS_CURRENCY_DEFAULT');
        $tax = (int) Configuration::get('PS_TAX');

        $final_url = 'comparators/'.$comparator_name.'/'.$comparator_name.'.datafeed.php?token='.$this->token.($multishop && ($id_shop_default != $id_shop) ? '&id_shop='.$id_shop : '');
        $default_datafeed_url = Tools::getShopDomainSsl(true, true).$this->_path.$final_url;
        $default_datafeed_crontab = dirname(__FILE__).'/'.$final_url;
        $prefered_datafeed_type = isset($comparator_obj->datafeed_type) ? $comparator_obj->datafeed_type[0] : 'html';
        $default_datafeed_file = Tools::getShopDomainSsl(true, true).$this->_path.'comparators/'.$comparator_name.'/';

        $onecronjob_code = '<br />
            <h4>* '.$this->l('To take updated each data feed you need to add them to the "Global Cron Job" (scheduled task)').':</h4>
            <a class="btn btn-default" href="'.AdminController::$currentIndex.'&configure='.$this->name.'&updateGeneralSettings&token='.Tools::getAdminTokenLite('AdminModules').'#module_form">
                <i class="icon-calendar"></i> '.$this->l('Global Cron Job (push here)').'
            </a>';

        $output = '
		<style type="text/css">
			#mao_ajax_running { display: none; z-index: 200000; background-color: #000; opacity: 0.5; height: 100%; width: 100%; position: fixed; top: 0; left: 0 }
			#mao_ajax_running img { position: absolute; top: 50%; left: 50% }
			#mao_ajax_progress { display: none; z-index: 210000; height: 25px; width: 100%; position: fixed; top: 0; left: 0 }
			#mao_ajax_progress .progress-bar { font-size: 18px; line-height: 23px; }
		</style>

		<script type="text/javascript">
			var mao_comparator_name = "'.$comparator_name.'";
			var mao_datafeed_type = "'.$prefered_datafeed_type.'";
			var mao_compression_type = "'.$compression_type.'";
			var mao_id_shop = '.$id_shop.';
			var mao_id_currency = '.$id_currency.';
			var mao_id_lang = '.$id_lang.';
			var mao_id_country = '.$id_country.';
			var mao_tax = '.$tax.';
			var mao_multishop = '.$multishop.';
			var mao_datafeed_default_link = "'.$default_datafeed_url.'";
			var mao_datafeed_default_crontab = "'.$default_datafeed_crontab.'";
			var mao_datafeed_default_file = "'.$default_datafeed_file.'";

			var mao_reload_redirect = "'.AdminController::$currentIndex.'&configure='.$this->name.'&token='.Tools::getAdminTokenLite('AdminModules').'&name='.Tools::strtoupper($comparator_name).'&viewComparatorList";
			var mao_path = "'.$this->_path.'";
			var confirm_1 = "'.$this->l('Are you sure you want to proceed with this update?').'";
			var alert_1 = "'.$this->l('Sorry, an error occurred during update').'";
			var alert_2 = "'.$this->l('Sorry, you cannot update this file from the current shop context. First select the appropriate shop context').'";
		</script>
		<script type="text/javascript" src="'.$this->_path.'views/js/maofree_datafeed.js"></script>

		<div class="panel clearfix">
			<h3><i class="icon-sliders"></i> '.$this->l('Data feed settings').'</h3>';

        if (in_array('txt', $datafeed_type) || in_array('csv', $datafeed_type)) {
            $text_row_end = htmlspecialchars(str_replace(array('<br>', '<br/>', '<br />'), '\n', nl2br($comparator_obj->text_row_end)));

            $output .= '
			<div class="alert alert-warning">
				'.$this->l('If you choose to export the data to a dynamic html page, or to a static file like txt or csv, remember to notify to the employees of this comparator that the field separator is').'
				" <span style="color: purple; font-weight: bold; font-size: 140%;">'.($comparator_obj->field_separator === "\t" ? '\t' : htmlspecialchars($comparator_obj->field_separator)).'</span> "
				'.$this->l('the category separator is').'
				" <span style="color: purple; font-weight: bold; font-size: 140%;">'.htmlspecialchars($comparator_obj->cat_separator).'</span> "
				'.$this->l('and the end of row is').'
				" <span style="color: purple; font-weight: bold; font-size: 140%;">'.$text_row_end.'</span> "
			</div>';
        }

        $output .= '
			<div class="alert alert-info">
				<h4>'.$this->l('How to proceed').':</h4>
				<ol style="padding-left: 10px;">
					<li>'.$this->l('From here below choose some options to use to create the desired data feed. These options, if different from the default ones, will be added to the data feed\'s URL. The token parameter is required to increase the security of your data.').'</li>
					<li>'.$this->l('Pass the data feed\'s URL to employees of this comparator to give them the possibility of charge your products into their systems. You can choose between two type of data feed:').'
						<ol style="list-style-type: lower-alpha;">
							<li><span style="font-weight: bold;">'.$this->l('Dynamic').' (html)</span>: '.$this->l('the data feed is created only when the comparator processes a php script present in this module. This solution does not need to use a Cron Job task. Normally it is the most used by comparators.').'</li>
							<li><span style="font-weight: bold;">'.$this->l('Static').' (txt, csv, xml)</span>: '.$this->l('the products data is saved into a file which normally is compressed to occupy a smaller size. Usually the formats TXT and CSV are preferred, because the file is smaller than that in XML and are easier to process by a comparator.').'</li>
						</ol>
					</li>
					<li>'.$this->l('If you choose a static solution, to keep the data updated, you need to add a new Cron Job task, inserting the green URL here below, to the scheduled task page of your hosting, then set the frequency you want, possibly at night.').'</li>
					<li>'.$this->l('The static solution is recommended if the data feed contains many items, because it is possible to use an ajax or Cron Job solution to bypass the timeout limit of the server.').'</li>
				</ol>
				<br />
				<span style="font-weight: bold;">'.$this->l('Normally it is not necessary to change the default parameters, with the exception of the data feed type, so you only need to press on the button below to check if the data is ok or to create the first file.').'</span>
			</div>

			<div class="row">
				<div class="input-group col-md-12 pull-left" style="margin-top: 20px;">
					<div class="input-group-addon"><i class="icon-file-text"></i></div>
					<select name="datafeed_type" id="datafeed_type" autocomplete="off">';

        foreach ($datafeed_type as $type) {
            $output .= '<option value="'.$type.'"'.($type == $prefered_datafeed_type ? ' selected="selected"' : '').'>'.$this->l('Type').' : "'.$type.'"</option>';
        }

        $output .= '
					</select>

					<div class="input-group-addon"><i class="icon-money"></i></div>
					<select name="datafeed_currency" id="datafeed_currency" autocomplete="off">';

        foreach (Currency::getCurrencies(false, true) as $currency) {
            $output .= '<option value="'.(int) $currency['id_currency'].'"'.($id_currency == $currency['id_currency'] ? ' selected="selected"' : '').'>'.$currency['name'].'</option>';
        }

        $output .= '
					</select>

					<div class="input-group-addon"><i class="icon-flag"></i></div>
					<select name="datafeed_lang" id="datafeed_lang" autocomplete="off">';

        foreach (Language::getLanguages(true) as $lang) {
            $output .= '<option value="'.(int) $lang['id_lang'].'"'.($id_lang == $lang['id_lang'] ? ' selected="selected"' : '').'>'.$lang['name'].'</option>';
        }

        $output .= '
					</select>

					<div class="input-group-addon"><i class="icon-globe"></i></div>
					<select name="datafeed_country" id="datafeed_country" autocomplete="off">';

        foreach ($countries as $country) {
            $output .= '<option value="'.(int) $country['id_country'].'"'.($id_country == $country['id_country'] ? ' selected="selected"' : '').'>'.$country['name'].'</option>';
        }

        $output .= '
					</select>

                    <div class="input-group-addon"><i class="icon-globe"></i></div>
					<select name="datafeed_tax" id="datafeed_tax" autocomplete="off">
                        <option value="1"'.($tax == 1 ? ' selected="selected"' : '').'>'.$this->l('Tax included').'</option>
                        <option value="0"'.($tax == 0 ? ' selected="selected"' : '').'>'.$this->l('Tax excluded').'</option>
					</select>
				</div>
			</div>

			<div class="row">
				<div class="col-xs-12" style="margin: 10px 0; padding: 20px 0;">
                    <div id="datafeed-taxstatus" data-tax="'.$tax.'" style="display: none;"></div>

					<div id="datafeed-html" style="display: '.($prefered_datafeed_type == 'html' ? 'block' : 'none').';">
						<h4>* '.$this->l('URL to pass to this comparator (data feed\'s URL)').':</h4>
						<a style="color: red; font-weight: bold;" target="_blank" title="" href="'.$default_datafeed_url.'">'.$default_datafeed_url.'</a>
					</div>

					<div id="datafeed-txt" style="display: '.($prefered_datafeed_type == 'txt' ? 'block' : 'none').';">
						<h4>* '.$this->l('URL to pass to this comparator (data feed\'s URL)').':</h4>
						<p style="color: red; font-weight: bold;">'.$default_datafeed_file.'txt/'.$comparator_name.'-'.$id_shop.'_'.$id_currency.'_'.$id_lang.'_'.$id_country.'.txt'.$compression_type.'</p>

                        '.($onecronjob ? '<div style="display: none;">' : '').'
						    <br />
    						<h4>* '.$this->l('URL to execute a Cron Job task, activatable from the control panel of your hosting (scheduled task)').':</h4>
    						<span class="default_datafeed_url" style="color: green; font-weight: bold;">'.$default_datafeed_url.'</span><br /><br />
    						<h4>* '.$this->l('In the field of scheduled tasks (Web Cron Job) on your cPanel or Plesk not use the "CURL" command, but "WGET", as shown below').':</h4>
    						<em style="">wget -O /dev/null --no-check-certificate --max-redirect=100000 "<b>'.$this->l('put here the green URL written above within these quotes').'</b>"</em><br /><br />
                            <h4 style="color: silver;">* '.$this->l('Instruction to execute a cron from crontab of your server every day at 03:00 (only for expert)').':</h4>
    						<span style="color: silver; font-weight: bold;">0 3 * * * php -d</span> <span class="default_datafeed_crontab" style="color: silver; font-weight: bold;">'.$default_datafeed_crontab.'</span>
                        '.($onecronjob ? '</div>'.$onecronjob_code : '').'
					</div>

					<div id="datafeed-csv" style="display: '.($prefered_datafeed_type == 'csv' ? 'block' : 'none').';">
						<h4>* '.$this->l('URL to pass to this comparator (data feed\'s URL)').':</h4>
						<p style="color: red; font-weight: bold;">'.$default_datafeed_file.'csv/'.$comparator_name.'-'.$id_shop.'_'.$id_currency.'_'.$id_lang.'_'.$id_country.'.csv'.$compression_type.'</p>

                        '.($onecronjob ? '<div style="display: none;">' : '').'
                            <br />
    						<h4>* '.$this->l('URL to execute a Cron Job task, activatable from the control panel of your hosting (scheduled task)').':</h4>
    						<span class="default_datafeed_url" style="color: green; font-weight: bold;">'.$default_datafeed_url.'</span><br /><br />
    						<h4>* '.$this->l('In the field of scheduled tasks (Web Cron Job) on your cPanel or Plesk not use the "CURL" command, but "WGET", as shown below').':</h4>
    						<em style="">wget -O /dev/null --no-check-certificate --max-redirect=100000 "<b>'.$this->l('put here the green URL written above within these quotes').'</b>"</em><br /><br />
                            <h4 style="color: silver;">* '.$this->l('Instruction to execute a cron from crontab of your server every day at 03:00 (only for expert)').':</h4>
    						<span style="color: silver; font-weight: bold;">0 3 * * * php -d</span> <span class="default_datafeed_crontab" style="color: silver; font-weight: bold;">'.$default_datafeed_crontab.'</span>
                        '.($onecronjob ? '</div>'.$onecronjob_code : '').'
					</div>

					<div id="datafeed-xml" style="display: '.($prefered_datafeed_type == 'xml' ? 'block' : 'none').';">
						<h4>* '.$this->l('URL to pass to this comparator (data feed\'s URL)').':</h4>
						<p style="color: red; font-weight: bold;">'.$default_datafeed_file.'xml/'.$comparator_name.'-'.$id_shop.'_'.$id_currency.'_'.$id_lang.'_'.$id_country.'.xml'.$compression_type.'</p>

                        '.($onecronjob ? '<div style="display: none;">' : '').'
    						<br />
    						<h4>* '.$this->l('URL to execute a Cron Job task, activatable from the control panel of your hosting (scheduled task)').':</h4>
    						<span class="default_datafeed_url" style="color: green; font-weight: bold;">'.$default_datafeed_url.'</span><br /><br />
    						<h4>* '.$this->l('In the field of scheduled tasks (Web Cron Job) on your cPanel or Plesk not use the "CURL" command, but "WGET", as shown below').':</h4>
    						<em style="">wget -O /dev/null --no-check-certificate --max-redirect=100000 "<b>'.$this->l('put here the green URL written above within these quotes').'</b>"</em><br /><br />
                            <h4 style="color: silver;">* '.$this->l('Instruction to execute a cron from crontab of your server every day at 03:00 (only for expert)').':</h4>
    						<span style="color: silver; font-weight: bold;">0 3 * * * php -d</span> <span class="default_datafeed_crontab" style="color: silver; font-weight: bold;">'.$default_datafeed_crontab.'</span>
                        '.($onecronjob ? '</div>'.$onecronjob_code : '').'
					</div>
				</div>
			</div>

			<div class="panel-footer">
				<button class="btn btn-primary pull-right" id="datafeed_link" title="'.$this->l('Check or create the data you provide to this comparator').'"><i class="icon-external-link-square"></i> '.$this->l('Check or create its data feed').($multishop ? ' ('.$this->l('referred to this shop').')' : '').' <i class="icon-chevron-right"></i></button>
				<a class="btn btn-default" href="'.AdminController::$currentIndex.'&configure='.$this->name.'&token='.Tools::getAdminTokenLite('AdminModules').'"><i class="process-icon-back"></i> '.$this->l('Back').'</a>
			</div>
		</div>';

        return $output;
    }

    private function displayDynamicLastCheck($comparator)
    {
        $this->smarty->assign('mao_comparator_last_check', ($comparator['dynamic_last_check'] ? date('g:ia, F j, Y', strtotime($comparator['dynamic_last_check'])) : $this->l('never done')));

        return $this->display(__FILE__, 'mao_dynamic_last_check.tpl');
    }

    private function renderTxtList($comparator)
    {
        $comparator_name = $comparator['name'];
        $settings = $file_list = $date = array();
        $directory = _PS_MODULE_DIR_.$this->name.'/comparators/'.$comparator_name.'/txt';

        if (file_exists($directory)) {
            $scanned_directory = array_diff(scandir($directory), array('..', '.', 'index.php'));

            if (is_array($scanned_directory) && count($scanned_directory)) {
                foreach ($scanned_directory as $file) {
                    if (!strstr($file, '.txt')) {
                        continue;
                    }

                    $timestamp = filemtime($directory.'/'.$file);
                    $filesize = filesize($directory.'/'.$file);
                    $date[] = $timestamp;

                    $file_list[] = array('filename' => $file, 'filesize' => $filesize, 'date' => $timestamp);
                }

                if (is_array($file_list) && count($file_list)) {
                    array_multisort($date, SORT_DESC, $file_list);

                    foreach ($file_list as $row) {
                        if (Tools::strlen($row['filesize']) > 6) {
                            $divisor = 1000000;
                            $suffix = ' MB';
                        } elseif (Tools::strlen($row['filesize']) > 3 && Tools::strlen($row['filesize']) <= 6) {
                            $divisor = 1000;
                            $suffix = ' KB';
                        } else {
                            $divisor = 1;
                            $suffix = ' bytes';
                        }

                        $settings[] = array(
                            'filename' => $row['filename'],
                            'href_link' => _MODULE_DIR_.$this->name.'/comparators/'.$comparator_name.'/txt/'.$row['filename'].'?t='.time(),
                            'filesize' => Tools::ps_round($row['filesize'] / $divisor, 2).$suffix,
                            'date' => date($this->context->language->date_format_full, $row['date']),
                            'date_color' => '#1E90FF',
                            'updateicon' => $row['filename'],
                        );
                    }
                }
            }
        }

        $fields_list = array(
            'filename' => array(
                'title' => $this->l('File name'),
                'type' => 'download_link_file',
                'align' => 'left',
                'hreflink' => 'href_link',
                'search' => false,
                'orderby' => false,
            ),
            'filesize' => array(
                'title' => $this->l('File size'),
                'type' => 'text',
                'align' => 'center',
                'search' => false,
                'orderby' => false,
            ),
            'date' => array(
                'title' => $this->l('Creation date'),
                'type' => 'text',
                'align' => 'center',
                'class' => 'fixed-width-xl',
                'color' => 'date_color',
                'search' => false,
                'orderby' => false,
            ),
            'updateicon' => array(
                'title' => $this->l('Manual update'),
                'hint' => $this->l('Update this data feed manually (ajax method)'),
                'type' => 'update_link_file',
                'align' => 'center',
                'class' => 'fixed-width-xl',
                'search' => false,
                'orderby' => false,
            ),
        );

        $helper = new HelperList();
        $helper->shopLinkType = '';
        $helper->simple_header = false;
        $helper->no_link = true;
        $helper->listTotal = count($settings);
        $helper->identifier = 'filename';
        $helper->actions = array('delete');
        $helper->table = 'TxtList';
        $helper->imageType = 'jpg';
        $helper->show_toolbar = true;
        $helper->module = $this;
        $helper->title = '<i class="icon-files-o"></i> '.$this->l('List of data feed in format').' "<span style="color: #ff0000">txt</span>"';
        $helper->bulk_actions = array(
            'delete' => array(
                'text' => $this->l('Delete selected'),
                'icon' => 'icon-trash',
                'confirm' => $this->l('Delete selected items forever?'),
            ),
        );
        $helper->toolbar_btn['a'] = array(
            'desc' => '',
        );
        $helper->token = Tools::getAdminTokenLite('AdminModules');
        $helper->currentIndex = AdminController::$currentIndex.'&configure='.$this->name.'&name='.Tools::strtoupper($comparator_name);

        return $helper->generateList($settings, $fields_list);
    }

    private function renderCsvList($comparator)
    {
        $comparator_name = $comparator['name'];
        $settings = $file_list = $date = array();
        $directory = _PS_MODULE_DIR_.$this->name.'/comparators/'.$comparator_name.'/csv';

        if (file_exists($directory)) {
            $scanned_directory = array_diff(scandir($directory), array('..', '.', 'index.php'));

            if (is_array($scanned_directory) && count($scanned_directory)) {
                foreach ($scanned_directory as $file) {
                    if (!strstr($file, '.csv')) {
                        continue;
                    }

                    $timestamp = filemtime($directory.'/'.$file);
                    $filesize = filesize($directory.'/'.$file);
                    $date[] = $timestamp;

                    $file_list[] = array('filename' => $file, 'filesize' => $filesize, 'date' => $timestamp);
                }

                if (is_array($file_list) && count($file_list)) {
                    array_multisort($date, SORT_DESC, $file_list);

                    foreach ($file_list as $row) {
                        if (Tools::strlen($row['filesize']) > 6) {
                            $divisor = 1000000;
                            $suffix = ' MB';
                        } elseif (Tools::strlen($row['filesize']) > 3 && Tools::strlen($row['filesize']) <= 6) {
                            $divisor = 1000;
                            $suffix = ' KB';
                        } else {
                            $divisor = 1;
                            $suffix = ' bytes';
                        }

                        $settings[] = array(
                            'filename' => $row['filename'],
                            'href_link' => _MODULE_DIR_.$this->name.'/comparators/'.$comparator_name.'/csv/'.$row['filename'].'?t='.time(),
                            'filesize' => Tools::ps_round($row['filesize'] / $divisor, 2).$suffix,
                            'date' => date($this->context->language->date_format_full, $row['date']),
                            'date_color' => '#1E90FF',
                            'updateicon' => $row['filename'],
                        );
                    }
                }
            }
        }

        $fields_list = array(
            'filename' => array(
                'title' => $this->l('File name'),
                'type' => 'download_link_file',
                'align' => 'left',
                'hreflink' => 'href_link',
                'search' => false,
                'orderby' => false,
            ),
            'filesize' => array(
                'title' => $this->l('File size'),
                'type' => 'text',
                'align' => 'center',
                'search' => false,
                'orderby' => false,
            ),
            'date' => array(
                'title' => $this->l('Creation date'),
                'type' => 'text',
                'align' => 'center',
                'class' => 'fixed-width-xl',
                'color' => 'date_color',
                'search' => false,
                'orderby' => false,
            ),
            'updateicon' => array(
                'title' => $this->l('Manual update'),
                'hint' => $this->l('Update this data feed manually (ajax method)'),
                'type' => 'update_link_file',
                'align' => 'center',
                'class' => 'fixed-width-xl',
                'search' => false,
                'orderby' => false,
            ),
        );

        $helper = new HelperList();
        $helper->shopLinkType = '';
        $helper->simple_header = false;
        $helper->no_link = true;
        $helper->listTotal = count($settings);
        $helper->identifier = 'filename';
        $helper->actions = array('delete');
        $helper->table = 'CsvList';
        $helper->imageType = 'jpg';
        $helper->show_toolbar = true;
        $helper->module = $this;
        $helper->title = '<i class="icon-files-o"></i> '.$this->l('List of data feed in format').' "<span style="color: #ff0000">csv</span>"';
        $helper->bulk_actions = array(
            'delete' => array(
                'text' => $this->l('Delete selected'),
                'icon' => 'icon-trash',
                'confirm' => $this->l('Delete selected items forever?'),
            ),
        );
        $helper->toolbar_btn['a'] = array(
            'desc' => '',
        );
        $helper->token = Tools::getAdminTokenLite('AdminModules');
        $helper->currentIndex = AdminController::$currentIndex.'&configure='.$this->name.'&name='.Tools::strtoupper($comparator_name);

        return $helper->generateList($settings, $fields_list);
    }

    private function renderXmlList($comparator)
    {
        $comparator_name = $comparator['name'];
        $settings = $file_list = $date = array();
        $directory = _PS_MODULE_DIR_.$this->name.'/comparators/'.$comparator_name.'/xml';

        if (file_exists($directory)) {
            $scanned_directory = array_diff(scandir($directory), array('..', '.', 'index.php'));

            if (is_array($scanned_directory) && count($scanned_directory)) {
                foreach ($scanned_directory as $file) {
                    if (!strstr($file, '.xml')) {
                        continue;
                    }

                    $timestamp = filemtime($directory.'/'.$file);
                    $filesize = filesize($directory.'/'.$file);
                    $date[] = $timestamp;

                    $file_list[] = array('filename' => $file, 'filesize' => $filesize, 'date' => $timestamp);
                }

                if (is_array($file_list) && count($file_list)) {
                    array_multisort($date, SORT_DESC, $file_list);

                    foreach ($file_list as $row) {
                        if (Tools::strlen($row['filesize']) > 6) {
                            $divisor = 1000000;
                            $suffix = ' MB';
                        } elseif (Tools::strlen($row['filesize']) > 3 && Tools::strlen($row['filesize']) <= 6) {
                            $divisor = 1000;
                            $suffix = ' KB';
                        } else {
                            $divisor = 1;
                            $suffix = ' bytes';
                        }

                        $settings[] = array(
                            'filename' => $row['filename'],
                            'href_link' => _MODULE_DIR_.$this->name.'/comparators/'.$comparator_name.'/xml/'.$row['filename'].'?t='.time(),
                            'filesize' => Tools::ps_round($row['filesize'] / $divisor, 2).$suffix,
                            'date' => date($this->context->language->date_format_full, $row['date']),
                            'date_color' => '#1E90FF',
                            'updateicon' => $row['filename'],
                        );
                    }
                }
            }
        }

        $fields_list = array(
            'filename' => array(
                'title' => $this->l('File name'),
                'type' => 'download_link_file',
                'align' => 'left',
                'hreflink' => 'href_link',
                'search' => false,
                'orderby' => false,
            ),
            'filesize' => array(
                'title' => $this->l('File size'),
                'type' => 'text',
                'align' => 'center',
                'search' => false,
                'orderby' => false,
            ),
            'date' => array(
                'title' => $this->l('Creation date'),
                'type' => 'text',
                'align' => 'center',
                'class' => 'fixed-width-xl',
                'color' => 'date_color',
                'search' => false,
                'orderby' => false,
            ),
            'updateicon' => array(
                'title' => $this->l('Manual update'),
                'hint' => $this->l('Update this data feed manually (ajax method)'),
                'type' => 'update_link_file',
                'align' => 'center',
                'class' => 'fixed-width-xl',
                'search' => false,
                'orderby' => false,
            ),
        );

        $helper = new HelperList();
        $helper->shopLinkType = '';
        $helper->simple_header = false;
        $helper->no_link = true;
        $helper->listTotal = count($settings);
        $helper->identifier = 'filename';
        $helper->actions = array('delete');
        $helper->table = 'XmlList';
        $helper->imageType = 'jpg';
        $helper->show_toolbar = true;
        $helper->module = $this;
        $helper->title = '<i class="icon-files-o"></i> '.$this->l('List of data feed in format').' "<span style="color: #ff0000">xml</span>"';
        $helper->bulk_actions = array(
            'delete' => array(
                'text' => $this->l('Delete selected'),
                'icon' => 'icon-trash',
                'confirm' => $this->l('Delete selected items forever?'),
            ),
        );
        $helper->toolbar_btn['a'] = array(
            'desc' => '',
        );
        $helper->token = Tools::getAdminTokenLite('AdminModules');
        $helper->currentIndex = AdminController::$currentIndex.'&configure='.$this->name.'&name='.Tools::strtoupper($comparator_name);

        return $helper->generateList($settings, $fields_list);
    }

    public function hookDisplayLeftColumn($params)
    {
        if (!$this->isCached('maofree_comparator_block.tpl', $this->getCacheId('maofree_comparator_block'))) {
            $id_shop = (int) $this->context->shop->id;
            $comparators = $comparator_name = $comparator_title = $comparator_url = $comparator_nofollow = array();

            if (!isset(self::$cache_links)) {
                self::$cache_links = MaofreeComparatorClass::getComparatorName(self::MAIN_TABLE, $id_shop, true);
            }

            $comparators = self::$cache_links;

            if (is_array($comparators) && count($comparators)) {
                foreach ($comparators as $comparator) {
                    $name = $comparator['name'];
                    $url = $comparator['website_url'] ? explode('@', $comparator['website_url']) : array();
                    require_once dirname(__FILE__).'/comparators/'.$name.'/'.$name.'.php';
                    $comparator_obj = new $name();

                    if (
                        isset($comparator_obj->name) &&
                        is_array($comparator_obj->name) &&
                        isset($comparator_obj->website_url) &&
                        is_array($comparator_obj->website_url) &&
                        (count($comparator_obj->name) == count($comparator_obj->website_url))
                    ) {
                        for ($i = 0; $i < count($comparator_obj->name); ++$i) {
                            $comparator_name[] = $comparator_obj->name[$i];
                            $comparator_title[] = (isset($comparator_obj->title[$i]) && $comparator_obj->title[$i] ? $comparator_obj->title[$i] : $comparator_obj->name[$i]);
                            $comparator_url[] = isset($url[$i]) && $url[$i] ? $url[$i] : $comparator_obj->website_url[$i];
                            $comparator_nofollow[] = isset($comparator_obj->nofollow) ? (int) $comparator_obj->nofollow : 1;
                        }
                    }

                    unset($comparator_obj);
                }

                $this->smarty->assign(array(
                    'block_comparator_name' => $comparator_name,
                    'block_comparator_title' => $comparator_title,
                    'block_comparator_url' => $comparator_url,
                    'block_comparator_nofollow' => $comparator_nofollow,
                ));
            }
        }

        return $this->display(__FILE__, 'maofree_comparator_block.tpl', $this->getCacheId('maofree_comparator_block'));
    }

    public function hookDisplayRightColumn($params)
    {
        return $this->hookDisplayLeftColumn($params);
    }

    public function hookDisplayHome($params)
    {
        if (!$this->isCached('maofree_comparator_home.tpl', $this->getCacheId('maofree_comparator_home'))) {
            $id_shop = (int) $this->context->shop->id;
            $comparators = $comparator_name = $comparator_title = $comparator_url = $comparator_nofollow = array();

            if (!isset(self::$cache_links)) {
                self::$cache_links = MaofreeComparatorClass::getComparatorName(self::MAIN_TABLE, $id_shop, true);
            }

            $comparators = self::$cache_links;

            if (is_array($comparators) && count($comparators)) {
                foreach ($comparators as $comparator) {
                    $name = $comparator['name'];
                    $url = $comparator['website_url'] ? explode('@', $comparator['website_url']) : array();
                    require_once dirname(__FILE__).'/comparators/'.$name.'/'.$name.'.php';
                    $comparator_obj = new $name();

                    if (
                        isset($comparator_obj->name) &&
                        is_array($comparator_obj->name) &&
                        isset($comparator_obj->website_url) &&
                        is_array($comparator_obj->website_url) &&
                        (count($comparator_obj->name) == count($comparator_obj->website_url))
                    ) {
                        for ($i = 0; $i < count($comparator_obj->name); ++$i) {
                            $comparator_name[] = $comparator_obj->name[$i];
                            $comparator_title[] = (isset($comparator_obj->title[$i]) && $comparator_obj->title[$i] ? $comparator_obj->title[$i] : $comparator_obj->name[$i]);
                            $comparator_url[] = isset($url[$i]) && $url[$i] ? $url[$i] : $comparator_obj->website_url[$i];
                            $comparator_nofollow[] = isset($comparator_obj->nofollow) ? (int) $comparator_obj->nofollow : 1;
                        }
                    }

                    unset($comparator_obj);
                }

                $this->smarty->assign(array(
                    'home_comparator_name' => $comparator_name,
                    'home_comparator_title' => $comparator_title,
                    'home_comparator_url' => $comparator_url,
                    'home_comparator_nofollow' => $comparator_nofollow,
                ));
            }
        }

        return $this->display(__FILE__, 'maofree_comparator_home.tpl', $this->getCacheId('maofree_comparator_home'));
    }

    public function hookDisplayHeader($params)
    {
        if (!$this->active) {
            return;
        }

        $this->context->controller->addCSS(_MODULE_DIR_.$this->name.'/views/css/maofree_comparator.css', 'all');

        $comparators = array();
        $id_shop = (int) $this->context->shop->id;

        if (!isset(self::$cache_tracks)) {
            self::$cache_tracks = MaofreeComparatorClass::getComparatorName(self::MAIN_TABLE, $id_shop, false, true);
        }

        $comparators = self::$cache_tracks;

        if (is_array($comparators) && count($comparators)) {
            foreach ($comparators as $comparator) {
                $name = $comparator['name'];

                if (file_exists(dirname(__FILE__).'/comparators/'.$name.'/'.$name.'.php')) {
                    require_once dirname(__FILE__).'/comparators/'.$name.'/'.$name.'.php';

                    $comparator_obj = new $name();
                    $page_name = Dispatcher::getInstance()->getController();

                    if (method_exists($comparator_obj, 'comparatorUrlScript')) {
                        $url_script = $comparator_obj->comparatorUrlScript($id_shop, _MODULE_DIR_.$this->name.'/comparators/'.$name.'/');

                        if ($url_script) {
                            $this->context->controller->addJS($url_script);
                        }
                    }

                    if ($page_name == 'product' && method_exists($comparator_obj, 'comparatorUrlProductScript')) {
                        $url_product_script = $comparator_obj->comparatorUrlProductScript($id_shop, _MODULE_DIR_.$this->name.'/comparators/'.$name.'/');

                        if ($url_product_script) {
                            $this->context->controller->addJS($url_product_script);
                        }
                    }

                    if ($page_name == 'orderconfirmation' && method_exists($comparator_obj, 'comparatorUrlOrderConfirmationScript')) {
                        $url_orderconfirmation_script = $comparator_obj->comparatorUrlOrderConfirmationScript($id_shop, _MODULE_DIR_.$this->name.'/comparators/'.$name.'/');

                        if ($url_orderconfirmation_script) {
                            $this->context->controller->addJS($url_orderconfirmation_script);
                        }
                    }

                    unset($comparator_obj);
                }
            }
        }
    }

    public function hookDisplayFooter($params)
    {
        if (!$this->isCached('maofree_comparator_footer.tpl', $this->getCacheId('maofree_comparator_footer'))) {
            $id_shop = (int) $this->context->shop->id;
            $comparators = $comparator_name = $comparator_title = $comparator_url = $comparator_nofollow = array();

            if (!isset(self::$cache_links)) {
                self::$cache_links = MaofreeComparatorClass::getComparatorName(self::MAIN_TABLE, $id_shop, true);
            }

            $comparators = self::$cache_links;

            if (is_array($comparators) && count($comparators)) {
                foreach ($comparators as $comparator) {
                    $name = $comparator['name'];
                    $url = $comparator['website_url'] ? explode('@', $comparator['website_url']) : array();
                    require_once dirname(__FILE__).'/comparators/'.$name.'/'.$name.'.php';
                    $comparator_obj = new $name();

                    if (
                        isset($comparator_obj->name) &&
                        is_array($comparator_obj->name) &&
                        isset($comparator_obj->website_url) &&
                        is_array($comparator_obj->website_url) &&
                        (count($comparator_obj->name) == count($comparator_obj->website_url))
                    ) {
                        for ($i = 0; $i < count($comparator_obj->name); ++$i) {
                            $comparator_name[] = $comparator_obj->name[$i];
                            $comparator_title[] = (isset($comparator_obj->title[$i]) && $comparator_obj->title[$i] ? $comparator_obj->title[$i] : $comparator_obj->name[$i]);
                            $comparator_url[] = isset($url[$i]) && $url[$i] ? $url[$i] : $comparator_obj->website_url[$i];
                            $comparator_nofollow[] = isset($comparator_obj->nofollow) ? (int) $comparator_obj->nofollow : 1;
                        }
                    }

                    unset($comparator_obj);
                }

                $this->smarty->assign(array(
                    'footer_comparator_name' => $comparator_name,
                    'footer_comparator_title' => $comparator_title,
                    'footer_comparator_url' => $comparator_url,
                    'footer_comparator_nofollow' => $comparator_nofollow,
                ));
            }
        }

        return $this->display(__FILE__, 'maofree_comparator_footer.tpl', $this->getCacheId('maofree_comparator_footer'));
    }

    public function hookDisplayOrderConfirmation($params)
    {
        if (!$this->active) {
            return;
        }

        $order = $params['objOrder'];

        $id_shop = (int) $this->context->shop->id;
        $comparators = $tracking_codes = array();

        if (!isset(self::$cache_tracks)) {
            self::$cache_tracks = MaofreeComparatorClass::getComparatorName(self::MAIN_TABLE, $id_shop, false, true);
        }

        $comparators = self::$cache_tracks;

        if (Validate::isLoadedObject($order) && is_array($comparators) && count($comparators)) {
            foreach ($comparators as $comparator) {
                $name = $comparator['name'];

                if (file_exists(dirname(__FILE__).'/comparators/'.$name.'/'.$name.'.php')) {
                    require_once dirname(__FILE__).'/comparators/'.$name.'/'.$name.'.php';

                    $comparator_obj = new $name();

                    if (method_exists($comparator_obj, 'comparatorOrderConfirmation')) {
                        $result = $comparator_obj->comparatorOrderConfirmation($order);

                        if ($result) {
                            $tracking_codes[] = $result;
                        }
                    }

                    unset($comparator_obj);
                }
            }

            if (count($tracking_codes)) {
                $this->smarty->assign('tracking_codes', $tracking_codes);

                return $this->display(__FILE__, 'maofree_comparator_order.tpl');
            } else {
                return;
            }
        } else {
            return;
        }
    }

    public function hookActionCategoryDelete($params)
    {
        $all_cat = $params['deleted_children'];
        $all_cat[] = $params['category'];

        foreach (Shop::getContextListShopID() as $id_shop) {
            foreach ($all_cat as $cat) {
                $id_category = (int) $cat->id;

                if (!MaofreeComparatorClass::categoryExists($id_category, $id_shop)) {
                    Db::getInstance()->execute(
                        'DELETE FROM `'._DB_PREFIX_.pSQL(TAXONOMY_LANG_TABLE).'`
                        WHERE `id_shop` = '.$id_shop.'
                        AND `id_category` = '.$id_category
                    );
                    Db::getInstance()->execute(
                        'DELETE FROM `'._DB_PREFIX_.pSQL(TAXONOMY_COUNTRY_TABLE).'`
                        WHERE `id_shop` = '.$id_shop.'
                        AND `id_category` = '.$id_category
                    );
                }
            }
        }
    }

    public function hookActionCarrierUpdate($params)
    {
        $previous_id_carrier = (int) $params['id_carrier'];
        $new_carrier = $params['carrier'];

        if (Configuration::getGlobalValue('PS_MULTISHOP_FEATURE_ACTIVE')) {
            $shops = Shop::getShops(false, null, true);

            if (is_array($shops) && count($shops)) {
                foreach ($shops as $id_shop) {
                    $id_group = (int) Shop::getGroupFromShop($id_shop);
                    $config = Configuration::get('MAO_COMPARATOR_CARRIERSZONE', null, $id_group, (int) $id_shop);

                    if (isset($config) && $config) {
                        $changed = 0;
                        $carriers_zone = array();
                        $carriers_zone_array = explode(',', $config);

                        foreach ($carriers_zone_array as $row) {
                            $zone_array = explode('#', $row);

                            $carriers_zone[] = $zone_array[0].'#'.($previous_id_carrier == $zone_array[1] ? (int) $new_carrier->id : $zone_array[1]);

                            if ($previous_id_carrier == $zone_array[1]) {
                                ++$changed;
                            }
                        }

                        if ($changed && count($carriers_zone)) {
                            Configuration::updateValue(
                                'MAO_COMPARATOR_CARRIERSZONE',
                                implode(',', $carriers_zone),
                                false,
                                $id_group,
                                (int) $id_shop
                            );
                        }
                    }
                }
            }
        } else {
            $config = Configuration::get('MAO_COMPARATOR_CARRIERSZONE');

            if (isset($config) && $config) {
                $changed = 0;
                $carriers_zone = array();
                $carriers_zone_array = explode(',', $config);

                foreach ($carriers_zone_array as $row) {
                    $zone_array = explode('#', $row);

                    $carriers_zone[] = $zone_array[0].'#'.($previous_id_carrier == $zone_array[1] ? (int) $new_carrier->id : $zone_array[1]);

                    if ($previous_id_carrier == $zone_array[1]) {
                        ++$changed;
                    }
                }

                if ($changed && count($carriers_zone)) {
                    Configuration::updateValue('MAO_COMPARATOR_CARRIERSZONE', implode(',', $carriers_zone));
                }
            }
        }
    }

    public function ajaxProcessMaoManualUpdateFindCycles()
    {
        $response = array();
        $response['total_cycle'] = 0;

        if (
            Tools::getIsset('action') &&
            Tools::getValue('action') == 'MaoManualUpdateFindCycles' &&
            Tools::getIsset('filename') &&
            Tools::getValue('filename')
        ) {
            $filename = Tools::getValue('filename');

            $filename_array = explode('-', $filename);
            $extension_array = explode('.', $filename_array[1]);
            $data_array = explode('_', $extension_array[0]);

            $comparator_name = Tools::strtolower($filename_array[0]);
            $id_shop = $data_array[0];

            $settings = unserialize(Configuration::get('MAO_COMPARATOR_SETTINGS'));
            $comparator = MaofreeComparatorClass::getValuesFromTable(self::MAIN_TABLE, $id_shop, $comparator_name);
            $products_cycle = $settings['products_cycle'];
            $selectall = (int) $comparator['selected_all'];
            $selectcat = $comparator['selected_categories'];
            $selectmanu = $comparator['selected_manufacturers'];
            $selectmanual = $comparator['selected_products_manually'];
            $removed_products_id = $comparator['removed_products'];

            $id_root_cat = (int) Configuration::get('PS_ROOT_CATEGORY');
            $id_home_cat = (int) Configuration::get('PS_HOME_CATEGORY');
            $stock_management = (int) Configuration::get('PS_STOCK_MANAGEMENT');
            $allow_out_of_stock_ordering = (int) Configuration::get('PS_ORDER_OUT_OF_STOCK');

            $num_totalproducts = MaofreeComparatorClass::getIdProducts(
                $id_shop,
                $selectall,
                $selectmanu,
                $selectcat,
                $selectmanual,
                $removed_products_id,
                true,
                (int) $comparator['on_stock_only'],
                $stock_management,
                $allow_out_of_stock_ordering,
                $id_root_cat,
                $id_home_cat,
                null,
                null,
                true
            );

            if (isset($num_totalproducts) && $num_totalproducts) {
                $response['total_cycle'] = ceil($num_totalproducts / $products_cycle);
            } else {
                MaofreeComparatorClass::updateValue(self::MAIN_TABLE, $id_shop, $comparator_name, 'type_error', 'noproducts_error');
                MaofreeComparatorClass::updateValue(self::MAIN_TABLE, $id_shop, $comparator_name, 'last_check', date('Y-m-j H:i:s'));
            }
        }

        die(Tools::jsonEncode($response));
    }

    public function ajaxProcessMaoManualUpdate()
    {
        $response = array();
        $response['current_cycle'] = $response['items_done'] = 0;
        $response['errors_cycle'] = '';

        if (
            Tools::getIsset('action') &&
            Tools::getValue('action') == 'MaoManualUpdate' &&
            Tools::getIsset('filename') &&
            Tools::getValue('filename') &&
            Tools::getIsset('totalcycle') &&
            Tools::getValue('totalcycle') &&
            Tools::getIsset('currentcycle') &&
            Tools::getIsset('tax') &&
            Tools::getIsset('itemsdone')
        ) {
            $filename = Tools::getValue('filename');
            $total_cycle = Tools::getValue('totalcycle');
            $current_cycle = Tools::getValue('currentcycle');
            $tax = (int) Tools::getValue('tax');
            $items_done = (int) Tools::getValue('itemsdone');

            $filename_array = explode('-', $filename);
            $extension_array = explode('.', $filename_array[1]);
            $data_array = explode('_', $extension_array[0]);

            $comparator_name = Tools::strtolower($filename_array[0]);
            $datafeed_type = $extension_array[1];
            $token = $this->token;
            $id_shop = $data_array[0];
            $id_currency = $data_array[1];
            $id_lang = $data_array[2];
            $id_country = $data_array[3];

            $data = MaofreeDatafeedClass::makeDatafeed(
                $comparator_name,
                $token,
                $datafeed_type,
                $id_shop,
                $id_lang,
                $id_country,
                null,
                $id_currency,
                $current_cycle,
                $total_cycle,
                $tax,
                $items_done,
                0,
                true
            );

            $response['errors_cycle'] = is_array($data) && $data['errors'] ? $data['errors'] : 'witherrors';
            $response['current_cycle'] = is_array($data) && $data['force_last_cycle'] ? $total_cycle : ++$current_cycle;
            $response['items_done'] = is_array($data) && isset($data['items_done']) && $data['items_done'] ? (int) $data['items_done'] : 0;

            // error code report $response['errors_cycle']
        } else {
            $response['errors_cycle'] = 'witherrors';
        }

        die(Tools::jsonEncode($response));
    }

    public function ajaxProcessMaoSearchCategory()
    {
        $response = array();
        $response['message'] = '';
        $response['items'] = 0;

        if (
            Tools::getIsset('action') &&
            Tools::getValue('action') == 'MaoSearchCategory' &&
            Tools::getIsset('queryString') &&
            Tools::getValue('queryString') &&
            Tools::getIsset('id_lang') &&
            Tools::getValue('id_lang') &&
            Tools::getIsset('id_country') &&
            Tools::getValue('id_country') &&
            Tools::getIsset('taxonomy_files_id') &&
            Tools::getIsset('comparator_name') &&
            Tools::getValue('comparator_name')
        ) {
            $response['message'] = '<ul>';

            $queryString = Tools::getValue('queryString');
            $id_lang = (int) Tools::getValue('id_lang');
            $id_country = (int) Tools::getValue('id_country');
            $taxonomy_files_id = (int) Tools::getValue('taxonomy_files_id');
            $comparator_name = Tools::getValue('comparator_name');

            if (Tools::strlen($queryString) > 1) {
                require_once dirname(__FILE__).'/comparators/'.$comparator_name.'/'.$comparator_name.'.php';

                $comparator_obj = new $comparator_name();
                $taxonomy_files = $comparator_obj->taxonomy_files[$taxonomy_files_id];
                $taxonomy_type = $taxonomy_files['type'];

                if ($taxonomy_type == 'language') {
                    $language = new Language($id_lang);
                    $iso_code = $language->language_code;
                } else {
                    $country = new Country($id_country, $id_lang);
                    $iso_code = $country->iso_code;
                }

                $result = $comparator_obj->readTaxonomyFile($iso_code, $taxonomy_files_id);

                if (is_array($result) && $result) {
                    foreach ($result as $row) {
                        // to avoit to use mb_stripos
                        if (stripos(mb_strtolower($row), mb_strtolower($queryString)) === false) {
                            continue;
                        }

                        $response['message'] .= '<li>'.trim($row).'</li>';
                        ++$response['items'];
                    }

                    if (!$response['items']) {
                        $response['message'] .= '<li class="messagealert">'.$this->l('ERROR: I have not found any category').'</li>';
                    }
                } else {
                    $response['message'] .= '<li class="messagealert">'.$this->l('ERROR: I have not found any data from taxonomy file').'</li>';
                }
            } else {
                $response['message'] .= '<li class="messagealert">'.$this->l('Digit at least 2 characters').'</li>';
            }

            $response['message'] .= '</ul>';
        }

        die(Tools::jsonEncode($response));
    }

    public function ajaxProcessMaoStatusCronJob()
    {
        $response = array();
        $response['active'] = 0;
        $response['errors'] = $this->l('There is an ajax error');

        if (
            Tools::getIsset('action') &&
            Tools::getValue('action') == 'MaoStatusCronJob' &&
            Tools::getIsset('cronjob') &&
            Tools::getValue('cronjob')
        ) {
            $cronjob = (int) Tools::getValue('cronjob');
            $is_active = (int) MaofreeComparatorClass::checkStatusCronJob(self::CRONJOBS_TABLE, $cronjob);

            MaofreeComparatorClass::setStatusCronJob(self::CRONJOBS_TABLE, $cronjob, !$is_active);

            $response['errors'] = '';
            $response['active'] = !$is_active;
        }

        die(Tools::jsonEncode($response));
    }

    public function ajaxProcessMaoRemoveCronJob()
    {
        $response = array();
        $response['remove'] = 0;
        $response['errors'] = $this->l('There is an ajax error');

        if (
            Tools::getIsset('action') &&
            Tools::getValue('action') == 'MaoRemoveCronJob' &&
            Tools::getIsset('cronjob') &&
            Tools::getValue('cronjob')
        ) {
            $cronjob = (int) Tools::getValue('cronjob');

            MaofreeComparatorClass::removeCronJob(self::CRONJOBS_TABLE, $cronjob);

            $response['errors'] = '';
            $response['remove'] = 1;
        }

        die(Tools::jsonEncode($response));
    }

    private function moduleClearCache()
    {
        parent::_clearCache('maofree_comparator_block.tpl', 'maofree_comparator_block');
        parent::_clearCache('maofree_comparator_footer.tpl', 'maofree_comparator_footer');
        parent::_clearCache('maofree_comparator_home.tpl', 'maofree_comparator_home');
    }
}
