/* gfileutf8.c - UTF-8 wrappers for C library functions
 *
 * Copyright 2004 Tor Lillqvist
 *
 * GLib is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * GLib is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with GLib; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "config.h"

#include "glib.h"
#include "gfileutf8.h"

#ifdef G_OS_UNIX
#include <unistd.h>
#endif

#ifdef G_OS_WIN32
#include <wchar.h>
#include <io.h>
#endif

#if !defined (G_OS_UNIX) && !defined (G_OS_WIN32)
#error Please port this to your operating system
#endif

#include <sys/stat.h>

int
g_open_utf8 (const gchar *pathname_utf8,
	     int          flags,
	     int          mode)
{
  gchar *pathname;
  int retval;

#ifdef G_OS_WIN32
  if (G_WIN32_HAS_WIDECHAR_API ())
    {
      wchar_t *pathname = g_utf8_to_utf16 (pathname_utf8, -1, NULL, NULL, NULL);
      int retval = _wopen (pathname, flags, mode);
      
      g_free (pathname);

      return retval;
    }
#endif

  pathname = g_filename_from_utf8 (pathname_utf8, -1, NULL, NULL, NULL);
  retval = open (pathname, flags, mode);

  g_free (pathname);

  return retval;
}

int
g_rename_utf8 (const gchar *oldpath_utf8,
	       const gchar *newpath_utf8)
{
  gchar *oldpath;
  gchar *newpath;
  int retval;

#ifdef G_OS_WIN32
  if (G_WIN32_HAS_WIDECHAR_API ())
    {
      wchar_t *oldpath = g_utf8_to_utf16 (oldpath_utf8, -1, NULL, NULL, NULL);
      wchar_t *newpath = g_utf8_to_utf16 (newpath_utf8, -1, NULL, NULL, NULL);
      int retval = _wrename (oldpath, newpath);
      
      g_free (oldpath);
      g_free (newpath);
      
      return retval;
    }
#endif

  oldpath = g_filename_from_utf8 (oldpath_utf8, -1, NULL, NULL, NULL);
  newpath = g_filename_from_utf8 (newpath_utf8, -1, NULL, NULL, NULL);
  retval = rename (oldpath, newpath);

  g_free (oldpath);
  g_free (newpath);

  return retval;
}

int
g_mkdir_utf8 (const gchar *pathname_utf8,
	      int          mode)
{
  gchar *pathname;
  int retval;

#ifdef G_OS_WIN32
  if (G_WIN32_HAS_WIDECHAR_API ())
    {
      wchar_t *pathname = g_utf8_to_utf16 (pathname_utf8, -1, NULL, NULL, NULL);
      int retval = _wmkdir (pathname);
      
      g_free (pathname);
      
      return retval;
    }
#endif

  pathname = g_filename_from_utf8 (pathname_utf8, -1, NULL, NULL, NULL);
#ifdef G_OS_WIN32
  retval = mkdir (pathname);
#else
  retval = mkdir (pathname, mode);
#endif

  g_free (pathname);

  return retval;
}

int
g_stat_utf8 (const gchar *pathname_utf8,
	     struct stat *buf)
{
  gchar *pathname;
  int retval;

#ifdef G_OS_WIN32
  if (G_WIN32_HAS_WIDECHAR_API ())
    {
      wchar_t *pathname = g_utf8_to_utf16 (pathname_utf8, -1, NULL, NULL, NULL);
      int retval = _wstat (pathname, (struct _stat *) buf);

      g_free (pathname);

      return retval;
    }
#endif

  pathname = g_filename_from_utf8 (pathname_utf8, -1, NULL, NULL, NULL);
  retval = stat (pathname, buf);

  g_free (pathname);

  return retval;
}

int
g_unlink_utf8 (const gchar *pathname_utf8)
{
  gchar *pathname;
  int retval;

#ifdef G_OS_WIN32
  if (G_WIN32_HAS_WIDECHAR_API ())
    {
      wchar_t *pathname = g_utf8_to_utf16 (pathname_utf8, -1, NULL, NULL, NULL);
      int retval = _wunlink (pathname);
  
      g_free (pathname);

      return retval;
    }
#endif

  pathname = g_filename_from_utf8 (pathname_utf8, -1, NULL, NULL, NULL);
  retval = unlink (pathname);

  g_free (pathname);

  return retval;
}

FILE *
g_fopen_utf8 (const gchar *pathname_utf8,
	      const gchar *mode_utf8)
{
  gchar *pathname;
  gchar *mode;
  FILE *retval;

#ifdef G_OS_WIN32
  if (G_WIN32_HAS_WIDECHAR_API ())
    {
      wchar_t *pathname = g_utf8_to_utf16 (pathname_utf8, -1, NULL, NULL, NULL);
      wchar_t *mode = g_utf8_to_utf16 (mode_utf8, -1, NULL, NULL, NULL);
      FILE *retval = _wfopen (pathname, mode);

      g_free (pathname);
      g_free (mode);

      return retval;
    }
#endif

  pathname = g_filename_from_utf8 (pathname_utf8, -1, NULL, NULL, NULL);
  mode = g_filename_from_utf8 (mode_utf8, -1, NULL, NULL, NULL);
  retval = fopen (pathname, mode);

  g_free (pathname);
  g_free (mode);

  return retval;
}

FILE *
g_freopen_utf8 (const gchar *pathname_utf8,
		const gchar *mode_utf8,
		FILE        *stream)
{
  gchar *pathname;
  gchar *mode;
  FILE *retval;

#ifdef G_OS_WIN32
  if (G_WIN32_HAS_WIDECHAR_API ())
    {
      wchar_t *pathname = g_utf8_to_utf16 (pathname_utf8, -1, NULL, NULL, NULL);
      wchar_t *mode = g_utf8_to_utf16 (mode_utf8, -1, NULL, NULL, NULL);
      FILE *retval = _wfreopen (pathname, mode, stream);

      g_free (pathname);
      g_free (mode);

      return retval;
    }
#endif

  pathname = g_filename_from_utf8 (pathname_utf8, -1, NULL, NULL, NULL);
  mode = g_filename_from_utf8 (mode_utf8, -1, NULL, NULL, NULL);
  retval = freopen (pathname, mode, stream);

  g_free (pathname);
  g_free (mode);

  return retval;
}
