/*
 * Copyright (C) 2008 Ole Andr Vadla Ravns <ole.andre.ravnas@tandberg.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "testutil.h"

#include <string.h>
#include <gst/rtp/gstrtpbuffer.h>

struct _GtuClockWaitContext
{
  GMutex * mutex;
  GCond * cond;

  GThread * worker_thread;

  GstTestClock * test_clock;
  GstClockID clock_id;
  gboolean complete;
  GstClockReturn clock_ret;
  GstClockTimeDiff * jitter;
};

static gpointer gst_test_util_wait_for_clock_id_worker (gpointer data);

void
gtu_dump_rtp_buffers (const GstBuffer ** rtp_buffers,
                      guint num_rtp_buffers)
{
  guint i;

  for (i = 0; i < num_rtp_buffers; i++)
  {
    GstBuffer * buf = (GstBuffer *) rtp_buffers[i];

    g_print ("buffer[%d]:\n"
        "  payload_type=%u\n"
        "  marker=%d\n"
        "  seq=%u\n"
        "  timestamp=%u\n"
        "  ssrc=%u\n",
        i,
        gst_rtp_buffer_get_payload_type (buf),
        gst_rtp_buffer_get_marker (buf),
        gst_rtp_buffer_get_seq (buf),
        gst_rtp_buffer_get_timestamp (buf),
        gst_rtp_buffer_get_ssrc (buf));
  }
}

void
gtu_shift_buffer_timestamps_from_first (GstBuffer ** buffers,
                                        guint num_buffers)
{
  guint i;

  GstClockTime offset = GST_BUFFER_TIMESTAMP (buffers[0]);

  for (i = 0; i < num_buffers; i++)
    GST_BUFFER_TIMESTAMP (buffers[i]) -= offset;
}

GtuClockWaitContext *
gst_test_util_wait_for_clock_id_begin (GstTestClock * test_clock,
                                       GstClockID clock_id,
                                       GstClockTimeDiff * jitter)
{
  GtuClockWaitContext * ctx;

  ctx = g_new (GtuClockWaitContext, 1);

  ctx->mutex = g_mutex_new ();
  ctx->cond = g_cond_new ();

  ctx->test_clock = gst_object_ref (test_clock);
  ctx->clock_id = clock_id;
  ctx->complete = FALSE;
  ctx->clock_ret = GST_CLOCK_ERROR;
  ctx->jitter = jitter;

  ctx->worker_thread = g_thread_create (gst_test_util_wait_for_clock_id_worker,
      ctx, TRUE, NULL);
  g_assert (ctx->worker_thread != NULL);

  while (!gst_test_clock_has_id (test_clock, clock_id))
    g_usleep (G_USEC_PER_SEC / 10 / 10);

  return ctx;
}

gboolean
gst_test_util_clock_wait_context_has_completed (GtuClockWaitContext * ctx)
{
  gboolean result;

  g_mutex_lock (ctx->mutex);
  result = ctx->complete;
  g_mutex_unlock (ctx->mutex);

  return result;
}

GstClockReturn
gst_test_util_wait_for_clock_id_end (GtuClockWaitContext * ctx)
{
  GstClockReturn result;
  GTimeVal deadline;
  gboolean processed = FALSE;

  g_get_current_time (&deadline);
  g_time_val_add (&deadline, 20 * G_USEC_PER_SEC);

  g_mutex_lock (ctx->mutex);

  while (!ctx->complete)
  {
    GTimeVal next_wakeup;
    gboolean signalled;

    g_get_current_time (&next_wakeup);
    g_time_val_add (&next_wakeup, G_USEC_PER_SEC / 10 / 2);

    signalled = g_cond_timed_wait (ctx->cond, ctx->mutex, &next_wakeup);
    if (!signalled)
    {
      GstClockID clock_id;

      g_assert (next_wakeup.tv_sec <= deadline.tv_sec);

      if (!processed)
      {
        clock_id = gst_test_clock_process_next_clock_id (ctx->test_clock);
        if (clock_id != NULL)
        {
          g_assert (clock_id == ctx->clock_id);
          processed = TRUE;
        }
      }
    }
  }

  g_mutex_unlock (ctx->mutex);

  g_thread_join (ctx->worker_thread);

  gst_object_unref (ctx->test_clock);

  g_cond_free (ctx->cond);
  g_mutex_free (ctx->mutex);

  result = ctx->clock_ret;
  g_free (ctx);

  return result;
}

static gpointer
gst_test_util_wait_for_clock_id_worker (gpointer data)
{
  GtuClockWaitContext * ctx = data;

  ctx->clock_ret = gst_clock_id_wait (ctx->clock_id, ctx->jitter);

  g_mutex_lock (ctx->mutex);
  ctx->complete = TRUE;
  g_cond_signal (ctx->cond);
  g_mutex_unlock (ctx->mutex);

  return NULL;
}
