/*[
 * This file is part of the GOCAD Software.
 * 
 * Copyright 1989-2008 Paradigm B.V. and/or its affiliates and subsidiaries.
 * All rights reserved.
 * 
 * Portions of software code were developed by Association Scientifique pour la
 * Geologie et ses Applications (ASGA). All rights reserved.
 * 
 * Use of this software is subject to the terms and conditions of the license
 * agreement with Paradigm Geophysical B.V. and/or its affiliates and
 * subsidiaries (collectively "Paradigm").
 * 
 * Warning: This computer program is protected by copyright law and international
 * treaties. Unauthorized reproduction or distribution of this program, or any
 * portion of it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
]*/

#ifndef Gocad_ascii_as_catalog_h
#define Gocad_ascii_as_catalog_h

#include <Gocad/ascii/enter_scope.h>
#include <Gocad/utils/containers/list.h>
#include <Gocad/utils/containers/set.h>
#include <Gocad/utils/misc/catalog.h>
#include <Gocad/utils/misc/version.h>

#include <Gocad/utils/os/iosfwd.h>

namespace Gocad {
    class istreamb;
    class ostreamb;
    class Vertex;
    class GObj;
    class Atomic;
    class Atom;
    class Style;
    class AsciiConverterMap;
    class Property;
    class CopyCString ;
    class UniqueCString ;
    class COSUri ;
} // namespace Gocad

/***************************************************************************/

#define GOCAD_ASCII_SEPARATOR "END"
#define GOCAD_ASCII_VERSION 1.0

namespace Gocad {

class _Gocad_ascii_decl AsciiCatalog  : public Catalog {

public:
    AsciiCatalog(float version = GOCAD_ASCII_VERSION);
    AsciiCatalog(AsciiCatalog *);
    AsciiCatalog(const AsciiCatalog &);
    virtual ~AsciiCatalog();

    Style* style() const;
    void set_style(Style* s);

    bool save(const PtrList<GObj>& gobjs, const CString& path);
    bool save(const PtrList<GObj>& gobjs, ostreamb& out);
    bool save(GObj* gobj, ostreamb& out);
    bool save(GObj* gobj, const CString& path);

    bool load(PtrList<GObj>& gobjs, const CString& path);
    bool load(PtrList<GObj>& gobjs, istreamb& in);

    PtrList<GObj>* load(const CString& path);
    PtrList<GObj>* load(istreamb& in);

    GObj* load(istreamb& in, const TypeId& id);
    GObj* load_gobj(istreamb& in) ;

    PtrList<GObj>& gobjs();

    GObj* find(const CString& name) const ;

    const CString& filename() const;

    void error(const Format& message);

    void warning(const Format& message);

    const VersionNumber& latest_version(const CString& type_name) const;

    void* create(const TypeId& id);
    void* create_converter(const TypeId& id);
    void* create_converter(const CString& type_name);

    void* create_converter(
         const CString& type_name, const VersionNumber& version
    ) const;

    void write_header(GObj* gobj, ostreamb& out) const ;
    void write_header(const TypeId& type_id, ostreamb& out) const ;

    void write_trailer(ostreamb& out) const ;

    void write_style(GObj*, ostreamb& out) const ;

    void write_style_attributes(Style* style, ostreamb& out, Set<CopyCString>& written) const ;

    bool read_style(istreamb&, const CString& code, GObj* gobj) ;

    bool read_style(istreamb&, Style* style) ;

    bool get(istreamb&, char&) const;

    bool next(istreamb&);

    void eat_line(istreamb&) const;

    void eat_whitespace(istreamb&, bool stop_at_eol = false) const;

    void bad_keyword(const char* code, istreamb &in);

    const char* code(istreamb& in);

    bool putback(const char* code) ;

    const char* peek(istreamb&);

    void read_quoted_string(istreamb&, char*) const;

    void read_quoted_string(istreamb&, CopyCString&) const;

    void write_quoted_string(ostreamb& out, const CString& string, bool enforce_backward=false) const;

    bool get_path(istreamb& in, CopyCString& path) const;
    bool get_path(CopyCString& path) const ;

    CopyCString create_path(
            const GObj* gobj, const CString& suffix, bool share=false
    ) const ;

    GOCAD_SUPPORT_2_0_DECL const char* stdio_open_mode(const UniqueCString& path) ;
    GOCAD_SUPPORT_2_0_DECL GOCAD_IOS_OPENMODE stream_open_mode(const UniqueCString& path) ;

    const char* binary_stdio_open_mode( const UniqueCString& path ) ;
    GOCAD_IOS_OPENMODE binary_stream_open_mode(const UniqueCString& path) ;

    const char* ascii_stdio_open_mode( const UniqueCString& path ) ;
    GOCAD_IOS_OPENMODE ascii_stream_open_mode(const UniqueCString& path) ;

    void set_append_mode() ;

    void unset_append_mode() ;

    bool is_in_append_mode() const ;

    CopyCString temp_path(const CString& path) const ;

    void rename_temp_path(const CString& path) const ;

    COSUri neutral_uri_from_uri( const COSUri& uri ) ;

    void insert_v(void *v,InstId id) ;
    bool find_v(void *v,InstId &id) ;
    InstId idAt2IdVx(InstId idAt) ;
    void store_idAt2IdVx(InstId idAt, InstId idVx) ;

protected :
    enum { gobj_header=1, bad_header=-1, other_header=0 } ;

    AsciiConverterMap* map() ;
    int read_header(istreamb& in, char* className, float& version);
    bool find_data_path(const UniqueCString& path) ;
    void add_data_path(const UniqueCString& path) ;

private:
    void increment_line_number() const;

protected :
    PtrList<GObj> gobjs_;
    Style* style_;
    CopyCString filename_;
    int lineno_;
    int nerrors_;
    int nwarnings_;
    char* pb_code_;
    Set<UniqueCString>* data_paths_;
};

/****************************************************************************/

inline Style* AsciiCatalog::style() const {
    return style_;
}
inline PtrList<GObj>& AsciiCatalog::gobjs() {
    return gobjs_;
}
inline const CString& AsciiCatalog::filename() const {
    return filename_;
}
inline bool AsciiCatalog::is_in_append_mode() const {
    return data_paths_ != nullptr;
}
inline void AsciiCatalog::increment_line_number() const {
    AsciiCatalog* non_const_this = const_cast<AsciiCatalog*>( this);
    non_const_this->lineno_++;
}

} // namespace Gocad

#endif
