/*[
 * This file is part of the GOCAD Software.
 * 
 * Copyright 1989-2008 Paradigm B.V. and/or its affiliates and subsidiaries.
 * All rights reserved.
 * 
 * Portions of software code were developed by Association Scientifique pour la
 * Geologie et ses Applications (ASGA). All rights reserved.
 * 
 * Use of this software is subject to the terms and conditions of the license
 * agreement with Paradigm Geophysical B.V. and/or its affiliates and
 * subsidiaries (collectively "Paradigm").
 * 
 * Warning: This computer program is protected by copyright law and international
 * treaties. Unauthorized reproduction or distribution of this program, or any
 * portion of it, may result in severe civil and criminal penalties, and will be
 * prosecuted to the maximum extent possible under the law.
]*/

/*! \file Gocad/ascii/as_catalog.h
 * \brief NO BRIEF DOCUMENTATION
 *
 * NOT DOCUMENTED
 */


/* TRANSLATOR Gocad::Format */

#include <Gocad/ascii/as_catalog.h>

#include <Gocad/ascii/asciilib.h>
#include <Gocad/ascii/as_gobj.h>

#include <Gocad/geobase/base/geobaselib.h>
#include <Gocad/geobase/base/gobj.h>
#include <Gocad/geobase/base/gobjstyle.h>

#include <Gocad/utils/entity/entitylib.h>
#include <Gocad/utils/entity/entity_events.h>
#include <Gocad/utils/entity/viewer_entitymgr.h>
#include <Gocad/utils/entity/entitymgr.h>
#include <Gocad/utils/entity/entity_utils.h>
#include <Gocad/utils/gocad/softnotes.h>
#include <Gocad/utils/misc/messagelib.h>
#include <Gocad/utils/misc/style.h>
#include <Gocad/utils/misc/style_dump.h>
#include <Gocad/utils/os/iostreamb.h>
#include <Gocad/utils/os/memory.h>
#include <Gocad/utils/os/pathname.h>
#include <Gocad/utils/os/pfilebuf.h>
#include <Gocad/utils/os/string.h>
#include <Gocad/utils/os/system.h>
#include <Gocad/utils/os/debug_assert.h>
#include <Gocad/utils/cos/cos_uri.h>

#include <ctype.h>

#if 0
//
// This protection has been moved to xmasciilib to avoid denying
// save operations to developers writing batch programs
// (see incident 020220-000)
// TV.
//
#include <glm/glm.h>
#ifdef USE_GLM
GLM_USE_CORE_API();
#include <glm/gocad_base.gfd>
#endif
#endif

/***************************************************************************/

namespace Gocad {

typedef Table <InstId, InstId> TableIdAt2IdVx ;

class AsciiConverterMap : public ObjectMap {
public :
      AsciiConverterMap(InstId universal=1);
      ~AsciiConverterMap();

      bool find_v(void *v,InstId &);
      void insert_v(void *v,InstId id);
      void store_idAt2IdVx(InstId idAt, InstId idVx) ;
      InstId idAt2IdVx(InstId idAt) ;
protected :
      TableAddr2Id * vrtx2Id_ ;
      TableIdAt2IdVx * idAt2IdVx_ ;
};

AsciiConverterMap::AsciiConverterMap(InstId u) : ObjectMap(u) {
    vrtx2Id_ = new TableAddr2Id(256);
    idAt2IdVx_ = new TableIdAt2IdVx(256) ;
}

AsciiConverterMap::~AsciiConverterMap() {
    delete vrtx2Id_;
    delete idAt2IdVx_ ;
}

/*! \def GOCAD_ASCII_SEPARATOR
 * NOT DOCUMENTED
 */

/*! \def GOCAD_ASCII_VERSION
 * NOT DOCUMENTED
 */

/*! \class AsciiCatalog Gocad/ascii/as_catalog.h
 * \ingroup Gocadascii
 * \brief PLEASE PROVIDE A BRIEF DESCRIPTION OF THE CLASS.
 *
 * AsciiCatalog loads GObj from ASCII files and save GObjs on ASCII files.
 * Many GObjs can be saved in the same file.
 */

/*! \fn Style* AsciiCatalog::style() const
 * Returns or set the style of the ascii catalog.
 */

/*! \fn PtrList<GObj>& AsciiCatalog::gobjs()
 * Returns the list of GObjs read by the catalog so far.
 */

/*! \fn const CString& AsciiCatalog::filename() const
 * Returns the filename currently being used for reading or writing
 * the file. Return nullptr if no filename is known (e.g. a pipe)\
 */

/*! \fn bool AsciiCatalog::is_in_append_mode() const
 * Test if the catalog is in 'append mode'.
 */

/*! \fn void AsciiCatalog::increment_line_number() const
 * \internal Increments the line number. lineno_ should be mutable in fact.
 * This const member function wraps modification of line number for compilers
 * that does not support mutable keyword.
 */

/*!
 * NOT DOCUMENTED
 */
AsciiConverterMap* AsciiCatalog::map() {
    return static_cast<AsciiConverterMap*> (curMap_);
}

bool AsciiConverterMap::find_v(void *v,InstId &id) {
    return vrtx2Id_->find(id,v);
}

void AsciiConverterMap::insert_v(void *v,InstId id) {
    vrtx2Id_->insert(v,id);
}

void AsciiConverterMap::store_idAt2IdVx(InstId idAt, InstId idVx) {
    idAt2IdVx_-> insert(idAt, idVx) ;
}

InstId AsciiConverterMap::idAt2IdVx(InstId idAt) {
    InstId idVx ;
    if( idAt2IdVx_-> find(idVx,idAt) ) {
       return idVx ;
    } else {
       return idAt ;
    }
}

/************************************************************************/

/*!
 * NOT DOCUMENTED
 */
AsciiCatalog::AsciiCatalog(float version) :
    Catalog("AsciiCatalog", version),
    style_(nullptr),
    filename_(),
    lineno_(0),
    nerrors_(0),
    nwarnings_(0),
    pb_code_(nullptr),
    data_paths_(nullptr)
{
    delete curMap_;
    curMap_ = new AsciiConverterMap();
}

/*!
 * NOT DOCUMENTED
 */
AsciiCatalog::AsciiCatalog(AsciiCatalog *ac) :
    Catalog("AsciiCatalog", ac->version_),
    style_(nullptr),
    filename_(ac->filename_),
    lineno_(0),
    nerrors_(0),
    nwarnings_(0),
    pb_code_(nullptr),
    data_paths_(nullptr)
{
    delete curMap_;
    curMap_ = new AsciiConverterMap();
}

/*!
 * NOT DOCUMENTED
 */
AsciiCatalog::AsciiCatalog(const AsciiCatalog &ac) :
    Catalog("AsciiCatalog", ac.version_),
    style_(nullptr),
    filename_(ac.filename_),
    lineno_(0),
    nerrors_(0),
    nwarnings_(0),
    pb_code_(nullptr),
    data_paths_(nullptr)
{
    delete curMap_;
    curMap_ = new AsciiConverterMap();
}

/*!
 * NOT DOCUMENTED
 */
AsciiCatalog::~AsciiCatalog() {
    Resource::unref(style_);
    delete[] pb_code_;
    delete data_paths_;
}

/*!
 * Returns the latest version number available for ascii converters
 * of the given type identifier.
 */
const VersionNumber& AsciiCatalog::latest_version(const CString& name) const {
    return FactoryLib::instance()->latest_version(
        GocadAsciiConverterFactoriesDomain(), name
    );
}

/*!
 * NOT DOCUMENTED
 */
void AsciiCatalog::insert_v(void *v,InstId id) {
     map()-> insert_v(v,id) ;
}

/*!
 * NOT DOCUMENTED
 */
bool AsciiCatalog::find_v(void *v,InstId &id) {
     return map()->find_v(v,id) ;
}

/*!
 * NOT DOCUMENTED
 */
InstId AsciiCatalog::idAt2IdVx(InstId idAt) {
    return map()-> idAt2IdVx(idAt) ;
}

/*!
 * NOT DOCUMENTED
 */
void AsciiCatalog::store_idAt2IdVx(InstId idAt, InstId idVx) {
    map()-> store_idAt2IdVx(idAt, idVx) ;
}

/*!
 * Creates an ascii converter converter for the given type
 * identifier and the latest version number available for this
 * type.
 */
void* AsciiCatalog::create(const TypeId& id) {
    return create_converter(id.name()) ;
}

/*!
 * Creates an ascii converter converter for the given type
 * identifier and the latest version number available for this
 * type.
 */
void* AsciiCatalog::create_converter(const TypeId& id) {
    void * c = create_converter(id.name()) ;
    if( c == nullptr ) {
        // If somebody creates a derived class from a gocad type
        // the object cannot be saved! We need to look at the
        // best matching factory and not only The matching factory.

        // Temporary fix until we use TradingServices for finding factories
        // Trading services will do the best match between the object type
        // and the service type required.
        List<TypeId> base_ids ;
        id.list_bases(base_ids, true) ;
        for( List<TypeId>::iterator i(base_ids); i.more(); i.next() ) {
            const TypeId& base_id = i.cur() ;
            c = create_converter(base_id.name());
            if( c != nullptr ) break ;
        }
    }
    return c ;
}

/*!
 * Creates an ascii converter converter for the given type
 * identifier and the latest version number available for this
 * type.
 */
void* AsciiCatalog::create_converter(const CString& name) {
    return create_converter(name, latest_version(name));
}

/*!
 * Creates an ascii converter converter for the given type
 * identifier and version number.
 */
void* AsciiCatalog::create_converter(
    const CString& name, const VersionNumber& version
) const {
    FactoryLib* lib = FactoryLib::instance();
    AbstractVersionedFactoryFinder_var finder = lib->versioned_factory_finder(
        GocadAsciiConverterFactoriesDomain()
    );
    AsciiConverterFactory factory = finder->find_factory(name, version);
    AsciiConverter* converter = factory.create();
    if (converter != nullptr) {
        converter->init(TypeId(name), version);
    }
    return converter;
}

/*!
 * Returns or set the style of the ascii catalog.
 */
void AsciiCatalog::set_style(Style *s) {
    if( s != style_ ) {
        Resource::unref(style_);
        style_ = s;
        Resource::ref(style_);
    }
}

/*!
 * Returns true if there is more characters to read on the input
 * stream.  Will not find a putback code (use code() for this).
 */
bool AsciiCatalog::next(istreamb &in) {
    if( in.eof() ) return false;
    if( !in.good() ) {
        error(Format::tr("input error\n"));
        return false;
    }
    eat_whitespace(in);
    if( in.eof() ) return false;
    while( in.peek() == '#' ) {
	// temporary fix for #CODE to become CODE one day
        static char code[80];
        in.width(sizeof(code));
        in >> code;
	if( !Str::equal(code,"#TFACE") &&
	    !Str::equal(code,"#ILINE") &&
	    !Str::equal(code,"#TVOLUME")
	) {
            eat_line(in);
	} else {
            putback(CString(code).right(1).string());
            return true ;
	}
    }
    return !in.eof();
}

/*!
 * Retrieve on character from the input stream.
 */
bool AsciiCatalog::get(istreamb &in, char& c) const {
    in.get( c );
    if( in.good() == 0 ) {
        return false;
    }
    if (c == '\n') {
        increment_line_number();
    }
    return true;
}

/*!
 * Read and ignore white space characters from the stream.
 */
void AsciiCatalog::eat_whitespace(istreamb &in, bool stop_at_eol) const{
    char c ;
    while( in.get(c) ) {
      if( !isspace((unsigned char)c) ) {
         in.putback(c);
         break;
      } else if(c == '\n') {
          if (stop_at_eol) {
              in.putback(c);
              break;
          }
          increment_line_number();
      }
    }
    // added the line below because certain implementation
    // of the operator above arrives to eof and fail at the same time.
    if( in.fail() ) in.clear( ~ios::failbit & in.rdstate()) ;
}

/*!
 * Read and ignore a line of input from the stream.
 */
void AsciiCatalog::eat_line(istreamb &in) const{
    char c;
    in.get( c );
    while( ( in.good() != 0 ) && c != '\n') {
        in.get( c );
    }
    increment_line_number();
    if( in.eof() ) return;
    eat_whitespace(in);
}

/*!
 * Print a warning about an unrecognized keyword and eat
 * the rest of the input line
 */
void AsciiCatalog::bad_keyword(const char* code, istreamb &in) {
    warning(Format::tr("Ignoring unrecognized keyword: %1\n") % code);
    eat_line(in) ;
}

/*!
 * Reads the input stream and returns the next keyword code.
 *
 * Keyword might have been written to \p in between quotes, in which case the
 * returned string is the unquoted keyword.
 *
 * \note This method does not handle escaped double quotes within strings.
 */
const char* AsciiCatalog::code(istreamb &in) {
    if( !in ) return nullptr;

    static char code[80];
    if (pb_code_) {
        Str::copy(code,pb_code_) ;
        delete[] pb_code_;
        pb_code_ = nullptr;
        return code;
    }
    if( !next(in) ) return nullptr;

    // temporary fix for #CODE to become CODE one day
    if( pb_code_ != nullptr ) {
        Str::copy(code,pb_code_) ;
        delete[] pb_code_;
        pb_code_ = nullptr;
        return code;
    }

    if( in.peek() == '"' ) {
        static CopyCString unquoted_code;
        in.ignore();
        unquoted_code.getline(in, '"');
        return unquoted_code.string();
    }

    in.width(sizeof(code));
    in >> code;
    if( Str::equal(code, GOCAD_ASCII_SEPARATOR) ) {
         return nullptr;
    }
    return code;
}

/*!
 * Puts back the given code into the input stream so that it can be
 * retrieved with the next call to code().
 */
bool AsciiCatalog::putback(const char* code) {
   if (!pb_code_) {
       if (code) pb_code_ = Str::duplicate(code) ;
       return true ;
   } else {
       return false ;
   }
}

/*!
 * Peeks at the next code on the stream, but does not advance the
 * stream.
 */
const char* AsciiCatalog::peek(istreamb& in) {
   const char* c = code(in) ;
   if (c) putback(c);
   return c;
}

/*!
 * NOT DOCUMENTED
 */
PtrList<GObj>* AsciiCatalog::load(const CString& path) {
    PtrList<GObj>* gobjs = new PtrList<GObj>();
    if( load(*gobjs, path) ) {
        return gobjs;
    } else {
        delete gobjs;
        return nullptr;
    }
}

/*!
 * Load a list of GObjs from a file referenced by its path name or
 * its file descriptor or its input stream. The gobjs read are
 * appended into the given gobjs list.
 */
bool AsciiCatalog::load(PtrList<GObj> &gobjs, const CString& path) {
    GOCAD_ASSERT( path.defined(), return false; );

    CopyCString old_filename = filename_ ;
    filename_ = pfilebuf::input_filename(path) ;

    CopyCString filter_error_file;
    CopyCString npath = path.trim(" ");
#if 0
    // The code below create problems with filters, the
    // redirection of the standard error appears in the import
    // command as a second file (ie xyzproperty filter crash): LD
    if( npath[npath.length()-1] == '|') {
       // Reconstruct a shell command which includes an error file.
       npath.set_to_left(npath.length()-1) ;
       filter_error_file = System::tempnam_("ferr");
       npath = npath + " 2>" + filter_error_file + " |" ;
    }
#endif
    {
       // Suppress the first and last backslashes, just before the ".
       const char* p = path.string();
       const char* end = p + path.length();

       char* str = new char[path.length()+1];
       char* s = str;
       for( ; p < end; p++ ) {
          if( *p == '\\' && *(p+1) == '"' ) {
              // skip backslash
          } else {
              *s++ = *p ;
          }
       }
       *s = '\0' ;
       npath = CopyCString(str);
       delete [] str ;
    }

    lineno_ = 1;
    nerrors_ = 0;
    nwarnings_ = 0;

    if( System::isdir_(filename_) ) {
        Logger::fatal( Format::tr("Input file is a directory: %1\n") % path );

        filename_ = CString::null();
        return false;
    }

    if( !System::exists_(filename_) ) {
       Logger::fatal( Format::tr("Input file does not exist: %1\n") % path );

       filename_ = CString::null();
       return false;
    }

    pfilebuf pbuf;
    bool ok = pbuf.open(npath,ios::in) != 0;
    if( ok ) {
       istreamb in(pbuf.fbuf());
       ok = load(gobjs,in);
       pbuf.close() ;

       if( gobjs.count() == 0 && filter_error_file.defined() ) {
           ifstream filter_err(filter_error_file.string()) ;
           Format message = Format::tr("No objects loaded: \n");
           char line[1024] ;
           while( !filter_err.eof() ) {
              filter_err.getline(line,1024);
              message % line % "\n";
           }
           error(message);
       }
    } else {
        Logger::fatal( Format::tr("Unable to open ascii file: %1\n") % path );
    }

    if( filter_error_file.defined() ) {
       System::unlink_(filter_error_file);
    }

    filename_ = old_filename;
    return ok ;
}

/*!
 * Load a list of GObjs from a file referenced by its path name
 * or its file descriptor or its input stream.
 * The returned pointer list has to be destoyed by the application.
 */
PtrList<GObj> * AsciiCatalog::load(istreamb &in) {
    PtrList<GObj> * gobjs = new PtrList<GObj>();
    if( load(*gobjs, in) ) {
        return gobjs;
    } else {
       delete gobjs;
       return nullptr;
    }
}

/*!
 * Load a list of GObjs from a file referenced by its path name or
 * its file descriptor or its input stream. The gobjs read are
 * appended into the given gobjs list.
 */
bool AsciiCatalog::load(PtrList<GObj> &gobjs, istreamb &in) {
    lineno_ = 1;
    nerrors_ = nwarnings_ = 0 ; // we have to do it here (see comments below).

    bool notify = GeobaseLib::instance()->set_notify(false);
    ScopedBulkEventSender bulk_sender;
    // nerrors_==-1 means that we have a bad_header file
    // (probably old file)
    // we should stop immediately in this case.
    // we will not try to load other gobj in the file.
    gobjs_.remove_all();

    bool failure=false;
    while( next(in) && nerrors_!=-1 ) {
        GObj* gobj = load_gobj(in) ;
        if(nerrors_!=0){
            failure = true;
        }
        if( gobj ) {
           gobjs.append(gobj);

           // If one gobj was loaded successfully, then I reset
           // the number of errors to 0.
           nerrors_=nwarnings_=0;
        }
    }
    if(failure){
        Logger::fatal( Format::tr("Errors occur while loading. See log for details\n") );

    }
    GeobaseLib::instance()->set_notify(notify);
    return true;
}

/*!
 * Load one GObj of any type from the input stream. Stops after
 * the one GObj is read.  Returns nullptr if no GObjs left.
 */
GObj* AsciiCatalog::load_gobj(istreamb &in) {
    char class_name[80];
    float versionId;
    int code;

    // nerrors_ = nwarnings_ = 0;
    // We cannot do that here: if it is a bad file we will be coming
    // back to that routine over and over again, as we cannot read
    // correctly the header.

    do {
        code = read_header(in, class_name, versionId) ;
        if( code == bad_header ) {
            return nullptr ;
        }
    } while (code == other_header) ;

    // We should reset the nerrors here.
    nerrors_=nwarnings_=0;

    TypeId type(class_name);
    GObj* gobj = load(in,type);

    if( in.fail() || nerrors_ > 0 || ( gobj != nullptr && gobj->style() == nullptr) ) {
          if( gobj != nullptr && gobj->name() ) {
              error(Format::tr("failed while reading %1\n") % gobj->name());
          } else {
              error(Format::tr("failed while reading\n"));
          }
          GeobaseLib::instance()->delete_gobj(gobj);
          return nullptr;
    }
    if( gobj != nullptr ) {
       gobjs_.append(gobj);

       // Call this here to solve issue revealed by Probes: a "dummy" GObj is created
       // during load time that makes post processing of style fails its conversion
       // because geometry is not good.
       // Copied comments from its previous location (end of the read_style function)
       // not sure this should be done here still... remains of the past I think.
       // forces gobj-by-gobj events, rather than all-of-them at once. DD
       // Tried to remove this call in frame of SKUAGOCAD-51461 but had to keep it:
       // Needed for proper registration of group and group elements. JC
       GeobaseLib::instance()->add_gobj(gobj);
    }
    return gobj ;
}

/*!
 * Load the first gobj on the input stream. The class id is given
 * and is not read from the file, i.e. the Ascii header is assumed to
 * be read at that point.
 */
GObj * AsciiCatalog::load(istreamb &in,const TypeId& id) {
    // create the Ascii object to write the gobj
    AsciiConverter* ai = static_cast<AsciiConverter*>( create_converter(id));
    if( ai == nullptr ) {
        error(
            Format::tr("unable to find a ascii creator for %1\n")
            % id.name()
        );
        return nullptr;
    }

    ObjectMap * oldMap = curMap_;
    curMap_ = new AsciiConverterMap();

    GObj * gobj = (GObj*) ai->load(in,this);

    // This cannot go here; it must come earlier (during the reading
    // of the style).  Some routines (for instance, as_voxet.cc)
    // assume that the name is set before ai->load is called.
    // StyleCatalog* scatalog = StyleCatalog::instance();
    // scatalog->create(new GObjStyleInfo(gobj));

    delete ai;
    delete curMap_;
    curMap_ = oldMap;

    return gobj;
}

/*!
 * NOT DOCUMENTED
 */
int AsciiCatalog::read_header(
    istreamb &in, char *class_name, float &version
) {
    const char* gocad_marker = code(in);
    if (gocad_marker == nullptr) {
        // wrong error message when reading a file with only a coordinate system
        // for example. Not enough context is kept.
        // warning() << "ignoring redundant " << GOCAD_ASCII_SEPARATOR << endl;
        return bad_header;
    }
    CString marker(gocad_marker);
    if( marker == "GOCAD_COORD_SYS" ||
        marker == "GOCAD_COORDINATE_SYSTEM"
    ) {
        AsciiGObj::read_coord_sys(in,*this);
        return other_header ;
    } else if( marker == "VRTX" ) {
        error(
            Format::tr("unable to read directly old gocad version: use old2new option\n")
        );
        nerrors_=-1;
        return bad_header ;
    } else if( marker == "GOCAD" ) {
        in.width(80) ;
        in >> class_name;
        if (!in.good()) {
            error(Format::tr("unable to read object format\n"));
            return bad_header;
        }
    } else {
        // check if the gocad_marker == a class_name
        CString class_names ;
        AsciiLib::instance()-> style()-> find_attribute(
            "ascii_catalog_class_names", class_names
        ) ;

        bool valid = false ;
        for( CStringTokenItr i(class_names," "); i.more(); i.next() ) {
             CString cur = i.cur() ;
             if( cur.case_insensitive_equal(marker) ) {
                 valid = true ; break ;
             }
        }

        if( valid ) {
           // It is simply a warning. Not an error.
           // (see why in ::read(istreamb))
           warning(
               Format::tr("unknown file format: trying temporary format using %1 as a class name\n")
               % gocad_marker
           );
           Str::copy(class_name,gocad_marker);
        } else {
           eat_line(in);
           error(
               Format::tr("unknown file format. Ignoring line: %1 %2 ...\n")
               % lineno_ % gocad_marker
           );
           return bad_header;
        }
    }
    float versionId;
    in >> versionId;
    if( !in.good() ) {
        warning(Format::tr("unable to read version number, assuming 1.0\n"));
        in.clear();
        const float default_version = 1.;
        version = default_version;
    } else {
        version = versionId;
    }
    return gobj_header ;
}

/*!
 * Reads the style header information and associates it with the
 * given gobj. Code is the last code read from the stream
 * (e.g. HEADER);
 */
bool AsciiCatalog::read_style(istreamb& in, const CString& code, GObj* gobj) {
    GOCAD_DEBUG_ASSERT(gobj);

    StyleVar style = gobj->style();
    if (!style) {
        style = new Style;
    }
    if( code == "HEADER" ) {
        read_style(in, style.get());
    } else if( code == "HEADER{" ) {
        in.putback('{');
        read_style(in, style.get());
    } else {
        ostringstream buf;
        char c;

        while( get(in, c) && c != '\n' ) { buf.put(c); }
        CopyCString key_value_pair( buf );
        style->load_property(key_value_pair);
    }

    // Calling GObj::set_name would assign a new uid to the GObj
    // so deal with uid's before, but read the name nonetheless
    // for more informative messages.
    CopyCString name;
    const bool has_name = style->find_attribute("name", name);

    if (has_name) {
        // PR #5138 and PR #3186: check name validity in the style
        name = GeobaseLib::instance()->names().get_valid_name(
            name, GeobaseLib::Names::Object, false
        );
        // PR #23162: if the name is already taken, and we are importing
        // an homogeneous group, then the reparenting of the style
        // is done before the gobj is finally registered into the geobase.
        // Reparenting is then done on the wrong style, and crash occurs
        // when closing the project.
        // So we have to make sure the gobj is correctly registered with
        // its unique name from the start.

        style->set_name(name);
        gobj->set_name(name);
    } else {
        // no gobj name in ascii file
        GOCAD_BREAKPOINT();
    }

    if (gobj->style() != style.get()) {
        // GObj::set_style will properly parent the new style.
        // It also re-writes the new uid into the style.
        gobj->set_style(style.get());
    }

    return true;
}

/*!
 * Reads style information from the given stream and stores
 * in the given style.
 */
bool AsciiCatalog::read_style(istreamb& in, Style* style) {
    char fixed_buf[1024];
    int len = sizeof(fixed_buf)/sizeof(char);
    char* buf = fixed_buf;
    char* end = buf+len;
    char* p = buf;
    char c;

    while( get(in, c) && c != '{' );
    while( get(in, c) && c != '}' ) {
        if( p == end ) {
            // Reallocate the buffer
            char* nbuf = new char[len*2];
            Memory::copy(buf, nbuf, len*sizeof(char));
            p = nbuf+len;
            len *= 2;
            end = nbuf+len;
            if( buf != fixed_buf ) {
                delete [] buf;
            }
            buf = nbuf;
        }
        *p = c;
        if (*p == '\n') {
            if( p > buf && *(p-1) != '\\') {
                style->load_property(CString(buf, int(p-buf)));
                p = buf;
            } else {
                p++;
            }
        } else {
            p++;
        }
    }
    if( buf != fixed_buf ) {
        delete [] buf;
    }
    return true;
}

/*!
 * Writes the header line for the GObj to the given stream.
 */
void AsciiCatalog::write_header(GObj *gobj, ostreamb &out) const {
    write_header(gobj->type_id(), out);
}

/*!
 * Writes the header line for the TypeId to the given stream.
 */
void AsciiCatalog::write_header(const TypeId& type_id, ostreamb &out) const {
    out << "GOCAD " << type_id.name() << " " << version_ << "\n";
}

/*!
 * Writes the trailer line for the GObj to the given stream.
 */
void AsciiCatalog::write_trailer(ostreamb &out) const {
    out << GOCAD_ASCII_SEPARATOR << "\n";
}

namespace {

/*! \internal
 * Transformer that changes UID and EntityKeyset strings into the corresponding entity name.
 */
class EntityToNameTransformer : public StyleAttributeIterator::Transformer {
public:
    EntityToNameTransformer() {}
    virtual ~EntityToNameTransformer() {}

    virtual bool transform(
        const StyleAttributeIterator::value_type& original,
        StyleAttributeIterator::tr_value_type& transformed
    ) {
        const CString& attrib = original.first;
        if( attrib.empty() ) {
            return false;
        }

        { // the attribute value
            ostringstream new_value_str;
            if( !add_transformed_token(original.second, new_value_str) ) {
                // found some obsolete uid: skip this attribute
                return false;
            }
            transformed.second = CopyCString(new_value_str);
        }

        { // the attribute name
            ostringstream new_name_str;
            const char* last = attrib.string();
            for( CStringTokenItr it(attrib, ".*"); it.more(); it.next() ) {
                const CString& cur_token = it.cur();
                if( last != cur_token.string() ) {
                    new_name_str.write(last, cur_token.string() - last);
                }

                if( !add_transformed_token(cur_token, new_name_str) ) {
                    // found some obsolete uid: skip this attribute
                    return false;
                }

                last = cur_token.string() + cur_token.length();
            }

            if( last < attrib.string() + attrib.length() ) {
                new_name_str.write(last, attrib.string() + attrib.length() - last);
            } else {
                GOCAD_DEBUG_ASSERT( last == attrib.string() + attrib.length() );
            }

            transformed.first = CopyCString(new_name_str);
        }
        return true;
    }

    /*! \internal
     * Adds to \p str the suited transformed string for \p token.
     * \retval false if token could be properly converted and added to \p str
     * \retval false otherwise.
     */
    bool add_transformed_token(const CString& token, ostringstream& str) const {
        if( UniqueId::can_convert(token) ) {
            const EntityIdPtr ident = EntityLib::instance()->find_ident( UniqueId(token) );
            if( !ident ) {
                // the uid can not be converted: skip this attribute
                return false;
            }

            str << ident->name();
            return true;
        }

        if( EntityKeySet::can_convert(token) ) {
            CString name_value;
            EntityKeySet keyset(token);
            if( keyset.value(EntityId::NAME_KEY, name_value) == EntityKeySet::Defined ) {
                str << name_value;
                return true;
            } // else keep as-is (maybe not an EntityKeySet after all)
        } // else keep as-is

        str << token;
        return true;
    }
};

} // anonymous namespace

/*!
 * Writes out the style attributes to the given stream.
 */
void AsciiCatalog::write_style_attributes(Style* style, ostreamb& out, Set<CopyCString>& written) const {
    GOCAD_ASSERT_POINTER(style, return;);

    EntityToNameTransformer style_name_transformer;
    StyleAttributeIterator_var attributes_itr = style->attributes_iterator();
    attributes_itr->set_transformer(&style_name_transformer);
    StyleAttributeIterator::tr_value_type transformed;
    for( ; attributes_itr->more(); attributes_itr->next() ) {
        const StyleAttributeIterator::value_type& cur = attributes_itr->cur();
        CString attrib = cur.first;
        if(
            attrib.empty() ||
            !cur.second.defined() // exclude undefined values, but keep the empty ones
        ) {
            continue;
        }

        CString value;
        if( !attributes_itr->cur_transformed(transformed) ) {
            continue;
        }

        attrib = transformed.first;

        // As we write two styles, remove duplicates
        if( written.find(attrib) ) {
            continue;
        }
        written.add(attrib);

        value = transformed.second;

        out << attrib << ':' << value << '\n';
    }
}

/*!
 * Writes out the style header and attributes to the given stream.
 */
void AsciiCatalog::write_style(GObj* gobj,ostreamb &out) const {
    if( gobj != nullptr ) {
        // As we write two styles, remove duplicates
        Set<CopyCString> written;

        out << "HEADER {\n";
        write_style_attributes(gobj->style(),out, written);

        // this should be the default viewer of the active page
        // when saving:
        ViewerEntityPtr cur_viewer = ViewerEntityMgr::instance()->default_viewer();

        StyleInfoProxy sinfo = StyleCatalog::style_info(gobj, cur_viewer);
        if( sinfo.style() != nullptr && gobj->style() != sinfo.style() ) {
            write_style_attributes(sinfo.style() ,out, written);
        }

        out << "}\n";
    }
}

/*!
 * Returns the GObj with the given name if one has been read.
 * If no GObj with that name has been read, return nullptr.
 */
GObj* AsciiCatalog::find(const CString& name) const {
    UniqueCString nameId(name);
    for( PtrList<GObj>::iterator i(gobjs_); i.more(); i.next() ) {
         GObj* gobj = i.cur();
         if( gobj->name() != nullptr ) {
             if( *gobj->name() == nameId ) {
                 return gobj;
             }
         }
    }
    return nullptr;
}

/*!
 * Saves either a list og GObjs or a unique GObj on a file
 * or on a already open file which file descriptor or outout stream
 * is given
 */
bool AsciiCatalog::save(const PtrList<GObj> &gobjs,const CString& path) {
#if 0
    // make sure that if we do not have a license at least we do not
    // globber the old files. (PR 7232)
    // JCD.
    //
    // This protection has been moved to xmasciilib to avoid denying
    // save operations to developers writing batch programs
    // (see incident 020220-000)
    // TV.
    //
    GLM_BEGIN_REQUEST();
    if( GLM_CHECKOUT_FEATURE_FAILED(glm_gocad_base) ) return false;
    GLM_END_REQUEST();

    if( GLM_REQUEST_FAILED() ) return false ;
#endif

    CopyCString old_filename = filename_ ;
    filename_ = pfilebuf::output_filename(path);

    pfilebuf pbuf;
    int prot = System::fprot_(0666);
    bool ok = pbuf.open(path, ios::out|ios::trunc, prot) != 0;

    if (ok) {
        ostreamb out(pbuf.fbuf());
        ok = save(gobjs,out);
        pbuf.close() ;
    } else {
        // PR #4731: thanks to the job done by Thierry,
        // when saving 200 wells, only one error message will popup instead of 200.
        Logger::fatal(
            Format::tr("Unable to open given output ascii file %1\n")
            % path
        );

    }

    filename_ = old_filename ;
    return ok;
}

/*!
 * Saves either a list og GObjs or a unique GObj on a file
 * or on a already open file which file descriptor or outout stream
 * is given.
 * \retval false if it was not possible to create a converter for one GObj
 * \retval true otherwise
 */
bool AsciiCatalog::save(const PtrList<GObj> &gobjs,ostreamb &out) {
#if 0
    //
    // This protection has been moved to xmasciilib to avoid denying
    // save operations to developers writing batch programs
    // (see incident 020220-000)
    // TV.
    //
    GLM_BEGIN_REQUEST();
    if( GLM_CHECKOUT_FEATURE_FAILED(glm_gocad_base) ) return false;
    GLM_END_REQUEST();

    if( GLM_REQUEST_FAILED() ) return false ;
#endif

    bool ok = true ;
    for( PtrList<GObj>::iterator ig(gobjs); ig.more(); ig.next() ) {
        GObj * gobj = ig.cur();

        AsciiConverter* ai = static_cast<AsciiConverter*>(
             create_converter(gobj->type_id())
        );
        if( ai == nullptr ) {
           // If no converters, skip this object.
           ok = false ;
           Logger::warning(Format::tr("unable to find a ascii creator for %1\n") %*gobj);
           continue ;
        }

        ObjectMap * oldMap = curMap_;
        curMap_ = new AsciiConverterMap();

        ai->save(out,gobj,this);

        delete ai;
        delete curMap_;
        curMap_ = oldMap;
    }
    return ok;
}

/*!
 * Saves either a list og GObjs or a unique GObj on a file
 * or on a already open file which file descriptor or outout stream
 * is given
 */
bool AsciiCatalog::save(GObj *gobj,ostreamb &out) {
    PtrList<GObj> gobjs;
    gobjs.append(gobj);
    return save(gobjs,out);
}

/*!
 * Saves either a list og GObjs or a unique GObj on a file
 * or on a already open file which file descriptor or outout stream
 * is given
 */
bool AsciiCatalog::save(GObj *gobj,const CString& path) {
    PtrList<GObj> gobjs;
    gobjs.append(gobj);
    return save(gobjs,path);
}

/*!
 * Read an optionally quoted character string from the stream and
 * stores it in the character array. The character array is assumed to
 * be big enough, hence this is a dangerous routine. It is strongly
 * recommended that you use the CopyCString version instead.
 *
 * \note This method does not handle escaped double quotes within strings.
 */
void AsciiCatalog::read_quoted_string(istreamb &in,char *name) const {
     eat_whitespace(in);
     if( in.peek() == '"' ) {
         char c;
         char * n = name;
         get(in,c);
         while( get(in,c) && c != '"' ) *n++ = c;
         *n = '\0';
     } else {
         in >> name;
     }
}

/*!
 * Read an optionally quoted character string from the stream and
 * stores it in the CopyCString. The CopyCString will automatically
 * be allocated large enough to hold the string.
 *
 * \note This method does not handle escaped double quotes within strings.
 */
void AsciiCatalog::read_quoted_string(istreamb &in,CopyCString& name) const {
     eat_whitespace(in);
     if( in.peek() == '"' ) {
         in.ignore();
         name.getline(in,'"');
     } else {
         in >> name;
     }
}


/*!
 * Writes a string, surrounding it with double quotes if it does not begin with one.
 *
 * \note This method will not escape double quotes within the string.
 */
void AsciiCatalog::write_quoted_string(ostreamb& out, const CString& string, bool enforce_backward_compatibility) const {
    if (string.length() > 0 && (string[0] == '"' || string.index(' ') == -1) ) {
        out << string;
    } else if( enforce_backward_compatibility ) {
        if( AsciiLib::instance()->header_version() >= VersionNumber("2.7.0a1") ) {
            out << CopyCString("\"", string, "\"");
        } else {
            CopyCString copy(string);
            copy.replace(' ','_');
            out << copy;
        }
    } else {
        out << CopyCString("\"", string, "\"");
    }
}


/*!
 * Output stream for writing catalog errors.
 */
void AsciiCatalog::error(const Format& message) {
    if( nerrors_++ < 5 ) {
       Format full_message = Format::tr("%1 near line %2: %3\n");
       if (filename_.defined()) {
           full_message % filename_;
       } else {
           full_message % "ascii catalog";
       }
       full_message % lineno_ % message;
       Logger::error(full_message);
    } else {
       if (nerrors_ == 6) {
           Logger::error(Format::tr("Maximum error limit reached, no more will be displayed\n"));
       }
    }
}

/*!
 * Output stream for writing catalog warnings.
 */
void AsciiCatalog::warning(const Format& message) {
    if( nwarnings_++ < 50 ) {
       Format full_message = Format::tr("%1 near line %2: %3\n");
       if (filename_.defined()) {
           full_message % filename_;
       } else {
           full_message % "ascii catalog";
       }
       full_message % lineno_ % message;
       Logger::warning(full_message);
    } else {
       if (nwarnings_ == 51) {
           Logger::warning(Format::tr("Maximum warning limit reached, no more will be displayed\n"));
       }
    }
}

/*!
 * \copydoc AsciiCatalog::get_path(CopyCString& filename) const
 */
bool AsciiCatalog::get_path(istreamb &in, CopyCString& filename) const {
    CopyCString copy(filename);
    read_quoted_string(in, copy);
    if (get_path(copy)) {
        filename = copy;
        return true;
    } else {
        return false;
    }
}

/*!
 * Searches in various places looking for a file with the name path.
 * If the file is found, path is set to the correct path, and true
 * is returned. Otherwise, false is returned, and \p filename is left unchanged.
 * The current search path is the home directory followed by the directory in
 * which the catalog is currently loading objects (see filename()).
 */
bool AsciiCatalog::get_path(CopyCString& filename) const {
    // look in the directory specified by the path of the filename
    // given to load.
    // We need to do that first, otherwise we are not getting what the
    // user has asked for.

    const CString& path = this->filename() ;
    if (path.defined()) {
       CopyCString longname(PathName::dirname(path), PathName::slash(), filename);
       if ( System::exists_(longname) ) {
           filename = longname;
           return true;
       }
    }

    // if the file does not exist in the remote directory
    // look locally.
    if( System::exists_(filename) ) return true;

    return false;
}

/*!
 * Create a pathname to use for a data file associated with the given
 * GObj. Uses the current filename if one exists or the gobj name
 * otherwise. If the object is a member of a group, the group name
 * will also be included. The suffix is then appended to the name and
 * returned. If set_append_mode() is activated, the created path name
 * is also saved in case the same data file is requested again.
 * If share is false, the gobj name will be added to the current
 * filename if it does not already include it.
 * The path returned is a full path name that should be written
 * using the function PathName::local_path.
 */
CopyCString AsciiCatalog::create_path(
    const GObj* gobj, const CString& suffix, bool share
) const {
    CString save_name = gobj->short_name();
    // filename() will return a valid filename when we are saving a group
    const CString& path = filename();
    if (!path.defined()) {
        return concat(save_name, suffix);  
    }
    CString prefix =
       PathName::prefix(path).trim_right(PathName::slash().string());
    if (share || PathName::basename(prefix) == save_name ) {
        return concat(prefix, suffix);
    }
    return concat(prefix, save_name, suffix);
}

/*!
 * Returns either "w" or "a" (stdio_open_mode) / ios::out or ios::app
 * (stream_open_mode) depending on whether the given path
 * should be appended to or written over. Append is used when
 * set_append_mode is on and the file has been written to already.
 * Obsolete functions. Should be replaced by the mode-specific
 * equivalent function.
 */
const char* AsciiCatalog::stdio_open_mode(const UniqueCString& path) {
    GOCAD_OBSOLETE_CALL(
        "AsciiCatalog::stdio_open_mode",
        "AsciiCatalog::binary_stdio_open_mode or AsciiCatalog::ascii_stdio_open_mode"
    );
    if( find_data_path(path) ) {
        return "ab" ;
    }
    if( System::exists_(path) ) {
       return "rb+" ;
    }
    return "wb" ;
}

/*!
 * Returns either "w" or "a" (stdio_open_mode) / ios::out or ios::app
 * (stream_open_mode) depending on whether the given path
 * should be appended to or written over. Append is used when
 * set_append_mode is on and the file has been written to already.
 * Obsolete functions. Should be replaced by the mode-specific
 * equivalent function.
 */
GOCAD_IOS_OPENMODE AsciiCatalog::stream_open_mode(const UniqueCString& path) {
    GOCAD_OBSOLETE_CALL(
        "AsciiCatalog::stream_open_mode",
        "AsciiCatalog::binary_stream_open_mode or AsciiCatalog::ascii_stream_open_mode"
    );
    return find_data_path(path) ? ios::app : ios::out ;
}

/*!
 * Returns either "wb" or "ab" (stdio_open_mode) /
 * ios::out or ios::app | GOCAD_IOS_BINARY_OPENMODE ( for platforms supporting
 * GOCAD_IOS_BINARY_OPENMODE ) depending on whether the given path
 * should be appended to or written over. Append is used when
 * set_append_mode is on and the file has been written to already.
 */
const char* AsciiCatalog::binary_stdio_open_mode(const UniqueCString& path) {
    if( find_data_path(path) ) {
        return "ab" ;
    }
    if( System::exists_(path) ) {
       return "rb+" ;
    }
    return "wb" ;
}

/*!
 * Returns either "wb" or "ab" (stdio_open_mode) /
 * ios::out or ios::app | GOCAD_IOS_BINARY_OPENMODE ( for platforms supporting
 * GOCAD_IOS_BINARY_OPENMODE ) depending on whether the given path
 * should be appended to or written over. Append is used when
 * set_append_mode is on and the file has been written to already.
 */
GOCAD_IOS_OPENMODE AsciiCatalog::binary_stream_open_mode(
    const UniqueCString& path
) {
#if defined(_WIN32)
    return find_data_path(path) ?
        GOCAD_IOS_OPENMODE( ios::app | GOCAD_IOS_BINARY_OPENMODE ) : GOCAD_IOS_OPENMODE( ios::out | GOCAD_IOS_BINARY_OPENMODE );
#else
    return find_data_path(path) ? ios::app : ios::out ;
#endif
}

/*!
 * Returns either "w" or "a" (stdio_open_mode) /
 * ios::out or ios::app depending on whether the given path
 * should be appended to or written over. Append is used when
 * set_append_mode is on and the file has been written to already.
 */
const char* AsciiCatalog::ascii_stdio_open_mode(const UniqueCString& path) {
    if( find_data_path(path) ) {
        return "at" ;
    }
    if( System::exists_(path) ) {
       return "rt+" ;
    }
    return "wt" ;
}

/*!
 * Returns either "w" or "a" (stdio_open_mode) /
 * ios::out or ios::app depending on whether the given path
 * should be appended to or written over. Append is used when
 * set_append_mode is on and the file has been written to already.
 */
GOCAD_IOS_OPENMODE AsciiCatalog::ascii_stream_open_mode(const UniqueCString& path) {
    return find_data_path(path) ? ios::app : ios::out ;
}

/*!
 * NOT DOCUMENTED
 */
bool AsciiCatalog::find_data_path(const UniqueCString& path) {
     if (data_paths_ == nullptr) return false;
     if ( data_paths_->find(path) ) {
        return true ;
     } else {
        add_data_path(path);
        return false;
     }
}

/*!
 * NOT DOCUMENTED
 */
void AsciiCatalog::add_data_path(const UniqueCString& path) {
     if (data_paths_ ) {
	 data_paths_->add(path);
     }
}

/*!
 * Put the catalog into 'append mode' for data files. If the catalog
 * is already in 'append mode' no action is taken. Second and
 * subsequent requests for the same data file will cause the open_mode
 * routines to return "a" or ios::app, so that data files will be
 * appended to.
 */
void AsciiCatalog::set_append_mode() {
     if (data_paths_ == nullptr) {
	 data_paths_ = new Set<UniqueCString>(16) ;
     }
}

/*!
 * Take the catalog out of 'append mode' for data files and destroy
 * the list of data file paths.
 */
void AsciiCatalog::unset_append_mode() {
     if (data_paths_ != nullptr) {
        // rename any ".TMP" files
        for (Set<UniqueCString>::iterator i(*data_paths_) ; i.more() ; i.next()) {
           CopyCString s = i.cur();
           if (s.right(-4) == ".TMP") {
              CString t1 = s.left(s.length()-4) ;
              CString t = t1.left(t1.rindex('.')) ;
              // PR #10426 : use our System::rename_ here, not default
              // rename, which has different semantics on some platforms
              System::rename_(s, t) ;
           }
        }
     }
     delete data_paths_;
     data_paths_ = nullptr;
}

/*!
 * Returns the path of the temporary file to use in place of
 * of the given path, when you cannot write on the actual path
 * yet. When rename_temp_path or unset_append_mode is called,
 * these files will be renamed to their permanent counterparts.
 */
CopyCString AsciiCatalog::temp_path(const CString& path) const {
   // Put pid in tmp file names to reduce the chance of conflict when
   // users share directories and files.

   System::pid_t pid = System::getpid_() ;
   for (int i = 0 ; i < 100 ; i++) {
      ostringstream oss ;
      oss << path << "." << (pid+10000*i) << ".TMP" ;
      CopyCString tpath( oss );

      // Check to make sure that there is not already a file w/o write
      // permission in the way. If so, find a new name.
      if (
          System::access_(tpath,F_OK) != 0 ||
          System::access_(tpath,W_OK) == 0
      ) {
          return tpath ;
      }
   }
   return path + ".0.TMP" ;
}

/*!
 * Rename the file to its permanent counterpart unless the file
 * is being reserved via set_append_mode
 */
void AsciiCatalog::rename_temp_path(const CString& path) const {
    // Only do the rename if the temp path is not in the data_paths_ list.
    // If it is, we do not want to do it until unset_append_mode is called

    CopyCString tmppath = temp_path(path) ;
    if (data_paths_ == nullptr || !data_paths_-> find(tmppath)) {
        int status = System::rename_(tmppath, path);
        if( status != 0 ) {
            GOCAD_ERROR_MESSAGE( Format("rename failed"), ; );
        }
    }
}

/*!
 * Build a workflow-neutral URI from a property URI.
 */
COSUri AsciiCatalog::neutral_uri_from_uri( const COSUri& uri ) {
    const COSUriQuery& query = uri.query() ;

    COSNamingLib::Name new_path ;
    const COSNaming::Name& old_path = uri.path() ;
    for( int i=0 ; i<old_path.length()-1 ; i++ ) {
        const COSNaming::NameComponent& path_component = old_path[i] ;
        if( path_component.kind.empty() ) {
            new_path.append_component(path_component) ;
        }
    }
    new_path.append_component( old_path[old_path.length()-1] ) ;

    return COSUri(new_path, query) ;
}

} // namespace Gocad

