#!/usr/bin/env python
# -*- coding: utf-8 -*- 
#

# Copyright notice:
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

##
# @brief A Doxygen filter for LPC
# @details Based (loosely) on glslfilter by Sebastian Schäfer
# @copyright GNU Public License.
# @todo Causes many of these errors (which don't appear to affect output...): Original exception was: close failed in file object destructor: Error in sys.excepthook:

import getopt          # get command-line options
import os.path         # getting extension from file
import string          # string manipulation
import sys             # output and stuff
import re              # for regular expressions

# modifiers that could show up in front of an inherit statement...
valid_modifiers = ["private", "nosave", "static", "public", "virtual", "inherit"]

def writeLine(txt):
	sys.stdout.write(txt + "\n")


# strips inherits and converts them to a class statement following the last #include X.h
def parseInherit(filename, txt):
	temp = lambda x: x.find("inherit ") != -1
	inherits = filter(temp, txt)
	temp = lambda x: x.split(" ")[0] in valid_modifiers
	inherits = filter(temp, inherits)
	removed = len(txt)
	[txt.remove(x) for x in inherits]
	removed = removed - len(txt)
	temp = lambda x: x.find("#include ") == 0 and x.rfind(".h") #find the last .h include so we can start our class after
	includes = filter(temp, txt)

	#save the line of our last include
	try:
		first_include = txt.index(includes[0]) + 1
	except IndexError:
		#set this to 1, as we knock a line off on include and that'll send us negative otherwise
		first_include = 1

	inherit_replacements = []

	#strip basename from our inherit statements
	for inherit_str in inherits:
		pair = inherit_str.split("inherit")
		modifiers = pair[0]
		inherit_class = pair[1].split("/")[-1].split(".")[0].rstrip("\"\';\n \r")
		inherit_replacements.append(modifiers + inherit_class)
	
	## construct the class statement
	if len(inherit_replacements):
		replacement_text = " : "+", ".join(inherit_replacements)
	else:
		replacement_text = ""
	
	## re-pad the file to keep our line numbers the same, so we can reliably cross-refer
	while removed > 1:
		removed = removed - 1
		txt.insert(first_include, "\n")

	## insert the class-style inheritance statement doxygen wants after our last header include
	txt.insert(first_include-1, "public class "+os.path.basename(filename)+ replacement_text + " {\n")

	##check how the file ends, end the class statement appropriately
	if txt[-1].startswith("/"):
		txt.insert(len(txt)-1, "};\n")
	else:
		txt.append("\n};")

	showLines(txt)

## @returns the complete file content as an array of lines
def readFile(filename):
	f = open(filename)
	r = f.readlines()
	if f:
		f.close()
	return r
	
## dump all lines to stdout
def showLines(r):
	for s in r:
		sys.stdout.write(s)

## main method - open a file and see what can be done
def inherit_filter(filename):

	try:
		root, ext = os.path.splitext(filename)
		txt = readFile(filename)
		if (ext.lower() == ".c" or ext.lower() == ".c"):
			parseInherit(root, txt)
		else:
			showLines(txt)
	except IOError,e:
		pass
		#sys.stderr.write(e[1]+"\n")

if len(sys.argv) != 2:
	print "usage: ", sys.argv[0], " filename"
	sys.exit(1)

filename = sys.argv[1]
inherit_filter(filename)
sys.exit(0)