--------------------------------------------------------------------------------
-- example_design_rtl.vhdl
--------------------------------------------------------------------------------
--! \file example_design_rtl.vhdl
--! \brief Implementation of the example_design component.
--!
--! \class example_design
--! \ingroup my_project_lib
--! \brief Example design.
--!
--! Detailed description.
--!
--! Additional information.
--!
--------------------------------------------------------------------------------
-- $Revision: 6$
--------------------------------------------------------------------------------

-- hds interface_start
library ieee;
use     ieee.std_logic_1164.all;
use     ieee.numeric_std.all;

entity example_design is

    port
    (

        --! Clock
        CLK                     : in    std_logic;

        --! clock enable
        CE                      : in    std_logic;

        --! Input data
        DATA_IN                 : in    std_logic_vector (15 downto 0);

        --! Output data address
        DATA_OUT_ADDR           : in    std_logic_vector (5 downto 0);

        --! Output data
        DATA_OUT                : out   std_logic

    );

end example_design;
-- hds interface_end

--------------------------------------------------------------------------------
-- Start of architecture rtl for module my_project_lib.example_design
--------------------------------------------------------------------------------

library my_design_lib; --! Library of reusable components and functions for synthesis.
use     my_design_lib.design_components_pkg.all; --! Reusable components for synthesis.
use     my_design_lib.design_utilities_pkg.all; --! Useful functions for synthesis.


--! Implementation of the example_design component.
architecture rtl of example_design is
--! \cond

    signal      SREG_DATA_OUT           : std_logic_vector (DATA_IN'range); -- ' addressable_shift_register output

    signal      DATA_OUT_INT            : std_logic := '0';

    -- synthesis translate_off
    for all : addressable_shift_register use entity my_design_lib.addressable_shift_register;
    -- synthesis translate_on

--! \endcond
begin


    ----------------------------------------------------------------------------
    --! Addressable shift register to delay the input data
    ----------------------------------------------------------------------------
    U_SREG : addressable_shift_register

        generic map
        (

            -- Width of the data ports
            NBITS_DATA              => DATA_IN'length -- ',

            -- Depth of the shift register, 5 -> 32, 6 -> 64
            LOG2_DEPTH              => DATA_OUT_ADDR'length -- '

        )

        port map
        (

            -- Clock
            CLK                     => CLK,

            -- Input data
            DATA_IN                 => DATA_IN,

            -- Input data clock enable
            DATA_IN_CE              => CE,

            -- Output data
            DATA_OUT                => SREG_DATA_OUT,

            -- Output data address
            DATA_OUT_ADDR           => DATA_OUT_ADDR,

            -- Output data clock enable
            DATA_OUT_CE             => CE,

            -- Output data synchronous reset
            DATA_OUT_RST            => '0'

        );


    ----------------------------------------------------------------------------
    --! Process to test the output of the addressable shift register
    ----------------------------------------------------------------------------
    OUTPUT_REGISTER : process (CLK)
    begin

        -- Synchronous operation.
        if rising_edge (CLK) then

            if CE = '1' then

                DATA_OUT_INT <= COMBINE_AND (SREG_DATA_OUT);

            end if;

        end if;

    end process OUTPUT_REGISTER;


    ----------------------------------------------------------------------------
    -- Map internal signals to outputs
    ----------------------------------------------------------------------------
    DATA_OUT <= DATA_OUT_INT;


end architecture rtl;


--------------------------------------------------------------------------------
-- End of file example_design_rtl.vhdl
--------------------------------------------------------------------------------

