/*
 * Copyright (C) 2008 Ole Andr Vadla Ravns <ole.andre.ravnas@tandberg.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "gstpluginstaatest.h"
#include "testutil.h"
#include "gsttestclock.h"

static gpointer test_wait_pending_single_shot_id_worker (gpointer data);
static gpointer test_wait_pending_periodic_id_waiter_thread (gpointer data);
static gboolean test_async_wait_cb (GstClock *clock, GstClockTime time,
    GstClockID id, gpointer user_data);

static void
test_object_flags (void)
{
  GstClock * clock = gst_test_clock_new ();
  g_assert (GST_OBJECT_FLAG_IS_SET (clock, GST_CLOCK_FLAG_CAN_DO_SINGLE_SYNC));
  g_assert (
      GST_OBJECT_FLAG_IS_SET (clock, GST_CLOCK_FLAG_CAN_DO_SINGLE_ASYNC));
  g_assert (
      GST_OBJECT_FLAG_IS_SET (clock, GST_CLOCK_FLAG_CAN_DO_PERIODIC_SYNC));
  g_assert (
      GST_OBJECT_FLAG_IS_SET (clock, GST_CLOCK_FLAG_CAN_DO_PERIODIC_ASYNC));
  gst_object_unref (clock);
}

static void
test_resolution_query (void)
{
  GstClock * clock = gst_test_clock_new ();
  g_assert_cmpuint (gst_clock_get_resolution (clock), ==, 1);
  gst_object_unref (clock);
}

static void
test_start_time (void)
{
  GstClock * clock;

  clock = gst_test_clock_new ();
  g_assert_cmpuint (gst_clock_get_time (clock), ==, 0);
  gst_object_unref (clock);

  clock = gst_test_clock_new_with_start_time (GST_SECOND);
  g_assert_cmpuint (gst_clock_get_time (clock), ==, GST_SECOND);
  gst_object_unref (clock);
}

static void
test_set_time (void)
{
  GstClock * clock = gst_test_clock_new_with_start_time (GST_SECOND);
  gst_test_clock_set_time (GST_TEST_CLOCK (clock), GST_SECOND);
  g_assert_cmpuint (gst_clock_get_time (clock), ==, GST_SECOND);
  gst_test_clock_set_time (GST_TEST_CLOCK (clock), GST_SECOND + 1);
  g_assert_cmpuint (gst_clock_get_time (clock), ==, GST_SECOND + 1);
  gst_object_unref (clock);
}

static void
test_advance_time (void)
{
  GstClock * clock = gst_test_clock_new_with_start_time (GST_SECOND);
  gst_test_clock_advance_time (GST_TEST_CLOCK (clock), 0);
  g_assert_cmpuint (gst_clock_get_time (clock), ==, GST_SECOND);
  gst_test_clock_advance_time (GST_TEST_CLOCK (clock), 42 * GST_MSECOND);
  g_assert_cmpuint (gst_clock_get_time (clock), ==,
      GST_SECOND + (42 * GST_MSECOND));
  gst_object_unref (clock);
}

static void
test_wait_pending_single_shot_id (void)
{
  GstClock * clock;
  GstTestClock * test_clock;
  GstClockID clock_id;
  GThread * worker_thread;
  GstTestClockPendingID pending_id;

  clock = gst_test_clock_new_with_start_time (GST_SECOND);
  test_clock = GST_TEST_CLOCK (clock);

  clock_id = gst_clock_new_single_shot_id (clock, GST_SECOND);
  gst_clock_id_wait_async (clock_id, test_async_wait_cb, NULL);
  g_assert (gst_test_clock_wait_for_next_pending_id (test_clock, &pending_id));
  g_assert (pending_id.clock_id == clock_id);
  g_assert (pending_id.type == GST_CLOCK_ENTRY_SINGLE);
  g_assert_cmpuint (pending_id.time, ==, GST_SECOND);
  gst_test_clock_process_next_clock_id (test_clock);
  gst_clock_id_unref (clock_id);

  clock_id = gst_clock_new_single_shot_id (clock, 2 * GST_SECOND);
  worker_thread = g_thread_create (test_wait_pending_single_shot_id_worker,
      clock_id, TRUE, NULL);
  g_assert (gst_test_clock_wait_for_next_pending_id (test_clock, &pending_id));
  g_thread_join (worker_thread);
  g_assert (pending_id.clock_id == clock_id);
  g_assert (pending_id.type == GST_CLOCK_ENTRY_SINGLE);
  g_assert_cmpuint (pending_id.time, ==, 2 * GST_SECOND);
  gst_clock_id_unref (clock_id);

  clock_id = gst_clock_new_single_shot_id (clock, 3 * GST_SECOND);
  worker_thread = g_thread_create (test_wait_pending_single_shot_id_worker,
      clock_id, TRUE, NULL);
  g_assert (gst_test_clock_wait_for_next_pending_id (test_clock, NULL));
  g_thread_join (worker_thread);
  gst_clock_id_unref (clock_id);

  gst_object_unref (clock);
}

static gpointer
test_wait_pending_single_shot_id_worker (gpointer data)
{
  GstClockID clock_id = data;

  g_usleep (G_USEC_PER_SEC / 10);
  gst_clock_id_wait_async (clock_id, test_async_wait_cb, NULL);

  return NULL;
}

static void
test_wait_pending_periodic_id (void)
{
  GstClock * clock;
  GstTestClock * test_clock;
  GstClockID clock_id;

  clock = gst_test_clock_new_with_start_time (GST_SECOND);
  test_clock = GST_TEST_CLOCK (clock);
  clock_id = gst_clock_new_periodic_id (clock, GST_SECOND, GST_MSECOND);

  {
    GThread * waiter_thread;

    waiter_thread = g_thread_create (
        test_wait_pending_periodic_id_waiter_thread, clock_id, TRUE, NULL);

    g_assert (gst_test_clock_wait_for_next_pending_id (test_clock, NULL));
    gst_test_clock_set_time (test_clock, GST_SECOND);
    g_assert (gst_test_clock_process_next_clock_id (test_clock) == clock_id);

    g_thread_join (waiter_thread);
  }

  {
    guint i;
    GThread * waiter_thread;

    for (i = 0; i < 3; i++)
    {
      g_assert (!gst_test_clock_peek_next_pending_id (test_clock, NULL));
      g_usleep (G_USEC_PER_SEC / 10 / 10);
    }

    waiter_thread = g_thread_create (
        test_wait_pending_periodic_id_waiter_thread, clock_id, TRUE, NULL);

    g_assert (gst_test_clock_wait_for_next_pending_id (test_clock, NULL));
    gst_clock_id_unschedule (clock_id);

    g_thread_join (waiter_thread);
  }

  gst_clock_id_unref (clock_id);
  gst_object_unref (clock);
}

static gpointer
test_wait_pending_periodic_id_waiter_thread (gpointer data)
{
  GstClockID clock_id = data;
  gst_clock_id_wait (clock_id, NULL);
  return NULL;
}

static void
test_single_shot_sync_past (void)
{
  GstClock * clock;
  GstTestClock * test_clock;
  GstClockID clock_id;
  GstClockTimeDiff jitter;
  GtuClockWaitContext * wait_ctx;

  clock = gst_test_clock_new_with_start_time (GST_SECOND);
  test_clock = GST_TEST_CLOCK (clock);

  clock_id = gst_clock_new_single_shot_id (clock, GST_SECOND - 1);
  wait_ctx =
      gst_test_util_wait_for_clock_id_begin (test_clock, clock_id, &jitter);
  g_assert (gst_test_util_wait_for_clock_id_end (wait_ctx) == GST_CLOCK_EARLY);
  g_assert_cmpint (jitter, ==, 1);
  gst_clock_id_unref (clock_id);

  gst_object_unref (clock);
}

static void
test_single_shot_sync_present (void)
{
  GstClock * clock;
  GstTestClock * test_clock;
  GstClockID clock_id;
  GstClockTimeDiff jitter;
  GtuClockWaitContext * wait_ctx;

  clock = gst_test_clock_new_with_start_time (GST_SECOND);
  test_clock = GST_TEST_CLOCK (clock);

  clock_id = gst_clock_new_single_shot_id (clock, GST_SECOND);
  wait_ctx =
      gst_test_util_wait_for_clock_id_begin (test_clock, clock_id, &jitter);
  g_assert (gst_test_util_wait_for_clock_id_end (wait_ctx) == GST_CLOCK_OK);
  g_assert_cmpint (jitter, ==, 0);
  gst_clock_id_unref (clock_id);

  gst_object_unref (clock);
}

static void
test_single_shot_sync_future (void)
{
  GstClock * clock;
  GstTestClock * test_clock;
  GstClockID clock_id;
  GstClockTimeDiff jitter;
  GtuClockWaitContext * wait_ctx;

  clock = gst_test_clock_new_with_start_time (GST_SECOND);
  test_clock = GST_TEST_CLOCK (clock);

  clock_id = gst_clock_new_single_shot_id (clock, 2 * GST_SECOND);
  wait_ctx =
      gst_test_util_wait_for_clock_id_begin (test_clock, clock_id, &jitter);
  gst_test_clock_advance_time (test_clock, GST_SECOND);
  g_assert (gst_test_util_wait_for_clock_id_end (wait_ctx) == GST_CLOCK_OK);
  g_assert_cmpint (jitter, ==, -GST_SECOND);
  gst_clock_id_unref (clock_id);

  gst_object_unref (clock);
}

static void
test_single_shot_sync_unschedule (void)
{
  GstClock * clock;
  GstTestClock * test_clock;
  GstClockID clock_id;
  GtuClockWaitContext * wait_ctx;

  clock = gst_test_clock_new_with_start_time (GST_SECOND);
  test_clock = GST_TEST_CLOCK (clock);

  clock_id = gst_clock_new_single_shot_id (clock, GST_SECOND);
  gst_clock_id_unschedule (clock_id);
  gst_clock_id_unref (clock_id);

  clock_id = gst_clock_new_single_shot_id (clock, 2 * GST_SECOND);
  wait_ctx =
      gst_test_util_wait_for_clock_id_begin (test_clock, clock_id, NULL);
  gst_clock_id_unschedule (clock_id);
  g_assert (gst_test_util_wait_for_clock_id_end (wait_ctx)
      == GST_CLOCK_UNSCHEDULED);
  gst_clock_id_unref (clock_id);

  gst_object_unref (clock);
}

static void
test_single_shot_sync_ordering (void)
{
  GstClock * clock;
  GstTestClock * test_clock;
  GstClockID clock_id_a, clock_id_b;
  GtuClockWaitContext * wait_ctx_a, * wait_ctx_b;

  clock = gst_test_clock_new_with_start_time (GST_SECOND);
  test_clock = GST_TEST_CLOCK (clock);

  clock_id_a = gst_clock_new_single_shot_id (clock, 3 * GST_SECOND);
  wait_ctx_a =
      gst_test_util_wait_for_clock_id_begin (test_clock, clock_id_a, NULL);

  gst_test_clock_advance_time (test_clock, GST_SECOND);

  clock_id_b = gst_clock_new_single_shot_id (clock, 2 * GST_SECOND);
  wait_ctx_b =
      gst_test_util_wait_for_clock_id_begin (test_clock, clock_id_b, NULL);

  gst_test_clock_advance_time (test_clock, GST_SECOND);

  g_assert (gst_test_util_wait_for_clock_id_end (wait_ctx_b) == GST_CLOCK_OK);
  g_assert (gst_test_util_wait_for_clock_id_end (wait_ctx_a) == GST_CLOCK_OK);

  gst_clock_id_unref (clock_id_b);
  gst_clock_id_unref (clock_id_a);

  gst_object_unref (clock);
}

static void
test_single_shot_async_past (void)
{
  GstClock * clock;
  GstClockID clock_id;
  gboolean wait_complete = FALSE;

  clock = gst_test_clock_new_with_start_time (GST_SECOND);
  clock_id = gst_clock_new_single_shot_id (clock, GST_SECOND - 1);
  g_assert (gst_clock_id_wait_async (clock_id, test_async_wait_cb,
      &wait_complete) == GST_CLOCK_OK);
  g_assert (!wait_complete);
  g_assert (gst_test_clock_process_next_clock_id (GST_TEST_CLOCK (clock))
      == clock_id);
  g_assert (wait_complete);
  g_assert (GST_CLOCK_ENTRY_STATUS (GST_CLOCK_ENTRY (clock_id))
      == GST_CLOCK_EARLY);
  gst_clock_id_unref (clock_id);
  gst_object_unref (clock);
}

static void
test_single_shot_async_present (void)
{
  GstClock * clock;
  GstClockID clock_id;
  gboolean wait_complete = FALSE;

  clock = gst_test_clock_new_with_start_time (GST_SECOND);
  clock_id = gst_clock_new_single_shot_id (clock, GST_SECOND);
  g_assert (gst_clock_id_wait_async (clock_id, test_async_wait_cb,
      &wait_complete) == GST_CLOCK_OK);
  g_assert (!wait_complete);
  g_assert (gst_test_clock_process_next_clock_id (GST_TEST_CLOCK (clock))
      == clock_id);
  g_assert (wait_complete);
  g_assert (GST_CLOCK_ENTRY_STATUS (GST_CLOCK_ENTRY (clock_id))
      == GST_CLOCK_OK);
  gst_clock_id_unref (clock_id);
  gst_object_unref (clock);
}

static void
test_single_shot_async_future (void)
{
  GstClock * clock;
  GstClockID clock_id;
  gboolean wait_complete = FALSE;

  clock = gst_test_clock_new_with_start_time (GST_SECOND);
  clock_id = gst_clock_new_single_shot_id (clock, 2 * GST_SECOND);
  g_assert (gst_clock_id_wait_async (clock_id, test_async_wait_cb,
      &wait_complete) == GST_CLOCK_OK);
  g_assert (gst_test_clock_process_next_clock_id (GST_TEST_CLOCK (clock))
      == NULL);
  g_assert (!wait_complete);
  g_assert (GST_CLOCK_ENTRY_STATUS (GST_CLOCK_ENTRY (clock_id))
      == GST_CLOCK_OK);

  gst_test_clock_advance_time (GST_TEST_CLOCK (clock), GST_SECOND - 1);
  g_assert (gst_test_clock_process_next_clock_id (GST_TEST_CLOCK (clock))
      == NULL);
  g_assert (!wait_complete);
  g_assert (GST_CLOCK_ENTRY_STATUS (GST_CLOCK_ENTRY (clock_id))
      == GST_CLOCK_OK);

  gst_test_clock_advance_time (GST_TEST_CLOCK (clock), 1);
  g_assert (gst_test_clock_process_next_clock_id (GST_TEST_CLOCK (clock))
      == clock_id);
  g_assert (wait_complete);
  g_assert (GST_CLOCK_ENTRY_STATUS (GST_CLOCK_ENTRY (clock_id))
      == GST_CLOCK_OK);

  gst_clock_id_unref (clock_id);
  gst_object_unref (clock);
}

static void
test_single_shot_async_unschedule (void)
{
  GstClock * clock;
  GstClockID clock_id;
  gboolean wait_complete = FALSE;

  clock = gst_test_clock_new_with_start_time (GST_SECOND);

  clock_id = gst_clock_new_single_shot_id (clock, 3 * GST_SECOND);
  g_assert (gst_clock_id_wait_async (clock_id, test_async_wait_cb,
      &wait_complete) == GST_CLOCK_OK);

  gst_clock_id_unschedule (clock_id);

  gst_test_clock_advance_time (GST_TEST_CLOCK (clock), 2 * GST_SECOND);
  g_assert (gst_test_clock_process_next_clock_id (GST_TEST_CLOCK (clock))
      == NULL);
  g_assert (!wait_complete);

  gst_clock_id_unref (clock_id);
  gst_object_unref (clock);
}

static void
test_periodic_sync (void)
{
  GstClock * clock;
  GstTestClock * test_clock;
  GstClockID clock_id;
  guint i;
  const GstClockTime interval = 4 * GST_MSECOND;

  clock = gst_test_clock_new ();
  test_clock = GST_TEST_CLOCK (clock);

  clock_id = gst_clock_new_periodic_id (clock, GST_SECOND, interval);

  for (i = 0; i < 3; i++)
  {
    GtuClockWaitContext * wait_ctx;
    GstTestClockPendingID pending_id;
    guint j;

    wait_ctx =
        gst_test_util_wait_for_clock_id_begin (test_clock, clock_id, NULL);

    g_assert (gst_test_clock_wait_for_next_pending_id (test_clock,
        &pending_id));
    g_assert (pending_id.clock_id == clock_id);
    g_assert (pending_id.type == GST_CLOCK_ENTRY_PERIODIC);
    g_assert_cmpuint (pending_id.time, ==, GST_SECOND + (i * interval));

    for (j = 0; j < 10; j++)
    {
      g_usleep (G_USEC_PER_SEC / 10 / 10);
      g_assert (!gst_test_util_clock_wait_context_has_completed (wait_ctx));
    }

    if (i == 0)
      gst_test_clock_advance_time (test_clock, GST_SECOND);
    else
      gst_test_clock_advance_time (test_clock, interval);

    gst_test_util_wait_for_clock_id_end (wait_ctx);
  }

  gst_clock_id_unref (clock_id);
  gst_object_unref (clock);
}

static void
test_periodic_async (void)
{
  GstClock * clock;
  GstClockID clock_id;
  gboolean wait_complete = FALSE;
  const GstClockTime interval = 4 * GST_MSECOND;

  clock = gst_test_clock_new ();
  clock_id = gst_clock_new_periodic_id (clock, gst_clock_get_time (clock),
      interval);
  g_assert (gst_clock_id_wait_async (clock_id, test_async_wait_cb,
      &wait_complete) == GST_CLOCK_OK);

  g_assert (gst_test_clock_process_next_clock_id (GST_TEST_CLOCK (clock))
      == clock_id);
  g_assert (wait_complete);
  wait_complete = FALSE;

  gst_test_clock_advance_time (GST_TEST_CLOCK (clock), interval - 1);
  g_assert (gst_test_clock_process_next_clock_id (GST_TEST_CLOCK (clock))
      == NULL);
  g_assert (!wait_complete);

  gst_test_clock_advance_time (GST_TEST_CLOCK (clock), 1);
  g_assert (gst_test_clock_process_next_clock_id (GST_TEST_CLOCK (clock))
      == clock_id);
  g_assert (wait_complete);
  wait_complete = FALSE;

  gst_test_clock_advance_time (GST_TEST_CLOCK (clock), interval - 1);
  g_assert (gst_test_clock_process_next_clock_id (GST_TEST_CLOCK (clock))
      == NULL);
  g_assert (!wait_complete);

  gst_test_clock_advance_time (GST_TEST_CLOCK (clock), 1);
  g_assert (gst_test_clock_process_next_clock_id (GST_TEST_CLOCK (clock))
      == clock_id);
  g_assert (wait_complete);
  wait_complete = FALSE;

  gst_clock_id_unref (clock_id);
  gst_object_unref (clock);
}

static void
test_periodic_uniqueness (void)
{
  GstClock * clock;
  GstTestClock * test_clock;
  GstClockID clock_id;
  guint i;
  const GstClockTime interval = 4 * GST_MSECOND;

  clock = gst_test_clock_new ();
  test_clock = GST_TEST_CLOCK (clock);

  clock_id = gst_clock_new_periodic_id (clock, 0, interval);

  for (i = 0; i < 3; i++)
  {
    GtuClockWaitContext * wait_ctx;
    guint j;

    wait_ctx =
        gst_test_util_wait_for_clock_id_begin (test_clock, clock_id, NULL);

    for (j = 0; j < 10; j++)
    {
      g_usleep (G_USEC_PER_SEC / 10 / 10);
      g_assert_cmpuint (gst_test_clock_peek_id_count (test_clock), ==, 1);
    }

    gst_test_clock_advance_time (test_clock, interval);
    gst_test_util_wait_for_clock_id_end (wait_ctx);
  }

  gst_clock_id_unref (clock_id);
  gst_object_unref (clock);
}

static gboolean
test_async_wait_cb (GstClock * clock,
                    GstClockTime time,
                    GstClockID id,
                    gpointer user_data)
{

  gboolean * wait_complete = user_data;

  if (wait_complete != NULL)
    *wait_complete = TRUE;

  return TRUE;
}

void
gstpluginstaatest_register_testclock_tests (void)
{
  g_test_add_func ("/GstTestClock/test-object-flags", &test_object_flags);
  g_test_add_func ("/GstTestClock/test-resolution-query",
      &test_resolution_query);
  g_test_add_func ("/GstTestClock/test-start-time", &test_start_time);
  g_test_add_func ("/GstTestClock/test-set-time", &test_set_time);
  g_test_add_func ("/GstTestClock/test-advance-time", &test_advance_time);
  g_test_add_func ("/GstTestClock/test-wait-pending-single-shot-id",
      &test_wait_pending_single_shot_id);
  g_test_add_func ("/GstTestClock/test-wait-pending-periodic-id",
      &test_wait_pending_periodic_id);
  g_test_add_func ("/GstTestClock/test-single-shot-sync-past",
      &test_single_shot_sync_past);
  g_test_add_func ("/GstTestClock/test-single-shot-sync-present",
      &test_single_shot_sync_present);
  g_test_add_func ("/GstTestClock/test-single-shot-sync-future",
      &test_single_shot_sync_future);
  g_test_add_func ("/GstTestClock/test-single-shot-sync-unschedule",
      &test_single_shot_sync_unschedule);
  g_test_add_func ("/GstTestClock/test-single-shot-sync-ordering",
      &test_single_shot_sync_ordering);
  g_test_add_func ("/GstTestClock/test-single-shot-async-past",
      &test_single_shot_async_past);
  g_test_add_func ("/GstTestClock/test-single-shot-async-present",
      &test_single_shot_async_present);
  g_test_add_func ("/GstTestClock/test-single-shot-async-future",
      &test_single_shot_async_future);
  g_test_add_func ("/GstTestClock/test-single-shot-async-unschedule",
      &test_single_shot_async_unschedule);
  g_test_add_func ("/GstTestClock/test-periodic-sync", &test_periodic_sync);
  g_test_add_func ("/GstTestClock/test-periodic-async", &test_periodic_async);
  g_test_add_func ("/GstTestClock/test-periodic-uniqueness",
      &test_periodic_uniqueness);
}

