/*
 * Copyright (C) 2008 Ole Andr Vadla Ravns <ole.andre.ravnas@tandberg.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "gsttestclock.h"

#define TEST_CLOCK_WAIT_TIMEOUT 20

enum
{
  PROP_0,
  PROP_START_TIME,
};

typedef struct _GstClockEntryContext GstClockEntryContext;

static void gst_test_clock_constructed (GObject * object);
static void gst_test_clock_dispose (GObject * object);
static void gst_test_clock_finalize (GObject * object);
static void gst_test_clock_set_property (GObject * object, guint property_id,
    const GValue * value, GParamSpec * pspec);
static void gst_test_clock_get_property (GObject * object, guint property_id,
    GValue * value, GParamSpec * pspec);

static GstClockTime gst_test_clock_get_resolution (GstClock * clock);
static GstClockTime gst_test_clock_get_internal_time (GstClock * clock);
static GstClockReturn gst_test_clock_wait_jitter (GstClock * clock,
    GstClockEntry * entry, GstClockTimeDiff * jitter);
static GstClockReturn gst_test_clock_wait_async (GstClock * clock,
    GstClockEntry * entry);
static void gst_test_clock_unschedule (GstClock * clock,
    GstClockEntry * entry);

static gboolean gst_test_clock_peek_next_pending_id_unlocked (
    GstTestClock * self, GstTestClockPendingID * pending_id);

static void gst_test_clock_add_entry (GstTestClock * self,
    GstClockEntry * entry, GstClockTimeDiff * jitter);
static void gst_test_clock_remove_entry (GstTestClock * self,
    GstClockEntry * entry);
GstClockEntryContext * gst_test_clock_lookup_entry_context (
    GstTestClock * self, GstClockEntry * clock_entry);

static gint gst_clock_entry_context_compare_func (gconstpointer a,
    gconstpointer b);

struct _GstClockEntryContext
{
  GstClockEntry * clock_entry;
  GstClockTimeDiff time_diff;
};

struct _GstTestClockPrivate
{
  GstClockTime start_time;
  GstClockTime internal_time;
  GList * entry_contexts;
  GCond * entry_added_cond;
  GCond * entry_processed_cond;
};

#define GST_TEST_CLOCK_GET_PRIVATE(o) ((o)->priv)

GST_BOILERPLATE (GstTestClock, gst_test_clock, GstClock, GST_TYPE_CLOCK);

static void
gst_test_clock_base_init (gpointer g_class)
{
}

static void
gst_test_clock_class_init (GstTestClockClass * g_class)
{
  GObjectClass * gobject_class = G_OBJECT_CLASS (g_class);
  GstClockClass * gstclock_class = GST_CLOCK_CLASS (g_class);
  GParamSpec * pspec;

  g_type_class_add_private (g_class, sizeof (GstTestClockPrivate));

  gobject_class->constructed = gst_test_clock_constructed;
  gobject_class->dispose = gst_test_clock_dispose;
  gobject_class->finalize = gst_test_clock_finalize;
  gobject_class->set_property = gst_test_clock_set_property;
  gobject_class->get_property = gst_test_clock_get_property;

  gstclock_class->get_resolution = gst_test_clock_get_resolution;
  gstclock_class->get_internal_time = gst_test_clock_get_internal_time;
  gstclock_class->wait_jitter = gst_test_clock_wait_jitter;
  gstclock_class->wait_async = gst_test_clock_wait_async;
  gstclock_class->unschedule = gst_test_clock_unschedule;

  pspec = g_param_spec_uint64 ("start-time", "Start Time",
      "Start Time of the Clock", 0, G_MAXUINT64, 0,
      G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS | G_PARAM_CONSTRUCT_ONLY);
  g_object_class_install_property (gobject_class, PROP_START_TIME, pspec);
}

static void
gst_test_clock_init (GstTestClock * self,
                     GstTestClockClass * g_class)
{
  GstTestClockPrivate * priv;

  self->priv = G_TYPE_INSTANCE_GET_PRIVATE (self, GST_TYPE_TEST_CLOCK,
      GstTestClockPrivate);

  priv = GST_TEST_CLOCK_GET_PRIVATE (self);
  priv->entry_added_cond = g_cond_new ();
  priv->entry_processed_cond = g_cond_new ();

  GST_OBJECT_FLAG_SET (self,
      GST_CLOCK_FLAG_CAN_DO_SINGLE_SYNC |
      GST_CLOCK_FLAG_CAN_DO_SINGLE_ASYNC |
      GST_CLOCK_FLAG_CAN_DO_PERIODIC_SYNC |
      GST_CLOCK_FLAG_CAN_DO_PERIODIC_ASYNC);
}

static void
gst_test_clock_constructed (GObject * object)
{
  GstTestClock * self = GST_TEST_CLOCK (object);
  GstTestClockPrivate * priv = GST_TEST_CLOCK_GET_PRIVATE (self);

  priv->internal_time = priv->start_time;
}

static void
gst_test_clock_dispose (GObject * object)
{
  GstTestClock * self = GST_TEST_CLOCK (object);
  GstTestClockPrivate * priv = GST_TEST_CLOCK_GET_PRIVATE (self);

  GST_OBJECT_LOCK (self);
  while (priv->entry_contexts != NULL)
  {
    GstClockEntryContext * ctx = priv->entry_contexts->data;
    gst_test_clock_remove_entry (self, ctx->clock_entry);
  }
  GST_OBJECT_UNLOCK (self);

  G_OBJECT_CLASS (parent_class)->dispose (object);
}

static void
gst_test_clock_finalize (GObject * object)
{
  GstTestClock * self = GST_TEST_CLOCK (object);
  GstTestClockPrivate * priv = GST_TEST_CLOCK_GET_PRIVATE (self);

  g_cond_free (priv->entry_added_cond);
  g_cond_free (priv->entry_processed_cond);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
gst_test_clock_set_property (GObject * object,
                             guint property_id,
                             const GValue * value,
                             GParamSpec * pspec)
{
  GstTestClock * self = GST_TEST_CLOCK (object);
  GstTestClockPrivate * priv = GST_TEST_CLOCK_GET_PRIVATE (self);

  switch (property_id)
  {
    case PROP_START_TIME:
      priv->start_time = g_value_get_uint64 (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
  }
}

static void
gst_test_clock_get_property (GObject * object,
                             guint property_id,
                             GValue * value,
                             GParamSpec * pspec)
{
  GstTestClock * self = GST_TEST_CLOCK (object);
  GstTestClockPrivate * priv = GST_TEST_CLOCK_GET_PRIVATE (self);

  switch (property_id)
  {
    case PROP_START_TIME:
      g_value_set_uint64 (value, priv->start_time);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
  }
}

static GstClockTime
gst_test_clock_get_resolution (GstClock * clock)
{
  return 1;
}

static GstClockTime
gst_test_clock_get_internal_time (GstClock * clock)
{
  GstTestClock * self = GST_TEST_CLOCK (clock);
  GstTestClockPrivate * priv = GST_TEST_CLOCK_GET_PRIVATE (self);
  GstClockTime result;

  GST_OBJECT_LOCK (self);
  result = priv->internal_time;
  GST_OBJECT_UNLOCK (self);

  return result;
}

static GstClockReturn
gst_test_clock_wait_jitter (GstClock * clock,
                            GstClockEntry * entry,
                            GstClockTimeDiff * jitter)
{
  GstTestClock * self = GST_TEST_CLOCK (clock);
  GstTestClockPrivate * priv = GST_TEST_CLOCK_GET_PRIVATE (self);

  GST_OBJECT_LOCK (self);

  if (gst_test_clock_lookup_entry_context (self, entry) == NULL)
    gst_test_clock_add_entry (self, entry, jitter);

  GST_CLOCK_ENTRY_STATUS (entry) = GST_CLOCK_BUSY;

  while (GST_CLOCK_ENTRY_STATUS (entry) == GST_CLOCK_BUSY)
    g_cond_wait (priv->entry_processed_cond, GST_OBJECT_GET_LOCK (self));

  GST_OBJECT_UNLOCK (self);

  return GST_CLOCK_ENTRY_STATUS (entry);
}

static GstClockReturn
gst_test_clock_wait_async (GstClock * clock,
                           GstClockEntry * entry)
{
  GstTestClock * self = GST_TEST_CLOCK (clock);

  GST_OBJECT_LOCK (self);
  gst_test_clock_add_entry (self, entry, NULL);
  GST_OBJECT_UNLOCK (self);

  return GST_CLOCK_OK;
}

static void
gst_test_clock_unschedule (GstClock * clock,
                           GstClockEntry * entry)
{
  GstTestClock * self = GST_TEST_CLOCK (clock);

  GST_OBJECT_LOCK (self);
  GST_CLOCK_ENTRY_STATUS (entry) = GST_CLOCK_UNSCHEDULED;
  gst_test_clock_remove_entry (self, entry);
  GST_OBJECT_UNLOCK (self);
}

GstClock *
gst_test_clock_new (void)
{
  return gst_test_clock_new_with_start_time (0);
}

GstClock *
gst_test_clock_new_with_start_time (GstClockTime start_time)
{
  return g_object_new (GST_TYPE_TEST_CLOCK, "start-time", start_time, NULL);
}

void
gst_test_clock_set_time (GstTestClock * self,
                         GstClockTime new_time)
{
  GstTestClockPrivate * priv = GST_TEST_CLOCK_GET_PRIVATE (self);

  g_assert (new_time >= priv->internal_time);

  GST_OBJECT_LOCK (self);
  priv->internal_time = new_time;
  GST_OBJECT_UNLOCK (self);
}

void
gst_test_clock_advance_time (GstTestClock * self,
                             GstClockTimeDiff delta)
{
  GstTestClockPrivate * priv = GST_TEST_CLOCK_GET_PRIVATE (self);

  g_assert (delta >= 0);

  GST_OBJECT_LOCK (self);
  priv->internal_time += delta;
  GST_OBJECT_UNLOCK (self);
}

guint
gst_test_clock_peek_id_count (GstTestClock * self)
{
  GstTestClockPrivate * priv = GST_TEST_CLOCK_GET_PRIVATE (self);
  guint result;

  GST_OBJECT_LOCK (self);
  result = g_list_length (priv->entry_contexts);
  GST_OBJECT_UNLOCK (self);

  return result;
}

gboolean
gst_test_clock_has_id (GstTestClock * self,
                       GstClockID id)
{
  gboolean result;

  GST_OBJECT_LOCK (self);
  result = gst_test_clock_lookup_entry_context (self, id) != NULL;
  GST_OBJECT_UNLOCK (self);

  return result;
}

gboolean
gst_test_clock_peek_next_pending_id (GstTestClock * self,
                                     GstTestClockPendingID * pending_id)
{
  gboolean result;

  GST_OBJECT_LOCK (self);
  result = gst_test_clock_peek_next_pending_id_unlocked (self, pending_id);
  GST_OBJECT_UNLOCK (self);

  return result;
}

gboolean
gst_test_clock_wait_for_next_pending_id (GstTestClock * self,
                                         GstTestClockPendingID * pending_id)
{
  GstTestClockPrivate * priv = GST_TEST_CLOCK_GET_PRIVATE (self);
  gboolean result = FALSE;
  GTimeVal deadline;
  gboolean signalled_before_timeout = TRUE;

  g_get_current_time (&deadline);
  g_time_val_add (&deadline, TEST_CLOCK_WAIT_TIMEOUT * G_USEC_PER_SEC);

  GST_OBJECT_LOCK (self);

  while (priv->entry_contexts == NULL)
  {
    signalled_before_timeout = g_cond_timed_wait (
        priv->entry_added_cond, GST_OBJECT_GET_LOCK (self), &deadline);
    if (!signalled_before_timeout)
      break;
  }

  if (signalled_before_timeout)
    result = gst_test_clock_peek_next_pending_id_unlocked (self, pending_id);

  GST_OBJECT_UNLOCK (self);

  return result;
}

static gboolean
gst_test_clock_peek_next_pending_id_unlocked (GstTestClock * self,
                                              GstTestClockPendingID * pending_id)
{
  GstTestClockPrivate * priv = GST_TEST_CLOCK_GET_PRIVATE (self);
  gboolean result = FALSE;

  if (priv->entry_contexts != NULL)
  {
    GstClockEntryContext * ctx = priv->entry_contexts->data;

    if (pending_id != NULL)
    {
      pending_id->clock_id = ctx->clock_entry;
      pending_id->type = ctx->clock_entry->type;
      pending_id->time = ctx->clock_entry->time;
    }

    result = TRUE;
  }

  return result;
}

GstClockID
gst_test_clock_process_next_clock_id (GstTestClock * self)
{
  GstTestClockPrivate * priv = GST_TEST_CLOCK_GET_PRIVATE (self);
  GstClockID result = NULL;
  GstClockEntryContext * ctx = NULL;
  GList * cur;

  GST_OBJECT_LOCK (self);

  for (cur = priv->entry_contexts; cur != NULL && result == NULL;
      cur = cur->next)
  {
    ctx = cur->data;

    if (priv->internal_time >= GST_CLOCK_ENTRY_TIME (ctx->clock_entry))
      result = ctx->clock_entry;
  }

  if (result != NULL)
  {
    GstClockEntry * entry = ctx->clock_entry;

    if (ctx->time_diff >= 0)
      GST_CLOCK_ENTRY_STATUS (entry) = GST_CLOCK_OK;
    else
      GST_CLOCK_ENTRY_STATUS (entry) = GST_CLOCK_EARLY;

    if (entry->func != NULL)
    {
      GST_OBJECT_UNLOCK (self);
      entry->func (GST_CLOCK (self), priv->internal_time, entry,
          entry->user_data);
      GST_OBJECT_LOCK (self);
    }

    gst_test_clock_remove_entry (self, entry);

    if (GST_CLOCK_ENTRY_TYPE (entry) == GST_CLOCK_ENTRY_PERIODIC)
    {
      GST_CLOCK_ENTRY_TIME (entry) += GST_CLOCK_ENTRY_INTERVAL (entry);

      if (entry->func != NULL)
        gst_test_clock_add_entry (self, entry, NULL);
    }
  }

  GST_OBJECT_UNLOCK (self);

  return result;
}

GstClockTime
gst_test_clock_get_next_entry_time (GstTestClock * self)
{
  GstTestClockPrivate * priv = GST_TEST_CLOCK_GET_PRIVATE (self);
  GstClockTime result = -1;
  GList * cur;

  GST_OBJECT_LOCK (self);

  for (cur = priv->entry_contexts; cur != NULL; cur = cur->next)
  {
    GstClockEntryContext * ctx = cur->data;
    GstClockEntry * entry = ctx->clock_entry;
    GstClockTime entry_time = GST_CLOCK_ENTRY_TIME (entry);

    if (entry_time < result)
      result = entry_time;
  }

  GST_OBJECT_UNLOCK (self);

  return result;
}
static void
gst_test_clock_add_entry (GstTestClock * self,
                          GstClockEntry * entry,
                          GstClockTimeDiff * jitter)
{
  GstTestClockPrivate * priv = GST_TEST_CLOCK_GET_PRIVATE (self);
  GstClockTime now;
  GstClockEntryContext * ctx;

  now = gst_clock_adjust_unlocked (GST_CLOCK (self), priv->internal_time);

  if (jitter != NULL)
    *jitter = GST_CLOCK_DIFF (GST_CLOCK_ENTRY_TIME (entry), now);

  ctx = g_slice_new (GstClockEntryContext);
  ctx->clock_entry = GST_CLOCK_ENTRY (gst_clock_id_ref (entry));
  ctx->time_diff = GST_CLOCK_DIFF (now, GST_CLOCK_ENTRY_TIME (entry));

  priv->entry_contexts = g_list_insert_sorted (priv->entry_contexts, ctx,
      gst_clock_entry_context_compare_func);

  g_cond_broadcast (priv->entry_added_cond);
}

static void
gst_test_clock_remove_entry (GstTestClock * self,
                             GstClockEntry * entry)
{
  GstTestClockPrivate * priv = GST_TEST_CLOCK_GET_PRIVATE (self);
  GstClockEntryContext * ctx;

  ctx = gst_test_clock_lookup_entry_context (self, entry);
  if (ctx != NULL)
  {
    gst_clock_id_unref (ctx->clock_entry);
    priv->entry_contexts = g_list_remove (priv->entry_contexts, ctx);

    g_cond_broadcast (priv->entry_processed_cond);
  }
}

GstClockEntryContext *
gst_test_clock_lookup_entry_context (GstTestClock * self,
                                     GstClockEntry * clock_entry)
{
  GstTestClockPrivate * priv = GST_TEST_CLOCK_GET_PRIVATE (self);
  GstClockEntryContext * result = NULL;
  GList * cur;

  for (cur = priv->entry_contexts; cur != NULL; cur = cur->next)
  {
    GstClockEntryContext * ctx = cur->data;

    if (ctx->clock_entry == clock_entry)
    {
      result = ctx;
      break;
    }
  }

  return result;
}

static gint
gst_clock_entry_context_compare_func (gconstpointer a,
                                      gconstpointer b)
{
  const GstClockEntryContext * ctx_a = a;
  const GstClockEntryContext * ctx_b = b;

  return gst_clock_id_compare_func (ctx_a->clock_entry, ctx_b->clock_entry);
}
