/*
 * TWAIN Plug-in
 * Copyright (C) 1999 Craig Setera
 * Craig Setera <setera@home.com>
 * 03/31/1999
 *
 * Updated for Mac OS X support
 * Brion Vibber <brion@pobox.com>
 * 07/22/2004
 *
 * Added for Win x64 support, changed data source selection.
 * Jens M. Plonka <jens.plonka@gmx.de>
 * 11/25/2011
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 * Based on (at least) the following plug-ins:
 * Screenshot
 * GIF
 * Randomize
 *
 * Any suggestions, bug-reports or patches are welcome.
 *
 * This plug-in interfaces to the TWAIN support library in order
 * to capture images from TWAIN devices directly into GIMP images.
 * The plug-in is capable of acquiring the following type of
 * images:
 * - B/W (1 bit images translated to grayscale B/W)
 * - Grayscale up to 16 bits per pixel
 * - RGB up to 16 bits per sample (24, 30, 36, etc.)
 * - Paletted images (both Gray and RGB)
 *
 * Prerequisites:
 * Should compile and run on both Win32 and Mac OS X 10.3 (possibly
 * also on 10.2).
 *
 * Known problems:
 * - Multiple image transfers will hang the plug-in.  The current
 *   configuration compiles with a maximum of single image transfers.
 * - On Mac OS X, canceling doesn't always close things out fully.
 * - Epson TWAIN driver on Mac OS X crashes the plugin when scanning.
 */

/*
 * Revision history
 *  (02/07/99)  v0.1   First working version (internal)
 *  (02/09/99)  v0.2   First release to anyone other than myself
 *  (02/15/99)  v0.3   Added image dump and read support for debugging
 *  (03/31/99)  v0.5   Added support for multi-byte samples and paletted
 *                     images.
 *  (07/23/04)  v0.6   Added Mac OS X support.
 *  (11/25/11)  v0.7   Added Win x64 support, changed data source selection.
 */

#include "config.h"

#include <glib.h>   /* Needed when compiling with gcc */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "libgimp/gimp.h"
#include "libgimp/stdplugins-intl.h"

#include "tw_platform.h"
#include "tw_func.h"
#include "tw_local.h"
#include "tw_util.h"

/******************************************************************
 * Application definitions
 ******************************************************************/
#define MAX_IMAGES          1

/*
 * Plug-in Parameter definitions
 */
#define NUMBER_IN_ARGS 1
#define IN_ARGS \
    { GIMP_PDB_INT32, "run-mode", "The run mode { RUN-INTERACTIVE (0), RUN-NONINTERACTIVE (1) }" }
#define NUMBER_OUT_ARGS 2
#define OUT_ARGS \
    { GIMP_PDB_INT32, "image-count", "Number of acquired images" }, \
    { GIMP_PDB_INT32ARRAY, "image-ids", "Array of acquired image identifiers" }
/******************************************************************
 * Global variables
 ******************************************************************/
pTW_SESSION  twSession = NULL;
static const GimpParamDef args[] = { IN_ARGS };
static const GimpParamDef returns[]  = { OUT_ARGS };

static char  *destBuf = NULL;

/* The list of images to be transferred */
GList        *images = NULL;

/******************************************************************
 * Forward declarations
 ******************************************************************/
void preTransferCallback (void *);
int  beginTransferCallback (pTW_IMAGEINFO, void *);
int  dataTransferCallback (pTW_IMAGEINFO, pTW_IMAGEMEMXFER, void *);
int  endTransferCallback (int, int, void *);
void postTransferCallback (int, void *);
pTW_SESSION initializeTwain (void);
void register_menu (pTW_IDENTITY dsIdentity, const gchar *menu_path);
void register_scanner_menus (void);
static void query (void);
static void run  (const gchar      *name,
       gint              nparams,
       const GimpParam  *param,
       gint             *nreturn_vals,
       GimpParam       **return_vals);

/* This plug-in's functions */
const GimpPlugInInfo PLUG_IN_INFO =
{
  NULL,    /* init_proc */
  NULL,    /* quit_proc */
  query,   /* query_proc */
  run,     /* run_proc */
};

/* The standard callback functions */
TXFR_CB_FUNCS standardCbFuncs =
{
  /* preTxfrCb   */ preTransferCallback,
  /* txfrBeginCb */ beginTransferCallback,
  /* txfrDataCb  */ dataTransferCallback,
  /* txfrEndCb   */ endTransferCallback,
  /* postTxfrCb  */ postTransferCallback
};

/******************************************************************
 * Type definitions
 ******************************************************************/

static char bitMasks[] = { 128, 64, 32, 16, 8, 4, 2, 1 };

/* Return values storage */
static GimpParam values[3];

/******************************************************************
 * Dump handling
 ******************************************************************/

#ifndef TWAIN_ALTERNATE_MAIN
MAIN ()
#endif /* TWAIN_ALTERNATE_MAIN */

/*
 * scan_image
 *
 * Callback function for OS dependant implementation inside the
 * event loop.
 */
int
scanImage (void)
{
  return getImage (twSession);
}

/*
 * get_app_identity
 *
 * Initialize and return our application's identity for
 * the TWAIN DataSourceManager.
 */
static pTW_IDENTITY
getAppIdentity (void)
{
  pTW_IDENTITY appIdentity = g_new (TW_IDENTITY, 1);

  /* Set up the application identity */
  appIdentity->Id = 0;
  appIdentity->Version.MajorNum = PLUG_IN_MAJOR;
  appIdentity->Version.MinorNum = PLUG_IN_MINOR;
  appIdentity->Version.Language = TWLG_USA;
  appIdentity->Version.Country = TWCY_USA;
  strcpy(appIdentity->Version.Info, PLUG_IN_VERSION);
  appIdentity->ProtocolMajor = TWON_PROTOCOLMAJOR;
  appIdentity->ProtocolMinor = TWON_PROTOCOLMINOR;
  /* Tell the data source manger that we only accept image data sources */
  appIdentity->SupportedGroups = DF_APP2 | DG_CONTROL | DG_IMAGE;
  strcpy (appIdentity->Manufacturer, PLUG_IN_COPYRIGHT);
  strcpy (appIdentity->ProductFamily, PRODUCT_FAMILY);
  strcpy (appIdentity->ProductName, PRODUCT_NAME);

  return appIdentity;
}

/*
 * initializeTwain
 *
 * Do the necessary TWAIN initialization.  This sets up
 * our TWAIN session information.  The session stuff is
 * something built by me on top of the standard TWAIN
 * datasource manager calls.
 */
pTW_SESSION
initializeTwain (void)
{
  pTW_IDENTITY appIdentity;

  /* Get our application's identity */
  appIdentity = getAppIdentity ();

  /* Create a new session object */
  twSession = newSession (appIdentity);

  /* Register our image transfer callback functions */
  twSession->transferFunctions = &standardCbFuncs;
  twSession->clientData = NULL;

  return twSession;
}

/*
 * register_menu
 *
 * Registers a menu item for the given scanner.
 */
void
register_menu (const pTW_IDENTITY dsIdentity, const gchar *menu_path)
{
  guint8 *iconfile;
  gchar *name;

  name = dsIdentity->ProductName;

  gimp_install_procedure (
      name,
      PLUG_IN_DESCRIPTION,
      PLUG_IN_HELP,
      dsIdentity->Manufacturer,
      PLUG_IN_COPYRIGHT,
      PLUG_IN_VERSION,
      name,
      NULL,
      GIMP_PLUGIN,
      NUMBER_IN_ARGS,
      NUMBER_OUT_ARGS,
      args, returns);

  gimp_plugin_menu_register (name, menu_path);
  iconfile = (guint8 *) g_build_filename (gimp_data_directory (),
                                      "twain",
                                      "twain-menu.png",
                                      NULL);
  gimp_plugin_icon_register (name, GIMP_ICON_TYPE_IMAGE_FILE, iconfile);
  g_free (iconfile);
}

/*
 * register_scanner_menus
 *
 * Initilizes communication with the Data Source Manager.
 * Communication will be kept open till shutdown of GIMP
 * because the id might change due to new installed data
 * sources wile GIMP is open!
 * The name of each scanner is the identifyer of the menu.
 */
void
register_scanner_menus (void)
{
  pTW_DATA_SOURCE dataSource;
  pTW_IDENTITY dsIdentity;

  // Get the list of available TWAIN data sources
  initializeTwain ();

  if (openDSM (twSession))
  {
    /* get all available data source as a linked list into twSession */
    dataSource = get_available_ds (twSession);

    /* for each found data source create an own menu item */
    while (dataSource != NULL)
    {
      dsIdentity = dataSource->dsIdentity;
      register_menu (dsIdentity, "<Toolbox>/File/Create");

      /* handle the next data source */
      dataSource = dataSource->dsNext;
    }

    closeDSM (twSession);
  }
}

/******************************************************************
 * GIMP Plug-in entry points
 ******************************************************************/

/*
 * query
 *
 * The plug-in is being queried. Install our procedure for acquiring.
 */
static void
query (void)
{
  /* Old style of scanner selection */
  gimp_install_procedure (MID_SELECT,
      PLUG_IN_DESCRIPTION,
      PLUG_IN_HELP,
      PLUG_IN_AUTHOR,
      PLUG_IN_COPYRIGHT,
      PLUG_IN_VERSION,
      N_("_Scanner/Camera..."),
      NULL,
      GIMP_PLUGIN,
      NUMBER_IN_ARGS,
      NUMBER_OUT_ARGS,
      args, returns);

  gimp_plugin_menu_register (MID_SELECT, "<Toolbox>/File/Create/Acquire");

  /* New style of scanner selection */
  register_scanner_menus ();
}

/*
 * run
 *
 * The plug-in is being requested to run.
 * Capture an image from a TWAIN datasource
 */
static void
run (const gchar      *name,
     gint              nparams,
     const GimpParam  *param,
     gint             *nreturn_vals,
     GimpParam       **return_vals)
{
  GList       *list;
  gint32       count = 0;
  gint32       i;
  GimpRunMode  run_mode = param[0].data.d_int32;

  /* Initialize the return values
   * Always return at least the status to the caller.
   */
  values[0].type = GIMP_PDB_STATUS;
  values[0].data.d_status = GIMP_PDB_SUCCESS;
  *nreturn_vals = 3;
  *return_vals = values;

  INIT_I18N ();

  /* Before we get any further, verify that we have
   * TWAIN and that there is actually a datasource
   * to be used in doing the acquire.
   */
  if (!twainIsAvailable ())
  {
    values[0].data.d_status = GIMP_PDB_EXECUTION_ERROR;
    return;
  }

  /* Set up the rest of the return parameters */
  values[1].type = GIMP_PDB_INT32;
  values[1].data.d_int32 = 0;
  values[2].type = GIMP_PDB_INT32ARRAY;
  values[2].data.d_int32array = NULL; /* Will be assigned when acquiring an image */

  /* How are we running today? */
  switch (run_mode)
  {
    case GIMP_RUN_NONINTERACTIVE:
      /* Non-interactive calls are not supported!
       * Bail if someone tries to call this way.
       */
      values[0].data.d_status = GIMP_PDB_CALLING_ERROR;
      return;

    default:
      /* Retrieve values from the last run...
       * Currently ignored
       */
      break;
  }

  /* Have we succeeded so far? */
  if (values[0].data.d_status == GIMP_PDB_SUCCESS)
  {
    if (twain_main (name))
    {
      count = g_list_length (images);
    }

    if (count > 0)
    {
      /* Return the list of image IDs */
      values[2].data.d_int32array = g_new (gint32, count);
      for (list = images, i = 0; list; list = g_list_next (list), i++)
      {
        values[2].data.d_int32array[i] = GPOINTER_TO_INT (list->data);
      }
      g_list_free (images);
    }
  }

  /* Retrun the number of images */
  values[1].data.d_int32      = count;

  /* Check to make sure we got at least one valid
   * image.
   */
  if (count == 0)
  {
    values[0].data.d_status = GIMP_PDB_EXECUTION_ERROR;
  }
}

/***********************************************************************
 * Image transfer callback functions
 ***********************************************************************/

/*
 * preTransferCallback
 *
 * This callback function is called before any images
 * are transferred.  Set up the one time only stuff.
 */
void
preTransferCallback (void *clientData)
{
  /* Initialize our progress dialog */
  gimp_progress_init (_("Transferring data from scanner/camera"));
}

/*
 * beginTransferCallback
 *
 * The following function is called at the beginning
 * of each image transfer.
 */
int
beginTransferCallback (pTW_IMAGEINFO imageInfo, void *clientData)
{
  int imageType, layerType;
  TW_INT32 width = imageInfo->ImageWidth;
  TW_INT32 length = imageInfo->ImageLength;

  pClientDataStruct theClientData = g_new (ClientDataStruct, 1);

  /* Decide on the image type */
  switch (imageInfo->PixelType)
  {
    case TWPT_BW:
    case TWPT_GRAY:
      /* Set up the image and layer types */
      imageType = GIMP_GRAY;
      layerType = GIMP_GRAY_IMAGE;
      break;

    case TWPT_RGB:
      /* Set up the image and layer types */
      imageType = GIMP_RGB;
      layerType = GIMP_RGB_IMAGE;
      break;

    case TWPT_PALETTE:
      /* Get the palette data */
      theClientData->paletteData = g_new (TW_PALETTE8, 1);
      twSession->twRC = DSM_GET_PALETTE(twSession, theClientData);
      if (twSession->twRC != TWRC_SUCCESS)
      {
        return FALSE;
      }

      switch (theClientData->paletteData->PaletteType)
      {
        case TWPA_RGB:
          /* Set up the image and layer types */
          imageType = GIMP_RGB;
          layerType = GIMP_RGB_IMAGE;
          break;

        case TWPA_GRAY:
          /* Set up the image and layer types */
          imageType = GIMP_GRAY;
          layerType = GIMP_GRAY_IMAGE;
          break;

        default:
          return FALSE;
      }
      break;

    default:
      /* We don't know how to deal with anything other than
       * the types listed above.  Bail for any other image
       * type.
       */
      return FALSE;
  }

  /* Create the GIMP image */
  theClientData->image_id = gimp_image_new (width, length, imageType);

  /* Set the actual resolution */
  gimp_image_set_resolution (theClientData->image_id,
      FIX32_TO_FLOAT(imageInfo->XResolution),
      FIX32_TO_FLOAT(imageInfo->YResolution));
  gimp_image_set_unit (theClientData->image_id, GIMP_UNIT_INCH);

  /* Create a layer */
  theClientData->layer_id = gimp_layer_new(theClientData->image_id,
      _("Background"), width, length, layerType, 100, GIMP_NORMAL_MODE);

  /* Add the layer to the image */
  gimp_image_insert_layer (theClientData->image_id,
           theClientData->layer_id, -1, 0);

  /* Update the progress dialog */
  theClientData->totalPixels = width * length;
  theClientData->completedPixels = 0;
  gimp_progress_update ((double) 0);

  /* Get our drawable */
  theClientData->drawable = gimp_drawable_get(theClientData->layer_id);

  /* Initialize a pixel region for writing to the image */
  gimp_pixel_rgn_init (&(theClientData->pixel_rgn), theClientData->drawable,
          0, 0, width, length, TRUE, FALSE);

  /* Store our client data for the data transfer callbacks */
  if (clientData)
  {
    g_free (clientData);
  }

  twSession->clientData = (void *) theClientData;

  /* Make sure to return TRUE to continue the image
   * transfer
   */
  return TRUE;
}

/*
 * bitTransferCallback
 *
 * The following function is called for each memory
 * block that is transferred from the data source if
 * the image type is Black/White.
 *
 * Black and white data is unpacked from bit data
 * into byte data and written into a gray scale GIMP
 * image.
 */
static int
bitTransferCallback(pTW_IMAGEINFO imageInfo,
        pTW_IMAGEMEMXFER imageMemXfer,
        void *clientData)
{
  int row, col, offset;
  char *srcBuf;
  int rows = imageMemXfer->Rows;
  int cols = imageMemXfer->Columns;
  pClientDataStruct theClientData = (pClientDataStruct) clientData;

  /* Allocate a buffer as necessary */
  if (!destBuf)
  {
    destBuf = g_new (char, rows * cols);
  }

  /* Unpack the image data from bits into bytes */
  srcBuf = (char *) imageMemXfer->Memory.TheMem;
  offset = 0;
  for (row = 0; row < rows; row++)
  {
    for (col = 0; col < cols; col++)
    {
      char byte = srcBuf[(row * imageMemXfer->BytesPerRow) + (col / 8)];
      destBuf[offset++] = ((byte & bitMasks[col % 8]) != 0) ? 255 : 0;
    }
  }

  /* Update the complete chunk */
  gimp_pixel_rgn_set_rect(&(theClientData->pixel_rgn),
        (guchar *) destBuf,
        imageMemXfer->XOffset, imageMemXfer->YOffset,
        cols, rows);

  /* Update the user on our progress */
  theClientData->completedPixels += (cols * rows);
  gimp_progress_update((double) theClientData->completedPixels /
           (double) theClientData->totalPixels);


  return TRUE;
}

/*
 * oneBytePerSampleTransferCallback
 *
 * The following function is called for each memory
 * block that is transferred from the data source if
 * the image type is Grayscale or RGB.  This transfer
 * mode is quicker than the modes that require translation
 * from a greater number of bits per sample down to the
 * 8 bits per sample understood by GIMP.
 */
static int
oneBytePerSampleTransferCallback(pTW_IMAGEINFO imageInfo,
         pTW_IMAGEMEMXFER imageMemXfer,
         void *clientData)
{
  int row;
  char *srcBuf;
  int bytesPerPixel = imageInfo->BitsPerPixel / 8;
  int rows = imageMemXfer->Rows;
  int cols = imageMemXfer->Columns;
  pClientDataStruct theClientData = (pClientDataStruct) clientData;

  /* Allocate a buffer as necessary */
  if (!destBuf)
  {
    destBuf = g_new (char, rows * cols * bytesPerPixel);
  }

  /* The bytes coming from the source may not be padded in
   * a way that GIMP is terribly happy with.  It is
   * possible to transfer row by row, but that is particularly
   * expensive in terms of performance.  It is much cheaper
   * to rearrange the data and transfer it in one large chunk.
   * The next chunk of code rearranges the incoming data into
   * a non-padded chunk for GIMP.
   */
  srcBuf = (char *) imageMemXfer->Memory.TheMem;
  for (row = 0; row < rows; row++)
  {
    /* Copy the current row */
    memcpy((destBuf + (row * bytesPerPixel * cols)),
     (srcBuf + (row * imageMemXfer->BytesPerRow)),
     (bytesPerPixel * cols));
  }

  /* Update the complete chunk */
  gimp_pixel_rgn_set_rect(&(theClientData->pixel_rgn),
        (guchar *) destBuf,
        imageMemXfer->XOffset, imageMemXfer->YOffset,
        cols, rows);

  /* Update the user on our progress */
  theClientData->completedPixels += (cols * rows);
  gimp_progress_update((double) theClientData->completedPixels /
           (double) theClientData->totalPixels);

  return TRUE;
}

/*
 * twoBytesPerSampleTransferCallback
 *
 * The following function is called for each memory
 * block that is transferred from the data source if
 * the image type is Grayscale or RGB.
 */
static int
twoBytesPerSampleTransferCallback (pTW_IMAGEINFO imageInfo,
         pTW_IMAGEMEMXFER imageMemXfer,
         void *clientData)
{
  static float ratio = 0.00390625; /* 1/256 */
  int row, col, sample;
  char *destByte;
  int rows = imageMemXfer->Rows;
  int cols = imageMemXfer->Columns;

  TW_UINT16 *samplePtr;

  pClientDataStruct theClientData = (pClientDataStruct) clientData;

  /* Allocate a buffer as necessary */
  if (!destBuf)
  {
    destBuf = g_new (char, rows * cols * imageInfo->SamplesPerPixel);
  }

  /* The bytes coming from the source may not be padded in
   * a way that GIMP is terribly happy with.  It is
   * possible to transfer row by row, but that is particularly
   * expensive in terms of performance.  It is much cheaper
   * to rearrange the data and transfer it in one large chunk.
   * The next chunk of code rearranges the incoming data into
   * a non-padded chunk for GIMP.  This function must also
   * reduce from multiple bytes per sample down to single byte
   * per sample.
   */
  /* Work through the rows */
  for (row = 0; row < rows; row++)
  {
    /* The start of this source row */
    samplePtr = (TW_UINT16 *)
      ((char *) imageMemXfer->Memory.TheMem + (row * imageMemXfer->BytesPerRow));

    /* The start of this dest row */
    destByte = destBuf + (row * imageInfo->SamplesPerPixel * cols);

    /* Work through the columns */
    for (col = 0; col < cols; col++)
    {
      /* Finally, work through each of the samples */
      for (sample = 0; sample < imageInfo->SamplesPerPixel; sample++)
      {
        /* Get the value */
        TW_UINT16 value = *samplePtr;

        /* Move the sample pointer */
        samplePtr++;

        /* Place in the destination */
        *destByte = (char) ((float) value * (float) ratio);
        destByte++;
      }
    }
  }

  /* Send the complete chunk */
  gimp_pixel_rgn_set_rect(&(theClientData->pixel_rgn),
        (guchar *) destBuf,
        imageMemXfer->XOffset, imageMemXfer->YOffset,
        cols, rows);

  /* Update the user on our progress */
  theClientData->completedPixels += (cols * rows);
  gimp_progress_update((double) theClientData->completedPixels /
           (double) theClientData->totalPixels);

  return TRUE;
}

/*
 * palettedTransferCallback
 *
 * The following function is called for each memory
 * block that is transferred from the data source if
 * the image type is paletted.  This does not create
 * an indexed image type in GIMP because for some
 * reason it does not allow creation of a specific
 * palette.  This function will create an RGB or Gray
 * image and use the palette to set the details of
 * the pixels.
 */
static int
palettedTransferCallback (pTW_IMAGEINFO imageInfo,
       pTW_IMAGEMEMXFER imageMemXfer,
       void *clientData)
{
  int channelsPerEntry;
  int row, col;
  int rows = imageMemXfer->Rows;
  int cols = imageMemXfer->Columns;
  char *destPtr = NULL, *srcPtr = NULL;

  /* Get the client data */
  pClientDataStruct theClientData = (pClientDataStruct) clientData;

  /* Look up the palette entry size */
  channelsPerEntry =
    (theClientData->paletteData->PaletteType == TWPA_RGB) ? 3 : 1;

  /* Allocate a buffer as necessary */
  if (!destBuf)
  {
    destBuf = g_new (char, rows * cols * channelsPerEntry);
  }

  /* Work through the rows */
  destPtr = destBuf;
  for (row = 0; row < rows; row++)
  {
    srcPtr = (char *) ((char *) imageMemXfer->Memory.TheMem +
           (row * imageMemXfer->BytesPerRow));

    /* Work through the columns */
    for (col = 0; col < cols; col++)
    {
      /* Get the palette index */
      int index = (unsigned char) *srcPtr;
      srcPtr++;

      switch (theClientData->paletteData->PaletteType)
      {
        case TWPA_GRAY:
          *destPtr = theClientData->paletteData->Colors[index].Channel1;
          destPtr++;
          break;

        case TWPA_RGB:
          *destPtr = theClientData->paletteData->Colors[index].Channel1;
          destPtr++;
          *destPtr = theClientData->paletteData->Colors[index].Channel2;
          destPtr++;
          *destPtr = theClientData->paletteData->Colors[index].Channel3;
          destPtr++;
      }
    }
  }

  /* Send the complete chunk */
  gimp_pixel_rgn_set_rect(&(theClientData->pixel_rgn),
        (guchar *) destBuf,
        imageMemXfer->XOffset, imageMemXfer->YOffset,
        cols, rows);

  /* Update the user on our progress */
  theClientData->completedPixels += (cols * rows);
  gimp_progress_update((double) theClientData->completedPixels /
           (double) theClientData->totalPixels);

  return TRUE;
}

/*
 * dataTransferCallback
 *
 * The following function is called for each memory
 * block that is transferred from the data source.
 */
int
dataTransferCallback(pTW_IMAGEINFO imageInfo,
    pTW_IMAGEMEMXFER imageMemXfer,
    void *clientData)
{
  /* Choose the appropriate transfer handler */
  switch (imageInfo->PixelType)
  {
    case TWPT_PALETTE:
      return palettedTransferCallback (imageInfo, imageMemXfer, clientData);

    case TWPT_BW:
      return bitTransferCallback (imageInfo, imageMemXfer, clientData);

    case TWPT_GRAY:
    case TWPT_RGB:
      switch (imageInfo->BitsPerPixel / imageInfo->SamplesPerPixel)
      {
        case 8:
          return oneBytePerSampleTransferCallback (imageInfo, imageMemXfer, clientData);

        case 16:
          return twoBytesPerSampleTransferCallback (imageInfo, imageMemXfer, clientData);

        default:
          return FALSE;
      }

    default:
      return FALSE;
  }
}

/*
 * endTransferCallback
 *
 * The following function is called at the end of the image transfer.
 * The caller will be handed the image transfer completion state.
 * The following values (defined in twain.h) are possible:
 *
 * TWRC_XFERDONE: Transfer completed successfully.
 * TWRC_CANCEL:   Transfer cancled by user.
 * TWRC_FAILURE:  Transfer failed.
 */
int
endTransferCallback (int completionState, int pendingCount, void *clientData)
{
  pClientDataStruct theClientData = (pClientDataStruct) clientData;

  /* Clean up and detach from the drawable */
  if (destBuf)
  {
    g_free (destBuf);
    destBuf = NULL;
  }
  gimp_drawable_flush (theClientData->drawable);
  gimp_drawable_detach (theClientData->drawable);

  /* Make sure to check our return code */
  if (completionState == TWRC_XFERDONE)
  {
    images = g_list_prepend (images, GINT_TO_POINTER (theClientData->image_id));
    /* Display the image */
    gimp_display_new (theClientData->image_id);
  }
  else
  {
    /* The transfer did not complete successfully */
    gimp_image_delete (theClientData->image_id);
  }

  gimp_displays_flush ();

  /* Shut down if we have received all of the possible images */
  return (values[1].data.d_int32 < MAX_IMAGES);
}

/*
 * postTransferCallback
 *
 * This callback function will be called
 * after all possible images have been
 * transferred.
 */
void
postTransferCallback (int pendingCount, void *clientData)
{
  /* Shut things down. */
  if (pendingCount != 0)
  {
    cancelPendingTransfers(twSession);
  }

  /* This will close the datasource and datasource
   * manager.  Then the message queue will be shut
   * down and the run() procedure will finally be
   * able to finish.
   */
  disableDS (twSession);
  closeDS (twSession);
  closeDSM (twSession);

  /* Post a message to close up the application */
  twainQuitApplication ();
}
