﻿using System;
using System.Collections.Generic;
using System.Configuration;
using System.IO;
using System.Linq;
using Utility.Resources;

namespace Utility
{
    [Serializable]
    public class SettingList : List<Setting>
    {
        public Setting this[string key]
        {
            get
            {
                var settings = GetSettings(key);
                if(settings.Count() < 1)
                {
                    throw new ConfigurationErrorsException(string.Format("The key '{0}' could not be found in this {1}.", key, GetType()));
                }
                if (settings.Count() > 1)
                {
                    throw new ConfigurationErrorsException((string.Format("The key '{0}' exists more than one time in this {1}.", key, GetType())));
                }
                return settings.First();
            }
            set
            {
                var settings = GetSettings(key);
                if(settings.Count() < 1)
                {
                    Add(value);
                }
                if (settings.Count() == 1)
                {
                    Remove(settings.First());
                    Add(value);
                }
                if (settings.Count() > 1)
                {
                    throw new ConfigurationErrorsException((string.Format("The key '{0}' exists more than one time in this {1}.", key, GetType())));
                }
            }
        }

        private IEnumerable<Setting> GetSettings(string key)
        {
            return from s in this where s.Key == key select s;
        }

        /// <summary>
        /// Determines whether this <see cref="SettingList"/> contains the specified key.
        /// </summary>
        /// <param name="key">The key.</param>
        /// <returns>
        ///   <c>true</c> if  this <see cref="SettingList"/> contains the specified key; otherwise, <c>false</c>.
        /// </returns>
        public bool ContainsKey(string key)
        {
            var keys = from s in this where s.Key == key select s;
            return keys.Any();
        }

        /// <summary>
        /// Validates the <see cref="Setting"/>s of this instance against its <see cref="SettingType"/>.
        /// </summary>
        /// <exception cref="ConfigurationErrorsException">When a setting value is invalid.</exception>
        public void Validate()
        {
            foreach (var setting in this)
            {
                switch (setting.SettingType)
                {
                    case SettingType.Directory:
                        if(!Directory.Exists(setting.Value))
                        {
                            throw new ConfigurationErrorsException(string.Format(Messages.DirectoryDoesNotExist, setting.Value));
                        }
                        break;
                }
            }
        }
    }
}