/**
 * \defgroup fskd FSK demodulation and clip detection.
 * \file fskd.h
 * Header file for CLIP detection.
 * \ingroup fskd
 */
#ifndef _FSKD_H_
#define _FSKD_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <stdint.h>

/*
   private defines
 */

#define FSKD_DELAY       4 /**< FSK demodulator delay */
#define FSKD_POLARITY   -1 /**< signal polarity after FSK demodulating */
#define QTHRESH         80 /**< initial quantization threshhold */

#define CBUFF_LEN 16        /**< circular buffer length, 2^k for better indexing */
#define buffInd(ind, n) ((ind+n) & (CBUFF_LEN-1)) /**< circular buffer indexing */

/**\{
 * \name Filter coefficients defines.
 *
 * Coefficients for Cauer filter used in FSK demodulator.
 * Normal form, biquad form (with a,b suffix). See filter design in Octave.
 */
#define B0a   (int16_t)(1.00000 * 32768)
#define B1a   (int16_t)(0.65559 * 32768)
#define B2a   (int16_t)(1.00000 * 32768)
#define B0b   (int16_t)(1.00000 * 32768)
#define B1b   (int16_t)(1.00000 * 32768)
#define B2b   (int16_t)(0.00000 * 32768)
#define A0a   (int16_t)(1.00000 * 32768)
#define A1a   (int16_t)(-0.99022 * 32768)
#define A2a   (int16_t)(0.68968 * 32768)
#define A0b   (int16_t)(1.00000 * 32768)
#define A1b   (int16_t)(-0.57872 * 32768)
#define A2b   (int16_t)(0.000000 * 32768)
#define B0    (int16_t)(0.064600 * 32768)
#define B1    (int16_t)(0.117553 * 32768)
#define B2    (int16_t)(0.117553 * 32768)
#define B3    (int16_t)(0.064600 * 32768)
#define A0    (int16_t)(1.000000 * 32768)
#define A1    (int16_t)(-1.41873 * 32768)
#define A2    (int16_t)(1.15232  * 32768)
#define A3    (int16_t)(-0.36920 * 32768)
/** \} */


static int16_t coeff_fir [] = {
  103, 52, -235, -794, -851, 827, 4498, 8499, 10249, 8499, 4498, 827, -851, -794, -235, 52, 103
};

/**\{
 * \name States for sample processing state machine.
 *
 */
#define STATE_SEIZURE_SIG 0x1
#define STATE_MARK_SIG    0x2
#define STATE_DSAMPLING   0x4
/** \} */

/**
 * \brief Structure containing decoded CLIP data.
 */
typedef struct
{
  char date[10];    /**< date and time */
  char clid[12];    /**< calling line identity */
  char clname[12];  /**< calling line party name */
  char provid[12];  /**< provider identity */
  uint8_t crc;      /**< crc - have to be zero*/
} s_clipdata;


/*
  function prototypes
*/
int16_t fskd(int16_t sample);
int processMessage(uint8_t* const message, s_clipdata* clipData);
int processSample(int16_t sample, uint8_t* const messageBuff);



#ifdef __cplusplus
}
#endif

#endif
