/**
 * \file fskd.c
 * FSK demodulation and CLIP message decoding.
 * \ingroup fskd
 */


#include <stdint.h>
#include <stdio.h>
#include "fskd.h"



/**
 * Character table for CLIP decoding.
 */
const char fsktab [256] = {
    '\0','\0','\0','\0','\0','\0','\0','\0','\0','\0','\0','\0','\0','\0','\0','\0',
    '\0','\0','\0','\0','\0','\0','\0','\0','\0','\0','\0','\0','\0','\0','\0','\0',
    ' ','!','"','#','$','%','&','\'','(',')','*','+',',','-','.','/',
    '0','1','2','3','4','5','6','7','8','9',':',';','<','=','>','?',
    '@','A','B','C','D','E','F','G','H','I','J','K','L','M','N','O',
    'P','Q','R','S','T','U','V','W','X','Y','Z','[','\\',']','^','_',
    '`','a','b','c','d','e','f','g','h','i','j','k','l','m','n','o',
    'p','q','r','s','t','u','v','w','x','y','z','{','|','}','~','\0'
} ;


/**
 * FSK demodulation.
 *
 * \param sample input sample
 * \return Sample of demodulated signal.
 *
 * Function for continuous FSK demodulation using non-coherent demodulator
 * with cauer filtering. Circular buffers are used to store the delayed samples.
 */
int16_t fskd(int16_t sample)
{
  static int32_t  xBuff[CBUFF_LEN]  = {0}; // kruhove buffery
  static int32_t  yBuff[CBUFF_LEN]  = {0};
  static int32_t  zBuff[CBUFF_LEN]  = {0};
  static int16_t inBuff[CBUFF_LEN]  = {0};
  static uint8_t buffIndex1 = 0;   // index v kruhovym bufferu


  inBuff[buffIndex1] = sample;

  xBuff[buffIndex1] = sample * inBuff[buffInd(buffIndex1,-FSKD_DELAY+1)]; // tohle ted musim filtrovat
  xBuff[buffIndex1] = xBuff[buffIndex1] >> 15;

  // filtruju bikvadama
  zBuff[buffIndex1] = B0a*xBuff[buffIndex1] + B1a*xBuff[buffInd(buffIndex1,-1)] + B2a*xBuff[buffInd(buffIndex1,-2)]
                                            - A1a*zBuff[buffInd(buffIndex1,-1)] - A2a*zBuff[buffInd(buffIndex1,-2)];
  zBuff[buffIndex1] = zBuff[buffIndex1] >> 15;
  yBuff[buffIndex1] = B0b*zBuff[buffIndex1] + B1b*zBuff[buffInd(buffIndex1,-1)] + B2b*zBuff[buffInd(buffIndex1,-2)]
                                            - A1b*yBuff[buffInd(buffIndex1,-1)] - A2b*yBuff[buffInd(buffIndex1,-2)];

  int i;
  for (i = 0; i < 4; i++)
  {
    yBuff[buffIndex1] += coeff_fir[i]*xBuff[buffInd(buffIndex1,-i)];
  }

  yBuff[buffIndex1] = yBuff[buffIndex1] >> 15;


  yBuff[buffIndex1] *= FSKD_POLARITY;

  buffIndex1 = buffInd(buffIndex1,1); // posunu se v kruhovym bufferu

  return yBuff[buffIndex1];
}



/**
 * CLIP message decoding.
 *
 * \param message pointer demodulated message
 * \param ClipData pointer to struct to store decoded values
 * \return 1 when known message type was received, 0 else.
 */
int processMessage(uint8_t* const message, s_clipdata* ClipData)
{

  char* p_text;
  char tempText[16];
  uint8_t i = 0;
  uint8_t j = 0;
  uint8_t crc = 0;
  uint8_t max = 0;
  uint8_t len = 0;

  uint8_t* p_message = message;

  switch (*p_message) {
    case 0x80: // je to vicenasobny prenos dat
      /**/
      break;

    default:
      return 0; // nemam znamej typ dat, koncim chybou
      break;
  }

  crc += *(p_message++);
  crc += *p_message;
  max = *(p_message++); // celkova delka je nasledujici byte

  j = 0;
  while (j < max) {
    switch (*p_message) {
      case 0x01: // datum
        p_text = ClipData->date;
        break;
      case 0x02: // cislo
      case 0x04: // reason for absence
        p_text = ClipData->clid;
        break;
      case 0x07: // jmeno
      case 0x08: // reason for absence
        p_text = ClipData->clname;
        break;
      case 0x30:
        p_text = ClipData->provid;
        break;
      default:
        p_text = tempText; // abych si nic neprepsal ...
        break;
    }

    crc += *(p_message++);      // kontrolni soucet
    crc += *p_message;
    len = *(p_message++);   // delka zpravy je nasledujici byte



    // zkopirujem si text
    for (i = 0; i < len; i++) {
      crc += *p_message;
      *(p_text++) = fsktab[*(p_message++)];
    }
    *p_text = '\0';

    j += len+2; // posouvam se, delka+(data, typ a delka)
  }

  ClipData->crc = crc + *p_message; // dohromady to ma dat nulu

  return 1;

}

/**
 * Input samples processing.
 *
 * \param sample input sample
 * \param messageBuff pointer to buffer for storing demodulated message
 * \return number of sampled bytes
 *
 * Fuction decodes extracts the CLIP message. It uses state machine where
 * the seizure signal is detected (FSK bps and signal mean is computed), mark signal is detected
 * and message data is sampled.
 */
int processSample(int16_t sample, uint8_t* const messageBuff)
{
  int8_t demodq = 0;
  static int8_t lastSample = 0; // posledni vzorek
  static int state = STATE_SEIZURE_SIG; // stavova masinka pri zpracovani
  static int counter = 0;
  static int sampleCounter = 0; // cislo vzorku
  static int32_t sigMean = 0;
  static int16_t thresh = QTHRESH;
  static int lastPos = 0;
  static uint32_t bps;
  int posDiff = 0;
  int tmp = 0;
  int i = 0;
  static int16_t samplePos[10]  = {3, 10, 16, 23, 29, 36, 42, 49, 55, 62};


  demodq = (fskd(sample) > thresh); // kvantovany vzorek
  sampleCounter++;


  switch (state) {
    case STATE_SEIZURE_SIG: // hledame zavadeci sekvenci, chci 10 stridavejch bitu
      if (counter > 0 && sampleCounter <= 1024)
        sigMean += sample;
      if (counter > 1 && sampleCounter == 1024)
        thresh = sigMean >> 10; // pouziju mean pro prahovani, vyrazne to snizuje chybovost!!

      if (demodq - lastSample == -1) { // mam spadovou hranu
        posDiff = sampleCounter - lastPos;
        if (posDiff > 11 && posDiff < 15) { // jsem v rozsahu
          if (counter == 0) {
            sampleCounter = 0;
            sigMean = 0;
          }
          bps += posDiff; // musim spocitat i bps!!!
          counter++;
        }
        else {
          bps = 0;
          counter = 0;
        }

        if (counter == 127) { // mam nejakej pocet stridavejch bitu, spocitam bps a jdu dal
                              // na tomhle cisle to hodne zavisi, asi proc?

          bps = bps << 1; // (bps*256)/counter(128)
          for (i = 0; i < 10; i++)
            samplePos[i] = ((bps*i+128) >> 9) + (bps >> 10); // tohle mi to zaokrouhli a posune do stredu bitu

          state = STATE_MARK_SIG;
          counter = 0;
          sampleCounter = 0; // znova pocitam vzorky
          lastPos = 0;
          bps = 0;
          break;
        }

        lastPos = sampleCounter;
      }


      break;

    case STATE_MARK_SIG: // hledam dlouhej stop bit

      if (lastSample == 1 && demodq == 1) { // jak dlouho je jednicka?
        counter++;
      }
      else if (lastSample == 1 && demodq == 0 && counter > 800) {// spadova hrana po dlouhy jednicce :)
        counter = 0;
        state = STATE_DSAMPLING;
        sampleCounter = 0; // zacinam znova pocitat vzorky
      }
      else if (sampleCounter > 5000) { // uz je to moc dlouhy, zacinam znova
        state = STATE_SEIZURE_SIG;
        counter = 0;
      }
      else
        counter = 0;


      break;

    case STATE_DSAMPLING: // sampluju data

      //if (sampleCounter == samplePos[0]); // start bit
      if (sampleCounter == samplePos[1])
        messageBuff[counter]  = (uint8_t)(demodq << 0);  // pozor, na zacatku nesmim orovat, je to glob. promenna
      else if (sampleCounter == samplePos[2])
        messageBuff[counter] |= (uint8_t)(demodq << 1);
      else if (sampleCounter == samplePos[3])
        messageBuff[counter] |= (uint8_t)(demodq << 2);
      else if (sampleCounter == samplePos[4])
        messageBuff[counter] |= (uint8_t)(demodq << 3);
      else if (sampleCounter == samplePos[5])
        messageBuff[counter] |= (uint8_t)(demodq << 4);
      else if (sampleCounter == samplePos[6])
        messageBuff[counter] |= (uint8_t)(demodq << 5);
      else if (sampleCounter == samplePos[7])
        messageBuff[counter] |= (uint8_t)(demodq << 6);
      else if (sampleCounter == samplePos[8])
        messageBuff[counter] |= (uint8_t)(demodq << 7);
      //else if (sampleCounter == samplePos[9]) // stop bit
        //tmp = 1; //jdem na dalsi bajt

      if ((sampleCounter > samplePos[9]) && (lastSample == 1) && (demodq == 0)) { // mam dalsi spadovou hranu (stop bit)
        counter++;
        sampleCounter = 0; // timhle tady muzu soupat s pozici samplovani
      }
      else if (sampleCounter > 100) { // dlouho nemam stopbit? jedu znova od zacatku
        tmp = counter+1; // tohle budu vracet +1 je crc

        lastSample = 0; // koncim musim reinicializovat staticky promenny
        state = STATE_SEIZURE_SIG; // pocatecni stav
        lastPos = 0;
        counter = 0;
        sampleCounter = 0;
        thresh = QTHRESH;

        return tmp; // vratim si pocet nasamplovanejch bajtu
      }

      break;

    default:
      state = STATE_SEIZURE_SIG;
      break;
  }

  lastSample = demodq; // zapamatuju posledni

  return 0;

}





