/*! @file
 * @brief Limited version of sprintf
 *
 *  FILE        :  _doprint.c \n
 *
 *  DESCRIPTION :  Source file for _doprint() routine \n
 *                 This routine is used to create all different
 *                 formatting routines for _Printf().
 *                 when SMALL is defined, a version without floating
 *                 point is created, and no length specifiers, etc. are
 *                 supported.
 *                 when MEDIUM is defined, a version without floating
 *                 point is created.
 *                 Default, an ANSI version is created.
 *
 *		   Define :                                           \n
 *		   SMALL  - No floats, no length specifiers           \n
 *                 MEDIUM - No floats                         \n
 *                 LARGE  - Full ANSI formatter (default)     \n
 *
 *  COPYRIGHT   :  1995-99 TASKING, Inc.
 *
 */

#include <stdarg.h>
#include <errno.h>
#include <string.h>
#include <stdio.h>
#include <string.h>
#include <limits.h>
#include "cygwin.h"
#undef FRACT_MAX
#include "defines.h"


#if defined(__CYGWIN__)
char *
_ltoa( char *p, unsigned long num, unsigned char radix )
{
	register unsigned int	digit;
	register unsigned long 	newnum;
	register char *		q;
	char *			hold;

	hold = p;
	q = p + 10 - 1 + 2;		/* should correspond to buffer size in _putnumber() in _doprint.c		*/

	*--q = '\0';		/* be sure to terminate the string with a NULL character	*/

	do
	{
		newnum = num / radix;
		digit = num - (newnum * radix);	/* avoid modulo operation */
		if ( digit > 9 )
			digit += 'A'-'0'-10;	/* Convert to hex A-F */
		*--q = digit + '0';
		num = newnum;
	} while ( newnum != 0 );

	while ( (*p++ = *q++) )			/* move the string to the left, including the NULL character	*/
		;

	return hold;
}
#endif

static char * g_dest;
static int  Space_left;

static void
my_put(char ch)
{
	// PR_3("%c(0x%02x)|",ch,ch);
	if (Space_left-- > 0)
		*g_dest++ = ch;
}
/* convert an unsigned int to a string			*/
extern char *_ltoa( char *p, unsigned long num, unsigned char radix );

static void _putstring( int packed, char *str);
static void _putnumber( unsigned char ch, unsigned char format,
			void *pvalue );

/* does the actual printing	*/
static void fmt_print( char * dest, const char *fmt, va_list ap )
{

	unsigned char ch;		/* character in format string	  */
	unsigned char format;		/* b7 - packed format		  */
					/* b6 - left alignment		  */
					/* b5 - zero fill		  */
					/* b4 - alternate print		  */
					/* b3,b2 - 0x00 --> sign if neg   */
					/*  	 - 0x04 --> always sign	  */
					/*       - 0x08 --> sign or space */
					/*       - 0x0C --> (invalid)     */
					/* b1,b0 - 0x01 --> short	  */
					/*       - 0x02 --> long	  */
					/*       - 0x03 --> double	  */
					/* specifier is found */

	int exit;
	g_dest = dest;
	// PR_2("MY_SP:fmt=<%s>\n",fmt);

	for( ;; )
	{
		exit = 0;
		do
		{
			ch = *fmt++;
			switch( ch )
			{
			case '%':
				if(*fmt != '%')
				{
					exit = 1;
					break;
				}
				else
					fmt++;
				/* no break */
			default:
				my_put( ch);
				break;
			case '\0':
				return;
			}
		}
		while( !exit );

		format = 0;

		exit = 0;
		do
		{
			ch = *fmt++;
			switch( ch )
			{
			case '-':
				format |= 0x40;	/* left alignment */
				break;
			case '+':
				format &= ~0x08;
				format |=  0x04;
				break;
			case ' ':
				/* do not overrule '+'	*/
				if( !(format & 0x04) )
					format |= 0x08;	/* sign or space */
				break;
			case '0':
				format |= 0x20;	/* zero fill */
				break;
			case '#':
				format |= 0x10;	/* alternate print */
				break;
			default:
				exit = 1;
				break;
			}
		}
		while( !exit );

		// ch = *fmt++;

		switch(ch)
		{
		case 'l':
			format |= 0x02;
			ch = *fmt++;
			break;
		case 'h':
			format |= 0x01;
			ch = *fmt++;
			break;
		case 'L':
			format |= 0x03;
			ch = *fmt++;
			break;
		default:
			break;
		}

		switch( ch )
		{
		case 'p':
			ch = 'x';
				/* as the memory space pointed to is unknown here,
				   we do not print it */
			/* no break */
		case 'c':	/* character			*/
		case 'd':	/* signed decimal integer	*/
		case 'i':
		case 'o':	/* unsigned octal int		*/
		case 'x':
		case 'X':	/* unsigned hexadecimal int	*/
		case 'u':	/* unsigned decimal int		*/
			        /* int, single character	*/
#if defined(__CYGWIN__)
			{
				long tmp;
				if ((format & 0x03) == 0x02) tmp = va_arg( ap, long );
				else	tmp = va_arg( ap, int );
				// PR_2("MYSP: tmp = %lu\n",tmp);
				_putnumber( ch, format, (void *)&tmp);

			}
#else
	//! @todo PEH / TOMVAL /MEGHOLM : compile _putnumber under C++
   #ifndef __cplusplus
			_putnumber(
				ch, format,
				(format & 0x03) == 0x02 ?  (void *)&va_arg( ap, long ) : (void *)&va_arg( ap, int ));
   #endif // __cplusplus
#endif
			break;
		case 'S':	/* packed string */
			format |= 0x80;
			/* no break */
		case 's':	/* string */
			_putstring( ch == 'S', va_arg( ap, char * ));
			break;
		case 'f':
		case 'e':
		case 'E':
		case 'g':
		case 'G':	/* floating point	*/

			_putstring( 0, "<no floats>");
			/* select next argument	*/
			va_arg( ap, double );
			break;
		default:
#if defined(__CYGWIN__)
			PR_3("\nMY_SP: Illegal format string:%c(0x%02x)\n",ch,ch);
#else
			errno = ERR_FORMAT;	/* illegal format string */
#endif
			return;
		}
	}
	// PR_1("\n");
}

/* prints a string to the output					  */
/* when 'min' > 'max', the result is undefined				  */
/* when 'left' != 0, left alignment is done				  */
/* when min=0, and max=0, strings are printed normal			  */
/* length of the strings are supposed to be smaller than INT_MAX chars    */
static void
_putstring( int packed, char *str)
{	char ch;  size_t count;

	for( count = 0;  ;  count++ )
	{
#if defined(__CYGWIN__)
		ch = *str++;
#else
		ch = packed ? _pstr_get((_packed char*)str, count) : *str++;
#endif
		if(!ch)  break;
		my_put( ch);
	}
}

static void
_putnumber(	unsigned char ch, unsigned char format, void *pvalue)
{
#if defined(__CYGWIN__)
	char buf[35];	/* room to print long value in	*/
#else
	char buf[LONG_DIGITS+3];	/* room to print long value in	*/
#endif
	char *buffer;			/* help variable		*/
	long value;
	char vsign;			/* sign of the value		*/


	if( (format & 0x02) )		/* 'l' ('L' not possible here)  */
	{
		value = *(long *)pvalue;
	}
	else
	{			/* short value ('h') is the same as int	*/
		value = *(int *)pvalue;
		if( ch != 'd' && ch != 'i' )	/* not a signed integer ? */
		{
			value &= UINT_MAX;	/* truncate to positive   */
		}				/* integer */
	}

	vsign = 0;	/* 0 - no sign to print, else char to print */

	switch( ch )
	{
	case 'c':
		buf[0] = value;
		buf[1] = '\0';
		break;
	case 'd':
	case 'i':
		if( value < 0 )
		{
			vsign = '-';
			if( value != LONG_MIN ) /* avoid signed overflow */
			{
				value = -value;
			}
		}
		/* no break */
	case 'u':
		_ltoa( buf, value, 10 );	/* convert to ascii	*/
		break;
	case 'o':
	case 'x':
	case 'X':
		_ltoa( buf, value, ch == 'o' ? 8 : 16 );

		if( ch == 'x' )		/* convert string to lowercase	*/
		{
			for( buffer = buf; *buffer; ++buffer )
			{	/* do not use 'tolower' */
				if( *buffer >= 'A' && *buffer <= 'Z' )
					*buffer += 'a' - 'A';
			}
		}
		break;
	}

	if( vsign )				/* sign character	*/
	{
		my_put( vsign);
	}

	/* now print the ascii value	*/
	_putstring( 0, buf);
}

void my_sprint( char *s, const char *format, ... )
{
	va_list		ap;
	Space_left = 10000; // Max length
	va_start( ap, format );
	fmt_print( s, format, ap );
	my_put('\0');
	va_end( ap );
}

void my_snprint( char *s, int Length, const char *format, ... )
{
	va_list		ap;
	Space_left = Length - 1; // Max length
	va_start( ap, format );
	fmt_print( s, format, ap );
	my_put('\0');
	va_end( ap );
}


extern char send_buf[MAX_SEND];

/*!
 * The request is copyed to @ref send_buf
 *
 * output : @ref send_buf
 */
void my_sprint_send_buf_4( const char *format, ... )
{
	char *s = &send_buf[4];
	va_list		ap;
	Space_left = sizeof(send_buf) - 6; // Max length
	va_start( ap, format );
	fmt_print( s, format, ap );
	my_put('\0');
	va_end( ap );
}


void my_sprint_send_buf_4_1( const int *int1 )
{
	my_sprint_send_buf_4("%d ", *int1 );
}


void my_sprint_send_buf_4_2( const int *int1, const int *int2 )
{
	my_sprint_send_buf_4("%d %d ", *int1, *int2);
}


void my_sprint_send_buf_4_3( const int *int1, const int *int2, const int *int3  )
{
	my_sprint_send_buf_4("%d %d %d ", *int1, *int2, *int3);
}

