/*-----------------------------------------------------------
 *                   COPYRIGHT (C) 2009.
 *          GN Otometrics. All rights reserved.
 *-----------------------------------------------------------
 *            P R O P R I E T A R Y   R I G H T S
 * This document must not be used, nor copied in any form or
 * handed over to third party, without prior written
 * permission from GN Otometrics.
 *-----------------------------------------------------------
 *
 * This source file is part of the 1081 Aurical Aud project.
 *
 * P4 Info: (P4-filetype must be "ktext" for these to work!)
 * Last edit by: $Author: pohansen $
 * Date:         $DateTime: 2010/12/20 13:07:10 $
 * Filename:     $File: //depot/1081/fw/head/dsp/main/src/appl/globaltypes.h $
 * Revision:     $Revision: #36 $
 *
 * File Info:
 * Original author: PEH
 * Creation date: 2009-03-02
 *
 * Doxygen Info: *//*!
 * @file
 * @brief Types used globally
 *
 * *//*
 *-----------------------------------------------------------*/


#ifndef _GLOBALSTYPES_H_
#define _GLOBALSTYPES_H_

//! Audiometer mode or REM mode
typedef enum { NONE_MODE = 0, AUD_MODE = 1, REM_MODE = 2 } e_module;


//! @brief   REM mode signal path mode
typedef enum {
	REM_FF_3_ROUTED_TO_CH4 = 0, REM_FF_1_2_ROUTED_TO_CH4_5 = 1
} e_rem_ff_mode;


//! @brief   Use of FF3
typedef enum
{
	FF3_MONITOR_TRANSDUCER,				//!< FF3 is used as operator stimulus monitor (sound via AD8)
	FF3_PATIENT_TRANSDUCER_BY_AD2,		//!< FF3 is used as patient stimulus transducer (AUD and REM mode) (sound via AD2 (SD_L1))
	FF3_PATIENT_TRANSDUCER_BY_AD3		//!< FF3 is used as patient stimulus transducer (AUD ) (sound via AD2 (SD_R1))
} e_FF3_transducer_function;


/*!  @brief   sig_type
 *
 * @warning Used for command parsing, don't change, only add new items at the end
 */
typedef enum {
	ST_SPEECH     = 0,		//!<	use 0 for external (non generated/speech) input
	ST_SPN        = 1,		//!<	1 Speech noise
	ST_WBN        = 2,		//!<	2 White band noise
	ST_NBNM       = 3,		//!<	3 Narrow band noise mask
	OLD_ST_FRESH  = 4,		//!<	4 Narrow band noise stimuli (Used to check unlegeal input parameter)
	ST_PTONE      = 5,		//!<	5 Puretone
	ST_WRB        = 6,		//!<	6 Warble tone
	ST_PTO_PULSED = 7,		//!<	7 Puretone pulsed
	ST_WRB_PULSED = 8,		//!<	8 Warble tone pulsed
	// OLD_ST_FRESH_PULSED,	//  	9 Narrow band noise stimuli pulsed
	// ST_CALIB,			//  	10 Calibration noise
	ST_TALKOVER = 11		//!<	11 Not a valid input, but used when signal mixing. Will have the enum "e_SRC" set to NO_SRC.
} e_sig_type;


/*! CHx_source_id
 *
 * @warning Used for command parsing, don't change, only add new items at the end
 */
typedef enum {
	NO_SRC=0,		// 0 At the moment only used when ST_TALKOVER chosen in the enum e_sig_type
	DSG1_SRC,		// 1
	DSG2_SRC,		// 2
	OPE_MIC_SRC,	// 3
	EXTA_SRC,		// 4
	EXTB_SRC,		// 5
	AUD0_SRC,		// 6
	AUD1_SRC,		// 7
	AUD2_SRC,		// 8
	AUD3_SRC,		// 9
	AUD4_SRC,		// 10
	AUD5_SRC		// 11
} e_SRC;


/*! @brief   Physical sockets as described internal in code
 *
 * @warning Values in  @ref e_phy_soc must be equal to values in @ref e_transducer_ret_calib_save_index
 */
typedef enum
{
	SOCKET_NONE   =  -1,//!< 0	Left signal is not routed anywhere
	SOCKET_ACLF_L =  0,	//!< 0	Air Conduction Low Frequency Left socket
	SOCKET_ACLF_R =  1,	//!< 1	Air Conduction Low Frequency Right socket

	SOCKET_ACIN_L =  2,	//!< 2	Air Conduction Insert phone Left socket (left stereo and mono)
	SOCKET_ACIN_R =  3,	//!< 3	Air Conduction Insert phone Right socket
	SOCKET_BONE   =  4,	//!< 4	Bone Conduction Mastoid socket
	SOCKET_FF_N1  =  5,	//!< 5	Free Field Number 1 (Left)
	SOCKET_FF_N2  =  6,	//!< 6 	Free Field Number 2 (Right)
	SOCKET_FF_N3  =  7	//!< 7 	Free Field Number 3
} e_phy_soc;



/*! @brief Save index used for saving RETSPLs
 *
 * @warning Values in  @ref e_phy_soc must be equal to values in @ref e_transducer_ret_calib_save_index
 */
typedef enum
{
	SAVE_RET_NONE   =  -1,//!< 0	Left signal is not routed anywhere
	SAVE_RET_ACLF   =  0,	//!< 0	Air Conduction Low Frequency Left socket
	SAVE_RET_ACIN   =  1,	//!< 1	Air Conduction Insert phone Left socket (left stereo and mono)
	SAVE_RET_BONE   =  2,	//!< 2	Bone Conduction socket
	SAVE_RET_FF_N1  =  3,	//!< 3	Free Field Number 1 (Left)
	SAVE_RET_FF_N2  =  4,	//!< 4 	Free Field Number 2 (Right)
	SAVE_RET_FF_N3  =  5,	//!< 5 	Free Field Number 3

	SAVE_RET_MASK_OFFS = 6    //!< 6   Not a real transducer, used in in NCH to offset between SPL and level to make mask

} e_transducer_ret_calib_save_index;


/*! @brief Save index used for saving transducer calibrations
 *
 * @warning Values in  @ref e_phy_soc must be equal to values in @ref e_transducer_ret_calib_save_index
 */
typedef enum e_transducer_calib_save_index
{
	SAVE_CAL_NONE   =  -1,	//!< 0	Left signal is not routed anywhere
	SAVE_CAL_ACLF_L =  0,	//!< 0	Air Conduction Low Frequency Left socket
	SAVE_CAL_ACLF_R =  1,	//!< 1	Air Conduction Low Frequency Right socket
	SAVE_CAL_ACIN_L =  2,	//!< 2	Air Conduction Insert phone Left socket (left stereo and mono)
	SAVE_CAL_ACIN_R =  3,	//!< 3	Air Conduction Insert phone Right socket
	SAVE_CAL_BONE   =  4,	//!< 4	Bone Conduction socket
	SAVE_CAL_FF_N1  =  5,	//!< 5	Free Field Number 1 (Left)
	SAVE_CAL_FF_N2  =  6,	//!< 6 	Free Field Number 2 (Right)
	SAVE_CAL_FF_N3  =  7,	//!< 7 	Free Field Number 3
	SAVE_CAL_MASK   =  8	//!< 8 	Not a transducer

} e_transducer_calib_save_index;


/*!  @brief   Channel transducers
 *
 * @warning Used for command parsing, don't change, only add new items at the end
 */
typedef enum e_socket_type
{
	NONE_TYPE = -1, 	//!<	-1 Valid for CH2
	ACLF_TYPE,		//!<	0 Low Frequency stereo sockets (ACLF_R + ACLF_L)
	ACIN_TYPE,		//!<	1 Insert phone stereo sockets (ACIN_R + ACIN_L)
	BC_TYPE,		//!<	2 Bone Conduction socket
	FF_TYPE			//!<	3 Free Field speaker socket - 2 speaker setup (FF_R_S,FF_L_S)

} e_socket_type;


typedef enum
{
	TR_NONE 	= -1,
	TR_HDA200 	=  0,
	TR_TDH39	=  1,
	TR_PD95		=  2,
	TR_ER3A		=  3,
	TR_OTOIEP	=  4,
	TR_B71_MAST	=  5,
	TR_ME70		=  6,	// If adding new ones here - remember updating cmd_int.c's syntax check for "NST"
	TR_FF		=  7
} t_transducer;

typedef enum
{
	eq_NONE = -1,	//!< -1 No equalizer
	eq_BB 	=  0,	//!<  0 Broadband equalizer
	eq_BBFF =  1,	//!<  1 Broadband FreeField equivalent equalizer
	eq_NB 	=  2	//!<  2 Narrowband equalizer

} e_equalizer_type;


/*!  @brief   Multi speaker transducers "bits"
 *
 * @warning Used for command parsing, don't change, only add new items at the end
 */
enum routing_single_free_field {
       No_ff_Select = 0,
       bFF_S1 =     0x01,
       bFF_S2 =     0x02,
       bFF_S3 =     0x04,
       bFF_S4 =     0x08 };


/*!  @brief   CH_route_id (cmd = RLR)
 *
 * @warning Used for command parsing, don't change, only add new items at the end
 */
typedef enum e_Route
{
	ROUTE_LEFT = 1,
	ROUTE_RIGHT= 2,
	ROUTE_BOTH = 3 // (ROUTE_LEFT|ROUTE_RIGHT)
} e_Route;



/*!  @brief   level_control_id
 *
 * Value 0..5  : Output level            v gain_lev index \n
 * Value 6..12 : Input gain
 *
 * @warning Used for command parsing, don't change, only add new items at the end
 */
enum e_lc_id {
// Output level (abs)       gain_level index
	CH1_LEVEL = 0,	//!<	0    Channel 1 HL dB level wanted
	CH2_LEVEL,		//!<	1    Channel 2 HL dB level wanted
	OPE_OUT_LEVEL,  //!<	2    Operator Mic. output level (Talkover| mixed)
// Output level (rel)
	MON_CH1_LEVEL,	//!<	3  0 Monitor channel 1 output control
	MON_CH2_LEVEL,	//!<	4  1 Monitor channel 2 output control
	PAT_RESP_LEVEL,	//!<	5  2 Patient response monitor output signal level
// Input gain
	EXTA_LEVEL,		//!<	6  3 Ext A (CD line in) input gain setting
	EXTB_LEVEL,		//!<	7  4 Ext B (CD line in) input gain setting
	AUDA_LEVEL,		//!<	8  5 C Audio stream A input gain setting
	AUDB_LEVEL,		//!<	9  6 C Audio stream B input gain setting
	OPE_DMIC_LEVEL,	//!<	10 7 Operator desktop microphone input signal gain
	OLD_OPE_HMIC_LEVEL,//!<	11 8 Operator headset microphone input signal gain
	PAT_MIC_LEVEL,	//!<	12 9 Patient microphone input signal gain
	OLD_ASS_OPERATOR_VOLUME, //!<	13 10 Assistemt headphone volume for operator mic.
	OLD_ASS_MONITOR_VOLUME	//!<	14 11 Assistemt headphone volume for monitor signal
};


/*!  @brief   SISI special test setup
 *
 * @warning Used for command parsing, don't change, only add new items at the end
 */
typedef struct SISI_setup_struct
{
	int		Modulation;	//!<	Modulation in 0.1 dB units
	int		Time_on;	//!<	time in ms
	int		Time_off;	//!<	time in ms

} SISI_setup_struct;



/*!  @brief   Talkover mode
 */
typedef enum { NO_TALKOVER, CLASIC_TALKOVER, MIX_TALKOVER } Talkover_mode;


/*!  @brief   Special test state
 *
 * @warning Used for command parsing, don't change, only add new items at the end
 */
typedef enum {
		SPEC_STOP = 0,
		SPEC_PAUSE,
		SPEC_RUNNING
} e_spec_tst_running;


/*!  @brief   Special test setup
 *
 * @warning Used for command parsing, don't change, only add new items at the end
 */
enum e_Audiometer_test_mode {
	ATM_MAN_THR	= 1,	//!<	1 : Manual Threshold
	ATM_SPEECH,			//!<	2 :	Speech
	ATM_SISI,			//!<	3 :	Sisi
	ATM_MHAHIS,			//!<	4 : Master Hearing Aid / Hearing instrument simulation test
	ATM_ABLB,			//!<	5 :	Ablb
	ATM_STENTER,		//!<	6 :	Stenger
	ATM_AUTO_THR,		//!<	7 :	Automatic Threshold
	ATM_AUTO_SCREEN,	//!<	8 :	Automatic  screening
	ATM_TONE_UCL,		//!<	9 :	UCL
	ATM_TONE_MCL,		//!<	10 : MCL
	ATM_MLD,			//!<	11 : MLD
	ATM_DLI				//!<	12 : DLI / Lcher
};


typedef enum
{
	UnknownMonitor = -1,
	NoneMonitor = 0,
	HeadsetMonitor = 1,
	DisktopMonitor = 2,
	Counseling_and_Operator = 3
} e_Monitor_system;


enum
{
	ABS_OUTPUT_LEVELS = OPE_OUT_LEVEL+1,
	LEV_RELATIVE = MON_CH1_LEVEL,							//!< Level "ID" is not at abs. sound level, but a reletive value

	GAIN_LEVEL_SIZE = (OLD_ASS_MONITOR_VOLUME - LEV_RELATIVE) + 1   	//!< Array size
};


typedef struct t_max_level
{
	int	 hl[2];				//!< Maximal hearing Level
	int	 talkforward[2];	//!< Maximal hearing Level	LEFT, RIGHT

	int	 current_hl[2];		//!< Current hearing Level at transducer
	int	 da0_max_hl[2];		//!< Maximal hearing Level if 0 dB was valid
} t_max_level;

/*!  @brief   This structure contains all setings set by interface commands in normal mode
 *
 */
typedef struct cmd_stim_setup_type
{
	e_sig_type 				sig_type[2];					//!< Signal type (cmd_parm.h)
	e_SRC					sig_src[2]; 					//!< Signal source selction (RS1/RS2)
	unsigned int 			gen_freq[2];					//!< Generator frequency in Hz
	int 					ch_lev[ABS_OUTPUT_LEVELS];		//!< Level control (CH1,CH2 and MIC out for TOK_) (value requested by PC
	e_Route					route_lr[2];					//!< Route L/R/L+R for ch1/ch2
	e_socket_type			transd[2];						//!< Transducer ch1/ch2
	BOOL					Correlated_mask[FF_TYPE+1];		//!< For each type of transducer, use correlated or non-correlated mask noise
	int 					wrb_freq;						//!< wrb_freq Hz [1;20];
	int 					wrb_mod;						//!< wrb_modulation in % [1;25]
	e_spec_tst_running		spec_running;					//!< Special mode stop running pause
	SISI_setup_struct		SISI;
	int 					alt_time;						//!< Alternating time
	Talkover_mode			Talkover;

	/*! index :                                                                               \n
	 * [0] Monitor_ch1_output_level, [1] Monitor_ch2_output_level,                            \n
	 * [2] Patient_response_signal_output_level,                                              \n
	 * [3] Ext_A_input_signal_gain, [4] Ext_B_input_signal_gain,                              \n
	 * [5] PC_Audio_stream_A_input_gain, [6] PC_Audio_stream_B_input_gain                     \n
	 * [7] Operator_Desktop_microphone_input_gain, [8] Operator_Headset_microphone_input_gain \n
	 * [9] Patient_microphone_input_gain
	 */
	int  gain_lev[GAIN_LEVEL_SIZE];

	BOOL stim_On[2];			//!< Ch1 = [0],  Ch2 = [1]. stim on /off
	BOOL Talkover_ass : 1;		//!< Talkover to ass. on /off
	BOOL Patient_mic  : 1;		//!< Patient_mic on /off

	BOOL Monitor      : 1;		//!< Monitor system On/Off
	BOOL Monitor_headset : 1;	//!< Monitor headset signal On/Off
	BOOL Monitor_speaker : 1;	//!< Monitor speaker signal On/Off

	BOOL Setup_Error     : 1;	//!< Not a valid setup combination
	BOOL Insert_force_mono : 1;	//!< Force insert tranducers to mono
	BOOL amp_modulation : 1;	//!< Amplitude modulation, (SISI or DLI)

	BOOL MLD_right_revers[2];	//!< MLD test mode : revers phase in right transducer
	BOOL Monitor_Ch[2];			//!< Monitor CH1 / CH2 signal On/Off

	enum e_Audiometer_test_mode    Audiometer_test_mode;
	e_module selected_module;			//!< AUD / REM mode
	e_rem_ff_mode rem_ff_mode;			//!< REM mode free field modes (arg from rrp cmd)

	int  pulse_on_duration;				//!<
	int  pulse_off_duration;			//!<

	int calib_rel_version;				//!< Relative version of the calibration data. Increment to force new evaluation.
	BOOL hw_init_force : 1;				//!< ReInit hardware
	BOOL level_lock_AKRX_reply : 1;		//!< Use stim_lock AKRX reply
	BOOL sound_card_mode : 1;			//!< Sound chard mode
	BOOL pediatric_talkback : 1;		//!< Pediatric talkback mode
	BOOL talkback_gate_enable : 1;		//!< Enable noise gate on talkback microphone
	BOOL talkback_lp_filer_enable : 1;	//!< Enable 4.5kHz LP filter on talkback microphone
	BOOL tone_on_min[2];				//!< Stim on minimum time
	int  proess_id[2];					//!< Application program process ID

} cmd_stim_setup_type;


//! Goald value for @ref common_setup .in_test_mode and @ref want_test_mode
#define TEST_MODE_VALUE 0x55AA

/*! @brief   setup / state commen for AUD & REM
 *
 */
typedef struct
{
	BOOL          HeartBeatMute      : 1;	//!< Mute by missing heartbeat
	BOOL          CoolDownMute       : 1;	//!< Mute by Output power extended
	BOOL          in_calib_mode      : 1;	//!< Calibration mode
	BOOL          AUD_module_enable  : 1;	//!< AUD module select able
	BOOL          REM_module_enable  : 1;	//!< REM module select able
	unsigned int  in_test_mode   : 16;	//!< Test-mode

} t_cmd_common_setup;

/*!  @brief   This structure contains hardware version, and eeprom layout version
 *
 */
typedef struct
{
    unsigned DataFmtVer		:8;	//!< @ref eDataFmtVer
    unsigned HardwareVer	:8; //!< @ref eHardwareVer
    unsigned equement_id	:2; //!< Must be "1" for a SPARC T2
    unsigned Reserved		:6;

    unsigned int CRC24_checksum;

} t_EEprom_header;


e_transducer_calib_save_index convert_socket_2_calib_enum( e_phy_soc socket );
e_transducer_ret_calib_save_index convert_socket_2_retspl_calib_enum( e_phy_soc socket );
e_phy_soc                         convert_e_transducer_ret_calib_save_index_2_e_phy_soc(e_transducer_ret_calib_save_index index);
e_socket_type                     convert_phy_soc_2_socket_type(e_phy_soc index);
e_phy_soc                         map_e_ch_transducer_2_e_phy_soc_id(const enum e_Route route_lr, const e_socket_type socket);

// e_transducer_calib_save_index convert_sockettype_2_calib_enum(e_socket_type socket_type);

#endif // _GLOBALSTYPES_H_


