/*
     NanoStore.h
     NanoStore

     Copyright (c) 2010 Webbo, L.L.C. All rights reserved.
     
     Redistribution and use in source and binary forms, with or without modification, are permitted
     provided that the following conditions are met:
     
     * Redistributions of source code must retain the above copyright notice, this list of conditions
     and the following disclaimer.
     * Redistributions in binary form must reproduce the above copyright notice, this list of conditions
     and the following disclaimer in the documentation and/or other materials provided with the distribution.
     * Neither the name of Webbo nor the names of its contributors may be used to endorse or promote
     products derived from this software without specific prior written permission.
     
     THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
     WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
     PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY
     DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
     PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
     CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
     OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
     SUCH DAMAGE.
*/

#import "NSFNanoObjectProtocol.h"
#import "NSFNanoObject.h"
#import "NSFNanoGlobals.h"
#import "NSFNanoStore.h"
#import "NSFNanoPredicate.h"
#import "NSFNanoExpression.h"
#import "NSFNanoSearch.h"
#import "NSFNanoResult.h"
#import "NSFNanoBag.h"
#import "NSFNanoEngine.h"
#import "NSFNanoGlobals.h"

/**
 @mainpage Welcome To NanoStore
 
 @section whatis_sec What is NanoStore?
 
 NanoStore is an open source, lightweight schema-less local key-value document store written in Objective-C for Mac OS X and iOS.
 
 Relational databases tend to have a rich understanding of the structure of your data, but requires some planing beforehand and some level of
 maintenance as well. NanoStore provides the flexibility that comes with key-value document stores, but still understands something about your data.
 Because the data is key-value based, it can be accessed quickly and can grow as much as needed... all without ever worrying about the schema.
 
 @section mainadv_sec Main advantages
 
 - No SQL knowledge required
 - Schema-less
 - Key-value based storage
 - Store your own custom objects
 - Bags, a free-form relational system
 - Dynamic queries
 - Full index support, inner-objects, embedded arrays and dictionaries
 - Convenience methods to access, manipulate and maintain SQLite databases
 - Full SQLite access available
 - Fast, in-place updates
 
 @section howitworks_sec How does it work?
 
 The basic unit of data in NanoStore is called NanoObject, which includes any type of object that conforms to the NSFNanoObjectProtocol protocol.
 
 <b>Structure of a NanoObject object</b>
 
 In its basic form, a NanoObject is nothing more than a wrapper around two properties:
 
 - A dictionary which contains the metadata (provided by the developer)
 - A key (UUID) that identifies the object (provided by NanoStore)
 
 The dictionary <i>must</i> be serializable, which means that only the following data types are allowed:
 
 - NSArray
 - NSDictionary
 - NSString
 - NSData (*)
 - NSDate
 - NSNumber
 
 (*) The data type NSData is allowed, but it will be excluded from the indexing process.
 
 To save and retrieve objects from the document store, NanoStore moves the data around by encapsulating it in NanoObjects. In order to store the objects in
 NanoStore the developer has three options:
 
 - Use the NSFNanoObject class directly
 - Expand the custom classes by inheriting from NSFNanoObject
 - Expand the custom classes by implementing the NSFNanoObjectProtocol protocol
 
 Regardless of the system used, NanoStore will be able to store and retrieve objects from the document store seamlessly. The beauty of this system is that
 NanoStore returns the object as it was stored, that is, instantiating an object of the class that was originally stored.
 
 @note
 If the document store is opened by another application that doesn't implement the object that was stored, NanoStore will instantiate a
 NSFNanoObject instead, thus allowing the app to retrieve the data seamlessly. If the object is then updated by this application, the original
 class name will be honored.
 
 <b>Example:</b>
 
 - App A stores an object of class <i>Car</i>.
 - App B retrieves the object, but since it doesn't know anything about the class <i>Car</i>, NanoStore returns a <i>NSFNanoObject</i>.
 - App B updates the object, perhaps adding a timestamp or additional information. NanoStore saves it as a <i>Car</i>, not as a <i>NSFNanoObject</i>.
 - App A retrieves the updated object as a <i>Car</i> object, in exactly the same format as it was originally stored.
 
 @section workingwithnanoobject_sec Working with a NanoObject

 There are three basic operations that NanoStore can perform with a NanoObject:
 
 - Add it to the document store
 - Update an existing object in the document store
 - Remove it from the document store
 
 To add an object, instantiate a NanoObject providing a dictionary and add it to the document store.
 
 @details <b>Example:</b>
 @code
 // Instantiate and open a NanoStore
 NSFNanoStore *nanoStore = [NSFNanoStore createAndOpenStoreWithType:NSFMemoryStoreType path:nil error:nil];
 
 // Assuming the dictionary exists, instantiate a NanoObject
 NSDictionary *info = ...;
 NSFNanoObject *object = [NSFNanoObject nanoObjectWithDictionary:info];
 
 // Add the NanoObject to the document store
 [nanoStore addObject:object error:nil];
 @endcode
 
 NanoStore will assign a UUID automatically when the NanoObject is instantiated. This means that requesting the key from the NanoObject will return
 a valid UUID. The same holds true for objects that inherit from NSFNanoObject. However, classes that implement the NSFNanoObjectProtocol protocol should
 make sure they return a valid key via \ref NSFNanoObjectProtocol::nanoObjectKey "- (NSString *)nanoObjectKey."
 
 @warning
 If an attempt is made to add or remove an object without a valid key, an exception of type \ref NSFGlobals::NSFNanoObjectBehaviorException
 "NSFNanoObjectBehaviorException" will be raised.
 
 To update an object, simply modify the object and add it to the document store. NanoStore will replace the existing object with the one being added.
 
 @details <b>Example:</b>
 @code
 // Instantiate and open a NanoStore
 NSFNanoStore *nanoStore = [NSFNanoStore createAndOpenStoreWithType:NSFMemoryStoreType path:nil error:nil];
 
 // Assuming the dictionary exists, instantiate a NanoObject
 NSDictionary *info = ...;
 NSFNanoObject *object = [NSFNanoObject nanoObjectWithDictionary:info];
 
 // Add the NanoObject to the document store
 [nanoStore addObject:object error:nil];
 
 // Obtain the dictionary and update the NanoObject
 NSMutableDictionary *updatedInfo = [[[object nanoObjectDictionaryRepresentation]mutableCopy]autorelease];
 [updatedInfo setObject:@"foo" forKey:@"SomeKey"];
 [object setInfo:updatedInfo];
 
 // Update the NanoObject in the document store
 [nanoStore addObject:object error:nil];
 @endcode
 
 To remove an object, there are several options available. The most common methods are found in NSFNanoStore:
 
 - (BOOL)removeObject:(id <NSFNanoObjectProtocol>)theObject error:(out NSError **)outError;
 - (BOOL)removeObjectsWithKeysInArray:(NSArray *)theKeys error:(out NSError **)outError;
 - (BOOL)removeObjectsInArray:(NSArray *)theObjects error:(out NSError **)outError;

 @details <b>Example:</b>
 @code
 // Instantiate and open a NanoStore
 NSFNanoStore *nanoStore = [NSFNanoStore createAndOpenStoreWithType:NSFMemoryStoreType path:nil error:nil];
 
 // Assuming the dictionary exists, instantiate a NanoObject
 NSDictionary *info = ...;
 NSFNanoObject *object = [NSFNanoObject nanoObjectWithDictionary:info];
 
 // Add the NanoObject to the document store
 [nanoStore addObject:object error:nil];
 
 // Remove the object
 [nanoStore removeObject:object error:nil];
 
 // ... or you could pass the key instead
 [nanoStore removeObjectsWithKeysInArray:[NSArray arrayWithObjects:[object nanoObjectKey], nil] error:nil];
 @endcode
 
 @section notaflatworld_sec It's not a flat World
 
 Most database solutions force the developer to think in a two-dimensional space (rows and columns), forcing the developer to plan the schema ahead of
 time. This situation is not ideal because in most cases schema refinements could be required, oftentimes impacting the code as well.
 
 NanoStore goes beyond that allowing the developer to store objects in their natural form. These objects must conform to the NSFNanoObjectProtocol
 protocol, providing NanoStore with the NSDictionary that will be stored. By using a dictionary data can be inspected very quickly, and it also allows the
 structure to be defined in a hierarchical fashion as well, due to the fact that it includes support for nested collections (of type NSDictionary and NSArray.)
 Each inner-object is indexed automatically, thus allowing to quickly find objects which contain a specific key and/or value.
 
 By default, NanoStore allows objects to be stored without any sense of relationship to other objects. This simple format, while powerful, is limited because
 the developer has to keep track of the relationships among objects. Some applications may need to relate objects, some of them perhaps of different nature or class
 type. This is exactly what NanoBag (represented by the NSFNanoBag class) does: it allows any object conforming to the NSFNanoObjectProtocol protocol to be
 added to the bag. By saving the bag with one single call, the new and/or modified are taken care of seamlessly.
 
 The NSFNanoBag API is rich, allowing the developer to add, remove, reload and undo its changes, deflate it (thus saving memory) and inflate it whenever it's
 required. In addition, it provides methods to obtain all bags, specific bags matching some keys, and bags containing a specific object
 (see NSFNanoStore for more information).
 
 @section wherearetheobjects_sec Where are my objects?
 
 While NSFNanoStore provides some convenience methods to obtain standard objects such as bags, the bulk of the search mechanism is handled by NSFNanoSearch.
 The steps involved to perform a search are quite simple:
 
 - 1) Instantiate a search object
 - 2) Configure the search via its accessors
 - 3) Obtain the results specifying whether objects or keys should be returned (*)
 
 (*) If introspecting the data is needed, request objects. You should request keys if you need to feed the result to another method, such as NSFNanoStore's
 <i>(BOOL)removeObjectsWithKeysInArray:(NSArray *)theKeys error:(out NSError **)outError</i> method.
 
 @details <b>Example: finding all objects with the attribute 'LastName' and value 'Doe'.</b>
 @code
 NSFNanoSearch *search = [NSFNanoSearch searchWithStore:nanoStore];
 
 search.attribute = @"LastName";
 search.match = NSFEqualTo;
 search.value = @"Doe";
 
 // Returns a dictionary with the UUID of the object (key) and the NanoObject (value).
 NSDictionary *searchResults = [search searchObjectsWithReturnType:NSFReturnObjects error:nil];
 @endcode
 
 @details <b>Example: removing all objects with the attribute 'LastName' and value 'Doe'.</b>
 @code
 NSFNanoSearch *search = [NSFNanoSearch searchWithStore:nanoStore];
 
 search.attribute = @"LastName";
 search.match = NSFEqualTo;
 search.value = @"Doe";
 
 // Returns an array of matching UUIDs
 NSArray *matchingKeys = [search searchObjectsWithReturnType:NSFReturnKeys error:nil];
 
 // Remove the NanoObjects matching the selected UUIDs
 NSError *outError = nil;
 if (YES == [nanoStore removeObjectsWithKeysInArray:matchingKeys error:&outError]) {
    NSLog(@"The matching objects have been removed.");
 } else {
    NSLog(@"An error has occurred while removing the matching objects. Reason: %@", [outError localizedDescription]);
 }
 @endcode
 
 Another cool feature is the possibility to invoke aggregated functions (count, avg, min, max and total) on the search results. Using the search snippet above,
 calculating the average salary of all people with last name equal to 'Doe' is very easy.
 
 @details <b>Example: calculating the average salary of all objects with the attribute 'LastName' and value 'Doe'.</b>
 @code
 NSFNanoSearch *search = [NSFNanoSearch searchWithStore:nanoStore];
 
 search.attribute = @"LastName";
 search.match = NSFEqualTo;
 search.value = @"Doe";
 
 float averageSalary = [[search aggregateOperation:NSFAverage onAttribute:@"Salary"]floatValue];
 @endcode
 
 @section needhelp_sec Need more help?
 There are two quick ways to find answers: reading the documentation and browsing the Unit tests.
 
 While several attempts have been made to make the documentation easy to read and understand, it's far from perfect. If you find that the documentation is
 incomplete, incorrect or needs some clarification, please file a bug. I'll appreciate it and correct it as soon as possible:
 
 - NanoStore Bug Tracker: http://code.google.com/p/nanostore/issues/list
 
 Other ways to be in touch:
 - NanoStore Developer List: http://groups.google.com/group/nanostore-dev
 - Twitter: http://twitter.com/nanostoredev

 */
