/*
     NSFNanoObject.h
     NanoStore
     
     Copyright (c) 2010 Webbo, L.L.C. All rights reserved.
     
     Redistribution and use in source and binary forms, with or without modification, are permitted
     provided that the following conditions are met:
     
     * Redistributions of source code must retain the above copyright notice, this list of conditions
     and the following disclaimer.
     * Redistributions in binary form must reproduce the above copyright notice, this list of conditions
     and the following disclaimer in the documentation and/or other materials provided with the distribution.
     * Neither the name of Webbo nor the names of its contributors may be used to endorse or promote
     products derived from this software without specific prior written permission.
     
     THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
     WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
     PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY
     DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
     PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
     CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
     OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
     SUCH DAMAGE.
 */

/*! @file NSFNanoObject.h
 @brief A generic class that implements all the basic behavior required of a NanoStore object.
 */

/** @class NSFNanoObject
 * A generic class that implements all the basic behavior required of a NanoStore object.
 * 
 * It is not possible to use instances of direct subclasses of NSObject (or any other class not implementing the NSFNanoObjectProtocol).
 * You may create custom subclasses of NSFNanoObject, although this is not always required. If no custom logic is needed, a complete object graph
 * can be formed with NSFNanoObject instances.
 *
 * @details <b>Example:</b>
 @code
 // Instantiate a NanoStore and open it
 NSFNanoStore *nanoStore = [NSFNanoStore createAndOpenStoreWithType:NSFMemoryStoreType path:nil error:nil];

 // Generate a NanoObject with a dictionary and a key
 NSDictionary *info = ...;
 NSFNanoObject *object = [NSFNanoObject nanoObjectWithDictionary:info];
 
 // Add it to the document store
 [nanoStore addObject:object error:nil];
 
 // Close the document store
 [nanoStore close];
 @endcode
 */

#import "NanoStore.h"

@interface NSFNanoObject : NSObject <NSFNanoObjectProtocol, NSCopying>
{
    NSString        *key;
    NSDictionary    *info;
    NSString        *originalClassString;
}

/** * The UUID of the NanoObject.  */
@property (copy, readonly) NSString *key;
/** * The user-supplied information of the NanoObject.  */
@property (retain, readwrite) NSDictionary *info;
/** * The class name used to store the NanoObject.  */
@property (copy, readonly) NSString *originalClassString;

/** @name Creating and Initializing a NanoObject
 */

//@{

/** * Creates and returns an object containing a given key and value.
 * @param theDictionary the information associated with the object. Must not be nil.
 * @return An initialized object upon success, nil otherwise.
 * @attention The dictionary must be serializable. For more information, please read the Property List Programming Guide.
 * @see - (id)initFromDictionaryRepresentation:(NSDictionary *)theDictionary;
 */

+ (NSFNanoObject*)nanoObjectWithDictionary:(NSDictionary *)theDictionary;

/** * Initializes a newly allocated object containing a given key and value.
 * @param theDictionary the information associated with the object. Must not be nil.
 * @return An initialized object upon success, nil otherwise.
 * @attention The dictionary must be serializable. For more information, please read the Property List Programming Guide.
 * @see + (NSFNanoObject*)nanoObjectWithDictionary:(NSDictionary *)theDictionary;
 */

- (id)initFromDictionaryRepresentation:(NSDictionary *)theDictionary;

//@}

/** @name Miscellaneous
 */

//@{

/** * Compares the receiving NanoObject to another NanoObject.
 * @param otherNanoObject is a NanoObject.
 * @return YES if the contents of otherNanoObject are equal to the contents of the receiving NanoObject, otherwise NO.
 */

- (BOOL)isEqualToNanoObject:(NSFNanoObject *)otherNanoObject;

/** * Returns a dictionary that contains the information stored in the object.
 * @see Properties info and key to find out the current state of the object.
 * @see - (NSString *)description;
 */

- (NSDictionary *)dictionaryRepresentation;

/** * Returns a string representation of the bag.
 * @see Properties info and key to find out the current state of the object.
 * @see - (NSDictionary *)dictionaryRepresentation;
 */

- (NSString *)description;

//@}

@end