/*
     NSFNanoBag.h
     NanoStore
     
     Copyright (c) 2010 Webbo, L.L.C. All rights reserved.
     
     Redistribution and use in source and binary forms, with or without modification, are permitted
     provided that the following conditions are met:
     
     * Redistributions of source code must retain the above copyright notice, this list of conditions
     and the following disclaimer.
     * Redistributions in binary form must reproduce the above copyright notice, this list of conditions
     and the following disclaimer in the documentation and/or other materials provided with the distribution.
     * Neither the name of Webbo nor the names of its contributors may be used to endorse or promote
     products derived from this software without specific prior written permission.
     
     THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
     WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
     PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY
     DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
     PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
     CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
     OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
     SUCH DAMAGE.
 */

/*! @file NSFNanoBag.h
 @brief A bag is a loose collection of objects stored in a document store.
 */

/** @class NSFNanoBag
 * A bag is a loose collection of objects stored in a document store.
 *
 * @note
 * The objects must conform to the NSFNanoObjectProtocol. For your convenience, NanoStore provides you with NSFNanoObject, which is the standard
 * way of storing and retrieving objects from/to a bag.
 *
 * @par
 * It's more efficient to make your storage objects NSFNanoObjectProtocol-compliant, thus eliminating the need to convert your objects to/from
 * objects of type NSFNanoObject.
 *
 * @details <b>Example:</b>
 @code
 // Instantiate a NanoStore and open it
 NSFNanoStore *nanoStore = [NSFNanoStore createAndOpenStoreWithType:NSFMemoryStoreType path:nil error:nil];
 [nanoStore removeAllObjectsFromStoreAndReturnError:nil];
 
 // Add some data to a bag
 NSFNanoBag *bag = [NSFNanoBag bagWithStore:nanoStore];
 NSDictionary *info = ...;
 NSFNanoObject *obj1 = [NSFNanoObject nanoObjectWithDictionary:info];
 NSFNanoObject *obj2 = [NSFNanoObject nanoObjectWithDictionary:info];
 NSFNanoObject *obj3 = [NSFNanoObject nanoObjectWithDictionary:info];
 [bag addObjectsFromArray:[NSArray arrayWithObjects:obj1, obj2, obj3, nil] error:nil];
 
 // Add the bag and its objects to the document store
 [nanoStore addObject:bag error:nil];
 
 // Obtain the bags from the document store
 NSArray *bags = [nanoStore bags];
 
 // Close the document store
 [nanoStore close];
 @endcode
*/

#import "NSFNanoObjectProtocol.h"

@interface NSFNanoBag : NSObject <NSFNanoObjectProtocol, NSCopying>
{
    NSFNanoStore            *store;
    NSString                *key;
    BOOL                    hasUnsavedChanges;
    
    @protected
    /** \cond */
    NSMutableDictionary     *savedObjects;
    NSMutableDictionary     *unsavedObjects;
    NSMutableDictionary     *removedObjects;
    /** \endcond */
}

/** * The store where the bag is located.  */
@property (assign, readonly) NSFNanoStore *store;
/** * The UUID of the bag.  */
@property (copy, readonly) NSString *key;
/** * Dictionary of NSString (key) and id<NSFNanoObjectProtocol> (value). */
@property (retain, readonly) NSDictionary *savedObjects;
/** * Dictionary of NSString (key) and id<NSFNanoObjectProtocol> (value). */
@property (retain, readonly) NSDictionary *unsavedObjects;
/** * Dictionary of NSString (key) and id<NSFNanoObjectProtocol> (value). */
@property (retain, readonly) NSDictionary *removedObjects;
/** * To determine whether the bag has uncommited changes.  */
@property (assign, readonly) BOOL hasUnsavedChanges;

/** @name Creating and Initializing Bags
 */

//@{

/** * Creates and returns a bag for a given document store.
 * @param theNanoStore the document store where the bag will be stored. Must not be nil.
 * @return An empty bag upon success, nil otherwise.
 * @see - (id)initBagWithStore:(NSFNanoStore *)theNanoStore;
 */

+ (NSFNanoBag *)bagWithStore:(NSFNanoStore *)theNanoStore;

/** * Initializes a newly allocated bag for a given document store.
 * @param theNanoStore the document store where the bag will be stored. Must not be nil.
 * @return An empty bag upon success, nil otherwise.
 * @see + (NSFNanoBag*)bagWithStore:(NSFNanoStore *)theNanoStore;
 */

- (id)initBagWithStore:(NSFNanoStore *)theNanoStore;

/** * Creates and returns a bag for a given document store by adding in it the objects contained in a given array.
 * @param theObjects an array of objects conforming to the NSFNanoObjectProtocol.
 * @param theNanoStore the document store where the bag will be stored.
 * @return A bag only containing the objects with conform to the NSFNanoObjectProtocol upon success, nil otherwise.
 * @see - (NSFNanoBag*)initBagWithObjects:(NSArray *)theObjects store:(NSFNanoStore *)theNanoStore;
 */

+ (NSFNanoBag *)bagWithObjects:(NSArray *)theObjects store:(NSFNanoStore *)theNanoStore;

/** * Initializes a newly allocated bag for a given document store by adding in it the objects contained in a given array.
 * @param theObjects an array of objects conforming to the NSFNanoObjectProtocol.
 * @param theNanoStore the document store where the bag will be stored.
 * @return A bag only containing the objects with conform to the NSFNanoObjectProtocol upon success, nil otherwise.
 * @see + (NSFNanoBag*)bagWithObjects:(NSArray *)theObjects store:(NSFNanoStore *)theNanoStore;
 */

- (id)initBagWithObjects:(NSArray *)theObjects store:(NSFNanoStore *)theNanoStore;

//@}

/** @name Adding and Removing Objects
 */

//@{

/** * Adds an NSFNanoObjectProtocol-compliant object to the bag.
 * @param theObject is added to the bag.
 * @param outError is used if an error occurs. May be NULL.
 * @return YES upon success, NO otherwise.
 * @warning This value cannot be nil and it must be NSFNanoObjectProtocol-compliant.
 * @throws NSFNonConformingNanoObjectProtocolException is thrown if the object is non-NSFNanoObjectProtocol compliant.
 * @see - (BOOL)addObjectsFromArray:(NSArray *)theObjects error:(out NSError **)outError;
 */

- (BOOL)addObject:(id <NSFNanoObjectProtocol>)theObject error:(out NSError **)outError;

/** * Adds a series of NSFNanoObjectProtocol-compliant objects to the bag.
 * @param theObjects is an array of objects to be added to the bag. The objects must be NSFNanoObjectProtocol-compliant.
 * @param outError is used if an error occurs. May be NULL.
 * @return YES upon success, NO otherwise.
 * @warning The objects of the array must be NSFNanoObjectProtocol-compliant.
 * @throws NSFNonConformingNanoObjectProtocolException is thrown if the object is non-NSFNanoObjectProtocol compliant.
 * @see - (BOOL)addObject:(id <NSFNanoObjectProtocol>)theObject error:(out NSError **)outError;
 */

- (BOOL)addObjectsFromArray:(NSArray *)theObjects error:(out NSError **)outError;

/** * Removes the specified object from the bag.
 * @param theObject the object to be removed from the bag.
 * @warning The object must be NSFNanoObjectProtocol-compliant.
 * @see - (void)removeObjectsInArray:(NSArray *)theObjects;
 * @see - (void)removeObjectWithKey:(NSString *)theObjectKey;
 * @see - (void)removeObjectsWithKeysInArray:(NSArray *)theKeys;
 */

- (void)removeObject:(id <NSFNanoObjectProtocol>)theObject;

/** * Removes the list of objects from the bag.
 * @param theObjects the list of objects to be removed from the bag.
 * @warning The objects of the array must be NSFNanoObjectProtocol-compliant.
 * @see - (void)removeObject:(id <NSFNanoObjectProtocol>)theObject;
 * @see - (void)removeObjectWithKey:(NSString *)theObjectKey;
 * @see - (void)removeObjectsWithKeysInArray:(NSArray *)theKeys;
 */

- (void)removeObjectsInArray:(NSArray *)theObjects;

/** * Removes the object with a given key from the bag.
 * @param theObjectKey the key of the object to be removed from the bag.
 * @warning The object referenced by theObjectKey must be NSFNanoObjectProtocol-compliant.
 * @see - (void)removeObject:(id <NSFNanoObjectProtocol>)theObject;
 * @see - (void)removeObjectsInArray:(NSArray *)theObjects;
 * @see - (void)removeObjectsWithKeysInArray:(NSArray *)theKeys;
 */

- (void)removeObjectWithKey:(NSString *)theObjectKey;

/** * Removes from the bag the objects specified by elements in a given array.
 * @param theKeys an array of objects specifying the keys to remove from the bag
 * @warning The objects referenced by theKeys must be NSFNanoObjectProtocol-compliant.
 * @see - (void)removeObject:(id <NSFNanoObjectProtocol>)theObject;
 * @see - (void)removeObjectsInArray:(NSArray *)theObjects;
 * @see - (void)removeObjectWithKey:(NSString *)theObjectKey;
 */

- (void)removeObjectsWithKeysInArray:(NSArray *)theKeys;

//@}

/** @name Saving, Reloading and Undoing
 */

//@{

/** * Saves the bag and its contents. Also, saves all the changes made since the last save.
 * @param outError is used if an error occurs. May be NULL.
 * @return YES upon success, NO otherwise.
 * @see Property hasUnsavedChanges to find out whether the bag has unsaved contents.
 * @see - (BOOL)reloadBagWithError:(out NSError **)outError;
 * @see - (BOOL)undoChangesWithError:(out NSError **)outError;
 */

- (BOOL)saveAndReturnError:(out NSError **)outError;

/** * Refreshes the bag to match the contents stored in the document store. The unsaved contents are preserved.
 * @param outError is used if an error occurs. May be NULL.
 * @return YES upon success, NO otherwise.
 * @see Properties savedObjects, unsavedObjects and removedObjects to find out the current state of the bag.
 * @see - (BOOL)saveAndReturnError:(out NSError **)outError;
 * @see - (BOOL)undoChangesWithError:(out NSError **)outError;
 */

- (BOOL)reloadBagWithError:(out NSError **)outError;

/** * Discards the changes made in the bag.
 * @param outError is used if an error occurs. May be NULL.
 * @return YES upon success, NO otherwise.
 * @see Properties savedObjects, unsavedObjects and removedObjects to find out the current state of the bag.
 * @see - (BOOL)saveAndReturnError:(out NSError **)outError;
 * @see - (BOOL)reloadBagWithError:(out NSError **)outError;
 */

- (BOOL)undoChangesWithError:(out NSError **)outError;

//@}

/** @name Inflating and Deflating
 */

//@{

/** * Inflates the bag by reconstructing the objects flattened with - (void)deflateBag;
 * @note To release memory, use - (void)deflateBag;
 * @see Properties savedObjects, unsavedObjects and removedObjects to find out the current state of the bag.
 * @see - (void)deflateBag;
 */

- (void)inflateBag;

/** * Releases memory by "flattening" the objects from the bag.
 * @note Deflating the bag affects the volatile version of the bag, not the version stored in the document store. Using - (void)inflateBag; reverses this action.
 * @see Properties savedObjects, unsavedObjects and removedObjects to find out the current state of the bag.
 * @see - (void)inflateBag;
 */

- (void)deflateBag;

//@}

/** @name Miscellaneous
 */

//@{

/** * Compares the receiving bag to another bag.
 * @param otherNanoBag is a bag.
 * @return YES if the contents of otherNanoBag are equal to the contents of the receiving bag, otherwise NO.
 */

- (BOOL)isEqualToNanoBag:(NSFNanoBag *)otherNanoBag;

/** * Returns a dictionary that contains the information stored in the bag.
 * @see Properties savedObjects, unsavedObjects and removedObjects to find out the current state of the bag.
 * @see - (NSString *)description;
 */

- (NSDictionary *)dictionaryRepresentation;

/** * Returns a string representation of the bag.
 * @see Properties savedObjects, unsavedObjects and removedObjects to find out the current state of the bag.
 * @see - (NSDictionary *)dictionaryRepresentation;
 */

- (NSString *)description;

//@}

@end