/*
     NSFNanoSearch.h
     NanoStore
     
     Copyright (c) 2010 Webbo, L.L.C. All rights reserved.
     
     Redistribution and use in source and binary forms, with or without modification, are permitted
     provided that the following conditions are met:
     
     * Redistributions of source code must retain the above copyright notice, this list of conditions
     and the following disclaimer.
     * Redistributions in binary form must reproduce the above copyright notice, this list of conditions
     and the following disclaimer in the documentation and/or other materials provided with the distribution.
     * Neither the name of Webbo nor the names of its contributors may be used to endorse or promote
     products derived from this software without specific prior written permission.
     
     THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
     WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
     PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY
     DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
     PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
     CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
     OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
     SUCH DAMAGE.
 */

/*! @file NSFNanoSearch.h
 @brief A unit that provides an API to retrieve data from the document store.
 */

/** @class NSFNanoSearch
 * A unit that provides an API to retrieve data from the document store.
 *
 * The search can be conducted in two ways: programatically via setters or by providing a SQL statement. In both cases,
 * it's necessary to indicate which object type should be returned. The type NSFReturnType provides two options: <i>NSFReturnObjects</i> and <i>NSFReturnKeys</i>.
 *
 *           -  <b>NSFReturnObjects</b> will return a dictionary with the key of the NanoObject (key) and the NanoObject itself (value).
 *           -  <b>NSFReturnKeys</b> will return an array of NanoObjects.
 *
 * @para <b>Some observations about retrieving data</b><br>
 * 
 * Given the following data set:
 * 
 *           - Number of dictionaries:    3.956
 *           - Number of attributes:    593.862
 * 
 * The table describing different timings to perform a simple value search (i.e. 'Barcelona') is included below, ordered from fastest to slowest:
 *
 *<table border="1" cellpadding="5">
 *<tr>
 *<th>Match type</th>
 *<th>Seconds</th>
 *</tr>
 *<tr><td>Equal to</td><td>0.295</td></tr>
 *<tr><td>Begins with</td><td>0.295</td></tr>
 *<tr><td>Contains</td><td>1.295</td></tr>
 *<tr><td>Contains (insensitive)</td><td>1.339</td></tr>
 *<tr><td>Ends with (insensitive)</td><td>1.341</td></tr>
 *<tr><td>Ends with</td><td>1.351</td></tr>
 *<tr><td>Equal to (insensitive)</td><td>1.890</td></tr>
 *<tr><td>Begins with (insensitive)</td><td>2.412</td></tr>
 *<tr><td>Greater than</td><td>18.246</td></tr>
 *<tr><td>Less than</td><td>27.677</td></tr>
 *</table>
 *
 * @details <b>Example:</b>
 @code
 // Instantiate a NanoStore and open it
 NSFNanoStore *nanoStore = [NSFNanoStore createAndOpenStoreWithType:NSFMemoryStoreType path:nil error:nil];
 
 // Generate a NanoObject with a dictionary and a key
 NSString *key = @"ABC-123";
 NSDictionary *info = ...;
 NSFNanoObject *nanoObject = [NSFNanoObject nanoObjectWithDictionary:info];
 
 // Add it to the document store
 [nanoStore addObject:nanoObject error:nil];
 
 // Instantiate a search and specify the attribute(s) we want to search for
 NSFNanoSearch *search = [NSFNanoSearch searchWithStore:nanoStore];
 search.key = key;
 
 // Perform the search and obtain the results
 NSDictionary *searchResults = [search searchObjectsWithReturnType:NSFReturnObjects error:nil];
 
 // Close the document store
 [nanoStore close];
 @endcode
 */

#import <Foundation/Foundation.h>

#import "NSFNanoGlobals.h"

@class NSFNanoStore, NSFNanoResult;

@interface NSFNanoSearch : NSObject
{
    NSFNanoStore        *nanoStore;
    NSArray             *attributesToBeReturned;
    
    NSString            *key;
    NSString            *attribute;
    id                  value;
    NSFMatchType        match;
    NSArray             *expressions;
    BOOL                groupValues;

    /** \cond */
    @protected
    NSFReturnType       returnedObjectType;
    NSString            *sql;
    /** \endcond */
}

/** * The document store used for searching. */
@property (assign, readonly) NSFNanoStore *nanoStore;
/** * The set of attributes to be returned on matching objects. */
@property (retain, readwrite) NSArray *attributesToBeReturned;
/** * The key used for searching. */
@property (copy, readwrite) NSString *key;
/** * The attribute used for searching. */
@property (copy, readwrite) NSString *attribute;
/** * The value used for searching. */
@property (copy, readwrite) id value;
/** * The comparison operator used for searching. */
@property (assign, readwrite) NSFMatchType match;
/** * The list of NSFNanoExpression objects used for searching. */
@property (retain, readwrite) NSArray *expressions;
/** * If set to YES, specifying NSFReturnKeys applies the DISTINCT function and groups the values. */
@property (assign, readwrite) BOOL groupValues;
/** * The SQL statement used for searching. Set when executeSQL: is invoked. */
@property (copy, readonly) NSString *sql;

/** @name Creating and Initializing a Search
 */

//@{

/** * Creates and returns a search element for a given document store.
 * @param theNanoStore the document store where the search will be performed. Must not be nil.
 * @return An search element upon success, nil otherwise.
 * @see - (id)initWithStore:(NSFNanoStore *)theNanoStore;
 */

+ (NSFNanoSearch *)searchWithStore:(NSFNanoStore *)theNanoStore;

/** * Initializes a newly allocated search element for a given document store.
 * @param theNanoStore the document store where the search will be performed. Must not be nil.
 * @return An search element upon success, nil otherwise.
 * @see + (NSFNanoSearch *)searchWithStore:(NSFNanoStore *)theNanoStore;
 */

- (id)initWithStore:(NSFNanoStore *)theNanoStore;

//@}

/** @name Searching
 */

//@{

/** * Performs a search using the values of the properties.
 * @param theReturnType the type of object to be returned. Can be <i>NSFReturnObjects</i> or <i>NSFReturnKeys</i>.
 * @param outError is used if an error occurs. May be NULL.
 * @return If theReturnType is NSFReturnObjects, a dictionary is returned. Otherwise, an array is returned.
 * @see - (id)searchObjectsAdded:(NSFDateMatchType)theDateMatch date:(NSDate *)theDate returnType:(NSFReturnType)theReturnType error:(out NSError **)outError;
 */

- (id)searchObjectsWithReturnType:(NSFReturnType)theReturnType error:(out NSError **)outError;

/** * Performs a search using the values of the properties before, on or after a given date.
 * @param theDateMatch the type of date comparison. Can be <i>NSFBeforeDate</i>, <i>NSFOnDate</i> or <i>NSFAfterDate</i>.
 * @param theDate the date to use as a pivot during the search.
 * @param theReturnType the type of object to be returned. Can be <i>NSFReturnObjects</i> or <i>NSFReturnKeys</i>.
 * @param outError is used if an error occurs. May be NULL.
 * @return If theReturnType is NSFReturnObjects, a dictionary is returned. Otherwise, an array is returned.
 * @see - (id)searchObjectsWithReturnType:(NSFReturnType)theReturnType error:(out NSError **)outError;
 */

- (id)searchObjectsAdded:(NSFDateMatchType)theDateMatch date:(NSDate *)theDate returnType:(NSFReturnType)theReturnType error:(out NSError **)outError;

/** * Returns the result of the aggregate function.
 * @param theFunctionType is the function type to be applied.
 * @param theAttribute is the attribute used in the function.
 * @returns An NSNumber containing the result of the aggregate function.
 * @details <b>Example:</b>
 @code
 * NSFNanoStore *nanoStore = [NSFNanoStore createAndOpenStoreWithType:NSFMemoryStoreType path:nil error:nil];
 *
 * // Assume we have saved data to the document store
 * ...
 * ...
 *
 * // Get the average for the attribute named 'SomeNumber'
 * NSNumber *result = [nanoStore aggregateOperation:NSFAverage onAttribute:@"SomeNumber"];
 @endcode
 */

- (NSNumber *)aggregateOperation:(NSFAggregateFunctionType)theFunctionType onAttribute:(NSString *)theAttribute;

/** * Performs a search with a given SQL statement.
 * @param theSQLStatement is the SQL statement to be executed. Must not be nil or an empty string.
 * @param theReturnType the type of object to be returned. Can be <i>NSFReturnObjects</i> or <i>NSFReturnKeys</i>.
 * @param outError is used if an error occurs. May be NULL.
 * @return If theReturnType is NSFReturnObjects, a dictionary is returned. Otherwise, an array is returned.
 * @note
 * Use this method when performing search on NanoObjects. If you need to perform more advanced SQL statements, you may want to use
 * <i>- (NSFNanoResult *)executeSQL:(NSString *)theSQLStatement error:(out NSError **)outError;</i> instead.
 * @par
 * The key difference between this method and <i>- (NSFNanoResult *)executeSQL:(NSString *)theSQLStatement;</i> is that this method performs
 * a check to make sure the columns specified match the ones required by the <i>NSFReturnType</i> type selected. If the column selection is wrong,
 * NanoStore rewrites the query by specifying the right set of columns while honoring the rest of the query.
 * @details <b>Example:</b>
 * @code
 * // Prepare a document store
 * NSFNanoStore *nanoStore = [NSFNanoStore createAndOpenStoreWithType:NSFMemoryStoreType path:nil error:nil];
 *
 * // Prepare some data and wrap it in a NanoObject
 * NSString *key = @"ABC-123";
 * NSDictionary *info = ...;
 * NSFNanoObject *nanoObject = [NSFNanoObject nanoObjectWithDictionary:info];
 *
 * // Add it to the document store
 * [nanoStore addObjectsFromArray:[NSArray arrayWithObject:nanoObject] error:nil];
 *
 * // Instantiate a search and specify the attribute(s) we want to search for
 * NSFNanoSearch *search = [NSFNanoSearch searchWithStore:nanoStore];
 *
 * // Perform the search
 * // The query will be rewritten as @"SELECT NSFKey, NSFPlist, NSFObjectClass FROM NSFKeys"
 * NSDictionary *results = [search executeSQL:@"SELECT foo, bar FROM NSFKeys" returnType:NSFReturnObjects error:nil];
 * @endcode
 * @see - (NSFNanoResult *)executeSQL:(NSString *)theSQLStatement error:(out NSError **)outError;
 */

- (id)executeSQL:(NSString *)theSQLStatement returnType:(NSFReturnType)theReturnType error:(out NSError **)outError;

/** * Performs a search with a given SQL statement.
 * @param theSQLStatement is the SQL statement to be executed. Must not be nil or an empty string.
 * @return Returns a NSFNanoResult.
 * @note
 * Use this method when you need to perform more advanced SQL statements. If you just want to query NanoObjects using your own SQL statement,
 * you may want to use <i>- (id)executeSQL:(NSString *)theSQLStatement returnType:(NSFReturnType)theReturnType error:(out NSError **)outError;</i> instead.
 * @par
 * The key difference between this method and <i>- (id)executeSQL:(NSString *)theSQLStatement returnType:(NSFReturnType)theReturnType error:(out NSError **)outError;</i>
 * is that this method doesn't perform any check at all. The SQL statement will be sent verbatim to SQLite.
 * @details <b>Example:</b>
 * @code
 * // Prepare a document store
 * NSFNanoStore *nanoStore = [NSFNanoStore createAndOpenStoreWithType:NSFMemoryStoreType path:nil error:nil];
 *
 * // Prepare some data and wrap it in a NanoObject
 * NSString *key = @"ABC-123";
 * NSDictionary *info = ...;
 * NSFNanoObject *nanoObject = [NSFNanoObject nanoObjectWithDictionary:info];
 *
 * // Add it to the document store
 * [nanoStore addObjectsFromArray:[NSArray arrayWithObject:nanoObject] error:nil];
 *
 * // Instantiate a search and specify the attribute(s) we want to search for
 * NSFNanoSearch *search = [NSFNanoSearch searchWithStore:nanoStore];
 *
 * // Perform the search
 * NSFNanoResult *result = [search executeSQL:@"SELECT COUNT(*) FROM NSFKEYS"];
 * @endcode
 * @see - (id)executeSQL:(NSString *)theSQLStatement returnType:(NSFReturnType)theReturnType error:(out NSError **)outError;
 */

- (NSFNanoResult *)executeSQL:(NSString *)theSQLStatement;

/** * Performs an analysis of the given SQL statement.
 * @param theSQLStatement is the SQL statement to be analyzed. Must not be nil or an empty string.
 * @return Returns a NSFNanoResult.
 * @note
 * Returns the sequence of virtual machine instructions with high-level information about what indices would have been used if the SQL statement had
 * been executed.
 *
 * @warning
 * The analysis generated by this method is intended for interactive analysis and troubleshooting only. The details of the output format
 * are subject to change from one release of SQLite to the next. Applications should not use this method in production code since the exact behavior
 * is undocumented, unspecified, and variable.
 *
 * For additional information about SQLite's Virtual Machine Opcodes, see http://www.sqlite.org/opcode.html
 *
 * The tutorial Virtual Database Engine of SQLite is available here: http://www.sqlite.org/vdbe.html
 *
 * @details <b>Example:</b>
 * @code
 * // Prepare a document store
 * NSFNanoStore *nanoStore = [NSFNanoStore createAndOpenStoreWithType:NSFMemoryStoreType path:nil error:nil];
 *
 * // Instantiate a search object
 * NSFNanoSearch *search = [NSFNanoSearch searchWithStore:nanoStore];
 *
 * // Perform the analysis
 * NSFNanoResult *results = [search explainSQL:@"SELECT * FROM NSFValues"];
 * @endcode
 * @see - (NSFNanoResult *)executeSQL:(NSString *)theSQLStatement;
 * @see - (id)executeSQL:(NSString *)theSQLStatement returnType:(NSFReturnType)theReturnType error:(out NSError **)outError;
 */

- (NSFNanoResult *)explainSQL:(NSString *)theSQLStatement;

//@}

/** @name Resetting Values
 */

//@{

/** * Resets the values to a know, default state.
 *      - key                 = nil;
 *      - attribute           = nil;
 *      - value               = nil;
 *      - match               = NSFContains;
 *      - object type         = NSFReturnObjects;
 *      - groupValues         = NO;
 *      - attributesReturned  = nil;
 *      - type returned       = NSFReturnObjects;
 *
 * @note
 * When invoked, it sets the values of search to its initial state. Resetting and performing a search will select all records.
 */

- (void)reset;

//@}

@end