using System.Text;
using System.Runtime.Serialization;
using System;
namespace DataObjects
{
	/// <summary>
	/// Implements a type that can hold a strongly typed value and metadata about its inheritance status
	/// </summary>
	/// <typeparam name="T">The type the Value property</typeparam>
	[DataContract]
	[KnownType(typeof(int))]
	[KnownType(typeof(int?))]
	[KnownType(typeof(decimal))]
	[KnownType(typeof(decimal?))]
	[KnownType(typeof(bool))]
	[KnownType(typeof(bool?))]
	[KnownType(typeof(TradingDefinitionsMeasuringType))]
	[KnownType(typeof(TradingDefinitionsMeasuringType?))]
	[KnownType(typeof(EntryOrderExecutionMode))]
	[KnownType(typeof(EntryOrderExecutionMode?))]
	public class Inheritable<T>
	{
		/// <summary>
		/// The value of type T
		/// </summary>
		[DataMember(Name = "P1")]
		public virtual T Value { get; set; }

		/// <summary>
		/// Gets or sets if the value is inherited
		/// </summary>
		[DataMember(Name = "P2")]
		public virtual bool IsInherited { get; set; }

		/// <summary>
		/// Gets or sets if the value is inherited
		/// </summary>
		[DataMember(Name = "P3")]
		public bool IsFetched { get; set; }

		/// <summary>
		/// The value, as appear in the database.
		/// If the row is missing from the database, should be represented as null.
		/// </summary>
		/// <remarks>
		/// This is represented as an object and not as T, due to the fact that they might not be the same,
		/// for example: T is an int and its database value is nullable int, and in another case T is a
		/// nullable int, as the database value. Generics does not fit well with all those scenarios
		/// (nullable inside a nullable will not compile), so we use an object.
		/// </remarks>
		[DataMember(Name = "P4")]
		public virtual object DBValue { get; set; }

		/// <summary>
		/// Initializas a new instance of the DataObjects.Inheritable of T class 
		/// </summary>
		public Inheritable()
		{
			Value = default(T);
			IsInherited = false;
		}

		/// <summary>
		/// Returns a System.String that represents the current DataObjects.Inheritable of T class 
		/// </summary>
		/// <returns></returns>		
		public override string ToString()
		{
			StringBuilder stringBuilder = new StringBuilder();

			stringBuilder.Append("Value");
			stringBuilder.Append("=");
			if (Value != null)
			{ 
				stringBuilder.Append(Value.ToString()); 
			}
			else
			{
				stringBuilder.Append("NULL"); 
			}
			stringBuilder.Append(";");
			stringBuilder.Append("DBValue");
			stringBuilder.Append("=");
			if (DBValue != null)
			{ 
				stringBuilder.Append(DBValue.ToString()); 
			}
			else
			{ 
				stringBuilder.Append("NULL"); 
			}
			stringBuilder.Append(";");
			stringBuilder.Append("IsInherited");
			stringBuilder.Append("=");
			stringBuilder.Append(IsInherited);
			stringBuilder.Append(";");

			return stringBuilder.ToString();
		}
	}
}
