#include <stdio.h>
#include <stdlib.h>

/**********************************************/
/**********************************************/

#ifndef _
#define _(x) (x)
#endif

#define INITGUID
#include <ole2.h>

#include "io-gdip-native.h"

typedef enum {
        /* image data hosed */
        GDK_PIXBUF_ERROR_CORRUPT_IMAGE,
        /* no mem to load image */
        GDK_PIXBUF_ERROR_INSUFFICIENT_MEMORY,
        /* bad option passed to save routine */
        GDK_PIXBUF_ERROR_BAD_OPTION,
        /* unsupported image type (sort of an ENOSYS) */
        GDK_PIXBUF_ERROR_UNKNOWN_TYPE,
        /* unsupported operation (load, save) for image type */
        GDK_PIXBUF_ERROR_UNSUPPORTED_OPERATION,
        GDK_PIXBUF_ERROR_FAILED
} GdkPixbufError;

#define GDK_PIXBUF_ERROR 0

#define LOAD_BUFFER_SIZE 65536

struct _GdipContext {
  GByteArray *buffer;
  IStream *stream;
  HGLOBAL hg;
};
typedef struct _GdipContext GdipContext;

static GdiplusStartupFunc GdiplusStartup;
static GdipCreateBitmapFromStreamFunc GdipCreateBitmapFromStream;
static GdipBitmapGetPixelFunc GdipBitmapGetPixel;
static GdipGetImageHeightFunc GdipGetImageHeight;
static GdipDisposeImageFunc GdipDisposeImage;
static GdipGetImageFlagsFunc GdipGetImageFlags;
static GdipGetImageWidthFunc GdipGetImageWidth;
static GdipImageGetFrameCountFunc GdipImageGetFrameCount;
static GdipImageSelectActiveFrameFunc GdipImageSelectActiveFrame;
static GdipGetPropertyItemSizeFunc GdipGetPropertyItemSize;
static GdipGetPropertyItemFunc GdipGetPropertyItem;
static GdipGetPropertyCountFunc GdipGetPropertyCount;
static GdipGetPropertyIdListFunc GdipGetPropertyIdList;
static GdipCreateBitmapFromScan0Func GdipCreateBitmapFromScan0;
static GdipSaveImageToStreamFunc GdipSaveImageToStream;
static GdipBitmapSetPixelFunc GdipBitmapSetPixel;
static GdipDrawImageIFunc GdipDrawImageI;
static GdipGetImageGraphicsContextFunc GdipGetImageGraphicsContext;
static GdipFlushFunc GdipFlush;
static GdipGraphicsClearFunc GdipGraphicsClear;
static GdipBitmapSetResolutionFunc GdipBitmapSetResolution;
static GdipGetImageHorizontalResolutionFunc GdipGetImageHorizontalResolution;
static GdipGetImageVerticalResolutionFunc GdipGetImageVerticalResolution;
static GdipLoadImageFromStreamFunc GdipLoadImageFromStream;
static GdipDeleteGraphicsFunc GdipDeleteGraphics;
static GdipGetImageEncodersFunc GdipGetImageEncoders;
static GdipGetImageEncodersSizeFunc GdipGetImageEncodersSize;
static GdipBitmapLockBitsFunc GdipBitmapLockBits;
static GdipBitmapUnlockBitsFunc GdipBitmapUnlockBits;
static GdipGetImagePixelFormatFunc GdipGetImagePixelFormat;
static GdipCloneBitmapAreaIFunc GdipCloneBitmapAreaI;

DEFINE_GUID(FrameDimensionTime, 0x6aedbd6d,0x3fb5,0x418a,0x83,0xa6,0x7f,0x45,0x22,0x9d,0xc8,0x72);
DEFINE_GUID(FrameDimensionPage, 0x7462dc86,0x6180,0x4c7e,0x8e,0x3f,0xee,0x73,0x33,0xa7,0xa4,0x83);

static void g_set_error_literal (GError **error, gint foo, gint code, const char* msg)
{
	g_set_error (error, foo, code, "%s", msg);
}

static void
gdip_set_error_from_hresult (GError **error, gint code, HRESULT hr, const char *format)
{
  gchar *msg;
  
  msg = g_win32_error_message (hr);
  
  if (msg) {
    g_set_error (error, GDK_PIXBUF_ERROR, code, format, msg);
    g_free (msg);
  }
}

static void
gdip_set_error_from_gpstatus (GError **error, gint code, GpStatus status)
{
  const char *msg;

  switch (status)
    {
#define CASE(x) case x: msg = #x; break
    CASE (GenericError);
    CASE (InvalidParameter);
    CASE (OutOfMemory);
    CASE (ObjectBusy);
    CASE (InsufficientBuffer);
    CASE (NotImplemented);
    CASE (Win32Error);
    CASE (WrongState);
    CASE (Aborted);
    CASE (FileNotFound);
    CASE (ValueOverflow);
    CASE (AccessDenied);
    CASE (UnknownImageFormat);
    CASE (FontFamilyNotFound);
    CASE (FontStyleNotFound);
    CASE (NotTrueTypeFont);
    CASE (UnsupportedGdiplusVersion);
    CASE (GdiplusNotInitialized);
    CASE (PropertyNotFound);
    CASE (PropertyNotSupported);
    CASE (ProfileNotFound);
    default:
      msg = "Unknown error";
    }
  g_set_error_literal (error, GDK_PIXBUF_ERROR, code, msg);
}

static gboolean
gdip_init (void)
{
  GdiplusStartupInput input;
  ULONG_PTR gdiplusToken = 0;
  static HINSTANCE gdipluslib = NULL;

  if (!gdipluslib)
    gdipluslib = LoadLibrary ("gdiplus.dll");
  else
    return TRUE; /* gdip_init() is idempotent */

  if (!gdipluslib)
    return FALSE;

#define LOOKUP(func) \
  G_STMT_START { \
    func = (func##Func) GetProcAddress (gdipluslib, #func); \
    if (!func) {\
      g_warning ("Couldn't find GDI+ function %s\n", #func); \
      return FALSE; \
    } \
  } G_STMT_END

  LOOKUP (GdiplusStartup);
  LOOKUP (GdipCreateBitmapFromStream);
  LOOKUP (GdipBitmapGetPixel);
  LOOKUP (GdipGetImageHeight);
  LOOKUP (GdipDisposeImage);
  LOOKUP (GdipGetImageFlags);
  LOOKUP (GdipGetImageWidth);
  LOOKUP (GdipImageGetFrameCount);
  LOOKUP (GdipImageSelectActiveFrame);
  LOOKUP (GdipGetPropertyItemSize);
  LOOKUP (GdipGetPropertyItem);
  LOOKUP (GdipGetPropertyCount);
  LOOKUP (GdipGetPropertyIdList);
  LOOKUP (GdipCreateBitmapFromScan0);
  LOOKUP (GdipSaveImageToStream);
  LOOKUP (GdipBitmapSetPixel);
  LOOKUP (GdipDrawImageI);
  LOOKUP (GdipGetImageGraphicsContext);
  LOOKUP (GdipFlush);
  LOOKUP (GdipGraphicsClear);
  LOOKUP (GdipBitmapSetResolution);
  LOOKUP (GdipGetImageHorizontalResolution);
  LOOKUP (GdipGetImageVerticalResolution);
  LOOKUP (GdipLoadImageFromStream);
  LOOKUP (GdipDeleteGraphics);
  LOOKUP (GdipGetImageEncoders);
  LOOKUP (GdipGetImageEncodersSize);
  LOOKUP (GdipBitmapLockBits);
  LOOKUP (GdipBitmapUnlockBits);
  LOOKUP (GdipGetImagePixelFormat);
  LOOKUP (GdipCloneBitmapAreaI);

#undef LOOKUP

  input.GdiplusVersion = 1;
  input.DebugEventCallback = NULL;
  input.SuppressBackgroundThread = input.SuppressExternalCodecs = FALSE;
  
  return (GdiplusStartup (&gdiplusToken, &input, NULL) == 0 ? TRUE : FALSE);
}

static gboolean
GetEncoderClsid (const WCHAR *format, CLSID *pClsid)
{
  UINT num, size;
  int j;
  ImageCodecInfo *pImageCodecInfo;
    
  if (Ok != GdipGetImageEncodersSize (&num, &size))
    return FALSE;
    
  pImageCodecInfo = (ImageCodecInfo *) g_malloc (size);
    
  if (Ok != GdipGetImageEncoders (num, size, pImageCodecInfo)) {
    g_free (pImageCodecInfo);
    return FALSE;
  }

  for (j = 0; j < num; j++) {
    if (wcscmp (pImageCodecInfo[j].MimeType, format) == 0) {
      *pClsid = pImageCodecInfo[j].Clsid;
      g_free (pImageCodecInfo);
      return TRUE;
    }
  }
 
  g_free (pImageCodecInfo);

  return FALSE;
}

static HGLOBAL
gdip_buffer_to_hglobal (const gchar *buffer, size_t size, GError **error)
{
  HGLOBAL hg = NULL;

  hg = GlobalAlloc (GPTR, size);

  if (!hg) {
    gdip_set_error_from_hresult (error, GDK_PIXBUF_ERROR_FAILED, GetLastError (), _("Could not allocate memory: %s"));
    return NULL;
  }

  CopyMemory (hg, buffer, size);

  return hg;
}

static GpBitmap *
gdip_buffer_to_bitmap (GdipContext *context, GError **error)
{
  HRESULT hr;
  HGLOBAL hg = NULL;
  GpBitmap *bitmap = NULL;
  IStream *stream = NULL;
  GpStatus status;
  guint64 size64 = context->buffer->len;

  hg = gdip_buffer_to_hglobal (context->buffer->data, context->buffer->len, error);

  if (!hg)
    return NULL;

  hr = CreateStreamOnHGlobal (hg, FALSE, (LPSTREAM *)&stream);

  if (!SUCCEEDED (hr)) {
    gdip_set_error_from_hresult (error, GDK_PIXBUF_ERROR_FAILED, hr, _("Could not create stream: %s"));
    GlobalFree (hg);
    return NULL;
  }
  
  IStream_SetSize (stream, *(ULARGE_INTEGER *)&size64);
  status = GdipCreateBitmapFromStream (stream, &bitmap);

  if (Ok != status) {
    gdip_set_error_from_gpstatus (error, GDK_PIXBUF_ERROR_FAILED, status);
    IStream_Release (stream);
    GlobalFree (hg);
	return NULL;
  }

  context->stream = stream;
  context->hg = hg;

  return bitmap;
}

static GpImage *
gdip_buffer_to_image (GdipContext *context, GError **error)
{
  HRESULT hr;
  HGLOBAL hg = NULL;
  GpImage *image = NULL;
  IStream *stream = NULL;
  GpStatus status;
  guint64 size64 = context->buffer->len;

  hg = gdip_buffer_to_hglobal (context->buffer->data, context->buffer->len, error);

  if (!hg)
    return NULL;

  hr = CreateStreamOnHGlobal (hg, FALSE, (LPSTREAM *)&stream);

  if (!SUCCEEDED (hr)) {
    gdip_set_error_from_hresult (error, GDK_PIXBUF_ERROR_FAILED, hr, _("Could not create stream: %s"));
    GlobalFree (hg);
    return NULL;
  }
  
  IStream_SetSize (stream, *(ULARGE_INTEGER *)&size64);
  status = GdipLoadImageFromStream (stream, &image);

  if (Ok != status) {
    gdip_set_error_from_gpstatus (error, GDK_PIXBUF_ERROR_FAILED, status);
    IStream_Release (stream);
    GlobalFree (hg);
	return NULL;
  }

  context->stream = stream;
  context->hg = hg;

  return image;
}

static void
gdip_bitmap_get_size (GpBitmap *bitmap, guint *width, guint *height)
{
  if (bitmap == NULL || width == NULL || height == NULL)
    return;

  *width = *height = 0;

  GdipGetImageWidth ((GpImage *) bitmap, width);
  GdipGetImageHeight ((GpImage *) bitmap, height);
}

static void
gdip_bitmap_get_has_alpha (GpBitmap *bitmap, gboolean *has_alpha)
{
  guint flags = 0;

  if (bitmap == NULL || has_alpha == NULL)
    return;

  GdipGetImageFlags ((GpImage *) bitmap, &flags);
  *has_alpha = (flags & ImageFlagsHasAlpha);
}

static gboolean
gdip_bitmap_get_n_frames (GpBitmap *bitmap, guint *n_frames, gboolean timeDimension)
{
  if (bitmap == NULL || n_frames == NULL)
    return FALSE;

  *n_frames = 1;

  return (Ok == GdipImageGetFrameCount ((GpImage *) bitmap, (timeDimension ? &FrameDimensionTime : &FrameDimensionPage), n_frames));
}

static gboolean
gdip_bitmap_select_frame (GpBitmap *bitmap, guint frame, gboolean timeDimension)
{
  if (bitmap == NULL)
    return FALSE;

  return (Ok == GdipImageSelectActiveFrame ((GpImage *)bitmap, (timeDimension ? &FrameDimensionTime : &FrameDimensionPage), frame));
}

/*************************************************************************/
/*************************************************************************/

static void
destroy_gdipcontext (GdipContext *context)
{
  if (context != NULL) {
	if (context->stream != NULL) {
		IStream_Release(context->stream);
		GlobalFree (context->hg);
	}
    g_byte_array_free (context->buffer, TRUE);
    g_free (context);
  }
}

static gpointer
gdk_pixbuf__gdip_image_begin_load (void)
{
  GdipContext *context   = g_new0 (GdipContext, 1);

  context->buffer        = g_byte_array_new ();

  return context;
}

static gboolean
gdk_pixbuf__gdip_image_load_increment (gpointer data,
                                       const guchar *buf, guint size,
                                       GError **error)
{
  GdipContext *context = (GdipContext *)data;
  GByteArray *image_buffer = context->buffer;

  g_byte_array_append (image_buffer, (guint8 *)buf, size);

  return TRUE;
}

static guchar*
gdip_bitmap_to_pixels (GpBitmap *bitmap, GError **error)
{
  guchar *cursor = NULL;
  gint rowstride;
  gboolean has_alpha = FALSE;
  gint n_channels = 0;

  guint width = 0, height = 0, x, y;

  gdip_bitmap_get_size (bitmap, &width, &height);
  gdip_bitmap_get_has_alpha (bitmap, &has_alpha);

  n_channels = has_alpha ? 4 : 3;
  rowstride = width * n_channels;
  rowstride = (rowstride + 3) & ~3; /* Always align rows to 32-bit boundaries */
  cursor = (guchar *)g_try_malloc (rowstride * height);

  if (!cursor) {
    g_set_error_literal (error, GDK_PIXBUF_ERROR, GDK_PIXBUF_ERROR_INSUFFICIENT_MEMORY, _("Couldn't load bitmap"));
    return NULL;
  }

  for (y = 0; y < height; y++) {
    for (x = 0; x < width; x++) {
      ARGB pixel;
      GpStatus status;
      guchar *b = cursor + (y * rowstride + (x * n_channels));
      
      if (Ok != (status = GdipBitmapGetPixel (bitmap, x, y, &pixel))) {
        gdip_set_error_from_gpstatus (error, GDK_PIXBUF_ERROR_FAILED, status);
        g_free (cursor);
        return NULL;
      }
      
      b[0] = (pixel & 0xff0000) >> 16;
      b[1] = (pixel & 0x00ff00) >> 8;
      b[2] = (pixel & 0x0000ff) >> 0;
      
      if (has_alpha)      
        b[3] = (pixel & 0xff000000) >> 24;
    }
  }

  return cursor;
}

static gboolean
stop_load (GpBitmap *bitmap, GdipContext *context, GError **error)
{
  guint       n_frames = 1, i;

  gdip_bitmap_get_n_frames (bitmap, &n_frames, TRUE);

  for (i = 0; i < n_frames; i++) {
    guchar *pixels;
    guint frame_delay = 0;

    gdip_bitmap_select_frame (bitmap, i, TRUE);
    
    pixels = gdip_bitmap_to_pixels (bitmap, error);
    
    if (!pixels) {
      GdipDisposeImage ((GpImage *)bitmap);
      destroy_gdipcontext (context);
      return FALSE;
    }

	g_free (pixels);
  }

  GdipDisposeImage ((GpImage *)bitmap);
  destroy_gdipcontext (context);
  
  return TRUE;
}

static gboolean
gdk_pixbuf__gdip_image_stop_load (gpointer data, GError **error)
{
  GdipContext *context = (GdipContext *)data;
  GpBitmap    *bitmap = NULL;

  bitmap = gdip_buffer_to_bitmap (context, error);

  if (!bitmap) {
    destroy_gdipcontext (context);
    g_set_error_literal (error, GDK_PIXBUF_ERROR, GDK_PIXBUF_ERROR_CORRUPT_IMAGE, _("Couldn't load bitmap"));
    return FALSE;
  }

  return stop_load (bitmap, context, error);
}

static gboolean
gdk_pixbuf__gdip_image_stop_vector_load (gpointer data, GError **error)
{
  GdipContext *context = (GdipContext *)data;

  GpImage *metafile;
  GpGraphics *graphics;
  GpBitmap *bitmap;
  GpStatus status;
  float metafile_xres, metafile_yres;
  guint width, height;

  metafile = gdip_buffer_to_image (context, error);
  if (!metafile) {
    destroy_gdipcontext (context);
    g_set_error_literal (error, GDK_PIXBUF_ERROR, GDK_PIXBUF_ERROR_CORRUPT_IMAGE, _("Couldn't load metafile"));
    return FALSE;
  }

  GdipGetImageWidth (metafile, &width);
  GdipGetImageHeight (metafile, &height);

  status = GdipCreateBitmapFromScan0 (width, height, 0, PixelFormat32bppARGB, NULL, &bitmap);
  if (Ok != status) {
    gdip_set_error_from_gpstatus (error, GDK_PIXBUF_ERROR_FAILED, status);
    GdipDisposeImage (metafile);
    
    return FALSE;
  }

  GdipGetImageHorizontalResolution (metafile, &metafile_xres);
  GdipGetImageVerticalResolution (metafile, &metafile_yres);
  GdipBitmapSetResolution (bitmap, metafile_xres, metafile_yres);

  status = GdipGetImageGraphicsContext ((GpImage *)bitmap, &graphics);
  if (Ok != status) {
    gdip_set_error_from_gpstatus (error, GDK_PIXBUF_ERROR_FAILED, status);
    GdipDisposeImage ((GpImage *)bitmap);
    GdipDisposeImage (metafile);
    
    return FALSE;
  }
  
  /* gotta clear the bitmap */
  GdipGraphicsClear (graphics, 0xffffffff);
  
  status = GdipDrawImageI (graphics, metafile, 0, 0);
  if (Ok != status) {
    gdip_set_error_from_gpstatus (error, GDK_PIXBUF_ERROR_FAILED, status);
    GdipDeleteGraphics (graphics);
    GdipDisposeImage ((GpImage *)bitmap);
    GdipDisposeImage (metafile);
    
    return FALSE;
  }
  
  GdipFlush (graphics, 1);
  
  GdipDeleteGraphics (graphics);
  GdipDisposeImage (metafile);

  return stop_load (bitmap, context, error);
}

/**********************************************/
/**********************************************/

#define LOAD_BUFFER_SIZE 65536

static void
_gdk_pixbuf_generic_image_load (FILE *f, GError **error)
{
	guchar buffer[LOAD_BUFFER_SIZE];
	size_t length;
	gpointer context;

	context = gdk_pixbuf__gdip_image_begin_load ();

	if (!context)
		return;
		
	while (!feof (f) && !ferror (f)) {
		length = fread (buffer, 1, sizeof (buffer), f);
		if (length > 0)
			if (!gdk_pixbuf__gdip_image_load_increment (context, buffer, length, error)) {
				gdk_pixbuf__gdip_image_stop_load (context, NULL);
				return;
			}
	}
		
	gdk_pixbuf__gdip_image_stop_load (context, error);
}

int main(int argc, char* argv[])
{
	FILE* f;
	gdip_init();

	f = fopen(argv[1], "rb");
	_gdk_pixbuf_generic_image_load(f, NULL);
	fclose(f);

	return 0;
}